/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that controls which HTTP methods CloudFront processes and forwards to your Amazon S3 bucket or your
 * custom origin. There are three choices:
 * </p>
 * <ul>
 * <li>
 * <p>
 * CloudFront forwards only <code>GET</code> and <code>HEAD</code> requests.
 * </p>
 * </li>
 * <li>
 * <p>
 * CloudFront forwards only <code>GET</code>, <code>HEAD</code>, and <code>OPTIONS</code> requests.
 * </p>
 * </li>
 * <li>
 * <p>
 * CloudFront forwards <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and <code>DELETE</code> requests.
 * </p>
 * </li>
 * </ul>
 * <p>
 * If you pick the third choice, you may need to restrict access to your Amazon S3 bucket or to your custom origin so
 * users can't perform operations that you don't want them to. For example, you might not want users to have permissions
 * to delete objects from your origin.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AllowedMethods implements SdkPojo, Serializable, ToCopyableBuilder<AllowedMethods.Builder, AllowedMethods> {
    private static final SdkField<Integer> QUANTITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Quantity")
            .getter(getter(AllowedMethods::quantity))
            .setter(setter(Builder::quantity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Quantity")
                    .unmarshallLocationName("Quantity").build()).build();

    private static final SdkField<List<String>> ITEMS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Items")
            .getter(getter(AllowedMethods::itemsAsStrings))
            .setter(setter(Builder::itemsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Items")
                    .unmarshallLocationName("Items").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Method")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Method").unmarshallLocationName("Method").build()).build())
                            .build()).build();

    private static final SdkField<CachedMethods> CACHED_METHODS_FIELD = SdkField
            .<CachedMethods> builder(MarshallingType.SDK_POJO)
            .memberName("CachedMethods")
            .getter(getter(AllowedMethods::cachedMethods))
            .setter(setter(Builder::cachedMethods))
            .constructor(CachedMethods::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CachedMethods")
                    .unmarshallLocationName("CachedMethods").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(QUANTITY_FIELD, ITEMS_FIELD,
            CACHED_METHODS_FIELD));

    private static final long serialVersionUID = 1L;

    private final Integer quantity;

    private final List<String> items;

    private final CachedMethods cachedMethods;

    private AllowedMethods(BuilderImpl builder) {
        this.quantity = builder.quantity;
        this.items = builder.items;
        this.cachedMethods = builder.cachedMethods;
    }

    /**
     * <p>
     * The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
     * <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
     * <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
     * <code>DELETE</code> requests).
     * </p>
     * 
     * @return The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
     *         <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
     *         <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
     *         <code>DELETE</code> requests).
     */
    public final Integer quantity() {
        return quantity;
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasItems} method.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
     *         origin.
     */
    public final List<Method> items() {
        return MethodsListCopier.copyStringToEnum(items);
    }

    /**
     * For responses, this returns true if the service returned a value for the Items property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasItems() {
        return items != null && !(items instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasItems} method.
     * </p>
     * 
     * @return A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
     *         origin.
     */
    public final List<String> itemsAsStrings() {
        return items;
    }

    /**
     * Returns the value of the CachedMethods property for this object.
     * 
     * @return The value of the CachedMethods property for this object.
     */
    public final CachedMethods cachedMethods() {
        return cachedMethods;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(quantity());
        hashCode = 31 * hashCode + Objects.hashCode(hasItems() ? itemsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(cachedMethods());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AllowedMethods)) {
            return false;
        }
        AllowedMethods other = (AllowedMethods) obj;
        return Objects.equals(quantity(), other.quantity()) && hasItems() == other.hasItems()
                && Objects.equals(itemsAsStrings(), other.itemsAsStrings())
                && Objects.equals(cachedMethods(), other.cachedMethods());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AllowedMethods").add("Quantity", quantity()).add("Items", hasItems() ? itemsAsStrings() : null)
                .add("CachedMethods", cachedMethods()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Quantity":
            return Optional.ofNullable(clazz.cast(quantity()));
        case "Items":
            return Optional.ofNullable(clazz.cast(itemsAsStrings()));
        case "CachedMethods":
            return Optional.ofNullable(clazz.cast(cachedMethods()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AllowedMethods, T> g) {
        return obj -> g.apply((AllowedMethods) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AllowedMethods> {
        /**
         * <p>
         * The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
         * <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
         * <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
         * <code>DELETE</code> requests).
         * </p>
         * 
         * @param quantity
         *        The number of HTTP methods that you want CloudFront to forward to your origin. Valid values are 2 (for
         *        <code>GET</code> and <code>HEAD</code> requests), 3 (for <code>GET</code>, <code>HEAD</code>, and
         *        <code>OPTIONS</code> requests) and 7 (for <code>GET, HEAD, OPTIONS, PUT, PATCH, POST</code>, and
         *        <code>DELETE</code> requests).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder quantity(Integer quantity);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemsWithStrings(Collection<String> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemsWithStrings(String... items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Collection<Method> items);

        /**
         * <p>
         * A complex type that contains the HTTP methods that you want CloudFront to process and forward to your origin.
         * </p>
         * 
         * @param items
         *        A complex type that contains the HTTP methods that you want CloudFront to process and forward to your
         *        origin.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder items(Method... items);

        /**
         * Sets the value of the CachedMethods property for this object.
         *
         * @param cachedMethods
         *        The new value for the CachedMethods property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cachedMethods(CachedMethods cachedMethods);

        /**
         * Sets the value of the CachedMethods property for this object.
         *
         * This is a convenience that creates an instance of the {@link CachedMethods.Builder} avoiding the need to
         * create one manually via {@link CachedMethods#builder()}.
         *
         * When the {@link Consumer} completes, {@link CachedMethods.Builder#build()} is called immediately and its
         * result is passed to {@link #cachedMethods(CachedMethods)}.
         * 
         * @param cachedMethods
         *        a consumer that will call methods on {@link CachedMethods.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cachedMethods(CachedMethods)
         */
        default Builder cachedMethods(Consumer<CachedMethods.Builder> cachedMethods) {
            return cachedMethods(CachedMethods.builder().applyMutation(cachedMethods).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private Integer quantity;

        private List<String> items = DefaultSdkAutoConstructList.getInstance();

        private CachedMethods cachedMethods;

        private BuilderImpl() {
        }

        private BuilderImpl(AllowedMethods model) {
            quantity(model.quantity);
            itemsWithStrings(model.items);
            cachedMethods(model.cachedMethods);
        }

        public final Integer getQuantity() {
            return quantity;
        }

        public final void setQuantity(Integer quantity) {
            this.quantity = quantity;
        }

        @Override
        @Transient
        public final Builder quantity(Integer quantity) {
            this.quantity = quantity;
            return this;
        }

        public final Collection<String> getItems() {
            if (items instanceof SdkAutoConstructList) {
                return null;
            }
            return items;
        }

        public final void setItems(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
        }

        @Override
        @Transient
        public final Builder itemsWithStrings(Collection<String> items) {
            this.items = MethodsListCopier.copy(items);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder itemsWithStrings(String... items) {
            itemsWithStrings(Arrays.asList(items));
            return this;
        }

        @Override
        @Transient
        public final Builder items(Collection<Method> items) {
            this.items = MethodsListCopier.copyEnumToString(items);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder items(Method... items) {
            items(Arrays.asList(items));
            return this;
        }

        public final CachedMethods.Builder getCachedMethods() {
            return cachedMethods != null ? cachedMethods.toBuilder() : null;
        }

        public final void setCachedMethods(CachedMethods.BuilderImpl cachedMethods) {
            this.cachedMethods = cachedMethods != null ? cachedMethods.build() : null;
        }

        @Override
        @Transient
        public final Builder cachedMethods(CachedMethods cachedMethods) {
            this.cachedMethods = cachedMethods;
            return this;
        }

        @Override
        public AllowedMethods build() {
            return new AllowedMethods(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
