/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A scanned resource returned by <code>ListResourceScanResources</code> or
 * <code>ListResourceScanRelatedResources</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScannedResource implements SdkPojo, Serializable, ToCopyableBuilder<ScannedResource.Builder, ScannedResource> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ScannedResource::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Map<String, String>> RESOURCE_IDENTIFIER_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ResourceIdentifier")
            .getter(getter(ScannedResource::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifier").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Boolean> MANAGED_BY_STACK_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ManagedByStack").getter(getter(ScannedResource::managedByStack)).setter(setter(Builder::managedByStack))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ManagedByStack").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            RESOURCE_IDENTIFIER_FIELD, MANAGED_BY_STACK_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final Map<String, String> resourceIdentifier;

    private final Boolean managedByStack;

    private ScannedResource(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.managedByStack = builder.managedByStack;
    }

    /**
     * <p>
     * The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported resources, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html">
     * Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
     * </p>
     * 
     * @return The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported resources,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
     *         >Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceIdentifier property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceIdentifier() {
        return resourceIdentifier != null && !(resourceIdentifier instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of up to 256 key-value pairs that identifies for the scanned resource. The key is the name of one of the
     * primary identifiers for the resource. (Primary identifiers are specified in the <code>primaryIdentifier</code>
     * list in the resource schema.) The value is the value of that primary identifier. For example, for a
     * <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is <code>TableName</code> so the key-value
     * pair could be <code>"TableName": "MyDDBTable"</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
     * >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceIdentifier} method.
     * </p>
     * 
     * @return A list of up to 256 key-value pairs that identifies for the scanned resource. The key is the name of one
     *         of the primary identifiers for the resource. (Primary identifiers are specified in the
     *         <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary
     *         identifier. For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is
     *         <code>TableName</code> so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
     *         >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
     */
    public final Map<String, String> resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * If <code>true</code>, the resource is managed by a CloudFormation stack.
     * </p>
     * 
     * @return If <code>true</code>, the resource is managed by a CloudFormation stack.
     */
    public final Boolean managedByStack() {
        return managedByStack;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIdentifier() ? resourceIdentifier() : null);
        hashCode = 31 * hashCode + Objects.hashCode(managedByStack());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScannedResource)) {
            return false;
        }
        ScannedResource other = (ScannedResource) obj;
        return Objects.equals(resourceType(), other.resourceType()) && hasResourceIdentifier() == other.hasResourceIdentifier()
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && Objects.equals(managedByStack(), other.managedByStack());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScannedResource").add("ResourceType", resourceType())
                .add("ResourceIdentifier", hasResourceIdentifier() ? resourceIdentifier() : null)
                .add("ManagedByStack", managedByStack()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "ManagedByStack":
            return Optional.ofNullable(clazz.cast(managedByStack()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("ResourceIdentifier", RESOURCE_IDENTIFIER_FIELD);
        map.put("ManagedByStack", MANAGED_BY_STACK_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ScannedResource, T> g) {
        return obj -> g.apply((ScannedResource) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScannedResource> {
        /**
         * <p>
         * The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported resources, see
         * <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
         * >Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
         * </p>
         * 
         * @param resourceType
         *        The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported
         *        resources, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
         *        >Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * A list of up to 256 key-value pairs that identifies for the scanned resource. The key is the name of one of
         * the primary identifiers for the resource. (Primary identifiers are specified in the
         * <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary
         * identifier. For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is
         * <code>TableName</code> so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
         * >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
         * </p>
         * 
         * @param resourceIdentifier
         *        A list of up to 256 key-value pairs that identifies for the scanned resource. The key is the name of
         *        one of the primary identifiers for the resource. (Primary identifiers are specified in the
         *        <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary
         *        identifier. For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is
         *        <code>TableName</code> so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
         *        >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(Map<String, String> resourceIdentifier);

        /**
         * <p>
         * If <code>true</code>, the resource is managed by a CloudFormation stack.
         * </p>
         * 
         * @param managedByStack
         *        If <code>true</code>, the resource is managed by a CloudFormation stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder managedByStack(Boolean managedByStack);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private Map<String, String> resourceIdentifier = DefaultSdkAutoConstructMap.getInstance();

        private Boolean managedByStack;

        private BuilderImpl() {
        }

        private BuilderImpl(ScannedResource model) {
            resourceType(model.resourceType);
            resourceIdentifier(model.resourceIdentifier);
            managedByStack(model.managedByStack);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Map<String, String> getResourceIdentifier() {
            if (resourceIdentifier instanceof SdkAutoConstructMap) {
                return null;
            }
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(Map<String, String> resourceIdentifier) {
            this.resourceIdentifier = JazzResourceIdentifierPropertiesCopier.copy(resourceIdentifier);
        }

        @Override
        public final Builder resourceIdentifier(Map<String, String> resourceIdentifier) {
            this.resourceIdentifier = JazzResourceIdentifierPropertiesCopier.copy(resourceIdentifier);
            return this;
        }

        public final Boolean getManagedByStack() {
            return managedByStack;
        }

        public final void setManagedByStack(Boolean managedByStack) {
            this.managedByStack = managedByStack;
        }

        @Override
        public final Builder managedByStack(Boolean managedByStack) {
            this.managedByStack = managedByStack;
            return this;
        }

        @Override
        public ScannedResource build() {
            return new ScannedResource(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
