/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about a resource in a generated template
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceDetail implements SdkPojo, Serializable, ToCopyableBuilder<ResourceDetail.Builder, ResourceDetail> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ResourceDetail::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> LOGICAL_RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogicalResourceId").getter(getter(ResourceDetail::logicalResourceId))
            .setter(setter(Builder::logicalResourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceId").build()).build();

    private static final SdkField<Map<String, String>> RESOURCE_IDENTIFIER_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("ResourceIdentifier")
            .getter(getter(ResourceDetail::resourceIdentifier))
            .setter(setter(Builder::resourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifier").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceStatus").getter(getter(ResourceDetail::resourceStatusAsString))
            .setter(setter(Builder::resourceStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceStatus").build()).build();

    private static final SdkField<String> RESOURCE_STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceStatusReason").getter(getter(ResourceDetail::resourceStatusReason))
            .setter(setter(Builder::resourceStatusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceStatusReason").build())
            .build();

    private static final SdkField<List<WarningDetail>> WARNINGS_FIELD = SdkField
            .<List<WarningDetail>> builder(MarshallingType.LIST)
            .memberName("Warnings")
            .getter(getter(ResourceDetail::warnings))
            .setter(setter(Builder::warnings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Warnings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<WarningDetail> builder(MarshallingType.SDK_POJO)
                                            .constructor(WarningDetail::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            LOGICAL_RESOURCE_ID_FIELD, RESOURCE_IDENTIFIER_FIELD, RESOURCE_STATUS_FIELD, RESOURCE_STATUS_REASON_FIELD,
            WARNINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final String logicalResourceId;

    private final Map<String, String> resourceIdentifier;

    private final String resourceStatus;

    private final String resourceStatusReason;

    private final List<WarningDetail> warnings;

    private ResourceDetail(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.logicalResourceId = builder.logicalResourceId;
        this.resourceIdentifier = builder.resourceIdentifier;
        this.resourceStatus = builder.resourceStatus;
        this.resourceStatusReason = builder.resourceStatusReason;
        this.warnings = builder.warnings;
    }

    /**
     * <p>
     * The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported resources, see <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html">
     * Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
     * </p>
     * 
     * @return The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported resources,
     *         see <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
     *         >Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The logical id for this resource in the final generated template.
     * </p>
     * 
     * @return The logical id for this resource in the final generated template.
     */
    public final String logicalResourceId() {
        return logicalResourceId;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceIdentifier property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceIdentifier() {
        return resourceIdentifier != null && !(resourceIdentifier instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * A list of up to 256 key-value pairs that identifies the resource in the generated template. The key is the name
     * of one of the primary identifiers for the resource. (Primary identifiers are specified in the
     * <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary identifier.
     * For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is <code>TableName</code>
     * so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more information, see <a href=
     * "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
     * >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceIdentifier} method.
     * </p>
     * 
     * @return A list of up to 256 key-value pairs that identifies the resource in the generated template. The key is
     *         the name of one of the primary identifiers for the resource. (Primary identifiers are specified in the
     *         <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary
     *         identifier. For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is
     *         <code>TableName</code> so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
     *         >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
     */
    public final Map<String, String> resourceIdentifier() {
        return resourceIdentifier;
    }

    /**
     * <p>
     * Status of the processing of a resource in a generated template.
     * </p>
     * <dl>
     * <dt>InProgress</dt>
     * <dd>
     * <p>
     * The resource processing is still in progress.
     * </p>
     * </dd>
     * <dt>Complete</dt>
     * <dd>
     * <p>
     * The resource processing is complete.
     * </p>
     * </dd>
     * <dt>Pending</dt>
     * <dd>
     * <p>
     * The resource processing is pending.
     * </p>
     * </dd>
     * <dt>Failed</dt>
     * <dd>
     * <p>
     * The resource processing has failed.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceStatus}
     * will return {@link GeneratedTemplateResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #resourceStatusAsString}.
     * </p>
     * 
     * @return Status of the processing of a resource in a generated template.</p>
     *         <dl>
     *         <dt>InProgress</dt>
     *         <dd>
     *         <p>
     *         The resource processing is still in progress.
     *         </p>
     *         </dd>
     *         <dt>Complete</dt>
     *         <dd>
     *         <p>
     *         The resource processing is complete.
     *         </p>
     *         </dd>
     *         <dt>Pending</dt>
     *         <dd>
     *         <p>
     *         The resource processing is pending.
     *         </p>
     *         </dd>
     *         <dt>Failed</dt>
     *         <dd>
     *         <p>
     *         The resource processing has failed.
     *         </p>
     *         </dd>
     * @see GeneratedTemplateResourceStatus
     */
    public final GeneratedTemplateResourceStatus resourceStatus() {
        return GeneratedTemplateResourceStatus.fromValue(resourceStatus);
    }

    /**
     * <p>
     * Status of the processing of a resource in a generated template.
     * </p>
     * <dl>
     * <dt>InProgress</dt>
     * <dd>
     * <p>
     * The resource processing is still in progress.
     * </p>
     * </dd>
     * <dt>Complete</dt>
     * <dd>
     * <p>
     * The resource processing is complete.
     * </p>
     * </dd>
     * <dt>Pending</dt>
     * <dd>
     * <p>
     * The resource processing is pending.
     * </p>
     * </dd>
     * <dt>Failed</dt>
     * <dd>
     * <p>
     * The resource processing has failed.
     * </p>
     * </dd>
     * </dl>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceStatus}
     * will return {@link GeneratedTemplateResourceStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #resourceStatusAsString}.
     * </p>
     * 
     * @return Status of the processing of a resource in a generated template.</p>
     *         <dl>
     *         <dt>InProgress</dt>
     *         <dd>
     *         <p>
     *         The resource processing is still in progress.
     *         </p>
     *         </dd>
     *         <dt>Complete</dt>
     *         <dd>
     *         <p>
     *         The resource processing is complete.
     *         </p>
     *         </dd>
     *         <dt>Pending</dt>
     *         <dd>
     *         <p>
     *         The resource processing is pending.
     *         </p>
     *         </dd>
     *         <dt>Failed</dt>
     *         <dd>
     *         <p>
     *         The resource processing has failed.
     *         </p>
     *         </dd>
     * @see GeneratedTemplateResourceStatus
     */
    public final String resourceStatusAsString() {
        return resourceStatus;
    }

    /**
     * <p>
     * The reason for the resource detail, providing more information if a failure happened.
     * </p>
     * 
     * @return The reason for the resource detail, providing more information if a failure happened.
     */
    public final String resourceStatusReason() {
        return resourceStatusReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Warnings property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasWarnings() {
        return warnings != null && !(warnings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The warnings generated for this resource.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasWarnings} method.
     * </p>
     * 
     * @return The warnings generated for this resource.
     */
    public final List<WarningDetail> warnings() {
        return warnings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(logicalResourceId());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIdentifier() ? resourceIdentifier() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceStatusReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasWarnings() ? warnings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceDetail)) {
            return false;
        }
        ResourceDetail other = (ResourceDetail) obj;
        return Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(logicalResourceId(), other.logicalResourceId())
                && hasResourceIdentifier() == other.hasResourceIdentifier()
                && Objects.equals(resourceIdentifier(), other.resourceIdentifier())
                && Objects.equals(resourceStatusAsString(), other.resourceStatusAsString())
                && Objects.equals(resourceStatusReason(), other.resourceStatusReason()) && hasWarnings() == other.hasWarnings()
                && Objects.equals(warnings(), other.warnings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceDetail").add("ResourceType", resourceType())
                .add("LogicalResourceId", logicalResourceId())
                .add("ResourceIdentifier", hasResourceIdentifier() ? resourceIdentifier() : null)
                .add("ResourceStatus", resourceStatusAsString()).add("ResourceStatusReason", resourceStatusReason())
                .add("Warnings", hasWarnings() ? warnings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "LogicalResourceId":
            return Optional.ofNullable(clazz.cast(logicalResourceId()));
        case "ResourceIdentifier":
            return Optional.ofNullable(clazz.cast(resourceIdentifier()));
        case "ResourceStatus":
            return Optional.ofNullable(clazz.cast(resourceStatusAsString()));
        case "ResourceStatusReason":
            return Optional.ofNullable(clazz.cast(resourceStatusReason()));
        case "Warnings":
            return Optional.ofNullable(clazz.cast(warnings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("LogicalResourceId", LOGICAL_RESOURCE_ID_FIELD);
        map.put("ResourceIdentifier", RESOURCE_IDENTIFIER_FIELD);
        map.put("ResourceStatus", RESOURCE_STATUS_FIELD);
        map.put("ResourceStatusReason", RESOURCE_STATUS_REASON_FIELD);
        map.put("Warnings", WARNINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ResourceDetail, T> g) {
        return obj -> g.apply((ResourceDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceDetail> {
        /**
         * <p>
         * The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported resources, see
         * <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
         * >Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
         * </p>
         * 
         * @param resourceType
         *        The type of the resource, such as <code>AWS::DynamoDB::Table</code>. For the list of supported
         *        resources, see <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resource-import-supported-resources.html"
         *        >Resource type support for imports and drift detection</a> In the <i>CloudFormation User Guide</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The logical id for this resource in the final generated template.
         * </p>
         * 
         * @param logicalResourceId
         *        The logical id for this resource in the final generated template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceId(String logicalResourceId);

        /**
         * <p>
         * A list of up to 256 key-value pairs that identifies the resource in the generated template. The key is the
         * name of one of the primary identifiers for the resource. (Primary identifiers are specified in the
         * <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary
         * identifier. For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is
         * <code>TableName</code> so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more
         * information, see <a href=
         * "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
         * >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
         * </p>
         * 
         * @param resourceIdentifier
         *        A list of up to 256 key-value pairs that identifies the resource in the generated template. The key is
         *        the name of one of the primary identifiers for the resource. (Primary identifiers are specified in the
         *        <code>primaryIdentifier</code> list in the resource schema.) The value is the value of that primary
         *        identifier. For example, for a <code>AWS::DynamoDB::Table</code> resource, the primary identifiers is
         *        <code>TableName</code> so the key-value pair could be <code>"TableName": "MyDDBTable"</code>. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/cloudformation-cli/latest/userguide/resource-type-schema.html#schema-properties-primaryidentifier"
         *        >primaryIdentifier</a> in the <i>CloudFormation Command Line Interface (CLI) User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifier(Map<String, String> resourceIdentifier);

        /**
         * <p>
         * Status of the processing of a resource in a generated template.
         * </p>
         * <dl>
         * <dt>InProgress</dt>
         * <dd>
         * <p>
         * The resource processing is still in progress.
         * </p>
         * </dd>
         * <dt>Complete</dt>
         * <dd>
         * <p>
         * The resource processing is complete.
         * </p>
         * </dd>
         * <dt>Pending</dt>
         * <dd>
         * <p>
         * The resource processing is pending.
         * </p>
         * </dd>
         * <dt>Failed</dt>
         * <dd>
         * <p>
         * The resource processing has failed.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param resourceStatus
         *        Status of the processing of a resource in a generated template.</p>
         *        <dl>
         *        <dt>InProgress</dt>
         *        <dd>
         *        <p>
         *        The resource processing is still in progress.
         *        </p>
         *        </dd>
         *        <dt>Complete</dt>
         *        <dd>
         *        <p>
         *        The resource processing is complete.
         *        </p>
         *        </dd>
         *        <dt>Pending</dt>
         *        <dd>
         *        <p>
         *        The resource processing is pending.
         *        </p>
         *        </dd>
         *        <dt>Failed</dt>
         *        <dd>
         *        <p>
         *        The resource processing has failed.
         *        </p>
         *        </dd>
         * @see GeneratedTemplateResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeneratedTemplateResourceStatus
         */
        Builder resourceStatus(String resourceStatus);

        /**
         * <p>
         * Status of the processing of a resource in a generated template.
         * </p>
         * <dl>
         * <dt>InProgress</dt>
         * <dd>
         * <p>
         * The resource processing is still in progress.
         * </p>
         * </dd>
         * <dt>Complete</dt>
         * <dd>
         * <p>
         * The resource processing is complete.
         * </p>
         * </dd>
         * <dt>Pending</dt>
         * <dd>
         * <p>
         * The resource processing is pending.
         * </p>
         * </dd>
         * <dt>Failed</dt>
         * <dd>
         * <p>
         * The resource processing has failed.
         * </p>
         * </dd>
         * </dl>
         * 
         * @param resourceStatus
         *        Status of the processing of a resource in a generated template.</p>
         *        <dl>
         *        <dt>InProgress</dt>
         *        <dd>
         *        <p>
         *        The resource processing is still in progress.
         *        </p>
         *        </dd>
         *        <dt>Complete</dt>
         *        <dd>
         *        <p>
         *        The resource processing is complete.
         *        </p>
         *        </dd>
         *        <dt>Pending</dt>
         *        <dd>
         *        <p>
         *        The resource processing is pending.
         *        </p>
         *        </dd>
         *        <dt>Failed</dt>
         *        <dd>
         *        <p>
         *        The resource processing has failed.
         *        </p>
         *        </dd>
         * @see GeneratedTemplateResourceStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GeneratedTemplateResourceStatus
         */
        Builder resourceStatus(GeneratedTemplateResourceStatus resourceStatus);

        /**
         * <p>
         * The reason for the resource detail, providing more information if a failure happened.
         * </p>
         * 
         * @param resourceStatusReason
         *        The reason for the resource detail, providing more information if a failure happened.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceStatusReason(String resourceStatusReason);

        /**
         * <p>
         * The warnings generated for this resource.
         * </p>
         * 
         * @param warnings
         *        The warnings generated for this resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warnings(Collection<WarningDetail> warnings);

        /**
         * <p>
         * The warnings generated for this resource.
         * </p>
         * 
         * @param warnings
         *        The warnings generated for this resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warnings(WarningDetail... warnings);

        /**
         * <p>
         * The warnings generated for this resource.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudformation.model.WarningDetail.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.cloudformation.model.WarningDetail#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudformation.model.WarningDetail.Builder#build()} is called
         * immediately and its result is passed to {@link #warnings(List<WarningDetail>)}.
         * 
         * @param warnings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudformation.model.WarningDetail.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #warnings(java.util.Collection<WarningDetail>)
         */
        Builder warnings(Consumer<WarningDetail.Builder>... warnings);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private String logicalResourceId;

        private Map<String, String> resourceIdentifier = DefaultSdkAutoConstructMap.getInstance();

        private String resourceStatus;

        private String resourceStatusReason;

        private List<WarningDetail> warnings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceDetail model) {
            resourceType(model.resourceType);
            logicalResourceId(model.logicalResourceId);
            resourceIdentifier(model.resourceIdentifier);
            resourceStatus(model.resourceStatus);
            resourceStatusReason(model.resourceStatusReason);
            warnings(model.warnings);
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final String getLogicalResourceId() {
            return logicalResourceId;
        }

        public final void setLogicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
        }

        @Override
        public final Builder logicalResourceId(String logicalResourceId) {
            this.logicalResourceId = logicalResourceId;
            return this;
        }

        public final Map<String, String> getResourceIdentifier() {
            if (resourceIdentifier instanceof SdkAutoConstructMap) {
                return null;
            }
            return resourceIdentifier;
        }

        public final void setResourceIdentifier(Map<String, String> resourceIdentifier) {
            this.resourceIdentifier = ResourceIdentifierPropertiesCopier.copy(resourceIdentifier);
        }

        @Override
        public final Builder resourceIdentifier(Map<String, String> resourceIdentifier) {
            this.resourceIdentifier = ResourceIdentifierPropertiesCopier.copy(resourceIdentifier);
            return this;
        }

        public final String getResourceStatus() {
            return resourceStatus;
        }

        public final void setResourceStatus(String resourceStatus) {
            this.resourceStatus = resourceStatus;
        }

        @Override
        public final Builder resourceStatus(String resourceStatus) {
            this.resourceStatus = resourceStatus;
            return this;
        }

        @Override
        public final Builder resourceStatus(GeneratedTemplateResourceStatus resourceStatus) {
            this.resourceStatus(resourceStatus == null ? null : resourceStatus.toString());
            return this;
        }

        public final String getResourceStatusReason() {
            return resourceStatusReason;
        }

        public final void setResourceStatusReason(String resourceStatusReason) {
            this.resourceStatusReason = resourceStatusReason;
        }

        @Override
        public final Builder resourceStatusReason(String resourceStatusReason) {
            this.resourceStatusReason = resourceStatusReason;
            return this;
        }

        public final List<WarningDetail.Builder> getWarnings() {
            List<WarningDetail.Builder> result = WarningDetailsCopier.copyToBuilder(this.warnings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setWarnings(Collection<WarningDetail.BuilderImpl> warnings) {
            this.warnings = WarningDetailsCopier.copyFromBuilder(warnings);
        }

        @Override
        public final Builder warnings(Collection<WarningDetail> warnings) {
            this.warnings = WarningDetailsCopier.copy(warnings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder warnings(WarningDetail... warnings) {
            warnings(Arrays.asList(warnings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder warnings(Consumer<WarningDetail.Builder>... warnings) {
            warnings(Stream.of(warnings).map(c -> WarningDetail.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public ResourceDetail build() {
            return new ResourceDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
