/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the target resources of a specific type in your import template (for example, all
 * <code>AWS::S3::Bucket</code> resources) and the properties you can provide during the import to identify resources of
 * that type.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResourceIdentifierSummary implements SdkPojo, Serializable,
        ToCopyableBuilder<ResourceIdentifierSummary.Builder, ResourceIdentifierSummary> {
    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(ResourceIdentifierSummary::resourceType))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<List<String>> LOGICAL_RESOURCE_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LogicalResourceIds")
            .getter(getter(ResourceIdentifierSummary::logicalResourceIds))
            .setter(setter(Builder::logicalResourceIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalResourceIds").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> RESOURCE_IDENTIFIERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ResourceIdentifiers")
            .getter(getter(ResourceIdentifierSummary::resourceIdentifiers))
            .setter(setter(Builder::resourceIdentifiers))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceIdentifiers").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_TYPE_FIELD,
            LOGICAL_RESOURCE_IDS_FIELD, RESOURCE_IDENTIFIERS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String resourceType;

    private final List<String> logicalResourceIds;

    private final List<String> resourceIdentifiers;

    private ResourceIdentifierSummary(BuilderImpl builder) {
        this.resourceType = builder.resourceType;
        this.logicalResourceIds = builder.logicalResourceIds;
        this.resourceIdentifiers = builder.resourceIdentifiers;
    }

    /**
     * <p>
     * The template resource type of the target resources, such as <code>AWS::S3::Bucket</code>.
     * </p>
     * 
     * @return The template resource type of the target resources, such as <code>AWS::S3::Bucket</code>.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * Returns true if the LogicalResourceIds property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasLogicalResourceIds() {
        return logicalResourceIds != null && !(logicalResourceIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The logical IDs of the target resources of the specified <code>ResourceType</code>, as defined in the import
     * template.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasLogicalResourceIds()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The logical IDs of the target resources of the specified <code>ResourceType</code>, as defined in the
     *         import template.
     */
    public final List<String> logicalResourceIds() {
        return logicalResourceIds;
    }

    /**
     * Returns true if the ResourceIdentifiers property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasResourceIdentifiers() {
        return resourceIdentifiers != null && !(resourceIdentifiers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource properties you can provide during the import to identify your target resources. For example,
     * <code>BucketName</code> is a possible identifier property for <code>AWS::S3::Bucket</code> resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasResourceIdentifiers()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The resource properties you can provide during the import to identify your target resources. For example,
     *         <code>BucketName</code> is a possible identifier property for <code>AWS::S3::Bucket</code> resources.
     */
    public final List<String> resourceIdentifiers() {
        return resourceIdentifiers;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(hasLogicalResourceIds() ? logicalResourceIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceIdentifiers() ? resourceIdentifiers() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResourceIdentifierSummary)) {
            return false;
        }
        ResourceIdentifierSummary other = (ResourceIdentifierSummary) obj;
        return Objects.equals(resourceType(), other.resourceType()) && hasLogicalResourceIds() == other.hasLogicalResourceIds()
                && Objects.equals(logicalResourceIds(), other.logicalResourceIds())
                && hasResourceIdentifiers() == other.hasResourceIdentifiers()
                && Objects.equals(resourceIdentifiers(), other.resourceIdentifiers());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResourceIdentifierSummary").add("ResourceType", resourceType())
                .add("LogicalResourceIds", hasLogicalResourceIds() ? logicalResourceIds() : null)
                .add("ResourceIdentifiers", hasResourceIdentifiers() ? resourceIdentifiers() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "LogicalResourceIds":
            return Optional.ofNullable(clazz.cast(logicalResourceIds()));
        case "ResourceIdentifiers":
            return Optional.ofNullable(clazz.cast(resourceIdentifiers()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResourceIdentifierSummary, T> g) {
        return obj -> g.apply((ResourceIdentifierSummary) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResourceIdentifierSummary> {
        /**
         * <p>
         * The template resource type of the target resources, such as <code>AWS::S3::Bucket</code>.
         * </p>
         * 
         * @param resourceType
         *        The template resource type of the target resources, such as <code>AWS::S3::Bucket</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The logical IDs of the target resources of the specified <code>ResourceType</code>, as defined in the import
         * template.
         * </p>
         * 
         * @param logicalResourceIds
         *        The logical IDs of the target resources of the specified <code>ResourceType</code>, as defined in the
         *        import template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceIds(Collection<String> logicalResourceIds);

        /**
         * <p>
         * The logical IDs of the target resources of the specified <code>ResourceType</code>, as defined in the import
         * template.
         * </p>
         * 
         * @param logicalResourceIds
         *        The logical IDs of the target resources of the specified <code>ResourceType</code>, as defined in the
         *        import template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder logicalResourceIds(String... logicalResourceIds);

        /**
         * <p>
         * The resource properties you can provide during the import to identify your target resources. For example,
         * <code>BucketName</code> is a possible identifier property for <code>AWS::S3::Bucket</code> resources.
         * </p>
         * 
         * @param resourceIdentifiers
         *        The resource properties you can provide during the import to identify your target resources. For
         *        example, <code>BucketName</code> is a possible identifier property for <code>AWS::S3::Bucket</code>
         *        resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifiers(Collection<String> resourceIdentifiers);

        /**
         * <p>
         * The resource properties you can provide during the import to identify your target resources. For example,
         * <code>BucketName</code> is a possible identifier property for <code>AWS::S3::Bucket</code> resources.
         * </p>
         * 
         * @param resourceIdentifiers
         *        The resource properties you can provide during the import to identify your target resources. For
         *        example, <code>BucketName</code> is a possible identifier property for <code>AWS::S3::Bucket</code>
         *        resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceIdentifiers(String... resourceIdentifiers);
    }

    static final class BuilderImpl implements Builder {
        private String resourceType;

        private List<String> logicalResourceIds = DefaultSdkAutoConstructList.getInstance();

        private List<String> resourceIdentifiers = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ResourceIdentifierSummary model) {
            resourceType(model.resourceType);
            logicalResourceIds(model.logicalResourceIds);
            resourceIdentifiers(model.resourceIdentifiers);
        }

        public final String getResourceType() {
            return resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        public final Collection<String> getLogicalResourceIds() {
            if (logicalResourceIds instanceof SdkAutoConstructList) {
                return null;
            }
            return logicalResourceIds;
        }

        @Override
        public final Builder logicalResourceIds(Collection<String> logicalResourceIds) {
            this.logicalResourceIds = LogicalResourceIdsCopier.copy(logicalResourceIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder logicalResourceIds(String... logicalResourceIds) {
            logicalResourceIds(Arrays.asList(logicalResourceIds));
            return this;
        }

        public final void setLogicalResourceIds(Collection<String> logicalResourceIds) {
            this.logicalResourceIds = LogicalResourceIdsCopier.copy(logicalResourceIds);
        }

        public final Collection<String> getResourceIdentifiers() {
            if (resourceIdentifiers instanceof SdkAutoConstructList) {
                return null;
            }
            return resourceIdentifiers;
        }

        @Override
        public final Builder resourceIdentifiers(Collection<String> resourceIdentifiers) {
            this.resourceIdentifiers = ResourceIdentifiersCopier.copy(resourceIdentifiers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceIdentifiers(String... resourceIdentifiers) {
            resourceIdentifiers(Arrays.asList(resourceIdentifiers));
            return this;
        }

        public final void setResourceIdentifiers(Collection<String> resourceIdentifiers) {
            this.resourceIdentifiers = ResourceIdentifiersCopier.copy(resourceIdentifiers);
        }

        @Override
        public ResourceIdentifierSummary build() {
            return new ResourceIdentifierSummary(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
