/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cleanrooms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Properties that define how a specific data column should be handled during synthetic data generation, including its
 * name, type, and role in predictive modeling.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SyntheticDataColumnProperties implements SdkPojo, Serializable,
        ToCopyableBuilder<SyntheticDataColumnProperties.Builder, SyntheticDataColumnProperties> {
    private static final SdkField<String> COLUMN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("columnName").getter(getter(SyntheticDataColumnProperties::columnName))
            .setter(setter(Builder::columnName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("columnName").build()).build();

    private static final SdkField<String> COLUMN_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("columnType").getter(getter(SyntheticDataColumnProperties::columnTypeAsString))
            .setter(setter(Builder::columnType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("columnType").build()).build();

    private static final SdkField<Boolean> IS_PREDICTIVE_VALUE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("isPredictiveValue").getter(getter(SyntheticDataColumnProperties::isPredictiveValue))
            .setter(setter(Builder::isPredictiveValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("isPredictiveValue").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COLUMN_NAME_FIELD,
            COLUMN_TYPE_FIELD, IS_PREDICTIVE_VALUE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String columnName;

    private final String columnType;

    private final Boolean isPredictiveValue;

    private SyntheticDataColumnProperties(BuilderImpl builder) {
        this.columnName = builder.columnName;
        this.columnType = builder.columnType;
        this.isPredictiveValue = builder.isPredictiveValue;
    }

    /**
     * <p>
     * The name of the data column as it appears in the dataset.
     * </p>
     * 
     * @return The name of the data column as it appears in the dataset.
     */
    public final String columnName() {
        return columnName;
    }

    /**
     * <p>
     * The data type of the column, which determines how the synthetic data generation algorithm processes and
     * synthesizes values for this column.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #columnType} will
     * return {@link SyntheticDataColumnType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #columnTypeAsString}.
     * </p>
     * 
     * @return The data type of the column, which determines how the synthetic data generation algorithm processes and
     *         synthesizes values for this column.
     * @see SyntheticDataColumnType
     */
    public final SyntheticDataColumnType columnType() {
        return SyntheticDataColumnType.fromValue(columnType);
    }

    /**
     * <p>
     * The data type of the column, which determines how the synthetic data generation algorithm processes and
     * synthesizes values for this column.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #columnType} will
     * return {@link SyntheticDataColumnType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #columnTypeAsString}.
     * </p>
     * 
     * @return The data type of the column, which determines how the synthetic data generation algorithm processes and
     *         synthesizes values for this column.
     * @see SyntheticDataColumnType
     */
    public final String columnTypeAsString() {
        return columnType;
    }

    /**
     * <p>
     * Indicates if this column contains predictive values that should be treated as target variables in machine
     * learning models. This affects how the synthetic data generation preserves statistical relationships.
     * </p>
     * 
     * @return Indicates if this column contains predictive values that should be treated as target variables in machine
     *         learning models. This affects how the synthetic data generation preserves statistical relationships.
     */
    public final Boolean isPredictiveValue() {
        return isPredictiveValue;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(columnName());
        hashCode = 31 * hashCode + Objects.hashCode(columnTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(isPredictiveValue());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SyntheticDataColumnProperties)) {
            return false;
        }
        SyntheticDataColumnProperties other = (SyntheticDataColumnProperties) obj;
        return Objects.equals(columnName(), other.columnName())
                && Objects.equals(columnTypeAsString(), other.columnTypeAsString())
                && Objects.equals(isPredictiveValue(), other.isPredictiveValue());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SyntheticDataColumnProperties").add("ColumnName", columnName())
                .add("ColumnType", columnTypeAsString()).add("IsPredictiveValue", isPredictiveValue()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "columnName":
            return Optional.ofNullable(clazz.cast(columnName()));
        case "columnType":
            return Optional.ofNullable(clazz.cast(columnTypeAsString()));
        case "isPredictiveValue":
            return Optional.ofNullable(clazz.cast(isPredictiveValue()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("columnName", COLUMN_NAME_FIELD);
        map.put("columnType", COLUMN_TYPE_FIELD);
        map.put("isPredictiveValue", IS_PREDICTIVE_VALUE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SyntheticDataColumnProperties, T> g) {
        return obj -> g.apply((SyntheticDataColumnProperties) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SyntheticDataColumnProperties> {
        /**
         * <p>
         * The name of the data column as it appears in the dataset.
         * </p>
         * 
         * @param columnName
         *        The name of the data column as it appears in the dataset.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder columnName(String columnName);

        /**
         * <p>
         * The data type of the column, which determines how the synthetic data generation algorithm processes and
         * synthesizes values for this column.
         * </p>
         * 
         * @param columnType
         *        The data type of the column, which determines how the synthetic data generation algorithm processes
         *        and synthesizes values for this column.
         * @see SyntheticDataColumnType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SyntheticDataColumnType
         */
        Builder columnType(String columnType);

        /**
         * <p>
         * The data type of the column, which determines how the synthetic data generation algorithm processes and
         * synthesizes values for this column.
         * </p>
         * 
         * @param columnType
         *        The data type of the column, which determines how the synthetic data generation algorithm processes
         *        and synthesizes values for this column.
         * @see SyntheticDataColumnType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SyntheticDataColumnType
         */
        Builder columnType(SyntheticDataColumnType columnType);

        /**
         * <p>
         * Indicates if this column contains predictive values that should be treated as target variables in machine
         * learning models. This affects how the synthetic data generation preserves statistical relationships.
         * </p>
         * 
         * @param isPredictiveValue
         *        Indicates if this column contains predictive values that should be treated as target variables in
         *        machine learning models. This affects how the synthetic data generation preserves statistical
         *        relationships.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isPredictiveValue(Boolean isPredictiveValue);
    }

    static final class BuilderImpl implements Builder {
        private String columnName;

        private String columnType;

        private Boolean isPredictiveValue;

        private BuilderImpl() {
        }

        private BuilderImpl(SyntheticDataColumnProperties model) {
            columnName(model.columnName);
            columnType(model.columnType);
            isPredictiveValue(model.isPredictiveValue);
        }

        public final String getColumnName() {
            return columnName;
        }

        public final void setColumnName(String columnName) {
            this.columnName = columnName;
        }

        @Override
        public final Builder columnName(String columnName) {
            this.columnName = columnName;
            return this;
        }

        public final String getColumnType() {
            return columnType;
        }

        public final void setColumnType(String columnType) {
            this.columnType = columnType;
        }

        @Override
        public final Builder columnType(String columnType) {
            this.columnType = columnType;
            return this;
        }

        @Override
        public final Builder columnType(SyntheticDataColumnType columnType) {
            this.columnType(columnType == null ? null : columnType.toString());
            return this;
        }

        public final Boolean getIsPredictiveValue() {
            return isPredictiveValue;
        }

        public final void setIsPredictiveValue(Boolean isPredictiveValue) {
            this.isPredictiveValue = isPredictiveValue;
        }

        @Override
        public final Builder isPredictiveValue(Boolean isPredictiveValue) {
            this.isPredictiveValue = isPredictiveValue;
            return this;
        }

        @Override
        public SyntheticDataColumnProperties build() {
            return new SyntheticDataColumnProperties(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
