/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.chime.internal.ChimeServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.chime.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.chime.model.AccessDeniedException;
import software.amazon.awssdk.services.chime.model.AssociatePhoneNumberWithUserRequest;
import software.amazon.awssdk.services.chime.model.AssociatePhoneNumberWithUserResponse;
import software.amazon.awssdk.services.chime.model.AssociateSigninDelegateGroupsWithAccountRequest;
import software.amazon.awssdk.services.chime.model.AssociateSigninDelegateGroupsWithAccountResponse;
import software.amazon.awssdk.services.chime.model.BadRequestException;
import software.amazon.awssdk.services.chime.model.BatchCreateRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.BatchCreateRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.BatchDeletePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.BatchDeletePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.BatchSuspendUserRequest;
import software.amazon.awssdk.services.chime.model.BatchSuspendUserResponse;
import software.amazon.awssdk.services.chime.model.BatchUnsuspendUserRequest;
import software.amazon.awssdk.services.chime.model.BatchUnsuspendUserResponse;
import software.amazon.awssdk.services.chime.model.BatchUpdatePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.BatchUpdatePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.BatchUpdateUserRequest;
import software.amazon.awssdk.services.chime.model.BatchUpdateUserResponse;
import software.amazon.awssdk.services.chime.model.ChimeException;
import software.amazon.awssdk.services.chime.model.ConflictException;
import software.amazon.awssdk.services.chime.model.CreateAccountRequest;
import software.amazon.awssdk.services.chime.model.CreateAccountResponse;
import software.amazon.awssdk.services.chime.model.CreateBotRequest;
import software.amazon.awssdk.services.chime.model.CreateBotResponse;
import software.amazon.awssdk.services.chime.model.CreateMeetingDialOutRequest;
import software.amazon.awssdk.services.chime.model.CreateMeetingDialOutResponse;
import software.amazon.awssdk.services.chime.model.CreatePhoneNumberOrderRequest;
import software.amazon.awssdk.services.chime.model.CreatePhoneNumberOrderResponse;
import software.amazon.awssdk.services.chime.model.CreateRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.CreateRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.CreateRoomRequest;
import software.amazon.awssdk.services.chime.model.CreateRoomResponse;
import software.amazon.awssdk.services.chime.model.CreateUserRequest;
import software.amazon.awssdk.services.chime.model.CreateUserResponse;
import software.amazon.awssdk.services.chime.model.DeleteAccountRequest;
import software.amazon.awssdk.services.chime.model.DeleteAccountResponse;
import software.amazon.awssdk.services.chime.model.DeleteEventsConfigurationRequest;
import software.amazon.awssdk.services.chime.model.DeleteEventsConfigurationResponse;
import software.amazon.awssdk.services.chime.model.DeletePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.DeletePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.DeleteRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.DeleteRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.DeleteRoomRequest;
import software.amazon.awssdk.services.chime.model.DeleteRoomResponse;
import software.amazon.awssdk.services.chime.model.DisassociatePhoneNumberFromUserRequest;
import software.amazon.awssdk.services.chime.model.DisassociatePhoneNumberFromUserResponse;
import software.amazon.awssdk.services.chime.model.DisassociateSigninDelegateGroupsFromAccountRequest;
import software.amazon.awssdk.services.chime.model.DisassociateSigninDelegateGroupsFromAccountResponse;
import software.amazon.awssdk.services.chime.model.ForbiddenException;
import software.amazon.awssdk.services.chime.model.GetAccountRequest;
import software.amazon.awssdk.services.chime.model.GetAccountResponse;
import software.amazon.awssdk.services.chime.model.GetAccountSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetAccountSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetBotRequest;
import software.amazon.awssdk.services.chime.model.GetBotResponse;
import software.amazon.awssdk.services.chime.model.GetEventsConfigurationRequest;
import software.amazon.awssdk.services.chime.model.GetEventsConfigurationResponse;
import software.amazon.awssdk.services.chime.model.GetGlobalSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetGlobalSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberOrderRequest;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberOrderResponse;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetPhoneNumberSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetRetentionSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetRetentionSettingsResponse;
import software.amazon.awssdk.services.chime.model.GetRoomRequest;
import software.amazon.awssdk.services.chime.model.GetRoomResponse;
import software.amazon.awssdk.services.chime.model.GetUserRequest;
import software.amazon.awssdk.services.chime.model.GetUserResponse;
import software.amazon.awssdk.services.chime.model.GetUserSettingsRequest;
import software.amazon.awssdk.services.chime.model.GetUserSettingsResponse;
import software.amazon.awssdk.services.chime.model.InviteUsersRequest;
import software.amazon.awssdk.services.chime.model.InviteUsersResponse;
import software.amazon.awssdk.services.chime.model.ListAccountsRequest;
import software.amazon.awssdk.services.chime.model.ListAccountsResponse;
import software.amazon.awssdk.services.chime.model.ListBotsRequest;
import software.amazon.awssdk.services.chime.model.ListBotsResponse;
import software.amazon.awssdk.services.chime.model.ListPhoneNumberOrdersRequest;
import software.amazon.awssdk.services.chime.model.ListPhoneNumberOrdersResponse;
import software.amazon.awssdk.services.chime.model.ListPhoneNumbersRequest;
import software.amazon.awssdk.services.chime.model.ListPhoneNumbersResponse;
import software.amazon.awssdk.services.chime.model.ListRoomMembershipsRequest;
import software.amazon.awssdk.services.chime.model.ListRoomMembershipsResponse;
import software.amazon.awssdk.services.chime.model.ListRoomsRequest;
import software.amazon.awssdk.services.chime.model.ListRoomsResponse;
import software.amazon.awssdk.services.chime.model.ListSupportedPhoneNumberCountriesRequest;
import software.amazon.awssdk.services.chime.model.ListSupportedPhoneNumberCountriesResponse;
import software.amazon.awssdk.services.chime.model.ListUsersRequest;
import software.amazon.awssdk.services.chime.model.ListUsersResponse;
import software.amazon.awssdk.services.chime.model.LogoutUserRequest;
import software.amazon.awssdk.services.chime.model.LogoutUserResponse;
import software.amazon.awssdk.services.chime.model.NotFoundException;
import software.amazon.awssdk.services.chime.model.PutEventsConfigurationRequest;
import software.amazon.awssdk.services.chime.model.PutEventsConfigurationResponse;
import software.amazon.awssdk.services.chime.model.PutRetentionSettingsRequest;
import software.amazon.awssdk.services.chime.model.PutRetentionSettingsResponse;
import software.amazon.awssdk.services.chime.model.RedactConversationMessageRequest;
import software.amazon.awssdk.services.chime.model.RedactConversationMessageResponse;
import software.amazon.awssdk.services.chime.model.RedactRoomMessageRequest;
import software.amazon.awssdk.services.chime.model.RedactRoomMessageResponse;
import software.amazon.awssdk.services.chime.model.RegenerateSecurityTokenRequest;
import software.amazon.awssdk.services.chime.model.RegenerateSecurityTokenResponse;
import software.amazon.awssdk.services.chime.model.ResetPersonalPinRequest;
import software.amazon.awssdk.services.chime.model.ResetPersonalPinResponse;
import software.amazon.awssdk.services.chime.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.chime.model.RestorePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.RestorePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.SearchAvailablePhoneNumbersRequest;
import software.amazon.awssdk.services.chime.model.SearchAvailablePhoneNumbersResponse;
import software.amazon.awssdk.services.chime.model.ServiceFailureException;
import software.amazon.awssdk.services.chime.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chime.model.ThrottledClientException;
import software.amazon.awssdk.services.chime.model.UnauthorizedClientException;
import software.amazon.awssdk.services.chime.model.UnprocessableEntityException;
import software.amazon.awssdk.services.chime.model.UpdateAccountRequest;
import software.amazon.awssdk.services.chime.model.UpdateAccountResponse;
import software.amazon.awssdk.services.chime.model.UpdateAccountSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdateAccountSettingsResponse;
import software.amazon.awssdk.services.chime.model.UpdateBotRequest;
import software.amazon.awssdk.services.chime.model.UpdateBotResponse;
import software.amazon.awssdk.services.chime.model.UpdateGlobalSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdateGlobalSettingsResponse;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberRequest;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberResponse;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdatePhoneNumberSettingsResponse;
import software.amazon.awssdk.services.chime.model.UpdateRoomMembershipRequest;
import software.amazon.awssdk.services.chime.model.UpdateRoomMembershipResponse;
import software.amazon.awssdk.services.chime.model.UpdateRoomRequest;
import software.amazon.awssdk.services.chime.model.UpdateRoomResponse;
import software.amazon.awssdk.services.chime.model.UpdateUserRequest;
import software.amazon.awssdk.services.chime.model.UpdateUserResponse;
import software.amazon.awssdk.services.chime.model.UpdateUserSettingsRequest;
import software.amazon.awssdk.services.chime.model.UpdateUserSettingsResponse;
import software.amazon.awssdk.services.chime.transform.AssociatePhoneNumberWithUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.AssociateSigninDelegateGroupsWithAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchCreateRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchDeletePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchSuspendUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchUnsuspendUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchUpdatePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.BatchUpdateUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateBotRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateMeetingDialOutRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreatePhoneNumberOrderRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteEventsConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeletePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DeleteRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DisassociatePhoneNumberFromUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.DisassociateSigninDelegateGroupsFromAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetBotRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetEventsConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetPhoneNumberOrderRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetPhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetPhoneNumberSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.GetUserSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.InviteUsersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListAccountsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListBotsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListPhoneNumberOrdersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListPhoneNumbersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListRoomMembershipsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListRoomsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListSupportedPhoneNumberCountriesRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.LogoutUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.PutEventsConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.PutRetentionSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RedactConversationMessageRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RedactRoomMessageRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RegenerateSecurityTokenRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.ResetPersonalPinRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.RestorePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.SearchAvailablePhoneNumbersRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateAccountRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateAccountSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateBotRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateGlobalSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdatePhoneNumberRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdatePhoneNumberSettingsRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateRoomMembershipRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateRoomRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.services.chime.transform.UpdateUserSettingsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ChimeClient}.
 *
 * @see ChimeClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChimeClient implements ChimeClient {
    private static final Logger log = Logger.loggerFor(DefaultChimeClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChimeClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "Chime" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Associates a phone number with the specified Amazon Chime user.
     * </p>
     *
     * @param associatePhoneNumberWithUserRequest
     * @return Result of the AssociatePhoneNumberWithUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.AssociatePhoneNumberWithUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/AssociatePhoneNumberWithUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociatePhoneNumberWithUserResponse associatePhoneNumberWithUser(
            AssociatePhoneNumberWithUserRequest associatePhoneNumberWithUserRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, AccessDeniedException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociatePhoneNumberWithUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociatePhoneNumberWithUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associatePhoneNumberWithUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associatePhoneNumberWithUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociatePhoneNumberWithUser");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociatePhoneNumberWithUserRequest, AssociatePhoneNumberWithUserResponse>()
                            .withOperationName("AssociatePhoneNumberWithUser").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associatePhoneNumberWithUserRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociatePhoneNumberWithUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associates the specified sign-in delegate groups with the specified Amazon Chime account.
     * </p>
     *
     * @param associateSigninDelegateGroupsWithAccountRequest
     * @return Result of the AssociateSigninDelegateGroupsWithAccount operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.AssociateSigninDelegateGroupsWithAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/AssociateSigninDelegateGroupsWithAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateSigninDelegateGroupsWithAccountResponse associateSigninDelegateGroupsWithAccount(
            AssociateSigninDelegateGroupsWithAccountRequest associateSigninDelegateGroupsWithAccountRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateSigninDelegateGroupsWithAccountResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, AssociateSigninDelegateGroupsWithAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                associateSigninDelegateGroupsWithAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                associateSigninDelegateGroupsWithAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateSigninDelegateGroupsWithAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateSigninDelegateGroupsWithAccountRequest, AssociateSigninDelegateGroupsWithAccountResponse>()
                            .withOperationName("AssociateSigninDelegateGroupsWithAccount").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration)
                            .withInput(associateSigninDelegateGroupsWithAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateSigninDelegateGroupsWithAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds up to 50 members to a chat room in an Amazon Chime Enterprise account. Members can be users or bots. The
     * member role designates whether the member is a chat room administrator or a general chat room member.
     * </p>
     *
     * @param batchCreateRoomMembershipRequest
     * @return Result of the BatchCreateRoomMembership operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.BatchCreateRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchCreateRoomMembership"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchCreateRoomMembershipResponse batchCreateRoomMembership(
            BatchCreateRoomMembershipRequest batchCreateRoomMembershipRequest) throws UnauthorizedClientException,
            NotFoundException, BadRequestException, ForbiddenException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchCreateRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchCreateRoomMembershipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchCreateRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchCreateRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchCreateRoomMembership");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchCreateRoomMembershipRequest, BatchCreateRoomMembershipResponse>()
                            .withOperationName("BatchCreateRoomMembership").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchCreateRoomMembershipRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchCreateRoomMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Moves phone numbers into the <b>Deletion queue</b>. Phone numbers must be disassociated from any users or Amazon
     * Chime Voice Connectors before they can be deleted.
     * </p>
     * <p>
     * Phone numbers remain in the <b>Deletion queue</b> for 7 days before they are deleted permanently.
     * </p>
     *
     * @param batchDeletePhoneNumberRequest
     * @return Result of the BatchDeletePhoneNumber operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.BatchDeletePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchDeletePhoneNumber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchDeletePhoneNumberResponse batchDeletePhoneNumber(BatchDeletePhoneNumberRequest batchDeletePhoneNumberRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchDeletePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchDeletePhoneNumberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchDeletePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchDeletePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchDeletePhoneNumber");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchDeletePhoneNumberRequest, BatchDeletePhoneNumberResponse>()
                            .withOperationName("BatchDeletePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchDeletePhoneNumberRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchDeletePhoneNumberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Suspends up to 50 users from a <code>Team</code> or <code>EnterpriseLWA</code> Amazon Chime account. For more
     * information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * Users suspended from a <code>Team</code> account are disassociated from the account,but they can continue to use
     * Amazon Chime as free users. To remove the suspension from suspended <code>Team</code> account users, invite them
     * to the <code>Team</code> account again. You can use the <a>InviteUsers</a> action to do so.
     * </p>
     * <p>
     * Users suspended from an <code>EnterpriseLWA</code> account are immediately signed out of Amazon Chime and can no
     * longer sign in. To remove the suspension from suspended <code>EnterpriseLWA</code> account users, use the
     * <a>BatchUnsuspendUser</a> action.
     * </p>
     * <p>
     * To sign out users without suspending them, use the <a>LogoutUser</a> action.
     * </p>
     *
     * @param batchSuspendUserRequest
     * @return Result of the BatchSuspendUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.BatchSuspendUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchSuspendUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchSuspendUserResponse batchSuspendUser(BatchSuspendUserRequest batchSuspendUserRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchSuspendUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchSuspendUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchSuspendUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchSuspendUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchSuspendUser");

            return clientHandler.execute(new ClientExecutionParams<BatchSuspendUserRequest, BatchSuspendUserResponse>()
                    .withOperationName("BatchSuspendUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchSuspendUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchSuspendUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the suspension from up to 50 previously suspended users for the specified Amazon Chime
     * <code>EnterpriseLWA</code> account. Only users on <code>EnterpriseLWA</code> accounts can be unsuspended using
     * this action. For more information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html"> Managing Your Amazon Chime Accounts
     * </a> in the account types, in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * Previously suspended users who are unsuspended using this action are returned to <code>Registered</code> status.
     * Users who are not previously suspended are ignored.
     * </p>
     *
     * @param batchUnsuspendUserRequest
     * @return Result of the BatchUnsuspendUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.BatchUnsuspendUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchUnsuspendUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchUnsuspendUserResponse batchUnsuspendUser(BatchUnsuspendUserRequest batchUnsuspendUserRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUnsuspendUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchUnsuspendUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUnsuspendUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUnsuspendUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUnsuspendUser");

            return clientHandler.execute(new ClientExecutionParams<BatchUnsuspendUserRequest, BatchUnsuspendUserResponse>()
                    .withOperationName("BatchUnsuspendUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchUnsuspendUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchUnsuspendUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates phone number product types or calling names. You can update one attribute at a time for each
     * <code>UpdatePhoneNumberRequestItem</code>. For example, you can update the product type or the calling name.
     * </p>
     * <p>
     * For toll-free numbers, you cannot use the Amazon Chime Business Calling product type. For numbers outside the
     * U.S., you must use the Amazon Chime SIP Media Application Dial-In product type.
     * </p>
     * <p>
     * Updates to outbound calling names can take up to 72 hours to complete. Pending updates to outbound calling names
     * must be complete before you can request another update.
     * </p>
     *
     * @param batchUpdatePhoneNumberRequest
     * @return Result of the BatchUpdatePhoneNumber operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.BatchUpdatePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchUpdatePhoneNumber" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public BatchUpdatePhoneNumberResponse batchUpdatePhoneNumber(BatchUpdatePhoneNumberRequest batchUpdatePhoneNumberRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUpdatePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchUpdatePhoneNumberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdatePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdatePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdatePhoneNumber");

            return clientHandler
                    .execute(new ClientExecutionParams<BatchUpdatePhoneNumberRequest, BatchUpdatePhoneNumberResponse>()
                            .withOperationName("BatchUpdatePhoneNumber").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(batchUpdatePhoneNumberRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new BatchUpdatePhoneNumberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates user details within the <a>UpdateUserRequestItem</a> object for up to 20 users for the specified Amazon
     * Chime account. Currently, only <code>LicenseType</code> updates are supported for this action.
     * </p>
     *
     * @param batchUpdateUserRequest
     * @return Result of the BatchUpdateUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.BatchUpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/BatchUpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchUpdateUserResponse batchUpdateUser(BatchUpdateUserRequest batchUpdateUserRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchUpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchUpdateUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchUpdateUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchUpdateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchUpdateUser");

            return clientHandler.execute(new ClientExecutionParams<BatchUpdateUserRequest, BatchUpdateUserResponse>()
                    .withOperationName("BatchUpdateUser").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchUpdateUserRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchUpdateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon Chime account under the administrator's AWS account. Only <code>Team</code> account types are
     * currently supported for this action. For more information about different account types, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/manage-chime-account.html">Managing Your Amazon Chime
     * Accounts</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     *
     * @param createAccountRequest
     * @return Result of the CreateAccount operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreateAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateAccountResponse createAccount(CreateAccountRequest createAccountRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAccount");

            return clientHandler.execute(new ClientExecutionParams<CreateAccountRequest, CreateAccountResponse>()
                    .withOperationName("CreateAccount").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createAccountRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a bot for an Amazon Chime Enterprise account.
     * </p>
     *
     * @param createBotRequest
     * @return Result of the CreateBot operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreateBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBotResponse createBot(CreateBotRequest createBotRequest) throws ServiceUnavailableException,
            ServiceFailureException, ForbiddenException, BadRequestException, UnauthorizedClientException,
            ResourceLimitExceededException, NotFoundException, ThrottledClientException, AwsServiceException, SdkClientException,
            ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBot");

            return clientHandler.execute(new ClientExecutionParams<CreateBotRequest, CreateBotResponse>()
                    .withOperationName("CreateBot").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createBotRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uses the join token and call metadata in a meeting request (From number, To number, and so forth) to initiate an
     * outbound call to a public switched telephone network (PSTN) and join them into a Chime meeting. Also ensures that
     * the From number belongs to the customer.
     * </p>
     * <p>
     * To play welcome audio or implement an interactive voice response (IVR), use the
     * <code>CreateSipMediaApplicationCall</code> action with the corresponding SIP media application ID.
     * </p>
     * <important>
     * <p>
     * <b>This API is not available in a dedicated namespace.</b>
     * </p>
     * </important>
     *
     * @param createMeetingDialOutRequest
     * @return Result of the CreateMeetingDialOut operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreateMeetingDialOut
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateMeetingDialOut" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateMeetingDialOutResponse createMeetingDialOut(CreateMeetingDialOutRequest createMeetingDialOutRequest)
            throws BadRequestException, ForbiddenException, ResourceLimitExceededException, ThrottledClientException,
            UnauthorizedClientException, AccessDeniedException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateMeetingDialOutResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateMeetingDialOutResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createMeetingDialOutRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createMeetingDialOutRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMeetingDialOut");

            return clientHandler.execute(new ClientExecutionParams<CreateMeetingDialOutRequest, CreateMeetingDialOutResponse>()
                    .withOperationName("CreateMeetingDialOut").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createMeetingDialOutRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateMeetingDialOutRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an order for phone numbers to be provisioned. For toll-free numbers, you cannot use the Amazon Chime
     * Business Calling product type. For numbers outside the U.S., you must use the Amazon Chime SIP Media Application
     * Dial-In product type.
     * </p>
     *
     * @param createPhoneNumberOrderRequest
     * @return Result of the CreatePhoneNumberOrder operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreatePhoneNumberOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreatePhoneNumberOrder" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreatePhoneNumberOrderResponse createPhoneNumberOrder(CreatePhoneNumberOrderRequest createPhoneNumberOrderRequest)
            throws BadRequestException, ForbiddenException, AccessDeniedException, UnauthorizedClientException,
            ThrottledClientException, ResourceLimitExceededException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreatePhoneNumberOrderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreatePhoneNumberOrderResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPhoneNumberOrderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPhoneNumberOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePhoneNumberOrder");

            return clientHandler
                    .execute(new ClientExecutionParams<CreatePhoneNumberOrderRequest, CreatePhoneNumberOrderResponse>()
                            .withOperationName("CreatePhoneNumberOrder").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createPhoneNumberOrderRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreatePhoneNumberOrderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a chat room for the specified Amazon Chime Enterprise account.
     * </p>
     *
     * @param createRoomRequest
     * @return Result of the CreateRoom operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateRoomResponse createRoom(CreateRoomRequest createRoomRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, UnauthorizedClientException, ResourceLimitExceededException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateRoomResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoom");

            return clientHandler.execute(new ClientExecutionParams<CreateRoomRequest, CreateRoomResponse>()
                    .withOperationName("CreateRoom").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createRoomRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRoomRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds a member to a chat room in an Amazon Chime Enterprise account. A member can be either a user or a bot. The
     * member role designates whether the member is a chat room administrator or a general chat room member.
     * </p>
     *
     * @param createRoomMembershipRequest
     * @return Result of the CreateRoomMembership operation returned by the service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreateRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateRoomMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateRoomMembershipResponse createRoomMembership(CreateRoomMembershipRequest createRoomMembershipRequest)
            throws ConflictException, UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException,
            ResourceLimitExceededException, ThrottledClientException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateRoomMembershipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateRoomMembership");

            return clientHandler.execute(new ClientExecutionParams<CreateRoomMembershipRequest, CreateRoomMembershipResponse>()
                    .withOperationName("CreateRoomMembership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createRoomMembershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateRoomMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a user under the specified Amazon Chime account.
     * </p>
     *
     * @param createUserRequest
     * @return Result of the CreateUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateUserResponse createUser(CreateUserRequest createUserRequest) throws UnauthorizedClientException,
            NotFoundException, ConflictException, ForbiddenException, BadRequestException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");

            return clientHandler.execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>()
                    .withOperationName("CreateUser").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(createUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified Amazon Chime account. You must suspend all users before deleting <code>Team</code> account.
     * You can use the <a>BatchSuspendUser</a> action to dodo.
     * </p>
     * <p>
     * For <code>EnterpriseLWA</code> and <code>EnterpriseAD</code> accounts, you must release the claimed domains for
     * your Amazon Chime account before deletion. As soon as you release the domain, all users under that account are
     * suspended.
     * </p>
     * <p>
     * Deleted accounts appear in your <code>Disabled</code> accounts list for 90 days. To restore deleted account from
     * your <code>Disabled</code> accounts list, you must contact AWS Support.
     * </p>
     * <p>
     * After 90 days, deleted accounts are permanently removed from your <code>Disabled</code> accounts list.
     * </p>
     *
     * @param deleteAccountRequest
     * @return Result of the DeleteAccount operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws UnprocessableEntityException
     *         The request was well-formed but was unable to be followed due to semantic errors.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DeleteAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAccountResponse deleteAccount(DeleteAccountRequest deleteAccountRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, UnprocessableEntityException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAccount");

            return clientHandler.execute(new ClientExecutionParams<DeleteAccountRequest, DeleteAccountResponse>()
                    .withOperationName("DeleteAccount").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteAccountRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the events configuration that allows a bot to receive outgoing events.
     * </p>
     *
     * @param deleteEventsConfigurationRequest
     * @return Result of the DeleteEventsConfiguration operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DeleteEventsConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteEventsConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteEventsConfigurationResponse deleteEventsConfiguration(
            DeleteEventsConfigurationRequest deleteEventsConfigurationRequest) throws ServiceUnavailableException,
            ServiceFailureException, ForbiddenException, BadRequestException, UnauthorizedClientException,
            ResourceLimitExceededException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEventsConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteEventsConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteEventsConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEventsConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEventsConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteEventsConfigurationRequest, DeleteEventsConfigurationResponse>()
                            .withOperationName("DeleteEventsConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteEventsConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteEventsConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Moves the specified phone number into the <b>Deletion queue</b>. A phone number must be disassociated from any
     * users or Amazon Chime Voice Connectors before it can be deleted.
     * </p>
     * <p>
     * Deleted phone numbers remain in the <b>Deletion queue</b> for 7 days before they are deleted permanently.
     * </p>
     *
     * @param deletePhoneNumberRequest
     * @return Result of the DeletePhoneNumber operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DeletePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeletePhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeletePhoneNumberResponse deletePhoneNumber(DeletePhoneNumberRequest deletePhoneNumberRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeletePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeletePhoneNumberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePhoneNumber");

            return clientHandler.execute(new ClientExecutionParams<DeletePhoneNumberRequest, DeletePhoneNumberResponse>()
                    .withOperationName("DeletePhoneNumber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deletePhoneNumberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeletePhoneNumberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a chat room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param deleteRoomRequest
     * @return Result of the DeleteRoom operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DeleteRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteRoomResponse deleteRoom(DeleteRoomRequest deleteRoomRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, UnauthorizedClientException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteRoomResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoom");

            return clientHandler.execute(new ClientExecutionParams<DeleteRoomRequest, DeleteRoomResponse>()
                    .withOperationName("DeleteRoom").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteRoomRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRoomRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes a member from a chat room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param deleteRoomMembershipRequest
     * @return Result of the DeleteRoomMembership operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DeleteRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DeleteRoomMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteRoomMembershipResponse deleteRoomMembership(DeleteRoomMembershipRequest deleteRoomMembershipRequest)
            throws UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteRoomMembershipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteRoomMembership");

            return clientHandler.execute(new ClientExecutionParams<DeleteRoomMembershipRequest, DeleteRoomMembershipResponse>()
                    .withOperationName("DeleteRoomMembership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteRoomMembershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteRoomMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates the primary provisioned phone number from the specified Amazon Chime user.
     * </p>
     *
     * @param disassociatePhoneNumberFromUserRequest
     * @return Result of the DisassociatePhoneNumberFromUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DisassociatePhoneNumberFromUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DisassociatePhoneNumberFromUser"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociatePhoneNumberFromUserResponse disassociatePhoneNumberFromUser(
            DisassociatePhoneNumberFromUserRequest disassociatePhoneNumberFromUserRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociatePhoneNumberFromUserResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociatePhoneNumberFromUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociatePhoneNumberFromUserRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociatePhoneNumberFromUserRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociatePhoneNumberFromUser");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociatePhoneNumberFromUserRequest, DisassociatePhoneNumberFromUserResponse>()
                            .withOperationName("DisassociatePhoneNumberFromUser").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociatePhoneNumberFromUserRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociatePhoneNumberFromUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disassociates the specified sign-in delegate groups from the specified Amazon Chime account.
     * </p>
     *
     * @param disassociateSigninDelegateGroupsFromAccountRequest
     * @return Result of the DisassociateSigninDelegateGroupsFromAccount operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.DisassociateSigninDelegateGroupsFromAccount
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/DisassociateSigninDelegateGroupsFromAccount"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateSigninDelegateGroupsFromAccountResponse disassociateSigninDelegateGroupsFromAccount(
            DisassociateSigninDelegateGroupsFromAccountRequest disassociateSigninDelegateGroupsFromAccountRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateSigninDelegateGroupsFromAccountResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, DisassociateSigninDelegateGroupsFromAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                disassociateSigninDelegateGroupsFromAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateSigninDelegateGroupsFromAccountRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateSigninDelegateGroupsFromAccount");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateSigninDelegateGroupsFromAccountRequest, DisassociateSigninDelegateGroupsFromAccountResponse>()
                            .withOperationName("DisassociateSigninDelegateGroupsFromAccount")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withInput(disassociateSigninDelegateGroupsFromAccountRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateSigninDelegateGroupsFromAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details for the specified Amazon Chime account, such as account type and supported licenses.
     * </p>
     *
     * @param getAccountRequest
     * @return Result of the GetAccount operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAccountResponse getAccount(GetAccountRequest getAccountRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccount");

            return clientHandler.execute(new ClientExecutionParams<GetAccountRequest, GetAccountResponse>()
                    .withOperationName("GetAccount").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getAccountRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves account settings for the specified Amazon Chime account ID, such as remote control and dialout
     * settings. For more information about these settings, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/policies.html">Use the Policies Page</a> in the <i>Amazon Chime
     * Administration Guide</i>.
     * </p>
     *
     * @param getAccountSettingsRequest
     * @return Result of the GetAccountSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetAccountSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetAccountSettingsResponse getAccountSettings(GetAccountSettingsRequest getAccountSettingsRequest)
            throws UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAccountSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountSettings");

            return clientHandler.execute(new ClientExecutionParams<GetAccountSettingsRequest, GetAccountSettingsResponse>()
                    .withOperationName("GetAccountSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getAccountSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details for the specified bot, such as bot email address, bot type, status, and display name.
     * </p>
     *
     * @param getBotRequest
     * @return Result of the GetBot operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetBotResponse getBot(GetBotRequest getBotRequest) throws ServiceUnavailableException, ServiceFailureException,
            ForbiddenException, UnauthorizedClientException, NotFoundException, BadRequestException, ThrottledClientException,
            AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetBotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetBot");

            return clientHandler.execute(new ClientExecutionParams<GetBotRequest, GetBotResponse>().withOperationName("GetBot")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getBotRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetBotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets details for an events configuration that allows a bot to receive outgoing events, such as an HTTPS endpoint
     * or Lambda function ARN.
     * </p>
     *
     * @param getEventsConfigurationRequest
     * @return Result of the GetEventsConfiguration operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetEventsConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetEventsConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetEventsConfigurationResponse getEventsConfiguration(GetEventsConfigurationRequest getEventsConfigurationRequest)
            throws ServiceUnavailableException, ServiceFailureException, ForbiddenException, BadRequestException,
            UnauthorizedClientException, ResourceLimitExceededException, NotFoundException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEventsConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEventsConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getEventsConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEventsConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEventsConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<GetEventsConfigurationRequest, GetEventsConfigurationResponse>()
                            .withOperationName("GetEventsConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getEventsConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetEventsConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves global settings for the administrator's AWS account, such as Amazon Chime Business Calling and Amazon
     * Chime Voice Connector settings.
     * </p>
     *
     * @param getGlobalSettingsRequest
     * @return Result of the GetGlobalSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetGlobalSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetGlobalSettingsResponse getGlobalSettings(GetGlobalSettingsRequest getGlobalSettingsRequest)
            throws UnauthorizedClientException, ForbiddenException, BadRequestException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetGlobalSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGlobalSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGlobalSettings");

            return clientHandler.execute(new ClientExecutionParams<GetGlobalSettingsRequest, GetGlobalSettingsResponse>()
                    .withOperationName("GetGlobalSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getGlobalSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetGlobalSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details for the specified phone number ID, such as associations, capabilities, and product type.
     * </p>
     *
     * @param getPhoneNumberRequest
     * @return Result of the GetPhoneNumber operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetPhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetPhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPhoneNumberResponse getPhoneNumber(GetPhoneNumberRequest getPhoneNumberRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetPhoneNumberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPhoneNumberRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPhoneNumber");

            return clientHandler.execute(new ClientExecutionParams<GetPhoneNumberRequest, GetPhoneNumberResponse>()
                    .withOperationName("GetPhoneNumber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPhoneNumberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPhoneNumberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details for the specified phone number order, such as the order creation timestamp, phone numbers in
     * E.164 format, product type, and order status.
     * </p>
     *
     * @param getPhoneNumberOrderRequest
     * @return Result of the GetPhoneNumberOrder operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetPhoneNumberOrder
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetPhoneNumberOrder" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetPhoneNumberOrderResponse getPhoneNumberOrder(GetPhoneNumberOrderRequest getPhoneNumberOrderRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPhoneNumberOrderResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPhoneNumberOrderResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPhoneNumberOrderRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPhoneNumberOrderRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPhoneNumberOrder");

            return clientHandler.execute(new ClientExecutionParams<GetPhoneNumberOrderRequest, GetPhoneNumberOrderResponse>()
                    .withOperationName("GetPhoneNumberOrder").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getPhoneNumberOrderRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetPhoneNumberOrderRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the phone number settings for the administrator's AWS account, such as the default outbound calling
     * name.
     * </p>
     *
     * @param getPhoneNumberSettingsRequest
     * @return Result of the GetPhoneNumberSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetPhoneNumberSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetPhoneNumberSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetPhoneNumberSettingsResponse getPhoneNumberSettings(GetPhoneNumberSettingsRequest getPhoneNumberSettingsRequest)
            throws UnauthorizedClientException, ForbiddenException, BadRequestException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetPhoneNumberSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetPhoneNumberSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPhoneNumberSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPhoneNumberSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPhoneNumberSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<GetPhoneNumberSettingsRequest, GetPhoneNumberSettingsResponse>()
                            .withOperationName("GetPhoneNumberSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(getPhoneNumberSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetPhoneNumberSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets the retention settings for the specified Amazon Chime Enterprise account. For more information about
     * retention settings, see <a href="https://docs.aws.amazon.com/chime/latest/ag/chat-retention.html">Managing Chat
     * Retention Policies</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     *
     * @param getRetentionSettingsRequest
     * @return Result of the GetRetentionSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetRetentionSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetRetentionSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRetentionSettingsResponse getRetentionSettings(GetRetentionSettingsRequest getRetentionSettingsRequest)
            throws UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRetentionSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRetentionSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRetentionSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRetentionSettings");

            return clientHandler.execute(new ClientExecutionParams<GetRetentionSettingsRequest, GetRetentionSettingsResponse>()
                    .withOperationName("GetRetentionSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getRetentionSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRetentionSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves room details, such as the room name, for a room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param getRoomRequest
     * @return Result of the GetRoom operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetRoomResponse getRoom(GetRoomRequest getRoomRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, UnauthorizedClientException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRoomResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRoom");

            return clientHandler.execute(new ClientExecutionParams<GetRoomRequest, GetRoomResponse>()
                    .withOperationName("GetRoom").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRoomRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRoomRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves details for the specified user ID, such as primary email address, license type,and personal meeting
     * PIN.
     * </p>
     * <p>
     * To retrieve user details with an email address instead of a user ID, use the <a>ListUsers</a> action, and then
     * filter by email address.
     * </p>
     *
     * @param getUserRequest
     * @return Result of the GetUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetUserResponse getUser(GetUserRequest getUserRequest) throws UnauthorizedClientException, NotFoundException,
            ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUser");

            return clientHandler.execute(new ClientExecutionParams<GetUserRequest, GetUserResponse>()
                    .withOperationName("GetUser").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves settings for the specified user ID, such as any associated phone number settings.
     * </p>
     *
     * @param getUserSettingsRequest
     * @return Result of the GetUserSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.GetUserSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/GetUserSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetUserSettingsResponse getUserSettings(GetUserSettingsRequest getUserSettingsRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetUserSettingsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetUserSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getUserSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getUserSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetUserSettings");

            return clientHandler.execute(new ClientExecutionParams<GetUserSettingsRequest, GetUserSettingsResponse>()
                    .withOperationName("GetUserSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getUserSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetUserSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Sends email to a maximum of 50 users, inviting them to the specified Amazon Chime <code>Team</code> account. Only
     * <code>Team</code> account types are currently supported for this action.
     * </p>
     *
     * @param inviteUsersRequest
     * @return Result of the InviteUsers operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.InviteUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/InviteUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public InviteUsersResponse inviteUsers(InviteUsersRequest inviteUsersRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InviteUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InviteUsersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(inviteUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, inviteUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InviteUsers");

            return clientHandler.execute(new ClientExecutionParams<InviteUsersRequest, InviteUsersResponse>()
                    .withOperationName("InviteUsers").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(inviteUsersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InviteUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the Amazon Chime accounts under the administrator's AWS account. You can filter accounts by account name
     * prefix. To find out which Amazon Chime account a user belongs to, you can filter by the user's email address,
     * which returns one account result.
     * </p>
     *
     * @param listAccountsRequest
     * @return Result of the ListAccounts operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListAccounts
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListAccounts" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListAccountsResponse listAccounts(ListAccountsRequest listAccountsRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListAccountsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListAccountsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAccountsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAccountsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAccounts");

            return clientHandler.execute(new ClientExecutionParams<ListAccountsRequest, ListAccountsResponse>()
                    .withOperationName("ListAccounts").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listAccountsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListAccountsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the bots associated with the administrator's Amazon Chime Enterprise account ID.
     * </p>
     *
     * @param listBotsRequest
     * @return Result of the ListBots operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListBots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListBots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBotsResponse listBots(ListBotsRequest listBotsRequest) throws ServiceUnavailableException,
            ServiceFailureException, ForbiddenException, UnauthorizedClientException, BadRequestException, NotFoundException,
            ThrottledClientException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBotsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBotsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBots");

            return clientHandler.execute(new ClientExecutionParams<ListBotsRequest, ListBotsResponse>()
                    .withOperationName("ListBots").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listBotsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the phone number orders for the administrator's Amazon Chime account.
     * </p>
     *
     * @param listPhoneNumberOrdersRequest
     * @return Result of the ListPhoneNumberOrders operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListPhoneNumberOrders
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListPhoneNumberOrders" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListPhoneNumberOrdersResponse listPhoneNumberOrders(ListPhoneNumberOrdersRequest listPhoneNumberOrdersRequest)
            throws UnauthorizedClientException, ForbiddenException, BadRequestException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPhoneNumberOrdersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListPhoneNumberOrdersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPhoneNumberOrdersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPhoneNumberOrdersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPhoneNumberOrders");

            return clientHandler.execute(new ClientExecutionParams<ListPhoneNumberOrdersRequest, ListPhoneNumberOrdersResponse>()
                    .withOperationName("ListPhoneNumberOrders").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPhoneNumberOrdersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPhoneNumberOrdersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the phone numbers for the specified Amazon Chime account, Amazon Chime user, Amazon Chime Voice Connector,
     * or Amazon Chime Voice Connector group.
     * </p>
     *
     * @param listPhoneNumbersRequest
     * @return Result of the ListPhoneNumbers operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListPhoneNumbers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListPhoneNumbers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListPhoneNumbersResponse listPhoneNumbers(ListPhoneNumbersRequest listPhoneNumbersRequest)
            throws UnauthorizedClientException, ForbiddenException, BadRequestException, NotFoundException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListPhoneNumbersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListPhoneNumbersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPhoneNumbersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPhoneNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPhoneNumbers");

            return clientHandler.execute(new ClientExecutionParams<ListPhoneNumbersRequest, ListPhoneNumbersResponse>()
                    .withOperationName("ListPhoneNumbers").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listPhoneNumbersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListPhoneNumbersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the membership details for the specified room in an Amazon Chime Enterprise account, such as the members'
     * IDs, email addresses, and names.
     * </p>
     *
     * @param listRoomMembershipsRequest
     * @return Result of the ListRoomMemberships operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListRoomMemberships
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListRoomMemberships" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRoomMembershipsResponse listRoomMemberships(ListRoomMembershipsRequest listRoomMembershipsRequest)
            throws NotFoundException, BadRequestException, ForbiddenException, UnauthorizedClientException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRoomMembershipsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListRoomMembershipsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoomMembershipsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoomMembershipsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRoomMemberships");

            return clientHandler.execute(new ClientExecutionParams<ListRoomMembershipsRequest, ListRoomMembershipsResponse>()
                    .withOperationName("ListRoomMemberships").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listRoomMembershipsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRoomMembershipsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the room details for the specified Amazon Chime Enterprise account. Optionally, filter the results by a
     * member ID (user ID or bot ID) to see a list of rooms that the member belongs to.
     * </p>
     *
     * @param listRoomsRequest
     * @return Result of the ListRooms operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListRooms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListRooms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListRoomsResponse listRooms(ListRoomsRequest listRoomsRequest) throws NotFoundException, BadRequestException,
            ForbiddenException, UnauthorizedClientException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListRoomsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListRoomsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listRoomsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRoomsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRooms");

            return clientHandler.execute(new ClientExecutionParams<ListRoomsRequest, ListRoomsResponse>()
                    .withOperationName("ListRooms").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listRoomsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListRoomsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists supported phone number countries.
     * </p>
     *
     * @param listSupportedPhoneNumberCountriesRequest
     * @return Result of the ListSupportedPhoneNumberCountries operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListSupportedPhoneNumberCountries
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListSupportedPhoneNumberCountries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListSupportedPhoneNumberCountriesResponse listSupportedPhoneNumberCountries(
            ListSupportedPhoneNumberCountriesRequest listSupportedPhoneNumberCountriesRequest) throws BadRequestException,
            ForbiddenException, AccessDeniedException, UnauthorizedClientException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListSupportedPhoneNumberCountriesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListSupportedPhoneNumberCountriesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listSupportedPhoneNumberCountriesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listSupportedPhoneNumberCountriesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListSupportedPhoneNumberCountries");

            return clientHandler
                    .execute(new ClientExecutionParams<ListSupportedPhoneNumberCountriesRequest, ListSupportedPhoneNumberCountriesResponse>()
                            .withOperationName("ListSupportedPhoneNumberCountries").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listSupportedPhoneNumberCountriesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListSupportedPhoneNumberCountriesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the users that belong to the specified Amazon Chime account. You can specify an email address to list only
     * the user that the email address belongs to.
     * </p>
     *
     * @param listUsersRequest
     * @return Result of the ListUsers operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListUsersResponse listUsers(ListUsersRequest listUsersRequest) throws UnauthorizedClientException, NotFoundException,
            ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListUsersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");

            return clientHandler.execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>()
                    .withOperationName("ListUsers").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listUsersRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Logs out the specified user from all of the devices they are currently logged into.
     * </p>
     *
     * @param logoutUserRequest
     * @return Result of the LogoutUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.LogoutUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/LogoutUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public LogoutUserResponse logoutUser(LogoutUserRequest logoutUserRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<LogoutUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                LogoutUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(logoutUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, logoutUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "LogoutUser");

            return clientHandler.execute(new ClientExecutionParams<LogoutUserRequest, LogoutUserResponse>()
                    .withOperationName("LogoutUser").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(logoutUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new LogoutUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an events configuration that allows a bot to receive outgoing events sent by Amazon Chime. Choose either
     * an HTTPS endpoint or a Lambda function ARN. For more information, see <a>Bot</a>.
     * </p>
     *
     * @param putEventsConfigurationRequest
     * @return Result of the PutEventsConfiguration operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.PutEventsConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/PutEventsConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutEventsConfigurationResponse putEventsConfiguration(PutEventsConfigurationRequest putEventsConfigurationRequest)
            throws ServiceUnavailableException, ServiceFailureException, ForbiddenException, BadRequestException,
            UnauthorizedClientException, ResourceLimitExceededException, NotFoundException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutEventsConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutEventsConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putEventsConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putEventsConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutEventsConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<PutEventsConfigurationRequest, PutEventsConfigurationResponse>()
                            .withOperationName("PutEventsConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(putEventsConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new PutEventsConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Puts retention settings for the specified Amazon Chime Enterprise account. We recommend using AWS CloudTrail to
     * monitor usage of this API for your account. For more information, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/cloudtrail.html">Logging Amazon Chime API Calls with AWS
     * CloudTrail</a> in the <i>Amazon Chime Administration Guide</i>.
     * </p>
     * <p>
     * To turn off existing retention settings, remove the number of days from the corresponding <b>RetentionDays</b>
     * field in the <b>RetentionSettings</b> object. For more information about retention settings, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/chat-retention.html">Managing Chat Retention Policies</a> in
     * the <i>Amazon Chime Administration Guide</i>.
     * </p>
     *
     * @param putRetentionSettingsRequest
     * @return Result of the PutRetentionSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.PutRetentionSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/PutRetentionSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutRetentionSettingsResponse putRetentionSettings(PutRetentionSettingsRequest putRetentionSettingsRequest)
            throws UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException, ConflictException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutRetentionSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, PutRetentionSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putRetentionSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putRetentionSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutRetentionSettings");

            return clientHandler.execute(new ClientExecutionParams<PutRetentionSettingsRequest, PutRetentionSettingsResponse>()
                    .withOperationName("PutRetentionSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(putRetentionSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutRetentionSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Redacts the specified message from the specified Amazon Chime conversation.
     * </p>
     *
     * @param redactConversationMessageRequest
     * @return Result of the RedactConversationMessage operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.RedactConversationMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RedactConversationMessage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RedactConversationMessageResponse redactConversationMessage(
            RedactConversationMessageRequest redactConversationMessageRequest) throws NotFoundException, ForbiddenException,
            UnauthorizedClientException, ThrottledClientException, BadRequestException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RedactConversationMessageResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RedactConversationMessageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(redactConversationMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, redactConversationMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RedactConversationMessage");

            return clientHandler
                    .execute(new ClientExecutionParams<RedactConversationMessageRequest, RedactConversationMessageResponse>()
                            .withOperationName("RedactConversationMessage").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(redactConversationMessageRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RedactConversationMessageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Redacts the specified message from the specified Amazon Chime channel.
     * </p>
     *
     * @param redactRoomMessageRequest
     * @return Result of the RedactRoomMessage operation returned by the service.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.RedactRoomMessage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RedactRoomMessage" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RedactRoomMessageResponse redactRoomMessage(RedactRoomMessageRequest redactRoomMessageRequest)
            throws NotFoundException, ForbiddenException, UnauthorizedClientException, ThrottledClientException,
            BadRequestException, ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException,
            ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RedactRoomMessageResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RedactRoomMessageResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(redactRoomMessageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, redactRoomMessageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RedactRoomMessage");

            return clientHandler.execute(new ClientExecutionParams<RedactRoomMessageRequest, RedactRoomMessageResponse>()
                    .withOperationName("RedactRoomMessage").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(redactRoomMessageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RedactRoomMessageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Regenerates the security token for a bot.
     * </p>
     *
     * @param regenerateSecurityTokenRequest
     * @return Result of the RegenerateSecurityToken operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.RegenerateSecurityToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RegenerateSecurityToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RegenerateSecurityTokenResponse regenerateSecurityToken(RegenerateSecurityTokenRequest regenerateSecurityTokenRequest)
            throws ServiceUnavailableException, ServiceFailureException, ForbiddenException, BadRequestException,
            UnauthorizedClientException, NotFoundException, ThrottledClientException, AwsServiceException, SdkClientException,
            ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RegenerateSecurityTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RegenerateSecurityTokenResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(regenerateSecurityTokenRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, regenerateSecurityTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RegenerateSecurityToken");

            return clientHandler
                    .execute(new ClientExecutionParams<RegenerateSecurityTokenRequest, RegenerateSecurityTokenResponse>()
                            .withOperationName("RegenerateSecurityToken").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(regenerateSecurityTokenRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RegenerateSecurityTokenRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Resets the personal meeting PIN for the specified user on an Amazon Chime account. Returns the <a>User</a> object
     * with the updated personal meeting PIN.
     * </p>
     *
     * @param resetPersonalPinRequest
     * @return Result of the ResetPersonalPIN operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.ResetPersonalPIN
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/ResetPersonalPIN" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ResetPersonalPinResponse resetPersonalPIN(ResetPersonalPinRequest resetPersonalPinRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ResetPersonalPinResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ResetPersonalPinResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(resetPersonalPinRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, resetPersonalPinRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ResetPersonalPIN");

            return clientHandler.execute(new ClientExecutionParams<ResetPersonalPinRequest, ResetPersonalPinResponse>()
                    .withOperationName("ResetPersonalPIN").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(resetPersonalPinRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ResetPersonalPinRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Moves a phone number from the <b>Deletion queue</b> back into the phone number <b>Inventory</b>.
     * </p>
     *
     * @param restorePhoneNumberRequest
     * @return Result of the RestorePhoneNumber operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ResourceLimitExceededException
     *         The request exceeds the resource limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.RestorePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/RestorePhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RestorePhoneNumberResponse restorePhoneNumber(RestorePhoneNumberRequest restorePhoneNumberRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ResourceLimitExceededException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestorePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestorePhoneNumberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restorePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restorePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestorePhoneNumber");

            return clientHandler.execute(new ClientExecutionParams<RestorePhoneNumberRequest, RestorePhoneNumberResponse>()
                    .withOperationName("RestorePhoneNumber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(restorePhoneNumberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RestorePhoneNumberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Searches for phone numbers that can be ordered. For US numbers, provide at least one of the following search
     * filters: <code>AreaCode</code>, <code>City</code>, <code>State</code>, or <code>TollFreePrefix</code>. If you
     * provide <code>City</code>, you must also provide <code>State</code>. Numbers outside the US only support the
     * <code>PhoneNumberType</code> filter, which you must use.
     * </p>
     *
     * @param searchAvailablePhoneNumbersRequest
     * @return Result of the SearchAvailablePhoneNumbers operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws AccessDeniedException
     *         You don't have permissions to perform the requested operation.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.SearchAvailablePhoneNumbers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/SearchAvailablePhoneNumbers"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SearchAvailablePhoneNumbersResponse searchAvailablePhoneNumbers(
            SearchAvailablePhoneNumbersRequest searchAvailablePhoneNumbersRequest) throws BadRequestException,
            ForbiddenException, AccessDeniedException, UnauthorizedClientException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SearchAvailablePhoneNumbersResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, SearchAvailablePhoneNumbersResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(searchAvailablePhoneNumbersRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchAvailablePhoneNumbersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchAvailablePhoneNumbers");

            return clientHandler
                    .execute(new ClientExecutionParams<SearchAvailablePhoneNumbersRequest, SearchAvailablePhoneNumbersResponse>()
                            .withOperationName("SearchAvailablePhoneNumbers").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(searchAvailablePhoneNumbersRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new SearchAvailablePhoneNumbersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates account details for the specified Amazon Chime account. Currently, only account name and default license
     * updates are supported for this action.
     * </p>
     *
     * @param updateAccountRequest
     * @return Result of the UpdateAccount operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateAccount
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateAccount" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateAccountResponse updateAccount(UpdateAccountRequest updateAccountRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAccountResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateAccountResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccountRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccount");

            return clientHandler.execute(new ClientExecutionParams<UpdateAccountRequest, UpdateAccountResponse>()
                    .withOperationName("UpdateAccount").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateAccountRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAccountRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the settings for the specified Amazon Chime account. You can update settings for remote control of shared
     * screens, or for the dial-out option. For more information about these settings, see <a
     * href="https://docs.aws.amazon.com/chime/latest/ag/policies.html">Use the Policies Page</a> in the <i>Amazon Chime
     * Administration Guide</i>.
     * </p>
     *
     * @param updateAccountSettingsRequest
     * @return Result of the UpdateAccountSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateAccountSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateAccountSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateAccountSettingsResponse updateAccountSettings(UpdateAccountSettingsRequest updateAccountSettingsRequest)
            throws UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException, ConflictException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateAccountSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateAccountSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccountSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccountSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateAccountSettingsRequest, UpdateAccountSettingsResponse>()
                    .withOperationName("UpdateAccountSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateAccountSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateAccountSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the status of the specified bot, such as starting or stopping the bot from running in your Amazon Chime
     * Enterprise account.
     * </p>
     *
     * @param updateBotRequest
     * @return Result of the UpdateBot operation returned by the service.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateBot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateBot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateBotResponse updateBot(UpdateBotRequest updateBotRequest) throws ServiceUnavailableException,
            ServiceFailureException, ForbiddenException, BadRequestException, UnauthorizedClientException, NotFoundException,
            ThrottledClientException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateBotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateBotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBot");

            return clientHandler.execute(new ClientExecutionParams<UpdateBotRequest, UpdateBotResponse>()
                    .withOperationName("UpdateBot").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateBotRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateBotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates global settings for the administrator's AWS account, such as Amazon Chime Business Calling and Amazon
     * Chime Voice Connector settings.
     * </p>
     *
     * @param updateGlobalSettingsRequest
     * @return Result of the UpdateGlobalSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateGlobalSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateGlobalSettings" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateGlobalSettingsResponse updateGlobalSettings(UpdateGlobalSettingsRequest updateGlobalSettingsRequest)
            throws UnauthorizedClientException, ForbiddenException, BadRequestException, ThrottledClientException,
            ServiceUnavailableException, ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGlobalSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGlobalSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGlobalSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateGlobalSettingsRequest, UpdateGlobalSettingsResponse>()
                    .withOperationName("UpdateGlobalSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateGlobalSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGlobalSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates phone number details, such as product type or calling name, for the specified phone number ID. You can
     * update one phone number detail at a time. For example, you can update either the product type or the calling name
     * in one action.
     * </p>
     * <p>
     * For toll-free numbers, you cannot use the Amazon Chime Business Calling product type. For numbers outside the
     * U.S., you must use the Amazon Chime SIP Media Application Dial-In product type.
     * </p>
     * <p>
     * Updates to outbound calling names can take 72 hours to complete. Pending updates to outbound calling names must
     * be complete before you can request another update.
     * </p>
     *
     * @param updatePhoneNumberRequest
     * @return Result of the UpdatePhoneNumber operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdatePhoneNumber
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdatePhoneNumber" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdatePhoneNumberResponse updatePhoneNumber(UpdatePhoneNumberRequest updatePhoneNumberRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ConflictException, ServiceUnavailableException, ServiceFailureException,
            AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePhoneNumberResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdatePhoneNumberResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePhoneNumberRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePhoneNumberRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePhoneNumber");

            return clientHandler.execute(new ClientExecutionParams<UpdatePhoneNumberRequest, UpdatePhoneNumberResponse>()
                    .withOperationName("UpdatePhoneNumber").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updatePhoneNumberRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdatePhoneNumberRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the phone number settings for the administrator's AWS account, such as the default outbound calling name.
     * You can update the default outbound calling name once every seven days. Outbound calling names can take up to 72
     * hours to update.
     * </p>
     *
     * @param updatePhoneNumberSettingsRequest
     * @return Result of the UpdatePhoneNumberSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdatePhoneNumberSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdatePhoneNumberSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdatePhoneNumberSettingsResponse updatePhoneNumberSettings(
            UpdatePhoneNumberSettingsRequest updatePhoneNumberSettingsRequest) throws UnauthorizedClientException,
            ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdatePhoneNumberSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdatePhoneNumberSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePhoneNumberSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePhoneNumberSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePhoneNumberSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdatePhoneNumberSettingsRequest, UpdatePhoneNumberSettingsResponse>()
                            .withOperationName("UpdatePhoneNumberSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updatePhoneNumberSettingsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdatePhoneNumberSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates room details, such as the room name, for a room in an Amazon Chime Enterprise account.
     * </p>
     *
     * @param updateRoomRequest
     * @return Result of the UpdateRoom operation returned by the service.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateRoom
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateRoom" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateRoomResponse updateRoom(UpdateRoomRequest updateRoomRequest) throws BadRequestException, ForbiddenException,
            NotFoundException, UnauthorizedClientException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRoomResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateRoomResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRoomRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRoomRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoom");

            return clientHandler.execute(new ClientExecutionParams<UpdateRoomRequest, UpdateRoomResponse>()
                    .withOperationName("UpdateRoom").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateRoomRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRoomRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates room membership details, such as the member role, for a room in an Amazon Chime Enterprise account. The
     * member role designates whether the member is a chat room administrator or a general chat room member. The member
     * role can be updated only for user IDs.
     * </p>
     *
     * @param updateRoomMembershipRequest
     * @return Result of the UpdateRoomMembership operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateRoomMembership
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateRoomMembership" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateRoomMembershipResponse updateRoomMembership(UpdateRoomMembershipRequest updateRoomMembershipRequest)
            throws UnauthorizedClientException, NotFoundException, BadRequestException, ForbiddenException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateRoomMembershipResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateRoomMembershipResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateRoomMembershipRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateRoomMembershipRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateRoomMembership");

            return clientHandler.execute(new ClientExecutionParams<UpdateRoomMembershipRequest, UpdateRoomMembershipResponse>()
                    .withOperationName("UpdateRoomMembership").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateRoomMembershipRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateRoomMembershipRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates user details for a specified user ID. Currently, only <code>LicenseType</code> updates are supported for
     * this action.
     * </p>
     *
     * @param updateUserRequest
     * @return Result of the UpdateUser operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserResponse updateUser(UpdateUserRequest updateUserRequest) throws UnauthorizedClientException,
            NotFoundException, ForbiddenException, BadRequestException, ThrottledClientException, ServiceUnavailableException,
            ServiceFailureException, AwsServiceException, SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateUserResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");

            return clientHandler.execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>()
                    .withOperationName("UpdateUser").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(updateUserRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the settings for the specified user, such as phone number settings.
     * </p>
     *
     * @param updateUserSettingsRequest
     * @return Result of the UpdateUserSettings operation returned by the service.
     * @throws UnauthorizedClientException
     *         The client is not currently authorized to make the request.
     * @throws NotFoundException
     *         One or more of the resources in the request does not exist in the system.
     * @throws ForbiddenException
     *         The client is permanently forbidden from making the request.
     * @throws BadRequestException
     *         The input parameters don't match the service's restrictions.
     * @throws ThrottledClientException
     *         The client exceeded its request rate limit.
     * @throws ServiceUnavailableException
     *         The service is currently unavailable.
     * @throws ServiceFailureException
     *         The service encountered an unexpected error.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ChimeException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ChimeClient.UpdateUserSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chime-2018-05-01/UpdateUserSettings" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateUserSettingsResponse updateUserSettings(UpdateUserSettingsRequest updateUserSettingsRequest)
            throws UnauthorizedClientException, NotFoundException, ForbiddenException, BadRequestException,
            ThrottledClientException, ServiceUnavailableException, ServiceFailureException, AwsServiceException,
            SdkClientException, ChimeException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateUserSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateUserSettingsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "UnauthorizedClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedClientException").httpStatusCode(401)
                        .exceptionBuilderSupplier(UnauthorizedClientException::builder).build());
            case "NotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotFoundException").httpStatusCode(404)
                        .exceptionBuilderSupplier(NotFoundException::builder).build());
            case "ServiceFailureException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceFailureException").httpStatusCode(500)
                        .exceptionBuilderSupplier(ServiceFailureException::builder).build());
            case "AccessDeniedException":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessDeniedException").httpStatusCode(403)
                        .exceptionBuilderSupplier(AccessDeniedException::builder).build());
            case "ConflictException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                        .exceptionBuilderSupplier(ConflictException::builder).build());
            case "UnprocessableEntityException":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnprocessableEntityException").httpStatusCode(422)
                        .exceptionBuilderSupplier(UnprocessableEntityException::builder).build());
            case "ThrottledClientException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ThrottledClientException").httpStatusCode(429)
                        .exceptionBuilderSupplier(ThrottledClientException::builder).build());
            case "ServiceUnavailableException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(503)
                        .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
            case "ForbiddenException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ForbiddenException").httpStatusCode(403)
                        .exceptionBuilderSupplier(ForbiddenException::builder).build());
            case "ResourceLimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceLimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceLimitExceededException::builder).build());
            case "BadRequestException":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequestException").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Chime");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUserSettings");

            return clientHandler.execute(new ClientExecutionParams<UpdateUserSettingsRequest, UpdateUserSettingsResponse>()
                    .withOperationName("UpdateUserSettings").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateUserSettingsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateUserSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ChimeServiceClientConfigurationBuilder serviceConfigBuilder = new ChimeServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ChimeException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    @Override
    public final ChimeServiceClientConfiguration serviceClientConfiguration() {
        return new ChimeServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
