/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration of an Amazon FSx for OpenZFS volume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OpenZFSVolumeConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<OpenZFSVolumeConfiguration.Builder, OpenZFSVolumeConfiguration> {
    private static final SdkField<String> PARENT_VOLUME_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParentVolumeId").getter(getter(OpenZFSVolumeConfiguration::parentVolumeId))
            .setter(setter(Builder::parentVolumeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentVolumeId").build()).build();

    private static final SdkField<String> VOLUME_PATH_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VolumePath").getter(getter(OpenZFSVolumeConfiguration::volumePath)).setter(setter(Builder::volumePath))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumePath").build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_RESERVATION_GIB_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacityReservationGiB")
            .getter(getter(OpenZFSVolumeConfiguration::storageCapacityReservationGiB))
            .setter(setter(Builder::storageCapacityReservationGiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacityReservationGiB")
                    .build()).build();

    private static final SdkField<Integer> STORAGE_CAPACITY_QUOTA_GIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("StorageCapacityQuotaGiB").getter(getter(OpenZFSVolumeConfiguration::storageCapacityQuotaGiB))
            .setter(setter(Builder::storageCapacityQuotaGiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StorageCapacityQuotaGiB").build())
            .build();

    private static final SdkField<Integer> RECORD_SIZE_KIB_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("RecordSizeKiB").getter(getter(OpenZFSVolumeConfiguration::recordSizeKiB))
            .setter(setter(Builder::recordSizeKiB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecordSizeKiB").build()).build();

    private static final SdkField<String> DATA_COMPRESSION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataCompressionType").getter(getter(OpenZFSVolumeConfiguration::dataCompressionTypeAsString))
            .setter(setter(Builder::dataCompressionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataCompressionType").build())
            .build();

    private static final SdkField<Boolean> COPY_TAGS_TO_SNAPSHOTS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyTagsToSnapshots").getter(getter(OpenZFSVolumeConfiguration::copyTagsToSnapshots))
            .setter(setter(Builder::copyTagsToSnapshots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsToSnapshots").build())
            .build();

    private static final SdkField<OpenZFSOriginSnapshotConfiguration> ORIGIN_SNAPSHOT_FIELD = SdkField
            .<OpenZFSOriginSnapshotConfiguration> builder(MarshallingType.SDK_POJO).memberName("OriginSnapshot")
            .getter(getter(OpenZFSVolumeConfiguration::originSnapshot)).setter(setter(Builder::originSnapshot))
            .constructor(OpenZFSOriginSnapshotConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginSnapshot").build()).build();

    private static final SdkField<Boolean> READ_ONLY_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ReadOnly").getter(getter(OpenZFSVolumeConfiguration::readOnly)).setter(setter(Builder::readOnly))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReadOnly").build()).build();

    private static final SdkField<List<OpenZFSNfsExport>> NFS_EXPORTS_FIELD = SdkField
            .<List<OpenZFSNfsExport>> builder(MarshallingType.LIST)
            .memberName("NfsExports")
            .getter(getter(OpenZFSVolumeConfiguration::nfsExports))
            .setter(setter(Builder::nfsExports))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NfsExports").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpenZFSNfsExport> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpenZFSNfsExport::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<OpenZFSUserOrGroupQuota>> USER_AND_GROUP_QUOTAS_FIELD = SdkField
            .<List<OpenZFSUserOrGroupQuota>> builder(MarshallingType.LIST)
            .memberName("UserAndGroupQuotas")
            .getter(getter(OpenZFSVolumeConfiguration::userAndGroupQuotas))
            .setter(setter(Builder::userAndGroupQuotas))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserAndGroupQuotas").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<OpenZFSUserOrGroupQuota> builder(MarshallingType.SDK_POJO)
                                            .constructor(OpenZFSUserOrGroupQuota::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> RESTORE_TO_SNAPSHOT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RestoreToSnapshot").getter(getter(OpenZFSVolumeConfiguration::restoreToSnapshot))
            .setter(setter(Builder::restoreToSnapshot))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RestoreToSnapshot").build()).build();

    private static final SdkField<Boolean> DELETE_INTERMEDIATE_SNAPHOTS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteIntermediateSnaphots")
            .getter(getter(OpenZFSVolumeConfiguration::deleteIntermediateSnaphots))
            .setter(setter(Builder::deleteIntermediateSnaphots))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteIntermediateSnaphots").build())
            .build();

    private static final SdkField<Boolean> DELETE_CLONED_VOLUMES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteClonedVolumes").getter(getter(OpenZFSVolumeConfiguration::deleteClonedVolumes))
            .setter(setter(Builder::deleteClonedVolumes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteClonedVolumes").build())
            .build();

    private static final SdkField<Boolean> DELETE_INTERMEDIATE_DATA_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteIntermediateData").getter(getter(OpenZFSVolumeConfiguration::deleteIntermediateData))
            .setter(setter(Builder::deleteIntermediateData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteIntermediateData").build())
            .build();

    private static final SdkField<String> SOURCE_SNAPSHOT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceSnapshotARN").getter(getter(OpenZFSVolumeConfiguration::sourceSnapshotARN))
            .setter(setter(Builder::sourceSnapshotARN))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceSnapshotARN").build()).build();

    private static final SdkField<String> DESTINATION_SNAPSHOT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DestinationSnapshot").getter(getter(OpenZFSVolumeConfiguration::destinationSnapshot))
            .setter(setter(Builder::destinationSnapshot))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationSnapshot").build())
            .build();

    private static final SdkField<String> COPY_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CopyStrategy").getter(getter(OpenZFSVolumeConfiguration::copyStrategyAsString))
            .setter(setter(Builder::copyStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyStrategy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PARENT_VOLUME_ID_FIELD,
            VOLUME_PATH_FIELD, STORAGE_CAPACITY_RESERVATION_GIB_FIELD, STORAGE_CAPACITY_QUOTA_GIB_FIELD, RECORD_SIZE_KIB_FIELD,
            DATA_COMPRESSION_TYPE_FIELD, COPY_TAGS_TO_SNAPSHOTS_FIELD, ORIGIN_SNAPSHOT_FIELD, READ_ONLY_FIELD, NFS_EXPORTS_FIELD,
            USER_AND_GROUP_QUOTAS_FIELD, RESTORE_TO_SNAPSHOT_FIELD, DELETE_INTERMEDIATE_SNAPHOTS_FIELD,
            DELETE_CLONED_VOLUMES_FIELD, DELETE_INTERMEDIATE_DATA_FIELD, SOURCE_SNAPSHOT_ARN_FIELD, DESTINATION_SNAPSHOT_FIELD,
            COPY_STRATEGY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String parentVolumeId;

    private final String volumePath;

    private final Integer storageCapacityReservationGiB;

    private final Integer storageCapacityQuotaGiB;

    private final Integer recordSizeKiB;

    private final String dataCompressionType;

    private final Boolean copyTagsToSnapshots;

    private final OpenZFSOriginSnapshotConfiguration originSnapshot;

    private final Boolean readOnly;

    private final List<OpenZFSNfsExport> nfsExports;

    private final List<OpenZFSUserOrGroupQuota> userAndGroupQuotas;

    private final String restoreToSnapshot;

    private final Boolean deleteIntermediateSnaphots;

    private final Boolean deleteClonedVolumes;

    private final Boolean deleteIntermediateData;

    private final String sourceSnapshotARN;

    private final String destinationSnapshot;

    private final String copyStrategy;

    private OpenZFSVolumeConfiguration(BuilderImpl builder) {
        this.parentVolumeId = builder.parentVolumeId;
        this.volumePath = builder.volumePath;
        this.storageCapacityReservationGiB = builder.storageCapacityReservationGiB;
        this.storageCapacityQuotaGiB = builder.storageCapacityQuotaGiB;
        this.recordSizeKiB = builder.recordSizeKiB;
        this.dataCompressionType = builder.dataCompressionType;
        this.copyTagsToSnapshots = builder.copyTagsToSnapshots;
        this.originSnapshot = builder.originSnapshot;
        this.readOnly = builder.readOnly;
        this.nfsExports = builder.nfsExports;
        this.userAndGroupQuotas = builder.userAndGroupQuotas;
        this.restoreToSnapshot = builder.restoreToSnapshot;
        this.deleteIntermediateSnaphots = builder.deleteIntermediateSnaphots;
        this.deleteClonedVolumes = builder.deleteClonedVolumes;
        this.deleteIntermediateData = builder.deleteIntermediateData;
        this.sourceSnapshotARN = builder.sourceSnapshotARN;
        this.destinationSnapshot = builder.destinationSnapshot;
        this.copyStrategy = builder.copyStrategy;
    }

    /**
     * <p>
     * The ID of the parent volume.
     * </p>
     * 
     * @return The ID of the parent volume.
     */
    public final String parentVolumeId() {
        return parentVolumeId;
    }

    /**
     * <p>
     * The path to the volume from the root volume. For example, <code>fsx/parentVolume/volume1</code>.
     * </p>
     * 
     * @return The path to the volume from the root volume. For example, <code>fsx/parentVolume/volume1</code>.
     */
    public final String volumePath() {
        return volumePath;
    }

    /**
     * <p>
     * The amount of storage in gibibytes (GiB) to reserve from the parent volume. You can't reserve more storage than
     * the parent volume has reserved.
     * </p>
     * 
     * @return The amount of storage in gibibytes (GiB) to reserve from the parent volume. You can't reserve more
     *         storage than the parent volume has reserved.
     */
    public final Integer storageCapacityReservationGiB() {
        return storageCapacityReservationGiB;
    }

    /**
     * <p>
     * The maximum amount of storage in gibibtyes (GiB) that the volume can use from its parent. You can specify a quota
     * larger than the storage on the parent volume.
     * </p>
     * 
     * @return The maximum amount of storage in gibibtyes (GiB) that the volume can use from its parent. You can specify
     *         a quota larger than the storage on the parent volume.
     */
    public final Integer storageCapacityQuotaGiB() {
        return storageCapacityQuotaGiB;
    }

    /**
     * <p>
     * The record size of an OpenZFS volume, in kibibytes (KiB). Valid values are 4, 8, 16, 32, 64, 128, 256, 512, or
     * 1024 KiB. The default is 128 KiB. Most workloads should use the default record size. For guidance on when to set
     * a custom record size, see the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     * 
     * @return The record size of an OpenZFS volume, in kibibytes (KiB). Valid values are 4, 8, 16, 32, 64, 128, 256,
     *         512, or 1024 KiB. The default is 128 KiB. Most workloads should use the default record size. For guidance
     *         on when to set a custom record size, see the <i>Amazon FSx for OpenZFS User Guide</i>.
     */
    public final Integer recordSizeKiB() {
        return recordSizeKiB;
    }

    /**
     * <p>
     * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
     * default.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm. Compared
     * to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage utilization.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to Z-Standard,
     * LZ4 is less compute-intensive and delivers higher write throughput speeds.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataCompressionType} will return {@link OpenZFSDataCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataCompressionTypeAsString}.
     * </p>
     * 
     * @return Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code>
     *         by default.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm.
     *         Compared to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage utilization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to
     *         Z-Standard, LZ4 is less compute-intensive and delivers higher write throughput speeds.
     *         </p>
     *         </li>
     * @see OpenZFSDataCompressionType
     */
    public final OpenZFSDataCompressionType dataCompressionType() {
        return OpenZFSDataCompressionType.fromValue(dataCompressionType);
    }

    /**
     * <p>
     * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
     * default.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm. Compared
     * to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage utilization.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to Z-Standard,
     * LZ4 is less compute-intensive and delivers higher write throughput speeds.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #dataCompressionType} will return {@link OpenZFSDataCompressionType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #dataCompressionTypeAsString}.
     * </p>
     * 
     * @return Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code>
     *         by default.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm.
     *         Compared to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage utilization.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to
     *         Z-Standard, LZ4 is less compute-intensive and delivers higher write throughput speeds.
     *         </p>
     *         </li>
     * @see OpenZFSDataCompressionType
     */
    public final String dataCompressionTypeAsString() {
        return dataCompressionType;
    }

    /**
     * <p>
     * A Boolean value indicating whether tags for the volume should be copied to snapshots. This value defaults to
     * <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied to snapshots where the
     * user doesn't specify tags. If this value is <code>true</code> and you specify one or more tags, only the
     * specified tags are copied to snapshots. If you specify one or more tags when creating the snapshot, no tags are
     * copied from the volume, regardless of this value.
     * </p>
     * 
     * @return A Boolean value indicating whether tags for the volume should be copied to snapshots. This value defaults
     *         to <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied to snapshots
     *         where the user doesn't specify tags. If this value is <code>true</code> and you specify one or more tags,
     *         only the specified tags are copied to snapshots. If you specify one or more tags when creating the
     *         snapshot, no tags are copied from the volume, regardless of this value.
     */
    public final Boolean copyTagsToSnapshots() {
        return copyTagsToSnapshots;
    }

    /**
     * <p>
     * The configuration object that specifies the snapshot to use as the origin of the data for the volume.
     * </p>
     * 
     * @return The configuration object that specifies the snapshot to use as the origin of the data for the volume.
     */
    public final OpenZFSOriginSnapshotConfiguration originSnapshot() {
        return originSnapshot;
    }

    /**
     * <p>
     * A Boolean value indicating whether the volume is read-only.
     * </p>
     * 
     * @return A Boolean value indicating whether the volume is read-only.
     */
    public final Boolean readOnly() {
        return readOnly;
    }

    /**
     * For responses, this returns true if the service returned a value for the NfsExports property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasNfsExports() {
        return nfsExports != null && !(nfsExports instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The configuration object for mounting a Network File System (NFS) file system.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNfsExports} method.
     * </p>
     * 
     * @return The configuration object for mounting a Network File System (NFS) file system.
     */
    public final List<OpenZFSNfsExport> nfsExports() {
        return nfsExports;
    }

    /**
     * For responses, this returns true if the service returned a value for the UserAndGroupQuotas property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasUserAndGroupQuotas() {
        return userAndGroupQuotas != null && !(userAndGroupQuotas instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An object specifying how much storage users or groups can use on the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasUserAndGroupQuotas} method.
     * </p>
     * 
     * @return An object specifying how much storage users or groups can use on the volume.
     */
    public final List<OpenZFSUserOrGroupQuota> userAndGroupQuotas() {
        return userAndGroupQuotas;
    }

    /**
     * <p>
     * Specifies the ID of the snapshot to which the volume was restored.
     * </p>
     * 
     * @return Specifies the ID of the snapshot to which the volume was restored.
     */
    public final String restoreToSnapshot() {
        return restoreToSnapshot;
    }

    /**
     * <p>
     * A Boolean value indicating whether snapshots between the current state and the specified snapshot should be
     * deleted when a volume is restored from snapshot.
     * </p>
     * 
     * @return A Boolean value indicating whether snapshots between the current state and the specified snapshot should
     *         be deleted when a volume is restored from snapshot.
     */
    public final Boolean deleteIntermediateSnaphots() {
        return deleteIntermediateSnaphots;
    }

    /**
     * <p>
     * A Boolean value indicating whether dependent clone volumes created from intermediate snapshots should be deleted
     * when a volume is restored from snapshot.
     * </p>
     * 
     * @return A Boolean value indicating whether dependent clone volumes created from intermediate snapshots should be
     *         deleted when a volume is restored from snapshot.
     */
    public final Boolean deleteClonedVolumes() {
        return deleteClonedVolumes;
    }

    /**
     * <p>
     * A Boolean value indicating whether snapshot data that differs between the current state and the specified
     * snapshot should be overwritten when a volume is restored from a snapshot.
     * </p>
     * 
     * @return A Boolean value indicating whether snapshot data that differs between the current state and the specified
     *         snapshot should be overwritten when a volume is restored from a snapshot.
     */
    public final Boolean deleteIntermediateData() {
        return deleteIntermediateData;
    }

    /**
     * Returns the value of the SourceSnapshotARN property for this object.
     * 
     * @return The value of the SourceSnapshotARN property for this object.
     */
    public final String sourceSnapshotARN() {
        return sourceSnapshotARN;
    }

    /**
     * <p>
     * The ID of the snapshot that's being copied or was most recently copied to the destination volume.
     * </p>
     * 
     * @return The ID of the snapshot that's being copied or was most recently copied to the destination volume.
     */
    public final String destinationSnapshot() {
        return destinationSnapshot;
    }

    /**
     * <p>
     * Specifies the strategy used when copying data from the snapshot to the new volume.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is faster than
     * copying data from the snapshot to a new volume and doesn't consume disk throughput. However, the origin snapshot
     * can't be deleted if there is a volume using its copied data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
     * </p>
     * <p>
     * Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot from another
     * FSx for OpenZFS file system. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
     * >CopySnapshotAndUpdateVolume</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #copyStrategy} will
     * return {@link OpenZFSCopyStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #copyStrategyAsString}.
     * </p>
     * 
     * @return Specifies the strategy used when copying data from the snapshot to the new volume. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is
     *         faster than copying data from the snapshot to a new volume and doesn't consume disk throughput. However,
     *         the origin snapshot can't be deleted if there is a volume using its copied data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
     *         </p>
     *         <p>
     *         Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot from
     *         another FSx for OpenZFS file system. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
     *         >CopySnapshotAndUpdateVolume</a>.
     *         </p>
     * @see OpenZFSCopyStrategy
     */
    public final OpenZFSCopyStrategy copyStrategy() {
        return OpenZFSCopyStrategy.fromValue(copyStrategy);
    }

    /**
     * <p>
     * Specifies the strategy used when copying data from the snapshot to the new volume.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is faster than
     * copying data from the snapshot to a new volume and doesn't consume disk throughput. However, the origin snapshot
     * can't be deleted if there is a volume using its copied data.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
     * </p>
     * <p>
     * Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot from another
     * FSx for OpenZFS file system. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
     * >CopySnapshotAndUpdateVolume</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #copyStrategy} will
     * return {@link OpenZFSCopyStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #copyStrategyAsString}.
     * </p>
     * 
     * @return Specifies the strategy used when copying data from the snapshot to the new volume. </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is
     *         faster than copying data from the snapshot to a new volume and doesn't consume disk throughput. However,
     *         the origin snapshot can't be deleted if there is a volume using its copied data.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
     *         </p>
     *         <p>
     *         Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot from
     *         another FSx for OpenZFS file system. For more information, see <a
     *         href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
     *         >CopySnapshotAndUpdateVolume</a>.
     *         </p>
     * @see OpenZFSCopyStrategy
     */
    public final String copyStrategyAsString() {
        return copyStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(parentVolumeId());
        hashCode = 31 * hashCode + Objects.hashCode(volumePath());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacityReservationGiB());
        hashCode = 31 * hashCode + Objects.hashCode(storageCapacityQuotaGiB());
        hashCode = 31 * hashCode + Objects.hashCode(recordSizeKiB());
        hashCode = 31 * hashCode + Objects.hashCode(dataCompressionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsToSnapshots());
        hashCode = 31 * hashCode + Objects.hashCode(originSnapshot());
        hashCode = 31 * hashCode + Objects.hashCode(readOnly());
        hashCode = 31 * hashCode + Objects.hashCode(hasNfsExports() ? nfsExports() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasUserAndGroupQuotas() ? userAndGroupQuotas() : null);
        hashCode = 31 * hashCode + Objects.hashCode(restoreToSnapshot());
        hashCode = 31 * hashCode + Objects.hashCode(deleteIntermediateSnaphots());
        hashCode = 31 * hashCode + Objects.hashCode(deleteClonedVolumes());
        hashCode = 31 * hashCode + Objects.hashCode(deleteIntermediateData());
        hashCode = 31 * hashCode + Objects.hashCode(sourceSnapshotARN());
        hashCode = 31 * hashCode + Objects.hashCode(destinationSnapshot());
        hashCode = 31 * hashCode + Objects.hashCode(copyStrategyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OpenZFSVolumeConfiguration)) {
            return false;
        }
        OpenZFSVolumeConfiguration other = (OpenZFSVolumeConfiguration) obj;
        return Objects.equals(parentVolumeId(), other.parentVolumeId()) && Objects.equals(volumePath(), other.volumePath())
                && Objects.equals(storageCapacityReservationGiB(), other.storageCapacityReservationGiB())
                && Objects.equals(storageCapacityQuotaGiB(), other.storageCapacityQuotaGiB())
                && Objects.equals(recordSizeKiB(), other.recordSizeKiB())
                && Objects.equals(dataCompressionTypeAsString(), other.dataCompressionTypeAsString())
                && Objects.equals(copyTagsToSnapshots(), other.copyTagsToSnapshots())
                && Objects.equals(originSnapshot(), other.originSnapshot()) && Objects.equals(readOnly(), other.readOnly())
                && hasNfsExports() == other.hasNfsExports() && Objects.equals(nfsExports(), other.nfsExports())
                && hasUserAndGroupQuotas() == other.hasUserAndGroupQuotas()
                && Objects.equals(userAndGroupQuotas(), other.userAndGroupQuotas())
                && Objects.equals(restoreToSnapshot(), other.restoreToSnapshot())
                && Objects.equals(deleteIntermediateSnaphots(), other.deleteIntermediateSnaphots())
                && Objects.equals(deleteClonedVolumes(), other.deleteClonedVolumes())
                && Objects.equals(deleteIntermediateData(), other.deleteIntermediateData())
                && Objects.equals(sourceSnapshotARN(), other.sourceSnapshotARN())
                && Objects.equals(destinationSnapshot(), other.destinationSnapshot())
                && Objects.equals(copyStrategyAsString(), other.copyStrategyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OpenZFSVolumeConfiguration").add("ParentVolumeId", parentVolumeId())
                .add("VolumePath", volumePath()).add("StorageCapacityReservationGiB", storageCapacityReservationGiB())
                .add("StorageCapacityQuotaGiB", storageCapacityQuotaGiB()).add("RecordSizeKiB", recordSizeKiB())
                .add("DataCompressionType", dataCompressionTypeAsString()).add("CopyTagsToSnapshots", copyTagsToSnapshots())
                .add("OriginSnapshot", originSnapshot()).add("ReadOnly", readOnly())
                .add("NfsExports", hasNfsExports() ? nfsExports() : null)
                .add("UserAndGroupQuotas", hasUserAndGroupQuotas() ? userAndGroupQuotas() : null)
                .add("RestoreToSnapshot", restoreToSnapshot()).add("DeleteIntermediateSnaphots", deleteIntermediateSnaphots())
                .add("DeleteClonedVolumes", deleteClonedVolumes()).add("DeleteIntermediateData", deleteIntermediateData())
                .add("SourceSnapshotARN", sourceSnapshotARN()).add("DestinationSnapshot", destinationSnapshot())
                .add("CopyStrategy", copyStrategyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ParentVolumeId":
            return Optional.ofNullable(clazz.cast(parentVolumeId()));
        case "VolumePath":
            return Optional.ofNullable(clazz.cast(volumePath()));
        case "StorageCapacityReservationGiB":
            return Optional.ofNullable(clazz.cast(storageCapacityReservationGiB()));
        case "StorageCapacityQuotaGiB":
            return Optional.ofNullable(clazz.cast(storageCapacityQuotaGiB()));
        case "RecordSizeKiB":
            return Optional.ofNullable(clazz.cast(recordSizeKiB()));
        case "DataCompressionType":
            return Optional.ofNullable(clazz.cast(dataCompressionTypeAsString()));
        case "CopyTagsToSnapshots":
            return Optional.ofNullable(clazz.cast(copyTagsToSnapshots()));
        case "OriginSnapshot":
            return Optional.ofNullable(clazz.cast(originSnapshot()));
        case "ReadOnly":
            return Optional.ofNullable(clazz.cast(readOnly()));
        case "NfsExports":
            return Optional.ofNullable(clazz.cast(nfsExports()));
        case "UserAndGroupQuotas":
            return Optional.ofNullable(clazz.cast(userAndGroupQuotas()));
        case "RestoreToSnapshot":
            return Optional.ofNullable(clazz.cast(restoreToSnapshot()));
        case "DeleteIntermediateSnaphots":
            return Optional.ofNullable(clazz.cast(deleteIntermediateSnaphots()));
        case "DeleteClonedVolumes":
            return Optional.ofNullable(clazz.cast(deleteClonedVolumes()));
        case "DeleteIntermediateData":
            return Optional.ofNullable(clazz.cast(deleteIntermediateData()));
        case "SourceSnapshotARN":
            return Optional.ofNullable(clazz.cast(sourceSnapshotARN()));
        case "DestinationSnapshot":
            return Optional.ofNullable(clazz.cast(destinationSnapshot()));
        case "CopyStrategy":
            return Optional.ofNullable(clazz.cast(copyStrategyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OpenZFSVolumeConfiguration, T> g) {
        return obj -> g.apply((OpenZFSVolumeConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OpenZFSVolumeConfiguration> {
        /**
         * <p>
         * The ID of the parent volume.
         * </p>
         * 
         * @param parentVolumeId
         *        The ID of the parent volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentVolumeId(String parentVolumeId);

        /**
         * <p>
         * The path to the volume from the root volume. For example, <code>fsx/parentVolume/volume1</code>.
         * </p>
         * 
         * @param volumePath
         *        The path to the volume from the root volume. For example, <code>fsx/parentVolume/volume1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumePath(String volumePath);

        /**
         * <p>
         * The amount of storage in gibibytes (GiB) to reserve from the parent volume. You can't reserve more storage
         * than the parent volume has reserved.
         * </p>
         * 
         * @param storageCapacityReservationGiB
         *        The amount of storage in gibibytes (GiB) to reserve from the parent volume. You can't reserve more
         *        storage than the parent volume has reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacityReservationGiB(Integer storageCapacityReservationGiB);

        /**
         * <p>
         * The maximum amount of storage in gibibtyes (GiB) that the volume can use from its parent. You can specify a
         * quota larger than the storage on the parent volume.
         * </p>
         * 
         * @param storageCapacityQuotaGiB
         *        The maximum amount of storage in gibibtyes (GiB) that the volume can use from its parent. You can
         *        specify a quota larger than the storage on the parent volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder storageCapacityQuotaGiB(Integer storageCapacityQuotaGiB);

        /**
         * <p>
         * The record size of an OpenZFS volume, in kibibytes (KiB). Valid values are 4, 8, 16, 32, 64, 128, 256, 512,
         * or 1024 KiB. The default is 128 KiB. Most workloads should use the default record size. For guidance on when
         * to set a custom record size, see the <i>Amazon FSx for OpenZFS User Guide</i>.
         * </p>
         * 
         * @param recordSizeKiB
         *        The record size of an OpenZFS volume, in kibibytes (KiB). Valid values are 4, 8, 16, 32, 64, 128, 256,
         *        512, or 1024 KiB. The default is 128 KiB. Most workloads should use the default record size. For
         *        guidance on when to set a custom record size, see the <i>Amazon FSx for OpenZFS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recordSizeKiB(Integer recordSizeKiB);

        /**
         * <p>
         * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
         * default.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm.
         * Compared to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage utilization.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to
         * Z-Standard, LZ4 is less compute-intensive and delivers higher write throughput speeds.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataCompressionType
         *        Specifies the method used to compress the data on the volume. The compression type is
         *        <code>NONE</code> by default.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression
         *        algorithm. Compared to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage
         *        utilization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to
         *        Z-Standard, LZ4 is less compute-intensive and delivers higher write throughput speeds.
         *        </p>
         *        </li>
         * @see OpenZFSDataCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenZFSDataCompressionType
         */
        Builder dataCompressionType(String dataCompressionType);

        /**
         * <p>
         * Specifies the method used to compress the data on the volume. The compression type is <code>NONE</code> by
         * default.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression algorithm.
         * Compared to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage utilization.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to
         * Z-Standard, LZ4 is less compute-intensive and delivers higher write throughput speeds.
         * </p>
         * </li>
         * </ul>
         * 
         * @param dataCompressionType
         *        Specifies the method used to compress the data on the volume. The compression type is
         *        <code>NONE</code> by default.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>NONE</code> - Doesn't compress the data on the volume. <code>NONE</code> is the default.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ZSTD</code> - Compresses the data in the volume using the Zstandard (ZSTD) compression
         *        algorithm. Compared to LZ4, Z-Standard provides a better compression ratio to minimize on-disk storage
         *        utilization.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LZ4</code> - Compresses the data in the volume using the LZ4 compression algorithm. Compared to
         *        Z-Standard, LZ4 is less compute-intensive and delivers higher write throughput speeds.
         *        </p>
         *        </li>
         * @see OpenZFSDataCompressionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenZFSDataCompressionType
         */
        Builder dataCompressionType(OpenZFSDataCompressionType dataCompressionType);

        /**
         * <p>
         * A Boolean value indicating whether tags for the volume should be copied to snapshots. This value defaults to
         * <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied to snapshots where
         * the user doesn't specify tags. If this value is <code>true</code> and you specify one or more tags, only the
         * specified tags are copied to snapshots. If you specify one or more tags when creating the snapshot, no tags
         * are copied from the volume, regardless of this value.
         * </p>
         * 
         * @param copyTagsToSnapshots
         *        A Boolean value indicating whether tags for the volume should be copied to snapshots. This value
         *        defaults to <code>false</code>. If it's set to <code>true</code>, all tags for the volume are copied
         *        to snapshots where the user doesn't specify tags. If this value is <code>true</code> and you specify
         *        one or more tags, only the specified tags are copied to snapshots. If you specify one or more tags
         *        when creating the snapshot, no tags are copied from the volume, regardless of this value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyTagsToSnapshots(Boolean copyTagsToSnapshots);

        /**
         * <p>
         * The configuration object that specifies the snapshot to use as the origin of the data for the volume.
         * </p>
         * 
         * @param originSnapshot
         *        The configuration object that specifies the snapshot to use as the origin of the data for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originSnapshot(OpenZFSOriginSnapshotConfiguration originSnapshot);

        /**
         * <p>
         * The configuration object that specifies the snapshot to use as the origin of the data for the volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link OpenZFSOriginSnapshotConfiguration.Builder} avoiding the need to create one manually via
         * {@link OpenZFSOriginSnapshotConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OpenZFSOriginSnapshotConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #originSnapshot(OpenZFSOriginSnapshotConfiguration)}.
         * 
         * @param originSnapshot
         *        a consumer that will call methods on {@link OpenZFSOriginSnapshotConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #originSnapshot(OpenZFSOriginSnapshotConfiguration)
         */
        default Builder originSnapshot(Consumer<OpenZFSOriginSnapshotConfiguration.Builder> originSnapshot) {
            return originSnapshot(OpenZFSOriginSnapshotConfiguration.builder().applyMutation(originSnapshot).build());
        }

        /**
         * <p>
         * A Boolean value indicating whether the volume is read-only.
         * </p>
         * 
         * @param readOnly
         *        A Boolean value indicating whether the volume is read-only.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder readOnly(Boolean readOnly);

        /**
         * <p>
         * The configuration object for mounting a Network File System (NFS) file system.
         * </p>
         * 
         * @param nfsExports
         *        The configuration object for mounting a Network File System (NFS) file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExports(Collection<OpenZFSNfsExport> nfsExports);

        /**
         * <p>
         * The configuration object for mounting a Network File System (NFS) file system.
         * </p>
         * 
         * @param nfsExports
         *        The configuration object for mounting a Network File System (NFS) file system.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nfsExports(OpenZFSNfsExport... nfsExports);

        /**
         * <p>
         * The configuration object for mounting a Network File System (NFS) file system.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport.Builder#build()} is called immediately and
         * its result is passed to {@link #nfsExports(List<OpenZFSNfsExport>)}.
         * 
         * @param nfsExports
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fsx.model.OpenZFSNfsExport.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #nfsExports(java.util.Collection<OpenZFSNfsExport>)
         */
        Builder nfsExports(Consumer<OpenZFSNfsExport.Builder>... nfsExports);

        /**
         * <p>
         * An object specifying how much storage users or groups can use on the volume.
         * </p>
         * 
         * @param userAndGroupQuotas
         *        An object specifying how much storage users or groups can use on the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAndGroupQuotas(Collection<OpenZFSUserOrGroupQuota> userAndGroupQuotas);

        /**
         * <p>
         * An object specifying how much storage users or groups can use on the volume.
         * </p>
         * 
         * @param userAndGroupQuotas
         *        An object specifying how much storage users or groups can use on the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userAndGroupQuotas(OpenZFSUserOrGroupQuota... userAndGroupQuotas);

        /**
         * <p>
         * An object specifying how much storage users or groups can use on the volume.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota.Builder#build()} is called
         * immediately and its result is passed to {@link #userAndGroupQuotas(List<OpenZFSUserOrGroupQuota>)}.
         * 
         * @param userAndGroupQuotas
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.fsx.model.OpenZFSUserOrGroupQuota.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userAndGroupQuotas(java.util.Collection<OpenZFSUserOrGroupQuota>)
         */
        Builder userAndGroupQuotas(Consumer<OpenZFSUserOrGroupQuota.Builder>... userAndGroupQuotas);

        /**
         * <p>
         * Specifies the ID of the snapshot to which the volume was restored.
         * </p>
         * 
         * @param restoreToSnapshot
         *        Specifies the ID of the snapshot to which the volume was restored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder restoreToSnapshot(String restoreToSnapshot);

        /**
         * <p>
         * A Boolean value indicating whether snapshots between the current state and the specified snapshot should be
         * deleted when a volume is restored from snapshot.
         * </p>
         * 
         * @param deleteIntermediateSnaphots
         *        A Boolean value indicating whether snapshots between the current state and the specified snapshot
         *        should be deleted when a volume is restored from snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteIntermediateSnaphots(Boolean deleteIntermediateSnaphots);

        /**
         * <p>
         * A Boolean value indicating whether dependent clone volumes created from intermediate snapshots should be
         * deleted when a volume is restored from snapshot.
         * </p>
         * 
         * @param deleteClonedVolumes
         *        A Boolean value indicating whether dependent clone volumes created from intermediate snapshots should
         *        be deleted when a volume is restored from snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteClonedVolumes(Boolean deleteClonedVolumes);

        /**
         * <p>
         * A Boolean value indicating whether snapshot data that differs between the current state and the specified
         * snapshot should be overwritten when a volume is restored from a snapshot.
         * </p>
         * 
         * @param deleteIntermediateData
         *        A Boolean value indicating whether snapshot data that differs between the current state and the
         *        specified snapshot should be overwritten when a volume is restored from a snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteIntermediateData(Boolean deleteIntermediateData);

        /**
         * Sets the value of the SourceSnapshotARN property for this object.
         *
         * @param sourceSnapshotARN
         *        The new value for the SourceSnapshotARN property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceSnapshotARN(String sourceSnapshotARN);

        /**
         * <p>
         * The ID of the snapshot that's being copied or was most recently copied to the destination volume.
         * </p>
         * 
         * @param destinationSnapshot
         *        The ID of the snapshot that's being copied or was most recently copied to the destination volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationSnapshot(String destinationSnapshot);

        /**
         * <p>
         * Specifies the strategy used when copying data from the snapshot to the new volume.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is faster
         * than copying data from the snapshot to a new volume and doesn't consume disk throughput. However, the origin
         * snapshot can't be deleted if there is a volume using its copied data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
         * </p>
         * <p>
         * Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot from
         * another FSx for OpenZFS file system. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
         * >CopySnapshotAndUpdateVolume</a>.
         * </p>
         * </note>
         * 
         * @param copyStrategy
         *        Specifies the strategy used when copying data from the snapshot to the new volume. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is
         *        faster than copying data from the snapshot to a new volume and doesn't consume disk throughput.
         *        However, the origin snapshot can't be deleted if there is a volume using its copied data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
         *        </p>
         *        <p>
         *        Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot
         *        from another FSx for OpenZFS file system. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
         *        >CopySnapshotAndUpdateVolume</a>.
         *        </p>
         * @see OpenZFSCopyStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenZFSCopyStrategy
         */
        Builder copyStrategy(String copyStrategy);

        /**
         * <p>
         * Specifies the strategy used when copying data from the snapshot to the new volume.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is faster
         * than copying data from the snapshot to a new volume and doesn't consume disk throughput. However, the origin
         * snapshot can't be deleted if there is a volume using its copied data.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
         * </p>
         * <p>
         * Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot from
         * another FSx for OpenZFS file system. For more information, see <a
         * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
         * >CopySnapshotAndUpdateVolume</a>.
         * </p>
         * </note>
         * 
         * @param copyStrategy
         *        Specifies the strategy used when copying data from the snapshot to the new volume. </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CLONE</code> - The new volume references the data in the origin snapshot. Cloning a snapshot is
         *        faster than copying data from the snapshot to a new volume and doesn't consume disk throughput.
         *        However, the origin snapshot can't be deleted if there is a volume using its copied data.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>FULL_COPY</code> - Copies all data from the snapshot to the new volume.
         *        </p>
         *        <p>
         *        Specify this option to create the volume from a snapshot on another FSx for OpenZFS file system.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The <code>INCREMENTAL_COPY</code> option is only for updating an existing volume by using a snapshot
         *        from another FSx for OpenZFS file system. For more information, see <a
         *        href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_CopySnapshotAndUpdateVolume.html"
         *        >CopySnapshotAndUpdateVolume</a>.
         *        </p>
         * @see OpenZFSCopyStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OpenZFSCopyStrategy
         */
        Builder copyStrategy(OpenZFSCopyStrategy copyStrategy);
    }

    static final class BuilderImpl implements Builder {
        private String parentVolumeId;

        private String volumePath;

        private Integer storageCapacityReservationGiB;

        private Integer storageCapacityQuotaGiB;

        private Integer recordSizeKiB;

        private String dataCompressionType;

        private Boolean copyTagsToSnapshots;

        private OpenZFSOriginSnapshotConfiguration originSnapshot;

        private Boolean readOnly;

        private List<OpenZFSNfsExport> nfsExports = DefaultSdkAutoConstructList.getInstance();

        private List<OpenZFSUserOrGroupQuota> userAndGroupQuotas = DefaultSdkAutoConstructList.getInstance();

        private String restoreToSnapshot;

        private Boolean deleteIntermediateSnaphots;

        private Boolean deleteClonedVolumes;

        private Boolean deleteIntermediateData;

        private String sourceSnapshotARN;

        private String destinationSnapshot;

        private String copyStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(OpenZFSVolumeConfiguration model) {
            parentVolumeId(model.parentVolumeId);
            volumePath(model.volumePath);
            storageCapacityReservationGiB(model.storageCapacityReservationGiB);
            storageCapacityQuotaGiB(model.storageCapacityQuotaGiB);
            recordSizeKiB(model.recordSizeKiB);
            dataCompressionType(model.dataCompressionType);
            copyTagsToSnapshots(model.copyTagsToSnapshots);
            originSnapshot(model.originSnapshot);
            readOnly(model.readOnly);
            nfsExports(model.nfsExports);
            userAndGroupQuotas(model.userAndGroupQuotas);
            restoreToSnapshot(model.restoreToSnapshot);
            deleteIntermediateSnaphots(model.deleteIntermediateSnaphots);
            deleteClonedVolumes(model.deleteClonedVolumes);
            deleteIntermediateData(model.deleteIntermediateData);
            sourceSnapshotARN(model.sourceSnapshotARN);
            destinationSnapshot(model.destinationSnapshot);
            copyStrategy(model.copyStrategy);
        }

        public final String getParentVolumeId() {
            return parentVolumeId;
        }

        public final void setParentVolumeId(String parentVolumeId) {
            this.parentVolumeId = parentVolumeId;
        }

        @Override
        public final Builder parentVolumeId(String parentVolumeId) {
            this.parentVolumeId = parentVolumeId;
            return this;
        }

        public final String getVolumePath() {
            return volumePath;
        }

        public final void setVolumePath(String volumePath) {
            this.volumePath = volumePath;
        }

        @Override
        public final Builder volumePath(String volumePath) {
            this.volumePath = volumePath;
            return this;
        }

        public final Integer getStorageCapacityReservationGiB() {
            return storageCapacityReservationGiB;
        }

        public final void setStorageCapacityReservationGiB(Integer storageCapacityReservationGiB) {
            this.storageCapacityReservationGiB = storageCapacityReservationGiB;
        }

        @Override
        public final Builder storageCapacityReservationGiB(Integer storageCapacityReservationGiB) {
            this.storageCapacityReservationGiB = storageCapacityReservationGiB;
            return this;
        }

        public final Integer getStorageCapacityQuotaGiB() {
            return storageCapacityQuotaGiB;
        }

        public final void setStorageCapacityQuotaGiB(Integer storageCapacityQuotaGiB) {
            this.storageCapacityQuotaGiB = storageCapacityQuotaGiB;
        }

        @Override
        public final Builder storageCapacityQuotaGiB(Integer storageCapacityQuotaGiB) {
            this.storageCapacityQuotaGiB = storageCapacityQuotaGiB;
            return this;
        }

        public final Integer getRecordSizeKiB() {
            return recordSizeKiB;
        }

        public final void setRecordSizeKiB(Integer recordSizeKiB) {
            this.recordSizeKiB = recordSizeKiB;
        }

        @Override
        public final Builder recordSizeKiB(Integer recordSizeKiB) {
            this.recordSizeKiB = recordSizeKiB;
            return this;
        }

        public final String getDataCompressionType() {
            return dataCompressionType;
        }

        public final void setDataCompressionType(String dataCompressionType) {
            this.dataCompressionType = dataCompressionType;
        }

        @Override
        public final Builder dataCompressionType(String dataCompressionType) {
            this.dataCompressionType = dataCompressionType;
            return this;
        }

        @Override
        public final Builder dataCompressionType(OpenZFSDataCompressionType dataCompressionType) {
            this.dataCompressionType(dataCompressionType == null ? null : dataCompressionType.toString());
            return this;
        }

        public final Boolean getCopyTagsToSnapshots() {
            return copyTagsToSnapshots;
        }

        public final void setCopyTagsToSnapshots(Boolean copyTagsToSnapshots) {
            this.copyTagsToSnapshots = copyTagsToSnapshots;
        }

        @Override
        public final Builder copyTagsToSnapshots(Boolean copyTagsToSnapshots) {
            this.copyTagsToSnapshots = copyTagsToSnapshots;
            return this;
        }

        public final OpenZFSOriginSnapshotConfiguration.Builder getOriginSnapshot() {
            return originSnapshot != null ? originSnapshot.toBuilder() : null;
        }

        public final void setOriginSnapshot(OpenZFSOriginSnapshotConfiguration.BuilderImpl originSnapshot) {
            this.originSnapshot = originSnapshot != null ? originSnapshot.build() : null;
        }

        @Override
        public final Builder originSnapshot(OpenZFSOriginSnapshotConfiguration originSnapshot) {
            this.originSnapshot = originSnapshot;
            return this;
        }

        public final Boolean getReadOnly() {
            return readOnly;
        }

        public final void setReadOnly(Boolean readOnly) {
            this.readOnly = readOnly;
        }

        @Override
        public final Builder readOnly(Boolean readOnly) {
            this.readOnly = readOnly;
            return this;
        }

        public final List<OpenZFSNfsExport.Builder> getNfsExports() {
            List<OpenZFSNfsExport.Builder> result = OpenZFSNfsExportsCopier.copyToBuilder(this.nfsExports);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNfsExports(Collection<OpenZFSNfsExport.BuilderImpl> nfsExports) {
            this.nfsExports = OpenZFSNfsExportsCopier.copyFromBuilder(nfsExports);
        }

        @Override
        public final Builder nfsExports(Collection<OpenZFSNfsExport> nfsExports) {
            this.nfsExports = OpenZFSNfsExportsCopier.copy(nfsExports);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nfsExports(OpenZFSNfsExport... nfsExports) {
            nfsExports(Arrays.asList(nfsExports));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nfsExports(Consumer<OpenZFSNfsExport.Builder>... nfsExports) {
            nfsExports(Stream.of(nfsExports).map(c -> OpenZFSNfsExport.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<OpenZFSUserOrGroupQuota.Builder> getUserAndGroupQuotas() {
            List<OpenZFSUserOrGroupQuota.Builder> result = OpenZFSUserAndGroupQuotasCopier.copyToBuilder(this.userAndGroupQuotas);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setUserAndGroupQuotas(Collection<OpenZFSUserOrGroupQuota.BuilderImpl> userAndGroupQuotas) {
            this.userAndGroupQuotas = OpenZFSUserAndGroupQuotasCopier.copyFromBuilder(userAndGroupQuotas);
        }

        @Override
        public final Builder userAndGroupQuotas(Collection<OpenZFSUserOrGroupQuota> userAndGroupQuotas) {
            this.userAndGroupQuotas = OpenZFSUserAndGroupQuotasCopier.copy(userAndGroupQuotas);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAndGroupQuotas(OpenZFSUserOrGroupQuota... userAndGroupQuotas) {
            userAndGroupQuotas(Arrays.asList(userAndGroupQuotas));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder userAndGroupQuotas(Consumer<OpenZFSUserOrGroupQuota.Builder>... userAndGroupQuotas) {
            userAndGroupQuotas(Stream.of(userAndGroupQuotas).map(c -> OpenZFSUserOrGroupQuota.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getRestoreToSnapshot() {
            return restoreToSnapshot;
        }

        public final void setRestoreToSnapshot(String restoreToSnapshot) {
            this.restoreToSnapshot = restoreToSnapshot;
        }

        @Override
        public final Builder restoreToSnapshot(String restoreToSnapshot) {
            this.restoreToSnapshot = restoreToSnapshot;
            return this;
        }

        public final Boolean getDeleteIntermediateSnaphots() {
            return deleteIntermediateSnaphots;
        }

        public final void setDeleteIntermediateSnaphots(Boolean deleteIntermediateSnaphots) {
            this.deleteIntermediateSnaphots = deleteIntermediateSnaphots;
        }

        @Override
        public final Builder deleteIntermediateSnaphots(Boolean deleteIntermediateSnaphots) {
            this.deleteIntermediateSnaphots = deleteIntermediateSnaphots;
            return this;
        }

        public final Boolean getDeleteClonedVolumes() {
            return deleteClonedVolumes;
        }

        public final void setDeleteClonedVolumes(Boolean deleteClonedVolumes) {
            this.deleteClonedVolumes = deleteClonedVolumes;
        }

        @Override
        public final Builder deleteClonedVolumes(Boolean deleteClonedVolumes) {
            this.deleteClonedVolumes = deleteClonedVolumes;
            return this;
        }

        public final Boolean getDeleteIntermediateData() {
            return deleteIntermediateData;
        }

        public final void setDeleteIntermediateData(Boolean deleteIntermediateData) {
            this.deleteIntermediateData = deleteIntermediateData;
        }

        @Override
        public final Builder deleteIntermediateData(Boolean deleteIntermediateData) {
            this.deleteIntermediateData = deleteIntermediateData;
            return this;
        }

        public final String getSourceSnapshotARN() {
            return sourceSnapshotARN;
        }

        public final void setSourceSnapshotARN(String sourceSnapshotARN) {
            this.sourceSnapshotARN = sourceSnapshotARN;
        }

        @Override
        public final Builder sourceSnapshotARN(String sourceSnapshotARN) {
            this.sourceSnapshotARN = sourceSnapshotARN;
            return this;
        }

        public final String getDestinationSnapshot() {
            return destinationSnapshot;
        }

        public final void setDestinationSnapshot(String destinationSnapshot) {
            this.destinationSnapshot = destinationSnapshot;
        }

        @Override
        public final Builder destinationSnapshot(String destinationSnapshot) {
            this.destinationSnapshot = destinationSnapshot;
            return this;
        }

        public final String getCopyStrategy() {
            return copyStrategy;
        }

        public final void setCopyStrategy(String copyStrategy) {
            this.copyStrategy = copyStrategy;
        }

        @Override
        public final Builder copyStrategy(String copyStrategy) {
            this.copyStrategy = copyStrategy;
            return this;
        }

        @Override
        public final Builder copyStrategy(OpenZFSCopyStrategy copyStrategy) {
            this.copyStrategy(copyStrategy == null ? null : copyStrategy.toString());
            return this;
        }

        @Override
        public OpenZFSVolumeConfiguration build() {
            return new OpenZFSVolumeConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
