/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Placement group configuration for an Amazon EMR cluster. The configuration specifies the placement strategy that can
 * be applied to instance roles during cluster creation.
 * </p>
 * <p>
 * To use this configuration, consider attaching managed policy AmazonElasticMapReducePlacementGroupPolicy to the Amazon
 * EMR role.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PlacementGroupConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<PlacementGroupConfig.Builder, PlacementGroupConfig> {
    private static final SdkField<String> INSTANCE_ROLE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceRole").getter(getter(PlacementGroupConfig::instanceRoleAsString))
            .setter(setter(Builder::instanceRole))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceRole").build()).build();

    private static final SdkField<String> PLACEMENT_STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PlacementStrategy").getter(getter(PlacementGroupConfig::placementStrategyAsString))
            .setter(setter(Builder::placementStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PlacementStrategy").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ROLE_FIELD,
            PLACEMENT_STRATEGY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceRole;

    private final String placementStrategy;

    private PlacementGroupConfig(BuilderImpl builder) {
        this.instanceRole = builder.instanceRole;
        this.placementStrategy = builder.placementStrategy;
    }

    /**
     * <p>
     * Role of the instance in the cluster.
     * </p>
     * <p>
     * Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceRole} will
     * return {@link InstanceRoleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceRoleAsString}.
     * </p>
     * 
     * @return Role of the instance in the cluster.</p>
     *         <p>
     *         Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
     * @see InstanceRoleType
     */
    public final InstanceRoleType instanceRole() {
        return InstanceRoleType.fromValue(instanceRole);
    }

    /**
     * <p>
     * Role of the instance in the cluster.
     * </p>
     * <p>
     * Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceRole} will
     * return {@link InstanceRoleType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceRoleAsString}.
     * </p>
     * 
     * @return Role of the instance in the cluster.</p>
     *         <p>
     *         Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
     * @see InstanceRoleType
     */
    public final String instanceRoleAsString() {
        return instanceRole;
    }

    /**
     * <p>
     * Amazon EC2 Placement Group strategy associated with instance role.
     * </p>
     * <p>
     * Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code> for the
     * <code>MASTER</code> instance role.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #placementStrategy}
     * will return {@link PlacementGroupStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #placementStrategyAsString}.
     * </p>
     * 
     * @return Amazon EC2 Placement Group strategy associated with instance role.</p>
     *         <p>
     *         Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code> for
     *         the <code>MASTER</code> instance role.
     * @see PlacementGroupStrategy
     */
    public final PlacementGroupStrategy placementStrategy() {
        return PlacementGroupStrategy.fromValue(placementStrategy);
    }

    /**
     * <p>
     * Amazon EC2 Placement Group strategy associated with instance role.
     * </p>
     * <p>
     * Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code> for the
     * <code>MASTER</code> instance role.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #placementStrategy}
     * will return {@link PlacementGroupStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #placementStrategyAsString}.
     * </p>
     * 
     * @return Amazon EC2 Placement Group strategy associated with instance role.</p>
     *         <p>
     *         Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code> for
     *         the <code>MASTER</code> instance role.
     * @see PlacementGroupStrategy
     */
    public final String placementStrategyAsString() {
        return placementStrategy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceRoleAsString());
        hashCode = 31 * hashCode + Objects.hashCode(placementStrategyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PlacementGroupConfig)) {
            return false;
        }
        PlacementGroupConfig other = (PlacementGroupConfig) obj;
        return Objects.equals(instanceRoleAsString(), other.instanceRoleAsString())
                && Objects.equals(placementStrategyAsString(), other.placementStrategyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PlacementGroupConfig").add("InstanceRole", instanceRoleAsString())
                .add("PlacementStrategy", placementStrategyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceRole":
            return Optional.ofNullable(clazz.cast(instanceRoleAsString()));
        case "PlacementStrategy":
            return Optional.ofNullable(clazz.cast(placementStrategyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PlacementGroupConfig, T> g) {
        return obj -> g.apply((PlacementGroupConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PlacementGroupConfig> {
        /**
         * <p>
         * Role of the instance in the cluster.
         * </p>
         * <p>
         * Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
         * </p>
         * 
         * @param instanceRole
         *        Role of the instance in the cluster.</p>
         *        <p>
         *        Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
         * @see InstanceRoleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceRoleType
         */
        Builder instanceRole(String instanceRole);

        /**
         * <p>
         * Role of the instance in the cluster.
         * </p>
         * <p>
         * Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
         * </p>
         * 
         * @param instanceRole
         *        Role of the instance in the cluster.</p>
         *        <p>
         *        Starting with Amazon EMR release 5.23.0, the only supported instance role is <code>MASTER</code>.
         * @see InstanceRoleType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceRoleType
         */
        Builder instanceRole(InstanceRoleType instanceRole);

        /**
         * <p>
         * Amazon EC2 Placement Group strategy associated with instance role.
         * </p>
         * <p>
         * Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code> for the
         * <code>MASTER</code> instance role.
         * </p>
         * 
         * @param placementStrategy
         *        Amazon EC2 Placement Group strategy associated with instance role.</p>
         *        <p>
         *        Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code>
         *        for the <code>MASTER</code> instance role.
         * @see PlacementGroupStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementGroupStrategy
         */
        Builder placementStrategy(String placementStrategy);

        /**
         * <p>
         * Amazon EC2 Placement Group strategy associated with instance role.
         * </p>
         * <p>
         * Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code> for the
         * <code>MASTER</code> instance role.
         * </p>
         * 
         * @param placementStrategy
         *        Amazon EC2 Placement Group strategy associated with instance role.</p>
         *        <p>
         *        Starting with Amazon EMR release 5.23.0, the only supported placement strategy is <code>SPREAD</code>
         *        for the <code>MASTER</code> instance role.
         * @see PlacementGroupStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlacementGroupStrategy
         */
        Builder placementStrategy(PlacementGroupStrategy placementStrategy);
    }

    static final class BuilderImpl implements Builder {
        private String instanceRole;

        private String placementStrategy;

        private BuilderImpl() {
        }

        private BuilderImpl(PlacementGroupConfig model) {
            instanceRole(model.instanceRole);
            placementStrategy(model.placementStrategy);
        }

        public final String getInstanceRole() {
            return instanceRole;
        }

        public final void setInstanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
        }

        @Override
        public final Builder instanceRole(String instanceRole) {
            this.instanceRole = instanceRole;
            return this;
        }

        @Override
        public final Builder instanceRole(InstanceRoleType instanceRole) {
            this.instanceRole(instanceRole == null ? null : instanceRole.toString());
            return this;
        }

        public final String getPlacementStrategy() {
            return placementStrategy;
        }

        public final void setPlacementStrategy(String placementStrategy) {
            this.placementStrategy = placementStrategy;
        }

        @Override
        public final Builder placementStrategy(String placementStrategy) {
            this.placementStrategy = placementStrategy;
            return this;
        }

        @Override
        public final Builder placementStrategy(PlacementGroupStrategy placementStrategy) {
            this.placementStrategy(placementStrategy == null ? null : placementStrategy.toString());
            return this;
        }

        @Override
        public PlacementGroupConfig build() {
            return new PlacementGroupConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
