/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Optional deployment parameters that control how many tasks run during a deployment and the ordering of stopping and
 * starting tasks.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentConfiguration.Builder, DeploymentConfiguration> {
    private static final SdkField<DeploymentCircuitBreaker> DEPLOYMENT_CIRCUIT_BREAKER_FIELD = SdkField
            .<DeploymentCircuitBreaker> builder(MarshallingType.SDK_POJO).memberName("deploymentCircuitBreaker")
            .getter(getter(DeploymentConfiguration::deploymentCircuitBreaker)).setter(setter(Builder::deploymentCircuitBreaker))
            .constructor(DeploymentCircuitBreaker::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentCircuitBreaker").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maximumPercent").getter(getter(DeploymentConfiguration::maximumPercent))
            .setter(setter(Builder::maximumPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumPercent").build()).build();

    private static final SdkField<Integer> MINIMUM_HEALTHY_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("minimumHealthyPercent").getter(getter(DeploymentConfiguration::minimumHealthyPercent))
            .setter(setter(Builder::minimumHealthyPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumHealthyPercent").build())
            .build();

    private static final SdkField<DeploymentAlarms> ALARMS_FIELD = SdkField.<DeploymentAlarms> builder(MarshallingType.SDK_POJO)
            .memberName("alarms").getter(getter(DeploymentConfiguration::alarms)).setter(setter(Builder::alarms))
            .constructor(DeploymentAlarms::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alarms").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            DEPLOYMENT_CIRCUIT_BREAKER_FIELD, MAXIMUM_PERCENT_FIELD, MINIMUM_HEALTHY_PERCENT_FIELD, ALARMS_FIELD));

    private static final long serialVersionUID = 1L;

    private final DeploymentCircuitBreaker deploymentCircuitBreaker;

    private final Integer maximumPercent;

    private final Integer minimumHealthyPercent;

    private final DeploymentAlarms alarms;

    private DeploymentConfiguration(BuilderImpl builder) {
        this.deploymentCircuitBreaker = builder.deploymentCircuitBreaker;
        this.maximumPercent = builder.maximumPercent;
        this.minimumHealthyPercent = builder.minimumHealthyPercent;
        this.alarms = builder.alarms;
    }

    /**
     * <note>
     * <p>
     * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
     * deployment type.
     * </p>
     * </note>
     * <p>
     * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
     * reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a failed
     * state and stop launching new tasks. If you use the rollback option, when a service deployment fails, the service
     * is rolled back to the last deployment that completed successfully. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling update</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>
     * </p>
     * 
     * @return <p>
     *         The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
     *         deployment type.
     *         </p>
     *         </note>
     *         <p>
     *         The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service
     *         can't reach a steady state. If you use the deployment circuit breaker, a service deployment will
     *         transition to a failed state and stop launching new tasks. If you use the rollback option, when a service
     *         deployment fails, the service is rolled back to the last deployment that completed successfully. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
     *         update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
     */
    public final DeploymentCircuitBreaker deploymentCircuitBreaker() {
        return deploymentCircuitBreaker;
    }

    /**
     * <p>
     * If a service is using the rolling update (<code>ECS</code>) deployment type, the <code>maximumPercent</code>
     * parameter represents an upper limit on the number of your service's tasks that are allowed in the
     * <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the
     * <code>desiredCount</code> (rounded down to the nearest integer). This parameter enables you to define the
     * deployment batch size. For example, if your service is using the <code>REPLICA</code> service scheduler and has a
     * <code>desiredCount</code> of four tasks and a <code>maximumPercent</code> value of 200%, the scheduler may start
     * four new tasks before stopping the four older tasks (provided that the cluster resources required to do this are
     * available). The default <code>maximumPercent</code> value for a service using the <code>REPLICA</code> service
     * scheduler is 200%.
     * </p>
     * <p>
     * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
     * and tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the default value and is used
     * to define the upper limit on the number of the tasks in the service that remain in the <code>RUNNING</code> state
     * while the container instances are in the <code>DRAINING</code> state. If the tasks in the service use the Fargate
     * launch type, the maximum percent value is not used, although it is returned when describing your service.
     * </p>
     * 
     * @return If a service is using the rolling update (<code>ECS</code>) deployment type, the
     *         <code>maximumPercent</code> parameter represents an upper limit on the number of your service's tasks
     *         that are allowed in the <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a
     *         percentage of the <code>desiredCount</code> (rounded down to the nearest integer). This parameter enables
     *         you to define the deployment batch size. For example, if your service is using the <code>REPLICA</code>
     *         service scheduler and has a <code>desiredCount</code> of four tasks and a <code>maximumPercent</code>
     *         value of 200%, the scheduler may start four new tasks before stopping the four older tasks (provided that
     *         the cluster resources required to do this are available). The default <code>maximumPercent</code> value
     *         for a service using the <code>REPLICA</code> service scheduler is 200%.</p>
     *         <p>
     *         If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
     *         deployment types and tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the
     *         default value and is used to define the upper limit on the number of the tasks in the service that remain
     *         in the <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state.
     *         If the tasks in the service use the Fargate launch type, the maximum percent value is not used, although
     *         it is returned when describing your service.
     */
    public final Integer maximumPercent() {
        return maximumPercent;
    }

    /**
     * <p>
     * If a service is using the rolling update (<code>ECS</code>) deployment type, the
     * <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that must
     * remain in the <code>RUNNING</code> state during a deployment, as a percentage of the <code>desiredCount</code>
     * (rounded up to the nearest integer). This parameter enables you to deploy without using additional cluster
     * capacity. For example, if your service has a <code>desiredCount</code> of four tasks and a
     * <code>minimumHealthyPercent</code> of 50%, the service scheduler may stop two existing tasks to free up cluster
     * capacity before starting two new tasks.
     * </p>
     * <p>
     * For services that <i>do not</i> use a load balancer, the following should be noted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A service is considered healthy if all essential containers within the tasks in the service pass their health
     * checks.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a task has no essential containers with a health check defined, the service scheduler will wait for 40 seconds
     * after a task reaches a <code>RUNNING</code> state before the task is counted towards the minimum healthy percent
     * total.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a task has one or more essential containers with a health check defined, the service scheduler will wait for
     * the task to reach a healthy status before counting it towards the minimum healthy percent total. A task is
     * considered healthy when all essential containers within the task have passed their health checks. The amount of
     * time the service scheduler can wait for is determined by the container health check settings.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For services that <i>do</i> use a load balancer, the following should be noted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If a task has no essential containers with a health check defined, the service scheduler will wait for the load
     * balancer target group health check to return a healthy status before counting the task towards the minimum
     * healthy percent total.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a task has an essential container with a health check defined, the service scheduler will wait for both the
     * task to reach a healthy status and the load balancer target group health check to return a healthy status before
     * counting the task towards the minimum healthy percent total.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
     * and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to the default
     * value and is used to define the lower limit on the number of the tasks in the service that remain in the
     * <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state. If a service is
     * using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and is running
     * tasks that use the Fargate launch type, the minimum healthy percent value is not used, although it is returned
     * when describing your service.
     * </p>
     * 
     * @return If a service is using the rolling update (<code>ECS</code>) deployment type, the
     *         <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that
     *         must remain in the <code>RUNNING</code> state during a deployment, as a percentage of the
     *         <code>desiredCount</code> (rounded up to the nearest integer). This parameter enables you to deploy
     *         without using additional cluster capacity. For example, if your service has a <code>desiredCount</code>
     *         of four tasks and a <code>minimumHealthyPercent</code> of 50%, the service scheduler may stop two
     *         existing tasks to free up cluster capacity before starting two new tasks. </p>
     *         <p>
     *         For services that <i>do not</i> use a load balancer, the following should be noted:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A service is considered healthy if all essential containers within the tasks in the service pass their
     *         health checks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a task has no essential containers with a health check defined, the service scheduler will wait for 40
     *         seconds after a task reaches a <code>RUNNING</code> state before the task is counted towards the minimum
     *         healthy percent total.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a task has one or more essential containers with a health check defined, the service scheduler will
     *         wait for the task to reach a healthy status before counting it towards the minimum healthy percent total.
     *         A task is considered healthy when all essential containers within the task have passed their health
     *         checks. The amount of time the service scheduler can wait for is determined by the container health check
     *         settings.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For services that <i>do</i> use a load balancer, the following should be noted:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If a task has no essential containers with a health check defined, the service scheduler will wait for
     *         the load balancer target group health check to return a healthy status before counting the task towards
     *         the minimum healthy percent total.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a task has an essential container with a health check defined, the service scheduler will wait for
     *         both the task to reach a healthy status and the load balancer target group health check to return a
     *         healthy status before counting the task towards the minimum healthy percent total.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
     *         deployment types and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b>
     *         value is set to the default value and is used to define the lower limit on the number of the tasks in the
     *         service that remain in the <code>RUNNING</code> state while the container instances are in the
     *         <code>DRAINING</code> state. If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or
     *         <code>EXTERNAL</code> deployment types and is running tasks that use the Fargate launch type, the minimum
     *         healthy percent value is not used, although it is returned when describing your service.
     */
    public final Integer minimumHealthyPercent() {
        return minimumHealthyPercent;
    }

    /**
     * <p>
     * Information about the CloudWatch alarms.
     * </p>
     * 
     * @return Information about the CloudWatch alarms.
     */
    public final DeploymentAlarms alarms() {
        return alarms;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentCircuitBreaker());
        hashCode = 31 * hashCode + Objects.hashCode(maximumPercent());
        hashCode = 31 * hashCode + Objects.hashCode(minimumHealthyPercent());
        hashCode = 31 * hashCode + Objects.hashCode(alarms());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentConfiguration)) {
            return false;
        }
        DeploymentConfiguration other = (DeploymentConfiguration) obj;
        return Objects.equals(deploymentCircuitBreaker(), other.deploymentCircuitBreaker())
                && Objects.equals(maximumPercent(), other.maximumPercent())
                && Objects.equals(minimumHealthyPercent(), other.minimumHealthyPercent())
                && Objects.equals(alarms(), other.alarms());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentConfiguration").add("DeploymentCircuitBreaker", deploymentCircuitBreaker())
                .add("MaximumPercent", maximumPercent()).add("MinimumHealthyPercent", minimumHealthyPercent())
                .add("Alarms", alarms()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentCircuitBreaker":
            return Optional.ofNullable(clazz.cast(deploymentCircuitBreaker()));
        case "maximumPercent":
            return Optional.ofNullable(clazz.cast(maximumPercent()));
        case "minimumHealthyPercent":
            return Optional.ofNullable(clazz.cast(minimumHealthyPercent()));
        case "alarms":
            return Optional.ofNullable(clazz.cast(alarms()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeploymentConfiguration, T> g) {
        return obj -> g.apply((DeploymentConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentConfiguration> {
        /**
         * <note>
         * <p>
         * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
         * deployment type.
         * </p>
         * </note>
         * <p>
         * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
         * reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a
         * failed state and stop launching new tasks. If you use the rollback option, when a service deployment fails,
         * the service is rolled back to the last deployment that completed successfully. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
         * update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
         * </p>
         * 
         * @param deploymentCircuitBreaker
         *        <p>
         *        The deployment circuit breaker can only be used for services using the rolling update (
         *        <code>ECS</code>) deployment type.
         *        </p>
         *        </note>
         *        <p>
         *        The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service
         *        can't reach a steady state. If you use the deployment circuit breaker, a service deployment will
         *        transition to a failed state and stop launching new tasks. If you use the rollback option, when a
         *        service deployment fails, the service is rolled back to the last deployment that completed
         *        successfully. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
         *        update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentCircuitBreaker(DeploymentCircuitBreaker deploymentCircuitBreaker);

        /**
         * <note>
         * <p>
         * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
         * deployment type.
         * </p>
         * </note>
         * <p>
         * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
         * reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a
         * failed state and stop launching new tasks. If you use the rollback option, when a service deployment fails,
         * the service is rolled back to the last deployment that completed successfully. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
         * update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentCircuitBreaker.Builder}
         * avoiding the need to create one manually via {@link DeploymentCircuitBreaker#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentCircuitBreaker.Builder#build()} is called immediately
         * and its result is passed to {@link #deploymentCircuitBreaker(DeploymentCircuitBreaker)}.
         * 
         * @param deploymentCircuitBreaker
         *        a consumer that will call methods on {@link DeploymentCircuitBreaker.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentCircuitBreaker(DeploymentCircuitBreaker)
         */
        default Builder deploymentCircuitBreaker(Consumer<DeploymentCircuitBreaker.Builder> deploymentCircuitBreaker) {
            return deploymentCircuitBreaker(DeploymentCircuitBreaker.builder().applyMutation(deploymentCircuitBreaker).build());
        }

        /**
         * <p>
         * If a service is using the rolling update (<code>ECS</code>) deployment type, the <code>maximumPercent</code>
         * parameter represents an upper limit on the number of your service's tasks that are allowed in the
         * <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the
         * <code>desiredCount</code> (rounded down to the nearest integer). This parameter enables you to define the
         * deployment batch size. For example, if your service is using the <code>REPLICA</code> service scheduler and
         * has a <code>desiredCount</code> of four tasks and a <code>maximumPercent</code> value of 200%, the scheduler
         * may start four new tasks before stopping the four older tasks (provided that the cluster resources required
         * to do this are available). The default <code>maximumPercent</code> value for a service using the
         * <code>REPLICA</code> service scheduler is 200%.
         * </p>
         * <p>
         * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         * types and tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to the default value
         * and is used to define the upper limit on the number of the tasks in the service that remain in the
         * <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state. If the tasks
         * in the service use the Fargate launch type, the maximum percent value is not used, although it is returned
         * when describing your service.
         * </p>
         * 
         * @param maximumPercent
         *        If a service is using the rolling update (<code>ECS</code>) deployment type, the
         *        <code>maximumPercent</code> parameter represents an upper limit on the number of your service's tasks
         *        that are allowed in the <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a
         *        percentage of the <code>desiredCount</code> (rounded down to the nearest integer). This parameter
         *        enables you to define the deployment batch size. For example, if your service is using the
         *        <code>REPLICA</code> service scheduler and has a <code>desiredCount</code> of four tasks and a
         *        <code>maximumPercent</code> value of 200%, the scheduler may start four new tasks before stopping the
         *        four older tasks (provided that the cluster resources required to do this are available). The default
         *        <code>maximumPercent</code> value for a service using the <code>REPLICA</code> service scheduler is
         *        200%.</p>
         *        <p>
         *        If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
         *        deployment types and tasks that use the EC2 launch type, the <b>maximum percent</b> value is set to
         *        the default value and is used to define the upper limit on the number of the tasks in the service that
         *        remain in the <code>RUNNING</code> state while the container instances are in the
         *        <code>DRAINING</code> state. If the tasks in the service use the Fargate launch type, the maximum
         *        percent value is not used, although it is returned when describing your service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPercent(Integer maximumPercent);

        /**
         * <p>
         * If a service is using the rolling update (<code>ECS</code>) deployment type, the
         * <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that must
         * remain in the <code>RUNNING</code> state during a deployment, as a percentage of the
         * <code>desiredCount</code> (rounded up to the nearest integer). This parameter enables you to deploy without
         * using additional cluster capacity. For example, if your service has a <code>desiredCount</code> of four tasks
         * and a <code>minimumHealthyPercent</code> of 50%, the service scheduler may stop two existing tasks to free up
         * cluster capacity before starting two new tasks.
         * </p>
         * <p>
         * For services that <i>do not</i> use a load balancer, the following should be noted:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A service is considered healthy if all essential containers within the tasks in the service pass their health
         * checks.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a task has no essential containers with a health check defined, the service scheduler will wait for 40
         * seconds after a task reaches a <code>RUNNING</code> state before the task is counted towards the minimum
         * healthy percent total.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a task has one or more essential containers with a health check defined, the service scheduler will wait
         * for the task to reach a healthy status before counting it towards the minimum healthy percent total. A task
         * is considered healthy when all essential containers within the task have passed their health checks. The
         * amount of time the service scheduler can wait for is determined by the container health check settings.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For services that <i>do</i> use a load balancer, the following should be noted:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If a task has no essential containers with a health check defined, the service scheduler will wait for the
         * load balancer target group health check to return a healthy status before counting the task towards the
         * minimum healthy percent total.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a task has an essential container with a health check defined, the service scheduler will wait for both
         * the task to reach a healthy status and the load balancer target group health check to return a healthy status
         * before counting the task towards the minimum healthy percent total.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         * types and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to
         * the default value and is used to define the lower limit on the number of the tasks in the service that remain
         * in the <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state. If a
         * service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
         * and is running tasks that use the Fargate launch type, the minimum healthy percent value is not used,
         * although it is returned when describing your service.
         * </p>
         * 
         * @param minimumHealthyPercent
         *        If a service is using the rolling update (<code>ECS</code>) deployment type, the
         *        <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that
         *        must remain in the <code>RUNNING</code> state during a deployment, as a percentage of the
         *        <code>desiredCount</code> (rounded up to the nearest integer). This parameter enables you to deploy
         *        without using additional cluster capacity. For example, if your service has a
         *        <code>desiredCount</code> of four tasks and a <code>minimumHealthyPercent</code> of 50%, the service
         *        scheduler may stop two existing tasks to free up cluster capacity before starting two new tasks. </p>
         *        <p>
         *        For services that <i>do not</i> use a load balancer, the following should be noted:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A service is considered healthy if all essential containers within the tasks in the service pass their
         *        health checks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a task has no essential containers with a health check defined, the service scheduler will wait for
         *        40 seconds after a task reaches a <code>RUNNING</code> state before the task is counted towards the
         *        minimum healthy percent total.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a task has one or more essential containers with a health check defined, the service scheduler will
         *        wait for the task to reach a healthy status before counting it towards the minimum healthy percent
         *        total. A task is considered healthy when all essential containers within the task have passed their
         *        health checks. The amount of time the service scheduler can wait for is determined by the container
         *        health check settings.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For services that <i>do</i> use a load balancer, the following should be noted:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If a task has no essential containers with a health check defined, the service scheduler will wait for
         *        the load balancer target group health check to return a healthy status before counting the task
         *        towards the minimum healthy percent total.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a task has an essential container with a health check defined, the service scheduler will wait for
         *        both the task to reach a healthy status and the load balancer target group health check to return a
         *        healthy status before counting the task towards the minimum healthy percent total.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
         *        deployment types and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b>
         *        value is set to the default value and is used to define the lower limit on the number of the tasks in
         *        the service that remain in the <code>RUNNING</code> state while the container instances are in the
         *        <code>DRAINING</code> state. If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or
         *        <code>EXTERNAL</code> deployment types and is running tasks that use the Fargate launch type, the
         *        minimum healthy percent value is not used, although it is returned when describing your service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumHealthyPercent(Integer minimumHealthyPercent);

        /**
         * <p>
         * Information about the CloudWatch alarms.
         * </p>
         * 
         * @param alarms
         *        Information about the CloudWatch alarms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(DeploymentAlarms alarms);

        /**
         * <p>
         * Information about the CloudWatch alarms.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentAlarms.Builder} avoiding the
         * need to create one manually via {@link DeploymentAlarms#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentAlarms.Builder#build()} is called immediately and its
         * result is passed to {@link #alarms(DeploymentAlarms)}.
         * 
         * @param alarms
         *        a consumer that will call methods on {@link DeploymentAlarms.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarms(DeploymentAlarms)
         */
        default Builder alarms(Consumer<DeploymentAlarms.Builder> alarms) {
            return alarms(DeploymentAlarms.builder().applyMutation(alarms).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private DeploymentCircuitBreaker deploymentCircuitBreaker;

        private Integer maximumPercent;

        private Integer minimumHealthyPercent;

        private DeploymentAlarms alarms;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentConfiguration model) {
            deploymentCircuitBreaker(model.deploymentCircuitBreaker);
            maximumPercent(model.maximumPercent);
            minimumHealthyPercent(model.minimumHealthyPercent);
            alarms(model.alarms);
        }

        public final DeploymentCircuitBreaker.Builder getDeploymentCircuitBreaker() {
            return deploymentCircuitBreaker != null ? deploymentCircuitBreaker.toBuilder() : null;
        }

        public final void setDeploymentCircuitBreaker(DeploymentCircuitBreaker.BuilderImpl deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker != null ? deploymentCircuitBreaker.build() : null;
        }

        @Override
        public final Builder deploymentCircuitBreaker(DeploymentCircuitBreaker deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker;
            return this;
        }

        public final Integer getMaximumPercent() {
            return maximumPercent;
        }

        public final void setMaximumPercent(Integer maximumPercent) {
            this.maximumPercent = maximumPercent;
        }

        @Override
        public final Builder maximumPercent(Integer maximumPercent) {
            this.maximumPercent = maximumPercent;
            return this;
        }

        public final Integer getMinimumHealthyPercent() {
            return minimumHealthyPercent;
        }

        public final void setMinimumHealthyPercent(Integer minimumHealthyPercent) {
            this.minimumHealthyPercent = minimumHealthyPercent;
        }

        @Override
        public final Builder minimumHealthyPercent(Integer minimumHealthyPercent) {
            this.minimumHealthyPercent = minimumHealthyPercent;
            return this;
        }

        public final DeploymentAlarms.Builder getAlarms() {
            return alarms != null ? alarms.toBuilder() : null;
        }

        public final void setAlarms(DeploymentAlarms.BuilderImpl alarms) {
            this.alarms = alarms != null ? alarms.build() : null;
        }

        @Override
        public final Builder alarms(DeploymentAlarms alarms) {
            this.alarms = alarms;
            return this;
        }

        @Override
        public DeploymentConfiguration build() {
            return new DeploymentConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
