/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information from your report plan about where to deliver your reports, specifically your Amazon S3 bucket
 * name, S3 key prefix, and the formats of your reports.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReportDeliveryChannel implements SdkPojo, Serializable,
        ToCopyableBuilder<ReportDeliveryChannel.Builder, ReportDeliveryChannel> {
    private static final SdkField<String> S3_BUCKET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3BucketName").getter(getter(ReportDeliveryChannel::s3BucketName)).setter(setter(Builder::s3BucketName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3BucketName").build()).build();

    private static final SdkField<String> S3_KEY_PREFIX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("S3KeyPrefix").getter(getter(ReportDeliveryChannel::s3KeyPrefix)).setter(setter(Builder::s3KeyPrefix))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3KeyPrefix").build()).build();

    private static final SdkField<List<String>> FORMATS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Formats")
            .getter(getter(ReportDeliveryChannel::formats))
            .setter(setter(Builder::formats))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Formats").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S3_BUCKET_NAME_FIELD,
            S3_KEY_PREFIX_FIELD, FORMATS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s3BucketName;

    private final String s3KeyPrefix;

    private final List<String> formats;

    private ReportDeliveryChannel(BuilderImpl builder) {
        this.s3BucketName = builder.s3BucketName;
        this.s3KeyPrefix = builder.s3KeyPrefix;
        this.formats = builder.formats;
    }

    /**
     * <p>
     * The unique name of the S3 bucket that receives your reports.
     * </p>
     * 
     * @return The unique name of the S3 bucket that receives your reports.
     */
    public final String s3BucketName() {
        return s3BucketName;
    }

    /**
     * <p>
     * The prefix for where Backup Audit Manager delivers your reports to Amazon S3. The prefix is this part of the
     * following path: s3://your-bucket-name/<code>prefix</code>/Backup/us-west-2/year/month/day/report-name. If not
     * specified, there is no prefix.
     * </p>
     * 
     * @return The prefix for where Backup Audit Manager delivers your reports to Amazon S3. The prefix is this part of
     *         the following path: s3://your-bucket-name/<code>prefix</code>
     *         /Backup/us-west-2/year/month/day/report-name. If not specified, there is no prefix.
     */
    public final String s3KeyPrefix() {
        return s3KeyPrefix;
    }

    /**
     * For responses, this returns true if the service returned a value for the Formats property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFormats() {
        return formats != null && !(formats instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the format of your reports: <code>CSV</code>, <code>JSON</code>, or both. If not specified, the default
     * format is <code>CSV</code>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFormats} method.
     * </p>
     * 
     * @return A list of the format of your reports: <code>CSV</code>, <code>JSON</code>, or both. If not specified, the
     *         default format is <code>CSV</code>.
     */
    public final List<String> formats() {
        return formats;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s3BucketName());
        hashCode = 31 * hashCode + Objects.hashCode(s3KeyPrefix());
        hashCode = 31 * hashCode + Objects.hashCode(hasFormats() ? formats() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReportDeliveryChannel)) {
            return false;
        }
        ReportDeliveryChannel other = (ReportDeliveryChannel) obj;
        return Objects.equals(s3BucketName(), other.s3BucketName()) && Objects.equals(s3KeyPrefix(), other.s3KeyPrefix())
                && hasFormats() == other.hasFormats() && Objects.equals(formats(), other.formats());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReportDeliveryChannel").add("S3BucketName", s3BucketName()).add("S3KeyPrefix", s3KeyPrefix())
                .add("Formats", hasFormats() ? formats() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S3BucketName":
            return Optional.ofNullable(clazz.cast(s3BucketName()));
        case "S3KeyPrefix":
            return Optional.ofNullable(clazz.cast(s3KeyPrefix()));
        case "Formats":
            return Optional.ofNullable(clazz.cast(formats()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ReportDeliveryChannel, T> g) {
        return obj -> g.apply((ReportDeliveryChannel) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReportDeliveryChannel> {
        /**
         * <p>
         * The unique name of the S3 bucket that receives your reports.
         * </p>
         * 
         * @param s3BucketName
         *        The unique name of the S3 bucket that receives your reports.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3BucketName(String s3BucketName);

        /**
         * <p>
         * The prefix for where Backup Audit Manager delivers your reports to Amazon S3. The prefix is this part of the
         * following path: s3://your-bucket-name/<code>prefix</code>/Backup/us-west-2/year/month/day/report-name. If not
         * specified, there is no prefix.
         * </p>
         * 
         * @param s3KeyPrefix
         *        The prefix for where Backup Audit Manager delivers your reports to Amazon S3. The prefix is this part
         *        of the following path: s3://your-bucket-name/<code>prefix</code>
         *        /Backup/us-west-2/year/month/day/report-name. If not specified, there is no prefix.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3KeyPrefix(String s3KeyPrefix);

        /**
         * <p>
         * A list of the format of your reports: <code>CSV</code>, <code>JSON</code>, or both. If not specified, the
         * default format is <code>CSV</code>.
         * </p>
         * 
         * @param formats
         *        A list of the format of your reports: <code>CSV</code>, <code>JSON</code>, or both. If not specified,
         *        the default format is <code>CSV</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formats(Collection<String> formats);

        /**
         * <p>
         * A list of the format of your reports: <code>CSV</code>, <code>JSON</code>, or both. If not specified, the
         * default format is <code>CSV</code>.
         * </p>
         * 
         * @param formats
         *        A list of the format of your reports: <code>CSV</code>, <code>JSON</code>, or both. If not specified,
         *        the default format is <code>CSV</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder formats(String... formats);
    }

    static final class BuilderImpl implements Builder {
        private String s3BucketName;

        private String s3KeyPrefix;

        private List<String> formats = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ReportDeliveryChannel model) {
            s3BucketName(model.s3BucketName);
            s3KeyPrefix(model.s3KeyPrefix);
            formats(model.formats);
        }

        public final String getS3BucketName() {
            return s3BucketName;
        }

        public final void setS3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
        }

        @Override
        public final Builder s3BucketName(String s3BucketName) {
            this.s3BucketName = s3BucketName;
            return this;
        }

        public final String getS3KeyPrefix() {
            return s3KeyPrefix;
        }

        public final void setS3KeyPrefix(String s3KeyPrefix) {
            this.s3KeyPrefix = s3KeyPrefix;
        }

        @Override
        public final Builder s3KeyPrefix(String s3KeyPrefix) {
            this.s3KeyPrefix = s3KeyPrefix;
            return this;
        }

        public final Collection<String> getFormats() {
            if (formats instanceof SdkAutoConstructList) {
                return null;
            }
            return formats;
        }

        public final void setFormats(Collection<String> formats) {
            this.formats = FormatListCopier.copy(formats);
        }

        @Override
        public final Builder formats(Collection<String> formats) {
            this.formats = FormatListCopier.copy(formats);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder formats(String... formats) {
            formats(Arrays.asList(formats));
            return this;
        }

        @Override
        public ReportDeliveryChannel build() {
            return new ReportDeliveryChannel(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
