/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains detailed information about a copy job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CopyJob implements SdkPojo, Serializable, ToCopyableBuilder<CopyJob.Builder, CopyJob> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(CopyJob::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> COPY_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CopyJobId").getter(getter(CopyJob::copyJobId)).setter(setter(Builder::copyJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyJobId").build()).build();

    private static final SdkField<String> SOURCE_BACKUP_VAULT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceBackupVaultArn").getter(getter(CopyJob::sourceBackupVaultArn))
            .setter(setter(Builder::sourceBackupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceBackupVaultArn").build())
            .build();

    private static final SdkField<String> SOURCE_RECOVERY_POINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceRecoveryPointArn").getter(getter(CopyJob::sourceRecoveryPointArn))
            .setter(setter(Builder::sourceRecoveryPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRecoveryPointArn").build())
            .build();

    private static final SdkField<String> DESTINATION_BACKUP_VAULT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DestinationBackupVaultArn").getter(getter(CopyJob::destinationBackupVaultArn))
            .setter(setter(Builder::destinationBackupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationBackupVaultArn").build())
            .build();

    private static final SdkField<String> DESTINATION_RECOVERY_POINT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DestinationRecoveryPointArn")
            .getter(getter(CopyJob::destinationRecoveryPointArn))
            .setter(setter(Builder::destinationRecoveryPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationRecoveryPointArn")
                    .build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(CopyJob::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(CopyJob::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<Instant> COMPLETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionDate").getter(getter(CopyJob::completionDate)).setter(setter(Builder::completionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionDate").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(CopyJob::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(CopyJob::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<Long> BACKUP_SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BackupSizeInBytes").getter(getter(CopyJob::backupSizeInBytes))
            .setter(setter(Builder::backupSizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupSizeInBytes").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(CopyJob::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<RecoveryPointCreator> CREATED_BY_FIELD = SdkField
            .<RecoveryPointCreator> builder(MarshallingType.SDK_POJO).memberName("CreatedBy").getter(getter(CopyJob::createdBy))
            .setter(setter(Builder::createdBy)).constructor(RecoveryPointCreator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(CopyJob::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<String> PARENT_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParentJobId").getter(getter(CopyJob::parentJobId)).setter(setter(Builder::parentJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentJobId").build()).build();

    private static final SdkField<Boolean> IS_PARENT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsParent").getter(getter(CopyJob::isParent)).setter(setter(Builder::isParent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsParent").build()).build();

    private static final SdkField<String> COMPOSITE_MEMBER_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CompositeMemberIdentifier").getter(getter(CopyJob::compositeMemberIdentifier))
            .setter(setter(Builder::compositeMemberIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompositeMemberIdentifier").build())
            .build();

    private static final SdkField<Long> NUMBER_OF_CHILD_JOBS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("NumberOfChildJobs").getter(getter(CopyJob::numberOfChildJobs))
            .setter(setter(Builder::numberOfChildJobs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NumberOfChildJobs").build()).build();

    private static final SdkField<Map<String, Long>> CHILD_JOBS_IN_STATE_FIELD = SdkField
            .<Map<String, Long>> builder(MarshallingType.MAP)
            .memberName("ChildJobsInState")
            .getter(getter(CopyJob::childJobsInStateAsStrings))
            .setter(setter(Builder::childJobsInStateWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChildJobsInState").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<Long> builder(MarshallingType.LONG)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> RESOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceName").getter(getter(CopyJob::resourceName)).setter(setter(Builder::resourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceName").build()).build();

    private static final SdkField<String> MESSAGE_CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageCategory").getter(getter(CopyJob::messageCategory)).setter(setter(Builder::messageCategory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageCategory").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            COPY_JOB_ID_FIELD, SOURCE_BACKUP_VAULT_ARN_FIELD, SOURCE_RECOVERY_POINT_ARN_FIELD,
            DESTINATION_BACKUP_VAULT_ARN_FIELD, DESTINATION_RECOVERY_POINT_ARN_FIELD, RESOURCE_ARN_FIELD, CREATION_DATE_FIELD,
            COMPLETION_DATE_FIELD, STATE_FIELD, STATUS_MESSAGE_FIELD, BACKUP_SIZE_IN_BYTES_FIELD, IAM_ROLE_ARN_FIELD,
            CREATED_BY_FIELD, RESOURCE_TYPE_FIELD, PARENT_JOB_ID_FIELD, IS_PARENT_FIELD, COMPOSITE_MEMBER_IDENTIFIER_FIELD,
            NUMBER_OF_CHILD_JOBS_FIELD, CHILD_JOBS_IN_STATE_FIELD, RESOURCE_NAME_FIELD, MESSAGE_CATEGORY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String copyJobId;

    private final String sourceBackupVaultArn;

    private final String sourceRecoveryPointArn;

    private final String destinationBackupVaultArn;

    private final String destinationRecoveryPointArn;

    private final String resourceArn;

    private final Instant creationDate;

    private final Instant completionDate;

    private final String state;

    private final String statusMessage;

    private final Long backupSizeInBytes;

    private final String iamRoleArn;

    private final RecoveryPointCreator createdBy;

    private final String resourceType;

    private final String parentJobId;

    private final Boolean isParent;

    private final String compositeMemberIdentifier;

    private final Long numberOfChildJobs;

    private final Map<String, Long> childJobsInState;

    private final String resourceName;

    private final String messageCategory;

    private CopyJob(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.copyJobId = builder.copyJobId;
        this.sourceBackupVaultArn = builder.sourceBackupVaultArn;
        this.sourceRecoveryPointArn = builder.sourceRecoveryPointArn;
        this.destinationBackupVaultArn = builder.destinationBackupVaultArn;
        this.destinationRecoveryPointArn = builder.destinationRecoveryPointArn;
        this.resourceArn = builder.resourceArn;
        this.creationDate = builder.creationDate;
        this.completionDate = builder.completionDate;
        this.state = builder.state;
        this.statusMessage = builder.statusMessage;
        this.backupSizeInBytes = builder.backupSizeInBytes;
        this.iamRoleArn = builder.iamRoleArn;
        this.createdBy = builder.createdBy;
        this.resourceType = builder.resourceType;
        this.parentJobId = builder.parentJobId;
        this.isParent = builder.isParent;
        this.compositeMemberIdentifier = builder.compositeMemberIdentifier;
        this.numberOfChildJobs = builder.numberOfChildJobs;
        this.childJobsInState = builder.childJobsInState;
        this.resourceName = builder.resourceName;
        this.messageCategory = builder.messageCategory;
    }

    /**
     * <p>
     * The account ID that owns the copy job.
     * </p>
     * 
     * @return The account ID that owns the copy job.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Uniquely identifies a copy job.
     * </p>
     * 
     * @return Uniquely identifies a copy job.
     */
    public final String copyJobId() {
        return copyJobId;
    }

    /**
     * <p>
     * An Amazon Resource Name (ARN) that uniquely identifies a source copy vault; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
     * </p>
     * 
     * @return An Amazon Resource Name (ARN) that uniquely identifies a source copy vault; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
     */
    public final String sourceBackupVaultArn() {
        return sourceBackupVaultArn;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a source recovery point; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     * </p>
     * 
     * @return An ARN that uniquely identifies a source recovery point; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     */
    public final String sourceRecoveryPointArn() {
        return sourceRecoveryPointArn;
    }

    /**
     * <p>
     * An Amazon Resource Name (ARN) that uniquely identifies a destination copy vault; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
     * </p>
     * 
     * @return An Amazon Resource Name (ARN) that uniquely identifies a destination copy vault; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
     */
    public final String destinationBackupVaultArn() {
        return destinationBackupVaultArn;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a destination recovery point; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     * </p>
     * 
     * @return An ARN that uniquely identifies a destination recovery point; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     */
    public final String destinationRecoveryPointArn() {
        return destinationRecoveryPointArn;
    }

    /**
     * <p>
     * The Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon EBS) volume or
     * an Amazon Relational Database Service (Amazon RDS) database.
     * </p>
     * 
     * @return The Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon EBS)
     *         volume or an Amazon Relational Database Service (Amazon RDS) database.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The date and time a copy job is created, in Unix format and Coordinated Universal Time (UTC). The value of
     * <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents Friday,
     * January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time a copy job is created, in Unix format and Coordinated Universal Time (UTC). The value
     *         of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
     *         represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The date and time a copy job is completed, in Unix format and Coordinated Universal Time (UTC). The value of
     * <code>CompletionDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents Friday,
     * January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time a copy job is completed, in Unix format and Coordinated Universal Time (UTC). The value
     *         of <code>CompletionDate</code> is accurate to milliseconds. For example, the value 1516925490.087
     *         represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant completionDate() {
        return completionDate;
    }

    /**
     * <p>
     * The current state of a copy job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CopyJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of a copy job.
     * @see CopyJobState
     */
    public final CopyJobState state() {
        return CopyJobState.fromValue(state);
    }

    /**
     * <p>
     * The current state of a copy job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CopyJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of a copy job.
     * @see CopyJobState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A detailed message explaining the status of the job to copy a resource.
     * </p>
     * 
     * @return A detailed message explaining the status of the job to copy a resource.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * The size, in bytes, of a copy job.
     * </p>
     * 
     * @return The size, in bytes, of a copy job.
     */
    public final Long backupSizeInBytes() {
        return backupSizeInBytes;
    }

    /**
     * <p>
     * Specifies the IAM role ARN used to copy the target recovery point; for example,
     * <code>arn:aws:iam::123456789012:role/S3Access</code>.
     * </p>
     * 
     * @return Specifies the IAM role ARN used to copy the target recovery point; for example,
     *         <code>arn:aws:iam::123456789012:role/S3Access</code>.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * Returns the value of the CreatedBy property for this object.
     * 
     * @return The value of the CreatedBy property for this object.
     */
    public final RecoveryPointCreator createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon EBS)
     * volume or an Amazon Relational Database Service (Amazon RDS) database.
     * </p>
     * 
     * @return The type of Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon
     *         EBS) volume or an Amazon Relational Database Service (Amazon RDS) database.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * This uniquely identifies a request to Backup to copy a resource. The return will be the parent (composite) job
     * ID.
     * </p>
     * 
     * @return This uniquely identifies a request to Backup to copy a resource. The return will be the parent
     *         (composite) job ID.
     */
    public final String parentJobId() {
        return parentJobId;
    }

    /**
     * <p>
     * This is a boolean value indicating this is a parent (composite) copy job.
     * </p>
     * 
     * @return This is a boolean value indicating this is a parent (composite) copy job.
     */
    public final Boolean isParent() {
        return isParent;
    }

    /**
     * <p>
     * This is the identifier of a resource within a composite group, such as nested (child) recovery point belonging to
     * a composite (parent) stack. The ID is transferred from the <a href=
     * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html#resources-section-structure-syntax"
     * > logical ID</a> within a stack.
     * </p>
     * 
     * @return This is the identifier of a resource within a composite group, such as nested (child) recovery point
     *         belonging to a composite (parent) stack. The ID is transferred from the <a href=
     *         "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html#resources-section-structure-syntax"
     *         > logical ID</a> within a stack.
     */
    public final String compositeMemberIdentifier() {
        return compositeMemberIdentifier;
    }

    /**
     * <p>
     * This is the number of child (nested) copy jobs.
     * </p>
     * 
     * @return This is the number of child (nested) copy jobs.
     */
    public final Long numberOfChildJobs() {
        return numberOfChildJobs;
    }

    /**
     * <p>
     * This returns the statistics of the included child (nested) copy jobs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChildJobsInState} method.
     * </p>
     * 
     * @return This returns the statistics of the included child (nested) copy jobs.
     */
    public final Map<CopyJobState, Long> childJobsInState() {
        return CopyJobChildJobsInStateCopier.copyStringToEnum(childJobsInState);
    }

    /**
     * For responses, this returns true if the service returned a value for the ChildJobsInState property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasChildJobsInState() {
        return childJobsInState != null && !(childJobsInState instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * This returns the statistics of the included child (nested) copy jobs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasChildJobsInState} method.
     * </p>
     * 
     * @return This returns the statistics of the included child (nested) copy jobs.
     */
    public final Map<String, Long> childJobsInStateAsStrings() {
        return childJobsInState;
    }

    /**
     * <p>
     * This is the non-unique name of the resource that belongs to the specified backup.
     * </p>
     * 
     * @return This is the non-unique name of the resource that belongs to the specified backup.
     */
    public final String resourceName() {
        return resourceName;
    }

    /**
     * <p>
     * This parameter is the job count for the specified message category.
     * </p>
     * <p>
     * Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>, and
     * <code>InvalidParameters</code>. See <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list of
     * MessageCategory strings.
     * </p>
     * <p>
     * The the value ANY returns count of all message categories.
     * </p>
     * <p>
     * <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum
     * </p>
     * 
     * @return This parameter is the job count for the specified message category.</p>
     *         <p>
     *         Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>,
     *         and <code>InvalidParameters</code>. See <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list
     *         of MessageCategory strings.
     *         </p>
     *         <p>
     *         The the value ANY returns count of all message categories.
     *         </p>
     *         <p>
     *         <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum
     */
    public final String messageCategory() {
        return messageCategory;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(copyJobId());
        hashCode = 31 * hashCode + Objects.hashCode(sourceBackupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(sourceRecoveryPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(destinationBackupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(destinationRecoveryPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(completionDate());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(backupSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(parentJobId());
        hashCode = 31 * hashCode + Objects.hashCode(isParent());
        hashCode = 31 * hashCode + Objects.hashCode(compositeMemberIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfChildJobs());
        hashCode = 31 * hashCode + Objects.hashCode(hasChildJobsInState() ? childJobsInStateAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resourceName());
        hashCode = 31 * hashCode + Objects.hashCode(messageCategory());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CopyJob)) {
            return false;
        }
        CopyJob other = (CopyJob) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(copyJobId(), other.copyJobId())
                && Objects.equals(sourceBackupVaultArn(), other.sourceBackupVaultArn())
                && Objects.equals(sourceRecoveryPointArn(), other.sourceRecoveryPointArn())
                && Objects.equals(destinationBackupVaultArn(), other.destinationBackupVaultArn())
                && Objects.equals(destinationRecoveryPointArn(), other.destinationRecoveryPointArn())
                && Objects.equals(resourceArn(), other.resourceArn()) && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(completionDate(), other.completionDate())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(backupSizeInBytes(), other.backupSizeInBytes())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(resourceType(), other.resourceType()) && Objects.equals(parentJobId(), other.parentJobId())
                && Objects.equals(isParent(), other.isParent())
                && Objects.equals(compositeMemberIdentifier(), other.compositeMemberIdentifier())
                && Objects.equals(numberOfChildJobs(), other.numberOfChildJobs())
                && hasChildJobsInState() == other.hasChildJobsInState()
                && Objects.equals(childJobsInStateAsStrings(), other.childJobsInStateAsStrings())
                && Objects.equals(resourceName(), other.resourceName())
                && Objects.equals(messageCategory(), other.messageCategory());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CopyJob").add("AccountId", accountId()).add("CopyJobId", copyJobId())
                .add("SourceBackupVaultArn", sourceBackupVaultArn()).add("SourceRecoveryPointArn", sourceRecoveryPointArn())
                .add("DestinationBackupVaultArn", destinationBackupVaultArn())
                .add("DestinationRecoveryPointArn", destinationRecoveryPointArn()).add("ResourceArn", resourceArn())
                .add("CreationDate", creationDate()).add("CompletionDate", completionDate()).add("State", stateAsString())
                .add("StatusMessage", statusMessage()).add("BackupSizeInBytes", backupSizeInBytes())
                .add("IamRoleArn", iamRoleArn()).add("CreatedBy", createdBy()).add("ResourceType", resourceType())
                .add("ParentJobId", parentJobId()).add("IsParent", isParent())
                .add("CompositeMemberIdentifier", compositeMemberIdentifier()).add("NumberOfChildJobs", numberOfChildJobs())
                .add("ChildJobsInState", hasChildJobsInState() ? childJobsInStateAsStrings() : null)
                .add("ResourceName", resourceName()).add("MessageCategory", messageCategory()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "CopyJobId":
            return Optional.ofNullable(clazz.cast(copyJobId()));
        case "SourceBackupVaultArn":
            return Optional.ofNullable(clazz.cast(sourceBackupVaultArn()));
        case "SourceRecoveryPointArn":
            return Optional.ofNullable(clazz.cast(sourceRecoveryPointArn()));
        case "DestinationBackupVaultArn":
            return Optional.ofNullable(clazz.cast(destinationBackupVaultArn()));
        case "DestinationRecoveryPointArn":
            return Optional.ofNullable(clazz.cast(destinationRecoveryPointArn()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "CompletionDate":
            return Optional.ofNullable(clazz.cast(completionDate()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "BackupSizeInBytes":
            return Optional.ofNullable(clazz.cast(backupSizeInBytes()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "ParentJobId":
            return Optional.ofNullable(clazz.cast(parentJobId()));
        case "IsParent":
            return Optional.ofNullable(clazz.cast(isParent()));
        case "CompositeMemberIdentifier":
            return Optional.ofNullable(clazz.cast(compositeMemberIdentifier()));
        case "NumberOfChildJobs":
            return Optional.ofNullable(clazz.cast(numberOfChildJobs()));
        case "ChildJobsInState":
            return Optional.ofNullable(clazz.cast(childJobsInStateAsStrings()));
        case "ResourceName":
            return Optional.ofNullable(clazz.cast(resourceName()));
        case "MessageCategory":
            return Optional.ofNullable(clazz.cast(messageCategory()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CopyJob, T> g) {
        return obj -> g.apply((CopyJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CopyJob> {
        /**
         * <p>
         * The account ID that owns the copy job.
         * </p>
         * 
         * @param accountId
         *        The account ID that owns the copy job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Uniquely identifies a copy job.
         * </p>
         * 
         * @param copyJobId
         *        Uniquely identifies a copy job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyJobId(String copyJobId);

        /**
         * <p>
         * An Amazon Resource Name (ARN) that uniquely identifies a source copy vault; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
         * </p>
         * 
         * @param sourceBackupVaultArn
         *        An Amazon Resource Name (ARN) that uniquely identifies a source copy vault; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceBackupVaultArn(String sourceBackupVaultArn);

        /**
         * <p>
         * An ARN that uniquely identifies a source recovery point; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
         * </p>
         * 
         * @param sourceRecoveryPointArn
         *        An ARN that uniquely identifies a source recovery point; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRecoveryPointArn(String sourceRecoveryPointArn);

        /**
         * <p>
         * An Amazon Resource Name (ARN) that uniquely identifies a destination copy vault; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
         * </p>
         * 
         * @param destinationBackupVaultArn
         *        An Amazon Resource Name (ARN) that uniquely identifies a destination copy vault; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:vault:aBackupVault</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationBackupVaultArn(String destinationBackupVaultArn);

        /**
         * <p>
         * An ARN that uniquely identifies a destination recovery point; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
         * </p>
         * 
         * @param destinationRecoveryPointArn
         *        An ARN that uniquely identifies a destination recovery point; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationRecoveryPointArn(String destinationRecoveryPointArn);

        /**
         * <p>
         * The Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon EBS) volume
         * or an Amazon Relational Database Service (Amazon RDS) database.
         * </p>
         * 
         * @param resourceArn
         *        The Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon EBS)
         *        volume or an Amazon Relational Database Service (Amazon RDS) database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The date and time a copy job is created, in Unix format and Coordinated Universal Time (UTC). The value of
         * <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents
         * Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param creationDate
         *        The date and time a copy job is created, in Unix format and Coordinated Universal Time (UTC). The
         *        value of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
         *        represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The date and time a copy job is completed, in Unix format and Coordinated Universal Time (UTC). The value of
         * <code>CompletionDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents
         * Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param completionDate
         *        The date and time a copy job is completed, in Unix format and Coordinated Universal Time (UTC). The
         *        value of <code>CompletionDate</code> is accurate to milliseconds. For example, the value
         *        1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionDate(Instant completionDate);

        /**
         * <p>
         * The current state of a copy job.
         * </p>
         * 
         * @param state
         *        The current state of a copy job.
         * @see CopyJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CopyJobState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of a copy job.
         * </p>
         * 
         * @param state
         *        The current state of a copy job.
         * @see CopyJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CopyJobState
         */
        Builder state(CopyJobState state);

        /**
         * <p>
         * A detailed message explaining the status of the job to copy a resource.
         * </p>
         * 
         * @param statusMessage
         *        A detailed message explaining the status of the job to copy a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The size, in bytes, of a copy job.
         * </p>
         * 
         * @param backupSizeInBytes
         *        The size, in bytes, of a copy job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupSizeInBytes(Long backupSizeInBytes);

        /**
         * <p>
         * Specifies the IAM role ARN used to copy the target recovery point; for example,
         * <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * </p>
         * 
         * @param iamRoleArn
         *        Specifies the IAM role ARN used to copy the target recovery point; for example,
         *        <code>arn:aws:iam::123456789012:role/S3Access</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * Sets the value of the CreatedBy property for this object.
         *
         * @param createdBy
         *        The new value for the CreatedBy property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(RecoveryPointCreator createdBy);

        /**
         * Sets the value of the CreatedBy property for this object.
         *
         * This is a convenience method that creates an instance of the {@link RecoveryPointCreator.Builder} avoiding
         * the need to create one manually via {@link RecoveryPointCreator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecoveryPointCreator.Builder#build()} is called immediately and
         * its result is passed to {@link #createdBy(RecoveryPointCreator)}.
         * 
         * @param createdBy
         *        a consumer that will call methods on {@link RecoveryPointCreator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #createdBy(RecoveryPointCreator)
         */
        default Builder createdBy(Consumer<RecoveryPointCreator.Builder> createdBy) {
            return createdBy(RecoveryPointCreator.builder().applyMutation(createdBy).build());
        }

        /**
         * <p>
         * The type of Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store (Amazon
         * EBS) volume or an Amazon Relational Database Service (Amazon RDS) database.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource to be copied; for example, an Amazon Elastic Block Store
         *        (Amazon EBS) volume or an Amazon Relational Database Service (Amazon RDS) database.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * This uniquely identifies a request to Backup to copy a resource. The return will be the parent (composite)
         * job ID.
         * </p>
         * 
         * @param parentJobId
         *        This uniquely identifies a request to Backup to copy a resource. The return will be the parent
         *        (composite) job ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentJobId(String parentJobId);

        /**
         * <p>
         * This is a boolean value indicating this is a parent (composite) copy job.
         * </p>
         * 
         * @param isParent
         *        This is a boolean value indicating this is a parent (composite) copy job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isParent(Boolean isParent);

        /**
         * <p>
         * This is the identifier of a resource within a composite group, such as nested (child) recovery point
         * belonging to a composite (parent) stack. The ID is transferred from the <a href=
         * "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html#resources-section-structure-syntax"
         * > logical ID</a> within a stack.
         * </p>
         * 
         * @param compositeMemberIdentifier
         *        This is the identifier of a resource within a composite group, such as nested (child) recovery point
         *        belonging to a composite (parent) stack. The ID is transferred from the <a href=
         *        "https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/resources-section-structure.html#resources-section-structure-syntax"
         *        > logical ID</a> within a stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder compositeMemberIdentifier(String compositeMemberIdentifier);

        /**
         * <p>
         * This is the number of child (nested) copy jobs.
         * </p>
         * 
         * @param numberOfChildJobs
         *        This is the number of child (nested) copy jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfChildJobs(Long numberOfChildJobs);

        /**
         * <p>
         * This returns the statistics of the included child (nested) copy jobs.
         * </p>
         * 
         * @param childJobsInState
         *        This returns the statistics of the included child (nested) copy jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childJobsInStateWithStrings(Map<String, Long> childJobsInState);

        /**
         * <p>
         * This returns the statistics of the included child (nested) copy jobs.
         * </p>
         * 
         * @param childJobsInState
         *        This returns the statistics of the included child (nested) copy jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder childJobsInState(Map<CopyJobState, Long> childJobsInState);

        /**
         * <p>
         * This is the non-unique name of the resource that belongs to the specified backup.
         * </p>
         * 
         * @param resourceName
         *        This is the non-unique name of the resource that belongs to the specified backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceName(String resourceName);

        /**
         * <p>
         * This parameter is the job count for the specified message category.
         * </p>
         * <p>
         * Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>, and
         * <code>InvalidParameters</code>. See <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list of
         * MessageCategory strings.
         * </p>
         * <p>
         * The the value ANY returns count of all message categories.
         * </p>
         * <p>
         * <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum
         * </p>
         * 
         * @param messageCategory
         *        This parameter is the job count for the specified message category.</p>
         *        <p>
         *        Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>,
         *        <code>AGGREGATE_ALL</code>, and <code>InvalidParameters</code>. See <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a
         *        list of MessageCategory strings.
         *        </p>
         *        <p>
         *        The the value ANY returns count of all message categories.
         *        </p>
         *        <p>
         *        <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageCategory(String messageCategory);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String copyJobId;

        private String sourceBackupVaultArn;

        private String sourceRecoveryPointArn;

        private String destinationBackupVaultArn;

        private String destinationRecoveryPointArn;

        private String resourceArn;

        private Instant creationDate;

        private Instant completionDate;

        private String state;

        private String statusMessage;

        private Long backupSizeInBytes;

        private String iamRoleArn;

        private RecoveryPointCreator createdBy;

        private String resourceType;

        private String parentJobId;

        private Boolean isParent;

        private String compositeMemberIdentifier;

        private Long numberOfChildJobs;

        private Map<String, Long> childJobsInState = DefaultSdkAutoConstructMap.getInstance();

        private String resourceName;

        private String messageCategory;

        private BuilderImpl() {
        }

        private BuilderImpl(CopyJob model) {
            accountId(model.accountId);
            copyJobId(model.copyJobId);
            sourceBackupVaultArn(model.sourceBackupVaultArn);
            sourceRecoveryPointArn(model.sourceRecoveryPointArn);
            destinationBackupVaultArn(model.destinationBackupVaultArn);
            destinationRecoveryPointArn(model.destinationRecoveryPointArn);
            resourceArn(model.resourceArn);
            creationDate(model.creationDate);
            completionDate(model.completionDate);
            state(model.state);
            statusMessage(model.statusMessage);
            backupSizeInBytes(model.backupSizeInBytes);
            iamRoleArn(model.iamRoleArn);
            createdBy(model.createdBy);
            resourceType(model.resourceType);
            parentJobId(model.parentJobId);
            isParent(model.isParent);
            compositeMemberIdentifier(model.compositeMemberIdentifier);
            numberOfChildJobs(model.numberOfChildJobs);
            childJobsInStateWithStrings(model.childJobsInState);
            resourceName(model.resourceName);
            messageCategory(model.messageCategory);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getCopyJobId() {
            return copyJobId;
        }

        public final void setCopyJobId(String copyJobId) {
            this.copyJobId = copyJobId;
        }

        @Override
        public final Builder copyJobId(String copyJobId) {
            this.copyJobId = copyJobId;
            return this;
        }

        public final String getSourceBackupVaultArn() {
            return sourceBackupVaultArn;
        }

        public final void setSourceBackupVaultArn(String sourceBackupVaultArn) {
            this.sourceBackupVaultArn = sourceBackupVaultArn;
        }

        @Override
        public final Builder sourceBackupVaultArn(String sourceBackupVaultArn) {
            this.sourceBackupVaultArn = sourceBackupVaultArn;
            return this;
        }

        public final String getSourceRecoveryPointArn() {
            return sourceRecoveryPointArn;
        }

        public final void setSourceRecoveryPointArn(String sourceRecoveryPointArn) {
            this.sourceRecoveryPointArn = sourceRecoveryPointArn;
        }

        @Override
        public final Builder sourceRecoveryPointArn(String sourceRecoveryPointArn) {
            this.sourceRecoveryPointArn = sourceRecoveryPointArn;
            return this;
        }

        public final String getDestinationBackupVaultArn() {
            return destinationBackupVaultArn;
        }

        public final void setDestinationBackupVaultArn(String destinationBackupVaultArn) {
            this.destinationBackupVaultArn = destinationBackupVaultArn;
        }

        @Override
        public final Builder destinationBackupVaultArn(String destinationBackupVaultArn) {
            this.destinationBackupVaultArn = destinationBackupVaultArn;
            return this;
        }

        public final String getDestinationRecoveryPointArn() {
            return destinationRecoveryPointArn;
        }

        public final void setDestinationRecoveryPointArn(String destinationRecoveryPointArn) {
            this.destinationRecoveryPointArn = destinationRecoveryPointArn;
        }

        @Override
        public final Builder destinationRecoveryPointArn(String destinationRecoveryPointArn) {
            this.destinationRecoveryPointArn = destinationRecoveryPointArn;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Instant getCompletionDate() {
            return completionDate;
        }

        public final void setCompletionDate(Instant completionDate) {
            this.completionDate = completionDate;
        }

        @Override
        public final Builder completionDate(Instant completionDate) {
            this.completionDate = completionDate;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(CopyJobState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final Long getBackupSizeInBytes() {
            return backupSizeInBytes;
        }

        public final void setBackupSizeInBytes(Long backupSizeInBytes) {
            this.backupSizeInBytes = backupSizeInBytes;
        }

        @Override
        public final Builder backupSizeInBytes(Long backupSizeInBytes) {
            this.backupSizeInBytes = backupSizeInBytes;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final RecoveryPointCreator.Builder getCreatedBy() {
            return createdBy != null ? createdBy.toBuilder() : null;
        }

        public final void setCreatedBy(RecoveryPointCreator.BuilderImpl createdBy) {
            this.createdBy = createdBy != null ? createdBy.build() : null;
        }

        @Override
        public final Builder createdBy(RecoveryPointCreator createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final String getParentJobId() {
            return parentJobId;
        }

        public final void setParentJobId(String parentJobId) {
            this.parentJobId = parentJobId;
        }

        @Override
        public final Builder parentJobId(String parentJobId) {
            this.parentJobId = parentJobId;
            return this;
        }

        public final Boolean getIsParent() {
            return isParent;
        }

        public final void setIsParent(Boolean isParent) {
            this.isParent = isParent;
        }

        @Override
        public final Builder isParent(Boolean isParent) {
            this.isParent = isParent;
            return this;
        }

        public final String getCompositeMemberIdentifier() {
            return compositeMemberIdentifier;
        }

        public final void setCompositeMemberIdentifier(String compositeMemberIdentifier) {
            this.compositeMemberIdentifier = compositeMemberIdentifier;
        }

        @Override
        public final Builder compositeMemberIdentifier(String compositeMemberIdentifier) {
            this.compositeMemberIdentifier = compositeMemberIdentifier;
            return this;
        }

        public final Long getNumberOfChildJobs() {
            return numberOfChildJobs;
        }

        public final void setNumberOfChildJobs(Long numberOfChildJobs) {
            this.numberOfChildJobs = numberOfChildJobs;
        }

        @Override
        public final Builder numberOfChildJobs(Long numberOfChildJobs) {
            this.numberOfChildJobs = numberOfChildJobs;
            return this;
        }

        public final Map<String, Long> getChildJobsInState() {
            if (childJobsInState instanceof SdkAutoConstructMap) {
                return null;
            }
            return childJobsInState;
        }

        public final void setChildJobsInState(Map<String, Long> childJobsInState) {
            this.childJobsInState = CopyJobChildJobsInStateCopier.copy(childJobsInState);
        }

        @Override
        public final Builder childJobsInStateWithStrings(Map<String, Long> childJobsInState) {
            this.childJobsInState = CopyJobChildJobsInStateCopier.copy(childJobsInState);
            return this;
        }

        @Override
        public final Builder childJobsInState(Map<CopyJobState, Long> childJobsInState) {
            this.childJobsInState = CopyJobChildJobsInStateCopier.copyEnumToString(childJobsInState);
            return this;
        }

        public final String getResourceName() {
            return resourceName;
        }

        public final void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        @Override
        public final Builder resourceName(String resourceName) {
            this.resourceName = resourceName;
            return this;
        }

        public final String getMessageCategory() {
            return messageCategory;
        }

        public final void setMessageCategory(String messageCategory) {
            this.messageCategory = messageCategory;
        }

        @Override
        public final Builder messageCategory(String messageCategory) {
            this.messageCategory = messageCategory;
            return this;
        }

        @Override
        public CopyJob build() {
            return new CopyJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
