/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mq;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mq.internal.MqServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mq.model.BadRequestException;
import software.amazon.awssdk.services.mq.model.ConflictException;
import software.amazon.awssdk.services.mq.model.CreateBrokerRequest;
import software.amazon.awssdk.services.mq.model.CreateBrokerResponse;
import software.amazon.awssdk.services.mq.model.CreateConfigurationRequest;
import software.amazon.awssdk.services.mq.model.CreateConfigurationResponse;
import software.amazon.awssdk.services.mq.model.CreateTagsRequest;
import software.amazon.awssdk.services.mq.model.CreateTagsResponse;
import software.amazon.awssdk.services.mq.model.CreateUserRequest;
import software.amazon.awssdk.services.mq.model.CreateUserResponse;
import software.amazon.awssdk.services.mq.model.DeleteBrokerRequest;
import software.amazon.awssdk.services.mq.model.DeleteBrokerResponse;
import software.amazon.awssdk.services.mq.model.DeleteTagsRequest;
import software.amazon.awssdk.services.mq.model.DeleteTagsResponse;
import software.amazon.awssdk.services.mq.model.DeleteUserRequest;
import software.amazon.awssdk.services.mq.model.DeleteUserResponse;
import software.amazon.awssdk.services.mq.model.DescribeBrokerEngineTypesRequest;
import software.amazon.awssdk.services.mq.model.DescribeBrokerEngineTypesResponse;
import software.amazon.awssdk.services.mq.model.DescribeBrokerInstanceOptionsRequest;
import software.amazon.awssdk.services.mq.model.DescribeBrokerInstanceOptionsResponse;
import software.amazon.awssdk.services.mq.model.DescribeBrokerRequest;
import software.amazon.awssdk.services.mq.model.DescribeBrokerResponse;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationRequest;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationResponse;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationRevisionRequest;
import software.amazon.awssdk.services.mq.model.DescribeConfigurationRevisionResponse;
import software.amazon.awssdk.services.mq.model.DescribeUserRequest;
import software.amazon.awssdk.services.mq.model.DescribeUserResponse;
import software.amazon.awssdk.services.mq.model.ForbiddenException;
import software.amazon.awssdk.services.mq.model.InternalServerErrorException;
import software.amazon.awssdk.services.mq.model.ListBrokersRequest;
import software.amazon.awssdk.services.mq.model.ListBrokersResponse;
import software.amazon.awssdk.services.mq.model.ListConfigurationRevisionsRequest;
import software.amazon.awssdk.services.mq.model.ListConfigurationRevisionsResponse;
import software.amazon.awssdk.services.mq.model.ListConfigurationsRequest;
import software.amazon.awssdk.services.mq.model.ListConfigurationsResponse;
import software.amazon.awssdk.services.mq.model.ListTagsRequest;
import software.amazon.awssdk.services.mq.model.ListTagsResponse;
import software.amazon.awssdk.services.mq.model.ListUsersRequest;
import software.amazon.awssdk.services.mq.model.ListUsersResponse;
import software.amazon.awssdk.services.mq.model.MqException;
import software.amazon.awssdk.services.mq.model.NotFoundException;
import software.amazon.awssdk.services.mq.model.PromoteRequest;
import software.amazon.awssdk.services.mq.model.PromoteResponse;
import software.amazon.awssdk.services.mq.model.RebootBrokerRequest;
import software.amazon.awssdk.services.mq.model.RebootBrokerResponse;
import software.amazon.awssdk.services.mq.model.UnauthorizedException;
import software.amazon.awssdk.services.mq.model.UpdateBrokerRequest;
import software.amazon.awssdk.services.mq.model.UpdateBrokerResponse;
import software.amazon.awssdk.services.mq.model.UpdateConfigurationRequest;
import software.amazon.awssdk.services.mq.model.UpdateConfigurationResponse;
import software.amazon.awssdk.services.mq.model.UpdateUserRequest;
import software.amazon.awssdk.services.mq.model.UpdateUserResponse;
import software.amazon.awssdk.services.mq.transform.CreateBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.CreateConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.CreateTagsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.CreateUserRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DeleteBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DeleteTagsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DeleteUserRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeBrokerEngineTypesRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeBrokerInstanceOptionsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeConfigurationRevisionRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.DescribeUserRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListBrokersRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListConfigurationRevisionsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.ListUsersRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.PromoteRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.RebootBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.UpdateBrokerRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.UpdateConfigurationRequestMarshaller;
import software.amazon.awssdk.services.mq.transform.UpdateUserRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MqAsyncClient}.
 *
 * @see MqAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMqAsyncClient implements MqAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMqAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MqServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMqAsyncClient(MqServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates a broker. Note: This API is asynchronous.
     * </p>
     * <p>
     * To create a broker, you must either use the AmazonMQFullAccess IAM policy or include the following EC2
     * permissions in your IAM policy.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ec2:CreateNetworkInterface
     * </p>
     * <p>
     * This permission is required to allow Amazon MQ to create an elastic network interface (ENI) on behalf of your
     * account.
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:CreateNetworkInterfacePermission
     * </p>
     * <p>
     * This permission is required to attach the ENI to the broker instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DeleteNetworkInterface
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DeleteNetworkInterfacePermission
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DetachNetworkInterface
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeInternetGateways
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeNetworkInterfaces
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeNetworkInterfacePermissions
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeRouteTables
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeSecurityGroups
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeSubnets
     * </p>
     * </li>
     * <li>
     * <p>
     * ec2:DescribeVpcs
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com//amazon-mq/latest/developer-guide/amazon-mq-setting-up.html#create-iam-user"
     * >Create an IAM User and Get Your Amazon Web Services Credentials</a> and <a href=
     * "https://docs.aws.amazon.com//amazon-mq/latest/developer-guide/connecting-to-amazon-mq.html#never-modify-delete-elastic-network-interface"
     * >Never Modify or Delete the Amazon MQ Elastic Network Interface</a> in the <i>Amazon MQ Developer Guide</i>.
     * </p>
     *
     * @param createBrokerRequest
     *        Creates a broker using the specified properties.
     * @return A Java Future containing the result of the CreateBroker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>UnauthorizedException HTTP Status Code 401: Unauthorized request. The provided credentials couldn't
     *         be validated.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. This broker name already exists. Retry your request
     *         with another name.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.CreateBroker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBrokerResponse> createBroker(CreateBrokerRequest createBrokerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBrokerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBroker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBrokerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateBrokerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBrokerRequest, CreateBrokerResponse>()
                            .withOperationName("CreateBroker").withMarshaller(new CreateBrokerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createBrokerRequest));
            CompletableFuture<CreateBrokerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new configuration for the specified configuration name. Amazon MQ uses the default configuration (the
     * engine type and version).
     * </p>
     *
     * @param createConfigurationRequest
     *        Creates a new configuration for the specified configuration name. Amazon MQ uses the default configuration
     *        (the engine type and version).
     * @return A Java Future containing the result of the CreateConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. This broker name already exists. Retry your request
     *         with another name.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.CreateConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateConfigurationResponse> createConfiguration(
            CreateConfigurationRequest createConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationRequest, CreateConfigurationResponse>()
                            .withOperationName("CreateConfiguration")
                            .withMarshaller(new CreateConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createConfigurationRequest));
            CompletableFuture<CreateConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Add a tag to a resource.
     * </p>
     *
     * @param createTagsRequest
     *        A map of the key-value pairs for the resource tag.
     * @return A Java Future containing the result of the CreateTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.CreateTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTagsResponse> createTags(CreateTagsRequest createTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTagsRequest, CreateTagsResponse>().withOperationName("CreateTags")
                            .withMarshaller(new CreateTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTagsRequest));
            CompletableFuture<CreateTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an ActiveMQ user.
     * </p>
     * <important>
     * <p>
     * Do not add personally identifiable information (PII) or other confidential or sensitive information in broker
     * usernames. Broker usernames are accessible to other Amazon Web Services services, including CloudWatch Logs.
     * Broker usernames are not intended to be used for private or sensitive data.
     * </p>
     * </important>
     *
     * @param createUserRequest
     *        Creates a new ActiveMQ user.
     * @return A Java Future containing the result of the CreateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. This broker name already exists. Retry your request
     *         with another name.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.CreateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/CreateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateUserResponse> createUser(CreateUserRequest createUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateUserRequest, CreateUserResponse>().withOperationName("CreateUser")
                            .withMarshaller(new CreateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createUserRequest));
            CompletableFuture<CreateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a broker. Note: This API is asynchronous.
     * </p>
     *
     * @param deleteBrokerRequest
     * @return A Java Future containing the result of the DeleteBroker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DeleteBroker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DeleteBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBrokerResponse> deleteBroker(DeleteBrokerRequest deleteBrokerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBrokerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBroker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBrokerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteBrokerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBrokerRequest, DeleteBrokerResponse>()
                            .withOperationName("DeleteBroker").withMarshaller(new DeleteBrokerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteBrokerRequest));
            CompletableFuture<DeleteBrokerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a tag from a resource.
     * </p>
     *
     * @param deleteTagsRequest
     * @return A Java Future containing the result of the DeleteTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DeleteTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DeleteTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTagsResponse> deleteTags(DeleteTagsRequest deleteTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTagsRequest, DeleteTagsResponse>().withOperationName("DeleteTags")
                            .withMarshaller(new DeleteTagsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTagsRequest));
            CompletableFuture<DeleteTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an ActiveMQ user.
     * </p>
     *
     * @param deleteUserRequest
     * @return A Java Future containing the result of the DeleteUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DeleteUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DeleteUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteUserResponse> deleteUser(DeleteUserRequest deleteUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteUserRequest, DeleteUserResponse>().withOperationName("DeleteUser")
                            .withMarshaller(new DeleteUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteUserRequest));
            CompletableFuture<DeleteUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified broker.
     * </p>
     *
     * @param describeBrokerRequest
     * @return A Java Future containing the result of the DescribeBroker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DescribeBroker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBrokerResponse> describeBroker(DescribeBrokerRequest describeBrokerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBrokerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBroker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBrokerResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBrokerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBrokerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBrokerRequest, DescribeBrokerResponse>()
                            .withOperationName("DescribeBroker")
                            .withMarshaller(new DescribeBrokerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBrokerRequest));
            CompletableFuture<DescribeBrokerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe available engine types and versions.
     * </p>
     *
     * @param describeBrokerEngineTypesRequest
     * @return A Java Future containing the result of the DescribeBrokerEngineTypes operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DescribeBrokerEngineTypes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeBrokerEngineTypes" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBrokerEngineTypesResponse> describeBrokerEngineTypes(
            DescribeBrokerEngineTypesRequest describeBrokerEngineTypesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBrokerEngineTypesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBrokerEngineTypesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBrokerEngineTypes");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBrokerEngineTypesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBrokerEngineTypesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBrokerEngineTypesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBrokerEngineTypesRequest, DescribeBrokerEngineTypesResponse>()
                            .withOperationName("DescribeBrokerEngineTypes")
                            .withMarshaller(new DescribeBrokerEngineTypesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBrokerEngineTypesRequest));
            CompletableFuture<DescribeBrokerEngineTypesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describe available broker instance options.
     * </p>
     *
     * @param describeBrokerInstanceOptionsRequest
     * @return A Java Future containing the result of the DescribeBrokerInstanceOptions operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DescribeBrokerInstanceOptions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeBrokerInstanceOptions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBrokerInstanceOptionsResponse> describeBrokerInstanceOptions(
            DescribeBrokerInstanceOptionsRequest describeBrokerInstanceOptionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBrokerInstanceOptionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeBrokerInstanceOptionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBrokerInstanceOptions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBrokerInstanceOptionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBrokerInstanceOptionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeBrokerInstanceOptionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBrokerInstanceOptionsRequest, DescribeBrokerInstanceOptionsResponse>()
                            .withOperationName("DescribeBrokerInstanceOptions")
                            .withMarshaller(new DescribeBrokerInstanceOptionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeBrokerInstanceOptionsRequest));
            CompletableFuture<DescribeBrokerInstanceOptionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified configuration.
     * </p>
     *
     * @param describeConfigurationRequest
     * @return A Java Future containing the result of the DescribeConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DescribeConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConfigurationResponse> describeConfiguration(
            DescribeConfigurationRequest describeConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConfigurationRequest, DescribeConfigurationResponse>()
                            .withOperationName("DescribeConfiguration")
                            .withMarshaller(new DescribeConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeConfigurationRequest));
            CompletableFuture<DescribeConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the specified configuration revision for the specified configuration.
     * </p>
     *
     * @param describeConfigurationRevisionRequest
     * @return A Java Future containing the result of the DescribeConfigurationRevision operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DescribeConfigurationRevision
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeConfigurationRevision"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeConfigurationRevisionResponse> describeConfigurationRevision(
            DescribeConfigurationRevisionRequest describeConfigurationRevisionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeConfigurationRevisionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeConfigurationRevisionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeConfigurationRevision");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeConfigurationRevisionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeConfigurationRevisionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeConfigurationRevisionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeConfigurationRevisionRequest, DescribeConfigurationRevisionResponse>()
                            .withOperationName("DescribeConfigurationRevision")
                            .withMarshaller(new DescribeConfigurationRevisionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeConfigurationRevisionRequest));
            CompletableFuture<DescribeConfigurationRevisionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about an ActiveMQ user.
     * </p>
     *
     * @param describeUserRequest
     * @return A Java Future containing the result of the DescribeUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.DescribeUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/DescribeUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeUserResponse> describeUser(DescribeUserRequest describeUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeUserRequest, DescribeUserResponse>()
                            .withOperationName("DescribeUser").withMarshaller(new DescribeUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeUserRequest));
            CompletableFuture<DescribeUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all brokers.
     * </p>
     *
     * @param listBrokersRequest
     * @return A Java Future containing the result of the ListBrokers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.ListBrokers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListBrokers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBrokersResponse> listBrokers(ListBrokersRequest listBrokersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBrokersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBrokersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBrokers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBrokersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBrokersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListBrokersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBrokersRequest, ListBrokersResponse>()
                            .withOperationName("ListBrokers").withMarshaller(new ListBrokersRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listBrokersRequest));
            CompletableFuture<ListBrokersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all revisions for the specified configuration.
     * </p>
     *
     * @param listConfigurationRevisionsRequest
     * @return A Java Future containing the result of the ListConfigurationRevisions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.ListConfigurationRevisions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListConfigurationRevisions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationRevisionsResponse> listConfigurationRevisions(
            ListConfigurationRevisionsRequest listConfigurationRevisionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationRevisionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationRevisionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationRevisions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationRevisionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationRevisionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationRevisionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationRevisionsRequest, ListConfigurationRevisionsResponse>()
                            .withOperationName("ListConfigurationRevisions")
                            .withMarshaller(new ListConfigurationRevisionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigurationRevisionsRequest));
            CompletableFuture<ListConfigurationRevisionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all configurations.
     * </p>
     *
     * @param listConfigurationsRequest
     * @return A Java Future containing the result of the ListConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.ListConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListConfigurations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListConfigurationsResponse> listConfigurations(ListConfigurationsRequest listConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationsRequest, ListConfigurationsResponse>()
                            .withOperationName("ListConfigurations")
                            .withMarshaller(new ListConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listConfigurationsRequest));
            CompletableFuture<ListConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists tags for a resource.
     * </p>
     *
     * @param listTagsRequest
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.ListTags
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listTagsRequest));
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of all ActiveMQ users.
     * </p>
     *
     * @param listUsersRequest
     * @return A Java Future containing the result of the ListUsers operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.ListUsers
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/ListUsers" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsersResponse> listUsers(ListUsersRequest listUsersRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUsersRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listUsersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsers");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListUsersResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsersResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsersRequest, ListUsersResponse>().withOperationName("ListUsers")
                            .withMarshaller(new ListUsersRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listUsersRequest));
            CompletableFuture<ListUsersResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Promotes a data replication replica broker to the primary broker role.
     * </p>
     *
     * @param promoteRequest
     *        Promotes a data replication replica broker to the primary broker role.
     * @return A Java Future containing the result of the Promote operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.Promote
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/Promote" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PromoteResponse> promote(PromoteRequest promoteRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(promoteRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, promoteRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Promote");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PromoteResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PromoteResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PromoteResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PromoteRequest, PromoteResponse>().withOperationName("Promote")
                            .withMarshaller(new PromoteRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(promoteRequest));
            CompletableFuture<PromoteResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Reboots a broker. Note: This API is asynchronous.
     * </p>
     *
     * @param rebootBrokerRequest
     * @return A Java Future containing the result of the RebootBroker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.RebootBroker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/RebootBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RebootBrokerResponse> rebootBroker(RebootBrokerRequest rebootBrokerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rebootBrokerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rebootBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RebootBroker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RebootBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RebootBrokerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RebootBrokerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RebootBrokerRequest, RebootBrokerResponse>()
                            .withOperationName("RebootBroker").withMarshaller(new RebootBrokerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rebootBrokerRequest));
            CompletableFuture<RebootBrokerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds a pending configuration change to a broker.
     * </p>
     *
     * @param updateBrokerRequest
     *        Updates the broker using the specified properties.
     * @return A Java Future containing the result of the UpdateBroker operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. This broker name already exists. Retry your request
     *         with another name.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.UpdateBroker
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateBroker" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateBrokerResponse> updateBroker(UpdateBrokerRequest updateBrokerRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateBrokerRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateBrokerRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateBroker");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateBrokerResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateBrokerResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateBrokerResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateBrokerRequest, UpdateBrokerResponse>()
                            .withOperationName("UpdateBroker").withMarshaller(new UpdateBrokerRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateBrokerRequest));
            CompletableFuture<UpdateBrokerResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the specified configuration.
     * </p>
     *
     * @param updateConfigurationRequest
     *        Updates the specified configuration.
     * @return A Java Future containing the result of the UpdateConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. This broker name already exists. Retry your request
     *         with another name.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.UpdateConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateConfiguration" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateConfigurationResponse> updateConfiguration(
            UpdateConfigurationRequest updateConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationRequest, UpdateConfigurationResponse>()
                            .withOperationName("UpdateConfiguration")
                            .withMarshaller(new UpdateConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateConfigurationRequest));
            CompletableFuture<UpdateConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the information for an ActiveMQ user.
     * </p>
     *
     * @param updateUserRequest
     *        Updates the information for an ActiveMQ user.
     * @return A Java Future containing the result of the UpdateUser operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>NotFoundException HTTP Status Code 404: Resource not found due to incorrect input. Correct your
     *         request and then retry it.</li>
     *         <li>BadRequestException HTTP Status Code 400: Bad request due to incorrect input. Correct your request
     *         and then retry it.</li>
     *         <li>InternalServerErrorException HTTP Status Code 500: Unexpected internal server error. Retrying your
     *         request might resolve the issue.</li>
     *         <li>ConflictException HTTP Status Code 409: Conflict. This broker name already exists. Retry your request
     *         with another name.</li>
     *         <li>ForbiddenException HTTP Status Code 403: Access forbidden. Correct your credentials and then retry
     *         your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MqException Base class for all service exceptions. Unknown exceptions will be thrown as an instance
     *         of this type.</li>
     *         </ul>
     * @sample MqAsyncClient.UpdateUser
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mq-2017-11-27/UpdateUser" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateUserResponse> updateUser(UpdateUserRequest updateUserRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateUserRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateUserRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "mq");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateUser");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateUserResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateUserResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateUserResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateUserRequest, UpdateUserResponse>().withOperationName("UpdateUser")
                            .withMarshaller(new UpdateUserRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateUserRequest));
            CompletableFuture<UpdateUserResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MqServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MqException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnauthorizedException")
                                .exceptionBuilderSupplier(UnauthorizedException::builder).httpStatusCode(401).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        MqServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = MqServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
