/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.mediaconvert;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.mediaconvert.internal.MediaConvertServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.mediaconvert.model.AssociateCertificateRequest;
import software.amazon.awssdk.services.mediaconvert.model.AssociateCertificateResponse;
import software.amazon.awssdk.services.mediaconvert.model.BadRequestException;
import software.amazon.awssdk.services.mediaconvert.model.CancelJobRequest;
import software.amazon.awssdk.services.mediaconvert.model.CancelJobResponse;
import software.amazon.awssdk.services.mediaconvert.model.ConflictException;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobResponse;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreateJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.CreatePresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreatePresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.CreateQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.CreateQueueResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeleteJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeleteJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeletePolicyRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeletePolicyResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeletePresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeletePresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.DeleteQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.DeleteQueueResponse;
import software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.mediaconvert.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.mediaconvert.model.DisassociateCertificateRequest;
import software.amazon.awssdk.services.mediaconvert.model.DisassociateCertificateResponse;
import software.amazon.awssdk.services.mediaconvert.model.ForbiddenException;
import software.amazon.awssdk.services.mediaconvert.model.GetJobRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetJobResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetPolicyRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetPolicyResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetPresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetPresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.GetQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.GetQueueResponse;
import software.amazon.awssdk.services.mediaconvert.model.InternalServerErrorException;
import software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListJobTemplatesResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListJobsRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListJobsResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListPresetsRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListPresetsResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListQueuesRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListQueuesResponse;
import software.amazon.awssdk.services.mediaconvert.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.mediaconvert.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.mediaconvert.model.MediaConvertException;
import software.amazon.awssdk.services.mediaconvert.model.NotFoundException;
import software.amazon.awssdk.services.mediaconvert.model.PutPolicyRequest;
import software.amazon.awssdk.services.mediaconvert.model.PutPolicyResponse;
import software.amazon.awssdk.services.mediaconvert.model.TagResourceRequest;
import software.amazon.awssdk.services.mediaconvert.model.TagResourceResponse;
import software.amazon.awssdk.services.mediaconvert.model.TooManyRequestsException;
import software.amazon.awssdk.services.mediaconvert.model.UntagResourceRequest;
import software.amazon.awssdk.services.mediaconvert.model.UntagResourceResponse;
import software.amazon.awssdk.services.mediaconvert.model.UpdateJobTemplateRequest;
import software.amazon.awssdk.services.mediaconvert.model.UpdateJobTemplateResponse;
import software.amazon.awssdk.services.mediaconvert.model.UpdatePresetRequest;
import software.amazon.awssdk.services.mediaconvert.model.UpdatePresetResponse;
import software.amazon.awssdk.services.mediaconvert.model.UpdateQueueRequest;
import software.amazon.awssdk.services.mediaconvert.model.UpdateQueueResponse;
import software.amazon.awssdk.services.mediaconvert.transform.AssociateCertificateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CancelJobRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreateJobRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreateJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreatePresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.CreateQueueRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeleteJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeletePresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DeleteQueueRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.DisassociateCertificateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetJobRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetPresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.GetQueueRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListJobTemplatesRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListJobsRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListPresetsRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListQueuesRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.PutPolicyRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UpdateJobTemplateRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UpdatePresetRequestMarshaller;
import software.amazon.awssdk.services.mediaconvert.transform.UpdateQueueRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link MediaConvertAsyncClient}.
 *
 * @see MediaConvertAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultMediaConvertAsyncClient implements MediaConvertAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultMediaConvertAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final MediaConvertServiceClientConfiguration serviceClientConfiguration;

    protected DefaultMediaConvertAsyncClient(MediaConvertServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * Associates an AWS Certificate Manager (ACM) Amazon Resource Name (ARN) with AWS Elemental MediaConvert.
     *
     * @param associateCertificateRequest
     * @return A Java Future containing the result of the AssociateCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and cannot fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested does not exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service could not complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.AssociateCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/AssociateCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateCertificateResponse> associateCertificate(
            AssociateCertificateRequest associateCertificateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateCertificateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AssociateCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateCertificateRequest, AssociateCertificateResponse>()
                            .withOperationName("AssociateCertificate")
                            .withMarshaller(new AssociateCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateCertificateRequest));
            CompletableFuture<AssociateCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Permanently cancel a job. Once you have canceled a job, you can't start it again.
     *
     * @param cancelJobRequest
     * @return A Java Future containing the result of the CancelJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.CancelJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CancelJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CancelJobResponse> cancelJob(CancelJobRequest cancelJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CancelJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CancelJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CancelJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CancelJobRequest, CancelJobResponse>().withOperationName("CancelJob")
                            .withMarshaller(new CancelJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(cancelJobRequest));
            CompletableFuture<CancelJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new transcoding job. For information about jobs and job settings, see the User Guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param createJobRequest
     * @return A Java Future containing the result of the CreateJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.CreateJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreateJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobResponse> createJob(CreateJobRequest createJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobRequest, CreateJobResponse>().withOperationName("CreateJob")
                            .withMarshaller(new CreateJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(createJobRequest));
            CompletableFuture<CreateJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new job template. For information about job templates see the User Guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param createJobTemplateRequest
     * @return A Java Future containing the result of the CreateJobTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.CreateJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreateJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateJobTemplateResponse> createJobTemplate(CreateJobTemplateRequest createJobTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createJobTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateJobTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateJobTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateJobTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateJobTemplateRequest, CreateJobTemplateResponse>()
                            .withOperationName("CreateJobTemplate")
                            .withMarshaller(new CreateJobTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createJobTemplateRequest));
            CompletableFuture<CreateJobTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new preset. For information about job templates see the User Guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param createPresetRequest
     * @return A Java Future containing the result of the CreatePreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.CreatePreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreatePreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePresetResponse> createPreset(CreatePresetRequest createPresetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createPresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePreset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreatePresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePresetRequest, CreatePresetResponse>()
                            .withOperationName("CreatePreset").withMarshaller(new CreatePresetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createPresetRequest));
            CompletableFuture<CreatePresetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create a new transcoding queue. For information about queues, see Working With Queues in the User Guide at
     * https://docs.aws.amazon.com/mediaconvert/latest/ug/working-with-queues.html
     *
     * @param createQueueRequest
     * @return A Java Future containing the result of the CreateQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.CreateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/CreateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateQueueResponse> createQueue(CreateQueueRequest createQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateQueueRequest, CreateQueueResponse>()
                            .withOperationName("CreateQueue").withMarshaller(new CreateQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createQueueRequest));
            CompletableFuture<CreateQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Permanently delete a job template you have created.
     *
     * @param deleteJobTemplateRequest
     * @return A Java Future containing the result of the DeleteJobTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.DeleteJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeleteJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteJobTemplateResponse> deleteJobTemplate(DeleteJobTemplateRequest deleteJobTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteJobTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteJobTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteJobTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteJobTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteJobTemplateRequest, DeleteJobTemplateResponse>()
                            .withOperationName("DeleteJobTemplate")
                            .withMarshaller(new DeleteJobTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteJobTemplateRequest));
            CompletableFuture<DeleteJobTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Permanently delete a policy that you created.
     *
     * @param deletePolicyRequest
     * @return A Java Future containing the result of the DeletePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.DeletePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyResponse> deletePolicy(DeletePolicyRequest deletePolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                            .withOperationName("DeletePolicy").withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePolicyRequest));
            CompletableFuture<DeletePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Permanently delete a preset you have created.
     *
     * @param deletePresetRequest
     * @return A Java Future containing the result of the DeletePreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.DeletePreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeletePreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePresetResponse> deletePreset(DeletePresetRequest deletePresetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deletePresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePreset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePresetRequest, DeletePresetResponse>()
                            .withOperationName("DeletePreset").withMarshaller(new DeletePresetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deletePresetRequest));
            CompletableFuture<DeletePresetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Permanently delete a queue you have created.
     *
     * @param deleteQueueRequest
     * @return A Java Future containing the result of the DeleteQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.DeleteQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DeleteQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteQueueResponse> deleteQueue(DeleteQueueRequest deleteQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteQueueRequest, DeleteQueueResponse>()
                            .withOperationName("DeleteQueue").withMarshaller(new DeleteQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteQueueRequest));
            CompletableFuture<DeleteQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Send an request with an empty body to the regional API endpoint to get your account API endpoint.
     *
     * @param describeEndpointsRequest
     *        DescribeEndpointsRequest
     * @return A Java Future containing the result of the DescribeEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException BadRequestException</li>
     *         <li>InternalServerErrorException InternalServiceException</li>
     *         <li>ForbiddenException AccessDeniedException</li>
     *         <li>NotFoundException ResourceNotFoundException</li>
     *         <li>TooManyRequestsException LimitExceededException</li>
     *         <li>ConflictException ResourceInUseException</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DescribeEndpoints"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEndpointsResponse> describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                            .withOperationName("DescribeEndpoints")
                            .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeEndpointsRequest));
            CompletableFuture<DescribeEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Removes an association between the Amazon Resource Name (ARN) of an AWS Certificate Manager (ACM) certificate and
     * an AWS Elemental MediaConvert resource.
     *
     * @param disassociateCertificateRequest
     * @return A Java Future containing the result of the DisassociateCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and cannot fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested does not exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service could not complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.DisassociateCertificate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/DisassociateCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateCertificateResponse> disassociateCertificate(
            DisassociateCertificateRequest disassociateCertificateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateCertificateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disassociateCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DisassociateCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateCertificateRequest, DisassociateCertificateResponse>()
                            .withOperationName("DisassociateCertificate")
                            .withMarshaller(new DisassociateCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateCertificateRequest));
            CompletableFuture<DisassociateCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve the JSON for a specific transcoding job.
     *
     * @param getJobRequest
     * @return A Java Future containing the result of the GetJob operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.GetJob
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetJob" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobResponse> getJob(GetJobRequest getJobRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJob");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetJobResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobRequest, GetJobResponse>().withOperationName("GetJob")
                            .withMarshaller(new GetJobRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getJobRequest));
            CompletableFuture<GetJobResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve the JSON for a specific job template.
     *
     * @param getJobTemplateRequest
     * @return A Java Future containing the result of the GetJobTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.GetJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetJobTemplate" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetJobTemplateResponse> getJobTemplate(GetJobTemplateRequest getJobTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getJobTemplateRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetJobTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetJobTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetJobTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetJobTemplateRequest, GetJobTemplateResponse>()
                            .withOperationName("GetJobTemplate")
                            .withMarshaller(new GetJobTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getJobTemplateRequest));
            CompletableFuture<GetJobTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve the JSON for your policy.
     *
     * @param getPolicyRequest
     * @return A Java Future containing the result of the GetPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.GetPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyResponse> getPolicy(GetPolicyRequest getPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>().withOperationName("GetPolicy")
                            .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPolicyRequest));
            CompletableFuture<GetPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve the JSON for a specific preset.
     *
     * @param getPresetRequest
     * @return A Java Future containing the result of the GetPreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.GetPreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetPreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPresetResponse> getPreset(GetPresetRequest getPresetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getPresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPreset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPresetRequest, GetPresetResponse>().withOperationName("GetPreset")
                            .withMarshaller(new GetPresetRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getPresetRequest));
            CompletableFuture<GetPresetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve the JSON for a specific queue.
     *
     * @param getQueueRequest
     * @return A Java Future containing the result of the GetQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.GetQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/GetQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetQueueResponse> getQueue(GetQueueRequest getQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetQueueRequest, GetQueueResponse>().withOperationName("GetQueue")
                            .withMarshaller(new GetQueueRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getQueueRequest));
            CompletableFuture<GetQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your job templates. This will return the templates themselves, not just
     * a list of them. To retrieve the next twenty templates, use the nextToken string returned with the array
     *
     * @param listJobTemplatesRequest
     * @return A Java Future containing the result of the ListJobTemplates operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.ListJobTemplates
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListJobTemplates" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobTemplatesResponse> listJobTemplates(ListJobTemplatesRequest listJobTemplatesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobTemplatesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobTemplatesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobTemplates");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobTemplatesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListJobTemplatesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobTemplatesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobTemplatesRequest, ListJobTemplatesResponse>()
                            .withOperationName("ListJobTemplates")
                            .withMarshaller(new ListJobTemplatesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listJobTemplatesRequest));
            CompletableFuture<ListJobTemplatesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your most recently created jobs. This array includes in-process,
     * completed, and errored jobs. This will return the jobs themselves, not just a list of the jobs. To retrieve the
     * twenty next most recent jobs, use the nextToken string returned with the array.
     *
     * @param listJobsRequest
     * @return A Java Future containing the result of the ListJobs operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.ListJobs
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListJobs" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListJobsResponse> listJobs(ListJobsRequest listJobsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listJobsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listJobsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListJobsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListJobsRequest, ListJobsResponse>().withOperationName("ListJobs")
                            .withMarshaller(new ListJobsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(listJobsRequest));
            CompletableFuture<ListJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your presets. This will return the presets themselves, not just a list
     * of them. To retrieve the next twenty presets, use the nextToken string returned with the array.
     *
     * @param listPresetsRequest
     * @return A Java Future containing the result of the ListPresets operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.ListPresets
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListPresets" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPresetsResponse> listPresets(ListPresetsRequest listPresetsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listPresetsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPresetsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPresets");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPresetsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListPresetsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPresetsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPresetsRequest, ListPresetsResponse>()
                            .withOperationName("ListPresets").withMarshaller(new ListPresetsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listPresetsRequest));
            CompletableFuture<ListPresetsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve a JSON array of up to twenty of your queues. This will return the queues themselves, not just a list of
     * them. To retrieve the next twenty queues, use the nextToken string returned with the array.
     *
     * @param listQueuesRequest
     * @return A Java Future containing the result of the ListQueues operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.ListQueues
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListQueues" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListQueuesResponse> listQueues(ListQueuesRequest listQueuesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listQueuesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listQueuesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListQueues");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListQueuesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListQueuesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListQueuesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListQueuesRequest, ListQueuesResponse>().withOperationName("ListQueues")
                            .withMarshaller(new ListQueuesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listQueuesRequest));
            CompletableFuture<ListQueuesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Retrieve the tags for a MediaConvert resource.
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Create or change your policy. For more information about policies, see the user guide at
     * http://docs.aws.amazon.com/mediaconvert/latest/ug/what-is.html
     *
     * @param putPolicyRequest
     * @return A Java Future containing the result of the PutPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.PutPolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/PutPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPolicyResponse> putPolicy(PutPolicyRequest putPolicyRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putPolicyRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPolicyRequest, PutPolicyResponse>().withOperationName("PutPolicy")
                            .withMarshaller(new PutPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(putPolicyRequest));
            CompletableFuture<PutPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Add tags to a MediaConvert queue, preset, or job template. For information about tagging, see the User Guide at
     * https://docs.aws.amazon.com/mediaconvert/latest/ug/tagging-resources.html
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Remove tags from a MediaConvert queue, preset, or job template. For information about tagging, see the User Guide
     * at https://docs.aws.amazon.com/mediaconvert/latest/ug/tagging-resources.html
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Modify one of your existing job templates.
     *
     * @param updateJobTemplateRequest
     * @return A Java Future containing the result of the UpdateJobTemplate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.UpdateJobTemplate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UpdateJobTemplate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateJobTemplateResponse> updateJobTemplate(UpdateJobTemplateRequest updateJobTemplateRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateJobTemplateRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateJobTemplateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateJobTemplate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateJobTemplateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateJobTemplateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateJobTemplateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateJobTemplateRequest, UpdateJobTemplateResponse>()
                            .withOperationName("UpdateJobTemplate")
                            .withMarshaller(new UpdateJobTemplateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateJobTemplateRequest));
            CompletableFuture<UpdateJobTemplateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Modify one of your existing presets.
     *
     * @param updatePresetRequest
     * @return A Java Future containing the result of the UpdatePreset operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.UpdatePreset
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UpdatePreset" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdatePresetResponse> updatePreset(UpdatePresetRequest updatePresetRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updatePresetRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updatePresetRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdatePreset");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdatePresetResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdatePresetResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdatePresetResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdatePresetRequest, UpdatePresetResponse>()
                            .withOperationName("UpdatePreset").withMarshaller(new UpdatePresetRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updatePresetRequest));
            CompletableFuture<UpdatePresetResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * Modify one of your existing queues.
     *
     * @param updateQueueRequest
     * @return A Java Future containing the result of the UpdateQueue operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BadRequestException The service can't process your request because of a problem in the request.
     *         Please check your request form and syntax.</li>
     *         <li>InternalServerErrorException The service encountered an unexpected condition and can't fulfill your
     *         request.</li>
     *         <li>ForbiddenException You don't have permissions for this action with the credentials you sent.</li>
     *         <li>NotFoundException The resource you requested doesn't exist.</li>
     *         <li>TooManyRequestsException Too many requests have been sent in too short of a time. The service limits
     *         the rate at which it will accept requests.</li>
     *         <li>ConflictException The service couldn't complete your request because there is a conflict with the
     *         current state of the resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>MediaConvertException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample MediaConvertAsyncClient.UpdateQueue
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/mediaconvert-2017-08-29/UpdateQueue" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateQueueResponse> updateQueue(UpdateQueueRequest updateQueueRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateQueueRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateQueueRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "MediaConvert");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateQueue");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateQueueResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateQueueResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateQueueResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateQueueRequest, UpdateQueueResponse>()
                            .withOperationName("UpdateQueue").withMarshaller(new UpdateQueueRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateQueueRequest));
            CompletableFuture<UpdateQueueResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final MediaConvertServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(MediaConvertException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ForbiddenException")
                                .exceptionBuilderSupplier(ForbiddenException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NotFoundException")
                                .exceptionBuilderSupplier(NotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerErrorException")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        MediaConvertServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = MediaConvertServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
