/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.licensemanager;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.licensemanager.internal.LicenseManagerServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.licensemanager.model.AcceptGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.AcceptGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.AccessDeniedException;
import software.amazon.awssdk.services.licensemanager.model.AuthorizationException;
import software.amazon.awssdk.services.licensemanager.model.CheckInLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CheckInLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.CheckoutBorrowLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CheckoutBorrowLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.CheckoutLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CheckoutLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.ConflictException;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantVersionRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateGrantVersionResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConversionTaskForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseConversionTaskForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseManagerReportGeneratorRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseManagerReportGeneratorResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseVersionRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateLicenseVersionResponse;
import software.amazon.awssdk.services.licensemanager.model.CreateTokenRequest;
import software.amazon.awssdk.services.licensemanager.model.CreateTokenResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseManagerReportGeneratorRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseManagerReportGeneratorResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.DeleteTokenRequest;
import software.amazon.awssdk.services.licensemanager.model.DeleteTokenResponse;
import software.amazon.awssdk.services.licensemanager.model.EntitlementNotAllowedException;
import software.amazon.awssdk.services.licensemanager.model.ExtendLicenseConsumptionRequest;
import software.amazon.awssdk.services.licensemanager.model.ExtendLicenseConsumptionResponse;
import software.amazon.awssdk.services.licensemanager.model.FailedDependencyException;
import software.amazon.awssdk.services.licensemanager.model.FilterLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.GetAccessTokenRequest;
import software.amazon.awssdk.services.licensemanager.model.GetAccessTokenResponse;
import software.amazon.awssdk.services.licensemanager.model.GetGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.GetGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConversionTaskRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseConversionTaskResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseManagerReportGeneratorRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseManagerReportGeneratorResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseResponse;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseUsageRequest;
import software.amazon.awssdk.services.licensemanager.model.GetLicenseUsageResponse;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.GetServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.model.InvalidParameterValueException;
import software.amazon.awssdk.services.licensemanager.model.InvalidResourceStateException;
import software.amazon.awssdk.services.licensemanager.model.LicenseManagerException;
import software.amazon.awssdk.services.licensemanager.model.LicenseUsageException;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListAssociationsForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.ListDistributedGrantsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListDistributedGrantsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListFailuresForLicenseConfigurationOperationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConfigurationsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConversionTasksRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseConversionTasksResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseManagerReportGeneratorsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseManagerReportGeneratorsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseVersionsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicenseVersionsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListLicensesRequest;
import software.amazon.awssdk.services.licensemanager.model.ListLicensesResponse;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedGrantsForOrganizationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedGrantsForOrganizationResponse;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedGrantsRequest;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedGrantsResponse;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedLicensesForOrganizationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedLicensesForOrganizationResponse;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedLicensesRequest;
import software.amazon.awssdk.services.licensemanager.model.ListReceivedLicensesResponse;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryRequest;
import software.amazon.awssdk.services.licensemanager.model.ListResourceInventoryResponse;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.ListTokensRequest;
import software.amazon.awssdk.services.licensemanager.model.ListTokensResponse;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.ListUsageForLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.NoEntitlementsAllowedException;
import software.amazon.awssdk.services.licensemanager.model.RateLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.RedirectException;
import software.amazon.awssdk.services.licensemanager.model.RejectGrantRequest;
import software.amazon.awssdk.services.licensemanager.model.RejectGrantResponse;
import software.amazon.awssdk.services.licensemanager.model.ResourceLimitExceededException;
import software.amazon.awssdk.services.licensemanager.model.ResourceNotFoundException;
import software.amazon.awssdk.services.licensemanager.model.ServerInternalException;
import software.amazon.awssdk.services.licensemanager.model.TagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.TagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UnsupportedDigitalSignatureMethodException;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UntagResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseConfigurationResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseManagerReportGeneratorRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseManagerReportGeneratorResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateLicenseSpecificationsForResourceResponse;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsRequest;
import software.amazon.awssdk.services.licensemanager.model.UpdateServiceSettingsResponse;
import software.amazon.awssdk.services.licensemanager.model.ValidationException;
import software.amazon.awssdk.services.licensemanager.transform.AcceptGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CheckInLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CheckoutBorrowLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CheckoutLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateGrantVersionRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseConversionTaskForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseManagerReportGeneratorRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateLicenseVersionRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.CreateTokenRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseManagerReportGeneratorRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.DeleteTokenRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ExtendLicenseConsumptionRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetAccessTokenRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseConversionTaskRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseManagerReportGeneratorRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetLicenseUsageRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.GetServiceSettingsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListAssociationsForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListDistributedGrantsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListFailuresForLicenseConfigurationOperationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseConversionTasksRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseManagerReportGeneratorsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicenseVersionsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListLicensesRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListReceivedGrantsForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListReceivedGrantsRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListReceivedLicensesForOrganizationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListReceivedLicensesRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListResourceInventoryRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListTokensRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.ListUsageForLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.RejectGrantRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseConfigurationRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseManagerReportGeneratorRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateLicenseSpecificationsForResourceRequestMarshaller;
import software.amazon.awssdk.services.licensemanager.transform.UpdateServiceSettingsRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link LicenseManagerAsyncClient}.
 *
 * @see LicenseManagerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultLicenseManagerAsyncClient implements LicenseManagerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultLicenseManagerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final LicenseManagerServiceClientConfiguration serviceClientConfiguration;

    protected DefaultLicenseManagerAsyncClient(LicenseManagerServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Accepts the specified grant.
     * </p>
     *
     * @param acceptGrantRequest
     * @return A Java Future containing the result of the AcceptGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.AcceptGrant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/AcceptGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<AcceptGrantResponse> acceptGrant(AcceptGrantRequest acceptGrantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(acceptGrantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, acceptGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AcceptGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AcceptGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    AcceptGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AcceptGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AcceptGrantRequest, AcceptGrantResponse>()
                            .withOperationName("AcceptGrant").withMarshaller(new AcceptGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(acceptGrantRequest));
            CompletableFuture<AcceptGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks in the specified license. Check in a license when it is no longer in use.
     * </p>
     *
     * @param checkInLicenseRequest
     * @return A Java Future containing the result of the CheckInLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CheckInLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CheckInLicense"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckInLicenseResponse> checkInLicense(CheckInLicenseRequest checkInLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(checkInLicenseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkInLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckInLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckInLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckInLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckInLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckInLicenseRequest, CheckInLicenseResponse>()
                            .withOperationName("CheckInLicense")
                            .withMarshaller(new CheckInLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(checkInLicenseRequest));
            CompletableFuture<CheckInLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks out the specified license for offline use.
     * </p>
     *
     * @param checkoutBorrowLicenseRequest
     * @return A Java Future containing the result of the CheckoutBorrowLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>NoEntitlementsAllowedException There are no entitlements found for this license, or the entitlement
     *         maximum count is reached.</li>
     *         <li>EntitlementNotAllowedException The entitlement is not allowed.</li>
     *         <li>UnsupportedDigitalSignatureMethodException The digital signature method is unsupported. Try your
     *         request again.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CheckoutBorrowLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CheckoutBorrowLicense"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckoutBorrowLicenseResponse> checkoutBorrowLicense(
            CheckoutBorrowLicenseRequest checkoutBorrowLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(checkoutBorrowLicenseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkoutBorrowLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckoutBorrowLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckoutBorrowLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckoutBorrowLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckoutBorrowLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckoutBorrowLicenseRequest, CheckoutBorrowLicenseResponse>()
                            .withOperationName("CheckoutBorrowLicense")
                            .withMarshaller(new CheckoutBorrowLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(checkoutBorrowLicenseRequest));
            CompletableFuture<CheckoutBorrowLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks out the specified license.
     * </p>
     * <note>
     * <p>
     * If the account that created the license is the same that is performing the check out, you must specify the
     * account as the beneficiary.
     * </p>
     * </note>
     *
     * @param checkoutLicenseRequest
     * @return A Java Future containing the result of the CheckoutLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>NoEntitlementsAllowedException There are no entitlements found for this license, or the entitlement
     *         maximum count is reached.</li>
     *         <li>UnsupportedDigitalSignatureMethodException The digital signature method is unsupported. Try your
     *         request again.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CheckoutLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CheckoutLicense"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CheckoutLicenseResponse> checkoutLicense(CheckoutLicenseRequest checkoutLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(checkoutLicenseRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, checkoutLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CheckoutLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CheckoutLicenseResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CheckoutLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CheckoutLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CheckoutLicenseRequest, CheckoutLicenseResponse>()
                            .withOperationName("CheckoutLicense")
                            .withMarshaller(new CheckoutLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(checkoutLicenseRequest));
            CompletableFuture<CheckoutLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a grant for the specified license. A grant shares the use of license entitlements with a specific Amazon
     * Web Services account, an organization, or an organizational unit (OU). For more information, see <a
     * href="https://docs.aws.amazon.com/license-manager/latest/userguide/granted-licenses.html">Granted licenses in
     * License Manager</a> in the <i>License Manager User Guide</i>.
     * </p>
     *
     * @param createGrantRequest
     * @return A Java Future containing the result of the CreateGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateGrant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGrantResponse> createGrant(CreateGrantRequest createGrantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGrantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGrantRequest, CreateGrantResponse>()
                            .withOperationName("CreateGrant").withMarshaller(new CreateGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGrantRequest));
            CompletableFuture<CreateGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of the specified grant. For more information, see <a
     * href="https://docs.aws.amazon.com/license-manager/latest/userguide/granted-licenses.html">Granted licenses in
     * License Manager</a> in the <i>License Manager User Guide</i>.
     * </p>
     *
     * @param createGrantVersionRequest
     * @return A Java Future containing the result of the CreateGrantVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateGrantVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateGrantVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGrantVersionResponse> createGrantVersion(CreateGrantVersionRequest createGrantVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGrantVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGrantVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGrantVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGrantVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGrantVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateGrantVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGrantVersionRequest, CreateGrantVersionResponse>()
                            .withOperationName("CreateGrantVersion")
                            .withMarshaller(new CreateGrantVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createGrantVersionRequest));
            CompletableFuture<CreateGrantVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a license.
     * </p>
     *
     * @param createLicenseRequest
     * @return A Java Future containing the result of the CreateLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseResponse> createLicense(CreateLicenseRequest createLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLicenseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseRequest, CreateLicenseResponse>()
                            .withOperationName("CreateLicense")
                            .withMarshaller(new CreateLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLicenseRequest));
            CompletableFuture<CreateLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a license configuration.
     * </p>
     * <p>
     * A license configuration is an abstraction of a customer license agreement that can be consumed and enforced by
     * License Manager. Components include specifications for the license type (licensing by instance, socket, CPU, or
     * vCPU), allowed tenancy (shared tenancy, Dedicated Instance, Dedicated Host, or all of these), license affinity to
     * host (how long a license must be associated with a host), and the number of licenses purchased and used.
     * </p>
     *
     * @param createLicenseConfigurationRequest
     * @return A Java Future containing the result of the CreateLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseConfigurationResponse> createLicenseConfiguration(
            CreateLicenseConfigurationRequest createLicenseConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLicenseConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseConfigurationRequest, CreateLicenseConfigurationResponse>()
                            .withOperationName("CreateLicenseConfiguration")
                            .withMarshaller(new CreateLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLicenseConfigurationRequest));
            CompletableFuture<CreateLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new license conversion task.
     * </p>
     *
     * @param createLicenseConversionTaskForResourceRequest
     * @return A Java Future containing the result of the CreateLicenseConversionTaskForResource operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseConversionTaskForResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseConversionTaskForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseConversionTaskForResourceResponse> createLicenseConversionTaskForResource(
            CreateLicenseConversionTaskForResourceRequest createLicenseConversionTaskForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLicenseConversionTaskForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createLicenseConversionTaskForResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseConversionTaskForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseConversionTaskForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateLicenseConversionTaskForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseConversionTaskForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseConversionTaskForResourceRequest, CreateLicenseConversionTaskForResourceResponse>()
                            .withOperationName("CreateLicenseConversionTaskForResource")
                            .withMarshaller(new CreateLicenseConversionTaskForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLicenseConversionTaskForResourceRequest));
            CompletableFuture<CreateLicenseConversionTaskForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a report generator.
     * </p>
     *
     * @param createLicenseManagerReportGeneratorRequest
     * @return A Java Future containing the result of the CreateLicenseManagerReportGenerator operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseManagerReportGenerator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseManagerReportGenerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseManagerReportGeneratorResponse> createLicenseManagerReportGenerator(
            CreateLicenseManagerReportGeneratorRequest createLicenseManagerReportGeneratorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLicenseManagerReportGeneratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createLicenseManagerReportGeneratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseManagerReportGenerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseManagerReportGeneratorResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateLicenseManagerReportGeneratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseManagerReportGeneratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseManagerReportGeneratorRequest, CreateLicenseManagerReportGeneratorResponse>()
                            .withOperationName("CreateLicenseManagerReportGenerator")
                            .withMarshaller(new CreateLicenseManagerReportGeneratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLicenseManagerReportGeneratorRequest));
            CompletableFuture<CreateLicenseManagerReportGeneratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new version of the specified license.
     * </p>
     *
     * @param createLicenseVersionRequest
     * @return A Java Future containing the result of the CreateLicenseVersion operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateLicenseVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateLicenseVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateLicenseVersionResponse> createLicenseVersion(
            CreateLicenseVersionRequest createLicenseVersionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createLicenseVersionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createLicenseVersionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateLicenseVersion");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateLicenseVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateLicenseVersionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateLicenseVersionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateLicenseVersionRequest, CreateLicenseVersionResponse>()
                            .withOperationName("CreateLicenseVersion")
                            .withMarshaller(new CreateLicenseVersionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createLicenseVersionRequest));
            CompletableFuture<CreateLicenseVersionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a long-lived token.
     * </p>
     * <p>
     * A refresh token is a JWT token used to get an access token. With an access token, you can call
     * AssumeRoleWithWebIdentity to get role credentials that you can use to call License Manager to manage the
     * specified license.
     * </p>
     *
     * @param createTokenRequest
     * @return A Java Future containing the result of the CreateToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.CreateToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/CreateToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTokenResponse> createToken(CreateTokenRequest createTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTokenRequest, CreateTokenResponse>()
                            .withOperationName("CreateToken").withMarshaller(new CreateTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createTokenRequest));
            CompletableFuture<CreateTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified grant.
     * </p>
     *
     * @param deleteGrantRequest
     * @return A Java Future containing the result of the DeleteGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteGrant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteGrantResponse> deleteGrant(DeleteGrantRequest deleteGrantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteGrantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteGrantRequest, DeleteGrantResponse>()
                            .withOperationName("DeleteGrant").withMarshaller(new DeleteGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteGrantRequest));
            CompletableFuture<DeleteGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified license.
     * </p>
     *
     * @param deleteLicenseRequest
     * @return A Java Future containing the result of the DeleteLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>ConflictException There was a conflict processing the request. Try your request again.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLicenseResponse> deleteLicense(DeleteLicenseRequest deleteLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLicenseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLicenseRequest, DeleteLicenseResponse>()
                            .withOperationName("DeleteLicense")
                            .withMarshaller(new DeleteLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLicenseRequest));
            CompletableFuture<DeleteLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified license configuration.
     * </p>
     * <p>
     * You cannot delete a license configuration that is in use.
     * </p>
     *
     * @param deleteLicenseConfigurationRequest
     * @return A Java Future containing the result of the DeleteLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteLicenseConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLicenseConfigurationResponse> deleteLicenseConfiguration(
            DeleteLicenseConfigurationRequest deleteLicenseConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLicenseConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLicenseConfigurationRequest, DeleteLicenseConfigurationResponse>()
                            .withOperationName("DeleteLicenseConfiguration")
                            .withMarshaller(new DeleteLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLicenseConfigurationRequest));
            CompletableFuture<DeleteLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified report generator.
     * </p>
     * <p>
     * This action deletes the report generator, which stops it from generating future reports. The action cannot be
     * reversed. It has no effect on the previous reports from this generator.
     * </p>
     *
     * @param deleteLicenseManagerReportGeneratorRequest
     * @return A Java Future containing the result of the DeleteLicenseManagerReportGenerator operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteLicenseManagerReportGenerator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteLicenseManagerReportGenerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteLicenseManagerReportGeneratorResponse> deleteLicenseManagerReportGenerator(
            DeleteLicenseManagerReportGeneratorRequest deleteLicenseManagerReportGeneratorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteLicenseManagerReportGeneratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteLicenseManagerReportGeneratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteLicenseManagerReportGenerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteLicenseManagerReportGeneratorResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteLicenseManagerReportGeneratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteLicenseManagerReportGeneratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteLicenseManagerReportGeneratorRequest, DeleteLicenseManagerReportGeneratorResponse>()
                            .withOperationName("DeleteLicenseManagerReportGenerator")
                            .withMarshaller(new DeleteLicenseManagerReportGeneratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteLicenseManagerReportGeneratorRequest));
            CompletableFuture<DeleteLicenseManagerReportGeneratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the specified token. Must be called in the license home Region.
     * </p>
     *
     * @param deleteTokenRequest
     * @return A Java Future containing the result of the DeleteToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>RedirectException This is not the correct Region for the resource. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.DeleteToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/DeleteToken" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTokenResponse> deleteToken(DeleteTokenRequest deleteTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTokenResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTokenRequest, DeleteTokenResponse>()
                            .withOperationName("DeleteToken").withMarshaller(new DeleteTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteTokenRequest));
            CompletableFuture<DeleteTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Extends the expiration date for license consumption.
     * </p>
     *
     * @param extendLicenseConsumptionRequest
     * @return A Java Future containing the result of the ExtendLicenseConsumption operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ExtendLicenseConsumption
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ExtendLicenseConsumption"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExtendLicenseConsumptionResponse> extendLicenseConsumption(
            ExtendLicenseConsumptionRequest extendLicenseConsumptionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(extendLicenseConsumptionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, extendLicenseConsumptionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExtendLicenseConsumption");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExtendLicenseConsumptionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExtendLicenseConsumptionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExtendLicenseConsumptionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExtendLicenseConsumptionRequest, ExtendLicenseConsumptionResponse>()
                            .withOperationName("ExtendLicenseConsumption")
                            .withMarshaller(new ExtendLicenseConsumptionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(extendLicenseConsumptionRequest));
            CompletableFuture<ExtendLicenseConsumptionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets a temporary access token to use with AssumeRoleWithWebIdentity. Access tokens are valid for one hour.
     * </p>
     *
     * @param getAccessTokenRequest
     * @return A Java Future containing the result of the GetAccessToken operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetAccessToken
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetAccessToken"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccessTokenResponse> getAccessToken(GetAccessTokenRequest getAccessTokenRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccessTokenRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccessTokenRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccessToken");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccessTokenResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccessTokenResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAccessTokenResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccessTokenRequest, GetAccessTokenResponse>()
                            .withOperationName("GetAccessToken")
                            .withMarshaller(new GetAccessTokenRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccessTokenRequest));
            CompletableFuture<GetAccessTokenResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified grant.
     * </p>
     *
     * @param getGrantRequest
     * @return A Java Future containing the result of the GetGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetGrant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetGrant" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetGrantResponse> getGrant(GetGrantRequest getGrantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getGrantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetGrantRequest, GetGrantResponse>().withOperationName("GetGrant")
                            .withMarshaller(new GetGrantRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                            .withMetricCollector(apiCallMetricCollector).withInput(getGrantRequest));
            CompletableFuture<GetGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified license.
     * </p>
     *
     * @param getLicenseRequest
     * @return A Java Future containing the result of the GetLicense operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicense
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicense" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseResponse> getLicense(GetLicenseRequest getLicenseRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLicenseRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicense");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetLicenseResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseRequest, GetLicenseResponse>().withOperationName("GetLicense")
                            .withMarshaller(new GetLicenseRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLicenseRequest));
            CompletableFuture<GetLicenseResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the specified license configuration.
     * </p>
     *
     * @param getLicenseConfigurationRequest
     * @return A Java Future containing the result of the GetLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseConfigurationResponse> getLicenseConfiguration(
            GetLicenseConfigurationRequest getLicenseConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLicenseConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseConfigurationRequest, GetLicenseConfigurationResponse>()
                            .withOperationName("GetLicenseConfiguration")
                            .withMarshaller(new GetLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLicenseConfigurationRequest));
            CompletableFuture<GetLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified license type conversion task.
     * </p>
     *
     * @param getLicenseConversionTaskRequest
     * @return A Java Future containing the result of the GetLicenseConversionTask operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseConversionTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseConversionTask"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseConversionTaskResponse> getLicenseConversionTask(
            GetLicenseConversionTaskRequest getLicenseConversionTaskRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLicenseConversionTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseConversionTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseConversionTask");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseConversionTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseConversionTaskResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseConversionTaskResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseConversionTaskRequest, GetLicenseConversionTaskResponse>()
                            .withOperationName("GetLicenseConversionTask")
                            .withMarshaller(new GetLicenseConversionTaskRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLicenseConversionTaskRequest));
            CompletableFuture<GetLicenseConversionTaskResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets information about the specified report generator.
     * </p>
     *
     * @param getLicenseManagerReportGeneratorRequest
     * @return A Java Future containing the result of the GetLicenseManagerReportGenerator operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseManagerReportGenerator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseManagerReportGenerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseManagerReportGeneratorResponse> getLicenseManagerReportGenerator(
            GetLicenseManagerReportGeneratorRequest getLicenseManagerReportGeneratorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLicenseManagerReportGeneratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getLicenseManagerReportGeneratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseManagerReportGenerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseManagerReportGeneratorResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetLicenseManagerReportGeneratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseManagerReportGeneratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseManagerReportGeneratorRequest, GetLicenseManagerReportGeneratorResponse>()
                            .withOperationName("GetLicenseManagerReportGenerator")
                            .withMarshaller(new GetLicenseManagerReportGeneratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLicenseManagerReportGeneratorRequest));
            CompletableFuture<GetLicenseManagerReportGeneratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets detailed information about the usage of the specified license.
     * </p>
     *
     * @param getLicenseUsageRequest
     * @return A Java Future containing the result of the GetLicenseUsage operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetLicenseUsage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetLicenseUsage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLicenseUsageResponse> getLicenseUsage(GetLicenseUsageRequest getLicenseUsageRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getLicenseUsageRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getLicenseUsageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLicenseUsage");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLicenseUsageResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetLicenseUsageResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLicenseUsageResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLicenseUsageRequest, GetLicenseUsageResponse>()
                            .withOperationName("GetLicenseUsage")
                            .withMarshaller(new GetLicenseUsageRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getLicenseUsageRequest));
            CompletableFuture<GetLicenseUsageResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gets the License Manager settings for the current Region.
     * </p>
     *
     * @param getServiceSettingsRequest
     * @return A Java Future containing the result of the GetServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.GetServiceSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/GetServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetServiceSettingsResponse> getServiceSettings(GetServiceSettingsRequest getServiceSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetServiceSettingsRequest, GetServiceSettingsResponse>()
                            .withOperationName("GetServiceSettings")
                            .withMarshaller(new GetServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getServiceSettingsRequest));
            CompletableFuture<GetServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the resource associations for the specified license configuration.
     * </p>
     * <p>
     * Resource associations need not consume licenses from a license configuration. For example, an AMI or a stopped
     * instance might not consume a license (depending on the license rules).
     * </p>
     *
     * @param listAssociationsForLicenseConfigurationRequest
     * @return A Java Future containing the result of the ListAssociationsForLicenseConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListAssociationsForLicenseConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListAssociationsForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociationsForLicenseConfigurationResponse> listAssociationsForLicenseConfiguration(
            ListAssociationsForLicenseConfigurationRequest listAssociationsForLicenseConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssociationsForLicenseConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listAssociationsForLicenseConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociationsForLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociationsForLicenseConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListAssociationsForLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAssociationsForLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociationsForLicenseConfigurationRequest, ListAssociationsForLicenseConfigurationResponse>()
                            .withOperationName("ListAssociationsForLicenseConfiguration")
                            .withMarshaller(new ListAssociationsForLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociationsForLicenseConfigurationRequest));
            CompletableFuture<ListAssociationsForLicenseConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the grants distributed for the specified license.
     * </p>
     *
     * @param listDistributedGrantsRequest
     * @return A Java Future containing the result of the ListDistributedGrants operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListDistributedGrants
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListDistributedGrants"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListDistributedGrantsResponse> listDistributedGrants(
            ListDistributedGrantsRequest listDistributedGrantsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listDistributedGrantsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDistributedGrantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDistributedGrants");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListDistributedGrantsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListDistributedGrantsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListDistributedGrantsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListDistributedGrantsRequest, ListDistributedGrantsResponse>()
                            .withOperationName("ListDistributedGrants")
                            .withMarshaller(new ListDistributedGrantsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listDistributedGrantsRequest));
            CompletableFuture<ListDistributedGrantsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license configuration operations that failed.
     * </p>
     *
     * @param listFailuresForLicenseConfigurationOperationsRequest
     * @return A Java Future containing the result of the ListFailuresForLicenseConfigurationOperations operation
     *         returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListFailuresForLicenseConfigurationOperations
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListFailuresForLicenseConfigurationOperations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> listFailuresForLicenseConfigurationOperations(
            ListFailuresForLicenseConfigurationOperationsRequest listFailuresForLicenseConfigurationOperationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                listFailuresForLicenseConfigurationOperationsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listFailuresForLicenseConfigurationOperationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListFailuresForLicenseConfigurationOperations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListFailuresForLicenseConfigurationOperationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListFailuresForLicenseConfigurationOperationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListFailuresForLicenseConfigurationOperationsRequest, ListFailuresForLicenseConfigurationOperationsResponse>()
                            .withOperationName("ListFailuresForLicenseConfigurationOperations")
                            .withMarshaller(new ListFailuresForLicenseConfigurationOperationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listFailuresForLicenseConfigurationOperationsRequest));
            CompletableFuture<ListFailuresForLicenseConfigurationOperationsResponse> whenCompleted = executeFuture.whenComplete((
                    r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license configurations for your account.
     * </p>
     *
     * @param listLicenseConfigurationsRequest
     * @return A Java Future containing the result of the ListLicenseConfigurations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseConfigurationsResponse> listLicenseConfigurations(
            ListLicenseConfigurationsRequest listLicenseConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLicenseConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicenseConfigurationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseConfigurationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLicenseConfigurationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseConfigurationsRequest, ListLicenseConfigurationsResponse>()
                            .withOperationName("ListLicenseConfigurations")
                            .withMarshaller(new ListLicenseConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLicenseConfigurationsRequest));
            CompletableFuture<ListLicenseConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the license type conversion tasks for your account.
     * </p>
     *
     * @param listLicenseConversionTasksRequest
     * @return A Java Future containing the result of the ListLicenseConversionTasks operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseConversionTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseConversionTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseConversionTasksResponse> listLicenseConversionTasks(
            ListLicenseConversionTasksRequest listLicenseConversionTasksRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLicenseConversionTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicenseConversionTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseConversionTasks");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseConversionTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLicenseConversionTasksResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseConversionTasksResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseConversionTasksRequest, ListLicenseConversionTasksResponse>()
                            .withOperationName("ListLicenseConversionTasks")
                            .withMarshaller(new ListLicenseConversionTasksRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLicenseConversionTasksRequest));
            CompletableFuture<ListLicenseConversionTasksResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the report generators for your account.
     * </p>
     *
     * @param listLicenseManagerReportGeneratorsRequest
     * @return A Java Future containing the result of the ListLicenseManagerReportGenerators operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseManagerReportGenerators
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseManagerReportGenerators"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseManagerReportGeneratorsResponse> listLicenseManagerReportGenerators(
            ListLicenseManagerReportGeneratorsRequest listLicenseManagerReportGeneratorsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLicenseManagerReportGeneratorsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listLicenseManagerReportGeneratorsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseManagerReportGenerators");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseManagerReportGeneratorsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListLicenseManagerReportGeneratorsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseManagerReportGeneratorsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseManagerReportGeneratorsRequest, ListLicenseManagerReportGeneratorsResponse>()
                            .withOperationName("ListLicenseManagerReportGenerators")
                            .withMarshaller(new ListLicenseManagerReportGeneratorsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLicenseManagerReportGeneratorsRequest));
            CompletableFuture<ListLicenseManagerReportGeneratorsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes the license configurations for the specified resource.
     * </p>
     *
     * @param listLicenseSpecificationsForResourceRequest
     * @return A Java Future containing the result of the ListLicenseSpecificationsForResource operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseSpecificationsForResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseSpecificationsForResourceResponse> listLicenseSpecificationsForResource(
            ListLicenseSpecificationsForResourceRequest listLicenseSpecificationsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLicenseSpecificationsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listLicenseSpecificationsForResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseSpecificationsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListLicenseSpecificationsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseSpecificationsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseSpecificationsForResourceRequest, ListLicenseSpecificationsForResourceResponse>()
                            .withOperationName("ListLicenseSpecificationsForResource")
                            .withMarshaller(new ListLicenseSpecificationsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLicenseSpecificationsForResourceRequest));
            CompletableFuture<ListLicenseSpecificationsForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all versions of the specified license.
     * </p>
     *
     * @param listLicenseVersionsRequest
     * @return A Java Future containing the result of the ListLicenseVersions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenseVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenseVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicenseVersionsResponse> listLicenseVersions(
            ListLicenseVersionsRequest listLicenseVersionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLicenseVersionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicenseVersionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenseVersions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicenseVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListLicenseVersionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicenseVersionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicenseVersionsRequest, ListLicenseVersionsResponse>()
                            .withOperationName("ListLicenseVersions")
                            .withMarshaller(new ListLicenseVersionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLicenseVersionsRequest));
            CompletableFuture<ListLicenseVersionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the licenses for your account.
     * </p>
     *
     * @param listLicensesRequest
     * @return A Java Future containing the result of the ListLicenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListLicenses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListLicenses" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListLicensesResponse> listLicenses(ListLicensesRequest listLicensesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listLicensesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listLicensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListLicenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListLicensesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListLicensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListLicensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListLicensesRequest, ListLicensesResponse>()
                            .withOperationName("ListLicenses").withMarshaller(new ListLicensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listLicensesRequest));
            CompletableFuture<ListLicensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists grants that are received. Received grants are grants created while specifying the recipient as this Amazon
     * Web Services account, your organization, or an organizational unit (OU) to which this member account belongs.
     * </p>
     *
     * @param listReceivedGrantsRequest
     * @return A Java Future containing the result of the ListReceivedGrants operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListReceivedGrants
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListReceivedGrants"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReceivedGrantsResponse> listReceivedGrants(ListReceivedGrantsRequest listReceivedGrantsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReceivedGrantsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReceivedGrantsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReceivedGrants");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReceivedGrantsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReceivedGrantsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReceivedGrantsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReceivedGrantsRequest, ListReceivedGrantsResponse>()
                            .withOperationName("ListReceivedGrants")
                            .withMarshaller(new ListReceivedGrantsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReceivedGrantsRequest));
            CompletableFuture<ListReceivedGrantsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the grants received for all accounts in the organization.
     * </p>
     *
     * @param listReceivedGrantsForOrganizationRequest
     * @return A Java Future containing the result of the ListReceivedGrantsForOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListReceivedGrantsForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListReceivedGrantsForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReceivedGrantsForOrganizationResponse> listReceivedGrantsForOrganization(
            ListReceivedGrantsForOrganizationRequest listReceivedGrantsForOrganizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReceivedGrantsForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listReceivedGrantsForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReceivedGrantsForOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReceivedGrantsForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListReceivedGrantsForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReceivedGrantsForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReceivedGrantsForOrganizationRequest, ListReceivedGrantsForOrganizationResponse>()
                            .withOperationName("ListReceivedGrantsForOrganization")
                            .withMarshaller(new ListReceivedGrantsForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReceivedGrantsForOrganizationRequest));
            CompletableFuture<ListReceivedGrantsForOrganizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists received licenses.
     * </p>
     *
     * @param listReceivedLicensesRequest
     * @return A Java Future containing the result of the ListReceivedLicenses operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListReceivedLicenses
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListReceivedLicenses"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReceivedLicensesResponse> listReceivedLicenses(
            ListReceivedLicensesRequest listReceivedLicensesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReceivedLicensesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listReceivedLicensesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReceivedLicenses");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReceivedLicensesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListReceivedLicensesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReceivedLicensesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReceivedLicensesRequest, ListReceivedLicensesResponse>()
                            .withOperationName("ListReceivedLicenses")
                            .withMarshaller(new ListReceivedLicensesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReceivedLicensesRequest));
            CompletableFuture<ListReceivedLicensesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the licenses received for all accounts in the organization.
     * </p>
     *
     * @param listReceivedLicensesForOrganizationRequest
     * @return A Java Future containing the result of the ListReceivedLicensesForOrganization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListReceivedLicensesForOrganization
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListReceivedLicensesForOrganization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListReceivedLicensesForOrganizationResponse> listReceivedLicensesForOrganization(
            ListReceivedLicensesForOrganizationRequest listReceivedLicensesForOrganizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listReceivedLicensesForOrganizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listReceivedLicensesForOrganizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListReceivedLicensesForOrganization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListReceivedLicensesForOrganizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListReceivedLicensesForOrganizationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListReceivedLicensesForOrganizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListReceivedLicensesForOrganizationRequest, ListReceivedLicensesForOrganizationResponse>()
                            .withOperationName("ListReceivedLicensesForOrganization")
                            .withMarshaller(new ListReceivedLicensesForOrganizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listReceivedLicensesForOrganizationRequest));
            CompletableFuture<ListReceivedLicensesForOrganizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists resources managed using Systems Manager inventory.
     * </p>
     *
     * @param listResourceInventoryRequest
     * @return A Java Future containing the result of the ListResourceInventory operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>FailedDependencyException A dependency required to run the API is missing.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListResourceInventory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListResourceInventory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListResourceInventoryResponse> listResourceInventory(
            ListResourceInventoryRequest listResourceInventoryRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listResourceInventoryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listResourceInventoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListResourceInventory");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListResourceInventoryResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListResourceInventoryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListResourceInventoryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListResourceInventoryRequest, ListResourceInventoryResponse>()
                            .withOperationName("ListResourceInventory")
                            .withMarshaller(new ListResourceInventoryRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listResourceInventoryRequest));
            CompletableFuture<ListResourceInventoryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags for the specified license configuration.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource")
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists your tokens.
     * </p>
     *
     * @param listTokensRequest
     * @return A Java Future containing the result of the ListTokens operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListTokens
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListTokens" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTokensResponse> listTokens(ListTokensRequest listTokensRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTokensRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTokensRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTokens");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTokensResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTokensResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTokensResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTokensRequest, ListTokensResponse>().withOperationName("ListTokens")
                            .withMarshaller(new ListTokensRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTokensRequest));
            CompletableFuture<ListTokensResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all license usage records for a license configuration, displaying license consumption details by resource
     * at a selected point in time. Use this action to audit the current license consumption for any license inventory
     * and configuration.
     * </p>
     *
     * @param listUsageForLicenseConfigurationRequest
     * @return A Java Future containing the result of the ListUsageForLicenseConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>FilterLimitExceededException The request uses too many filters or too many filter values.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.ListUsageForLicenseConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/ListUsageForLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListUsageForLicenseConfigurationResponse> listUsageForLicenseConfiguration(
            ListUsageForLicenseConfigurationRequest listUsageForLicenseConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listUsageForLicenseConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listUsageForLicenseConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListUsageForLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListUsageForLicenseConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListUsageForLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListUsageForLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListUsageForLicenseConfigurationRequest, ListUsageForLicenseConfigurationResponse>()
                            .withOperationName("ListUsageForLicenseConfiguration")
                            .withMarshaller(new ListUsageForLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listUsageForLicenseConfigurationRequest));
            CompletableFuture<ListUsageForLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Rejects the specified grant.
     * </p>
     *
     * @param rejectGrantRequest
     * @return A Java Future containing the result of the RejectGrant operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.RejectGrant
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/RejectGrant" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<RejectGrantResponse> rejectGrant(RejectGrantRequest rejectGrantRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(rejectGrantRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, rejectGrantRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RejectGrant");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RejectGrantResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    RejectGrantResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RejectGrantResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RejectGrantRequest, RejectGrantResponse>()
                            .withOperationName("RejectGrant").withMarshaller(new RejectGrantRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(rejectGrantRequest));
            CompletableFuture<RejectGrantResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds the specified tags to the specified license configuration.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the specified tags from the specified license configuration.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the attributes of an existing license configuration.
     * </p>
     *
     * @param updateLicenseConfigurationRequest
     * @return A Java Future containing the result of the UpdateLicenseConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseConfigurationResponse> updateLicenseConfiguration(
            UpdateLicenseConfigurationRequest updateLicenseConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLicenseConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateLicenseConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateLicenseConfigurationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseConfigurationRequest, UpdateLicenseConfigurationResponse>()
                            .withOperationName("UpdateLicenseConfiguration")
                            .withMarshaller(new UpdateLicenseConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLicenseConfigurationRequest));
            CompletableFuture<UpdateLicenseConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a report generator.
     * </p>
     * <p>
     * After you make changes to a report generator, it starts generating new reports within 60 minutes of being
     * updated.
     * </p>
     *
     * @param updateLicenseManagerReportGeneratorRequest
     * @return A Java Future containing the result of the UpdateLicenseManagerReportGenerator operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ValidationException The provided input is not valid. Try your request again.</li>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>ResourceNotFoundException The resource cannot be found.</li>
     *         <li>ResourceLimitExceededException Your resource limits have been exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseManagerReportGenerator
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseManagerReportGenerator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseManagerReportGeneratorResponse> updateLicenseManagerReportGenerator(
            UpdateLicenseManagerReportGeneratorRequest updateLicenseManagerReportGeneratorRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLicenseManagerReportGeneratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateLicenseManagerReportGeneratorRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseManagerReportGenerator");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseManagerReportGeneratorResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateLicenseManagerReportGeneratorResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseManagerReportGeneratorResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseManagerReportGeneratorRequest, UpdateLicenseManagerReportGeneratorResponse>()
                            .withOperationName("UpdateLicenseManagerReportGenerator")
                            .withMarshaller(new UpdateLicenseManagerReportGeneratorRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLicenseManagerReportGeneratorRequest));
            CompletableFuture<UpdateLicenseManagerReportGeneratorResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or removes the specified license configurations for the specified Amazon Web Services resource.
     * </p>
     * <p>
     * You can update the license specifications of AMIs, instances, and hosts. You cannot update the license
     * specifications for launch templates and CloudFormation templates, as they send license configurations to the
     * operation that creates the resource.
     * </p>
     *
     * @param updateLicenseSpecificationsForResourceRequest
     * @return A Java Future containing the result of the UpdateLicenseSpecificationsForResource operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>InvalidResourceStateException License Manager cannot allocate a license to a resource because of its
     *         state. </p>
     *         <p>
     *         For example, you cannot allocate a license to an instance in the process of shutting down.</li>
     *         <li>LicenseUsageException You do not have enough licenses available to support a new resource launch.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateLicenseSpecificationsForResource
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateLicenseSpecificationsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> updateLicenseSpecificationsForResource(
            UpdateLicenseSpecificationsForResourceRequest updateLicenseSpecificationsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateLicenseSpecificationsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateLicenseSpecificationsForResourceRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateLicenseSpecificationsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateLicenseSpecificationsForResourceResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateLicenseSpecificationsForResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateLicenseSpecificationsForResourceRequest, UpdateLicenseSpecificationsForResourceResponse>()
                            .withOperationName("UpdateLicenseSpecificationsForResource")
                            .withMarshaller(new UpdateLicenseSpecificationsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateLicenseSpecificationsForResourceRequest));
            CompletableFuture<UpdateLicenseSpecificationsForResourceResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates License Manager settings for the current Region.
     * </p>
     *
     * @param updateServiceSettingsRequest
     * @return A Java Future containing the result of the UpdateServiceSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidParameterValueException One or more parameter values are not valid.</li>
     *         <li>ServerInternalException The server experienced an internal error. Try again.</li>
     *         <li>AuthorizationException The Amazon Web Services user account does not have permission to perform the
     *         action. Check the IAM policy associated with this account.</li>
     *         <li>AccessDeniedException Access to resource denied.</li>
     *         <li>RateLimitExceededException Too many requests have been submitted. Try again after a brief wait.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>LicenseManagerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample LicenseManagerAsyncClient.UpdateServiceSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/license-manager-2018-08-01/UpdateServiceSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceSettingsResponse> updateServiceSettings(
            UpdateServiceSettingsRequest updateServiceSettingsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateServiceSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "License Manager");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceSettingsRequest, UpdateServiceSettingsResponse>()
                            .withOperationName("UpdateServiceSettings")
                            .withMarshaller(new UpdateServiceSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateServiceSettingsRequest));
            CompletableFuture<UpdateServiceSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final LicenseManagerServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(LicenseManagerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FailedDependencyException")
                                .exceptionBuilderSupplier(FailedDependencyException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RateLimitExceededException")
                                .exceptionBuilderSupplier(RateLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("FilterLimitExceededException")
                                .exceptionBuilderSupplier(FilterLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServerInternalException")
                                .exceptionBuilderSupplier(ServerInternalException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RedirectException")
                                .exceptionBuilderSupplier(RedirectException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("UnsupportedDigitalSignatureMethodException")
                                .exceptionBuilderSupplier(UnsupportedDigitalSignatureMethodException::builder)
                                .httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthorizationException")
                                .exceptionBuilderSupplier(AuthorizationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidResourceStateException")
                                .exceptionBuilderSupplier(InvalidResourceStateException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EntitlementNotAllowedException")
                                .exceptionBuilderSupplier(EntitlementNotAllowedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("NoEntitlementsAllowedException")
                                .exceptionBuilderSupplier(NoEntitlementsAllowedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LicenseUsageException")
                                .exceptionBuilderSupplier(LicenseUsageException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceLimitExceededException")
                                .exceptionBuilderSupplier(ResourceLimitExceededException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        LicenseManagerServiceClientConfigurationBuilder.BuilderInternal serviceConfigBuilder = LicenseManagerServiceClientConfigurationBuilder
                .builder(clientConfiguration.toBuilder());
        serviceConfigBuilder.overrideConfiguration(serviceClientConfiguration.overrideConfiguration());
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        return serviceConfigBuilder.buildSdkClientConfiguration();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
