/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies the execution engine (cluster) to run the notebook and perform the notebook execution, for example, an
 * Amazon EMR cluster.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExecutionEngineConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ExecutionEngineConfig.Builder, ExecutionEngineConfig> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(ExecutionEngineConfig::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(ExecutionEngineConfig::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> MASTER_INSTANCE_SECURITY_GROUP_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MasterInstanceSecurityGroupId")
            .getter(getter(ExecutionEngineConfig::masterInstanceSecurityGroupId))
            .setter(setter(Builder::masterInstanceSecurityGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MasterInstanceSecurityGroupId")
                    .build()).build();

    private static final SdkField<String> EXECUTION_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExecutionRoleArn").getter(getter(ExecutionEngineConfig::executionRoleArn))
            .setter(setter(Builder::executionRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionRoleArn").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, TYPE_FIELD,
            MASTER_INSTANCE_SECURITY_GROUP_ID_FIELD, EXECUTION_ROLE_ARN_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String type;

    private final String masterInstanceSecurityGroupId;

    private final String executionRoleArn;

    private ExecutionEngineConfig(BuilderImpl builder) {
        this.id = builder.id;
        this.type = builder.type;
        this.masterInstanceSecurityGroupId = builder.masterInstanceSecurityGroupId;
        this.executionRoleArn = builder.executionRoleArn;
    }

    /**
     * <p>
     * The unique identifier of the execution engine. For an Amazon EMR cluster, this is the cluster ID.
     * </p>
     * 
     * @return The unique identifier of the execution engine. For an Amazon EMR cluster, this is the cluster ID.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ExecutionEngineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
     * @see ExecutionEngineType
     */
    public final ExecutionEngineType type() {
        return ExecutionEngineType.fromValue(type);
    }

    /**
     * <p>
     * The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link ExecutionEngineType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
     * @see ExecutionEngineType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * An optional unique ID of an Amazon EC2 security group to associate with the master instance of the Amazon EMR
     * cluster for this notebook execution. For more information see <a
     * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
     * >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>EMR Management Guide</i>.
     * </p>
     * 
     * @return An optional unique ID of an Amazon EC2 security group to associate with the master instance of the Amazon
     *         EMR cluster for this notebook execution. For more information see <a href=
     *         "https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
     *         >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>EMR Management Guide</i>.
     */
    public final String masterInstanceSecurityGroupId() {
        return masterInstanceSecurityGroupId;
    }

    /**
     * <p>
     * The execution role ARN required for the notebook execution.
     * </p>
     * 
     * @return The execution role ARN required for the notebook execution.
     */
    public final String executionRoleArn() {
        return executionRoleArn;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(masterInstanceSecurityGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(executionRoleArn());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExecutionEngineConfig)) {
            return false;
        }
        ExecutionEngineConfig other = (ExecutionEngineConfig) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(masterInstanceSecurityGroupId(), other.masterInstanceSecurityGroupId())
                && Objects.equals(executionRoleArn(), other.executionRoleArn());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExecutionEngineConfig").add("Id", id()).add("Type", typeAsString())
                .add("MasterInstanceSecurityGroupId", masterInstanceSecurityGroupId())
                .add("ExecutionRoleArn", executionRoleArn()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "MasterInstanceSecurityGroupId":
            return Optional.ofNullable(clazz.cast(masterInstanceSecurityGroupId()));
        case "ExecutionRoleArn":
            return Optional.ofNullable(clazz.cast(executionRoleArn()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ExecutionEngineConfig, T> g) {
        return obj -> g.apply((ExecutionEngineConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ExecutionEngineConfig> {
        /**
         * <p>
         * The unique identifier of the execution engine. For an Amazon EMR cluster, this is the cluster ID.
         * </p>
         * 
         * @param id
         *        The unique identifier of the execution engine. For an Amazon EMR cluster, this is the cluster ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
         * </p>
         * 
         * @param type
         *        The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
         * @see ExecutionEngineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionEngineType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
         * </p>
         * 
         * @param type
         *        The type of execution engine. A value of <code>EMR</code> specifies an Amazon EMR cluster.
         * @see ExecutionEngineType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionEngineType
         */
        Builder type(ExecutionEngineType type);

        /**
         * <p>
         * An optional unique ID of an Amazon EC2 security group to associate with the master instance of the Amazon EMR
         * cluster for this notebook execution. For more information see <a
         * href="https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
         * >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>EMR Management Guide</i>.
         * </p>
         * 
         * @param masterInstanceSecurityGroupId
         *        An optional unique ID of an Amazon EC2 security group to associate with the master instance of the
         *        Amazon EMR cluster for this notebook execution. For more information see <a href=
         *        "https://docs.aws.amazon.com/emr/latest/ManagementGuide/emr-managed-notebooks-security-groups.html"
         *        >Specifying Amazon EC2 Security Groups for Amazon EMR Notebooks</a> in the <i>EMR Management
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder masterInstanceSecurityGroupId(String masterInstanceSecurityGroupId);

        /**
         * <p>
         * The execution role ARN required for the notebook execution.
         * </p>
         * 
         * @param executionRoleArn
         *        The execution role ARN required for the notebook execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder executionRoleArn(String executionRoleArn);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String type;

        private String masterInstanceSecurityGroupId;

        private String executionRoleArn;

        private BuilderImpl() {
        }

        private BuilderImpl(ExecutionEngineConfig model) {
            id(model.id);
            type(model.type);
            masterInstanceSecurityGroupId(model.masterInstanceSecurityGroupId);
            executionRoleArn(model.executionRoleArn);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(ExecutionEngineType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getMasterInstanceSecurityGroupId() {
            return masterInstanceSecurityGroupId;
        }

        public final void setMasterInstanceSecurityGroupId(String masterInstanceSecurityGroupId) {
            this.masterInstanceSecurityGroupId = masterInstanceSecurityGroupId;
        }

        @Override
        public final Builder masterInstanceSecurityGroupId(String masterInstanceSecurityGroupId) {
            this.masterInstanceSecurityGroupId = masterInstanceSecurityGroupId;
            return this;
        }

        public final String getExecutionRoleArn() {
            return executionRoleArn;
        }

        public final void setExecutionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
        }

        @Override
        public final Builder executionRoleArn(String executionRoleArn) {
            this.executionRoleArn = executionRoleArn;
            return this;
        }

        @Override
        public ExecutionEngineConfig build() {
            return new ExecutionEngineConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
