/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.waiters;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.elasticache.ElastiCacheClient;
import software.amazon.awssdk.services.elasticache.model.DescribeCacheClustersRequest;
import software.amazon.awssdk.services.elasticache.model.DescribeCacheClustersResponse;
import software.amazon.awssdk.services.elasticache.model.DescribeReplicationGroupsRequest;
import software.amazon.awssdk.services.elasticache.model.DescribeReplicationGroupsResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
public interface ElastiCacheWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition
     * {@code CacheClusterAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeCacheClustersRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterAvailable(
            DescribeCacheClustersRequest describeCacheClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition
     * {@code CacheClusterAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeCacheClustersRequest#builder()}
     *
     * @param describeCacheClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterAvailable(
            Consumer<DescribeCacheClustersRequest.Builder> describeCacheClustersRequest) {
        return waitUntilCacheClusterAvailable(DescribeCacheClustersRequest.builder().applyMutation(describeCacheClustersRequest)
                .build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition
     * {@code CacheClusterAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeCacheClustersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterAvailable(
            DescribeCacheClustersRequest describeCacheClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition
     * {@code CacheClusterAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeCacheClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterAvailable(
            Consumer<DescribeCacheClustersRequest.Builder> describeCacheClustersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilCacheClusterAvailable(DescribeCacheClustersRequest.builder().applyMutation(describeCacheClustersRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition {@code CacheClusterDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeCacheClustersRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterDeleted(
            DescribeCacheClustersRequest describeCacheClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition {@code CacheClusterDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeCacheClustersRequest#builder()}
     *
     * @param describeCacheClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterDeleted(
            Consumer<DescribeCacheClustersRequest.Builder> describeCacheClustersRequest) {
        return waitUntilCacheClusterDeleted(DescribeCacheClustersRequest.builder().applyMutation(describeCacheClustersRequest)
                .build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition {@code CacheClusterDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeCacheClustersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterDeleted(
            DescribeCacheClustersRequest describeCacheClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeCacheClusters} API until the desired condition {@code CacheClusterDeleted}
     * is met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeCacheClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterDeleted(
            Consumer<DescribeCacheClustersRequest.Builder> describeCacheClustersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilCacheClusterDeleted(DescribeCacheClustersRequest.builder().applyMutation(describeCacheClustersRequest)
                .build(), WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationGroupsRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupAvailable(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationGroupsRequest#builder()}
     *
     * @param describeReplicationGroupsRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupAvailable(
            Consumer<DescribeReplicationGroupsRequest.Builder> describeReplicationGroupsRequest) {
        return waitUntilReplicationGroupAvailable(DescribeReplicationGroupsRequest.builder()
                .applyMutation(describeReplicationGroupsRequest).build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationGroupsRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupAvailable(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupAvailable} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationGroupsRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupAvailable(
            Consumer<DescribeReplicationGroupsRequest.Builder> describeReplicationGroupsRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationGroupAvailable(
                DescribeReplicationGroupsRequest.builder().applyMutation(describeReplicationGroupsRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationGroupsRequest
     *        the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupDeleted(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeReplicationGroupsRequest#builder()}
     *
     * @param describeReplicationGroupsRequest
     *        The consumer that will configure the request to be used for polling
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupDeleted(
            Consumer<DescribeReplicationGroupsRequest.Builder> describeReplicationGroupsRequest) {
        return waitUntilReplicationGroupDeleted(DescribeReplicationGroupsRequest.builder()
                .applyMutation(describeReplicationGroupsRequest).build());
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state
     *
     * @param describeReplicationGroupsRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupDeleted(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link ElastiCacheClient#describeReplicationGroups} API until the desired condition
     * {@code ReplicationGroupDeleted} is met, or until it is determined that the resource will never enter into the
     * desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeReplicationGroupsRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupDeleted(
            Consumer<DescribeReplicationGroupsRequest.Builder> describeReplicationGroupsRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilReplicationGroupDeleted(
                DescribeReplicationGroupsRequest.builder().applyMutation(describeReplicationGroupsRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link ElastiCacheWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultElastiCacheWaiter.builder();
    }

    /**
     * Create an instance of {@link ElastiCacheWaiter} with the default configuration.
     * <p>
     * <b>A default {@link ElastiCacheClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link ElastiCacheWaiter}
     */
    static ElastiCacheWaiter create() {
        return DefaultElastiCacheWaiter.builder().build();
    }

    interface Builder {
        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link ElastiCacheClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(ElastiCacheClient client);

        /**
         * Builds an instance of {@link ElastiCacheWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link ElastiCacheWaiter}
         */
        ElastiCacheWaiter build();
    }
}
