/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.elasticache.ElastiCacheClient;
import software.amazon.awssdk.services.elasticache.model.DescribeCacheClustersRequest;
import software.amazon.awssdk.services.elasticache.model.DescribeCacheClustersResponse;
import software.amazon.awssdk.services.elasticache.model.DescribeReplicationGroupsRequest;
import software.amazon.awssdk.services.elasticache.model.DescribeReplicationGroupsResponse;
import software.amazon.awssdk.services.elasticache.model.ElastiCacheRequest;
import software.amazon.awssdk.services.elasticache.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElastiCacheWaiter implements ElastiCacheWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final ElastiCacheClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeCacheClustersResponse> cacheClusterAvailableWaiter;

    private final Waiter<DescribeCacheClustersResponse> cacheClusterDeletedWaiter;

    private final Waiter<DescribeReplicationGroupsResponse> replicationGroupAvailableWaiter;

    private final Waiter<DescribeReplicationGroupsResponse> replicationGroupDeletedWaiter;

    private DefaultElastiCacheWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElastiCacheClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.cacheClusterAvailableWaiter = Waiter.builder(DescribeCacheClustersResponse.class)
                .acceptors(cacheClusterAvailableWaiterAcceptors())
                .overrideConfiguration(cacheClusterAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.cacheClusterDeletedWaiter = Waiter.builder(DescribeCacheClustersResponse.class)
                .acceptors(cacheClusterDeletedWaiterAcceptors())
                .overrideConfiguration(cacheClusterDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.replicationGroupAvailableWaiter = Waiter.builder(DescribeReplicationGroupsResponse.class)
                .acceptors(replicationGroupAvailableWaiterAcceptors())
                .overrideConfiguration(replicationGroupAvailableWaiterConfig(builder.overrideConfiguration)).build();
        this.replicationGroupDeletedWaiter = Waiter.builder(DescribeReplicationGroupsResponse.class)
                .acceptors(replicationGroupDeletedWaiterAcceptors())
                .overrideConfiguration(replicationGroupDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterAvailable(
            DescribeCacheClustersRequest describeCacheClustersRequest) {
        return cacheClusterAvailableWaiter.run(() -> client
                .describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)));
    }

    @Override
    public WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterAvailable(
            DescribeCacheClustersRequest describeCacheClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        return cacheClusterAvailableWaiter.run(
                () -> client.describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)),
                cacheClusterAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterDeleted(
            DescribeCacheClustersRequest describeCacheClustersRequest) {
        return cacheClusterDeletedWaiter.run(() -> client
                .describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)));
    }

    @Override
    public WaiterResponse<DescribeCacheClustersResponse> waitUntilCacheClusterDeleted(
            DescribeCacheClustersRequest describeCacheClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        return cacheClusterDeletedWaiter.run(
                () -> client.describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)),
                cacheClusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupAvailable(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest) {
        return replicationGroupAvailableWaiter.run(() -> client
                .describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)));
    }

    @Override
    public WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupAvailable(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationGroupAvailableWaiter.run(
                () -> client.describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)),
                replicationGroupAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupDeleted(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest) {
        return replicationGroupDeletedWaiter.run(() -> client
                .describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)));
    }

    @Override
    public WaiterResponse<DescribeReplicationGroupsResponse> waitUntilReplicationGroupDeleted(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationGroupDeletedWaiter.run(
                () -> client.describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)),
                replicationGroupDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeCacheClustersResponse>> cacheClusterAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCacheClustersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-network"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "restore-failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCacheClustersResponse>> cacheClusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCacheClustersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "CacheClusterNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-network"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "restore-failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "snapshotting"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> replicationGroupAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> replicationGroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ReplicationGroupNotFoundFault")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration cacheClusterAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration cacheClusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationGroupAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationGroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElastiCacheWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElastiCacheRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElastiCacheWaiter.Builder {
        private ElastiCacheClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public ElastiCacheWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElastiCacheWaiter.Builder client(ElastiCacheClient client) {
            this.client = client;
            return this;
        }

        public ElastiCacheWaiter build() {
            return new DefaultElastiCacheWaiter(this);
        }
    }
}
