/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Optional object containing the parameters specific to an incremental export.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IncrementalExportSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<IncrementalExportSpecification.Builder, IncrementalExportSpecification> {
    private static final SdkField<Instant> EXPORT_FROM_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExportFromTime").getter(getter(IncrementalExportSpecification::exportFromTime))
            .setter(setter(Builder::exportFromTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportFromTime").build()).build();

    private static final SdkField<Instant> EXPORT_TO_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExportToTime").getter(getter(IncrementalExportSpecification::exportToTime))
            .setter(setter(Builder::exportToTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportToTime").build()).build();

    private static final SdkField<String> EXPORT_VIEW_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ExportViewType").getter(getter(IncrementalExportSpecification::exportViewTypeAsString))
            .setter(setter(Builder::exportViewType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExportViewType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXPORT_FROM_TIME_FIELD,
            EXPORT_TO_TIME_FIELD, EXPORT_VIEW_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Instant exportFromTime;

    private final Instant exportToTime;

    private final String exportViewType;

    private IncrementalExportSpecification(BuilderImpl builder) {
        this.exportFromTime = builder.exportFromTime;
        this.exportToTime = builder.exportToTime;
        this.exportViewType = builder.exportViewType;
    }

    /**
     * <p>
     * Time in the past which provides the inclusive start range for the export table's data, counted in seconds from
     * the start of the Unix epoch. The incremental export will reflect the table's state including and after this point
     * in time.
     * </p>
     * 
     * @return Time in the past which provides the inclusive start range for the export table's data, counted in seconds
     *         from the start of the Unix epoch. The incremental export will reflect the table's state including and
     *         after this point in time.
     */
    public final Instant exportFromTime() {
        return exportFromTime;
    }

    /**
     * <p>
     * Time in the past which provides the exclusive end range for the export table's data, counted in seconds from the
     * start of the Unix epoch. The incremental export will reflect the table's state just prior to this point in time.
     * If this is not provided, the latest time with data available will be used.
     * </p>
     * 
     * @return Time in the past which provides the exclusive end range for the export table's data, counted in seconds
     *         from the start of the Unix epoch. The incremental export will reflect the table's state just prior to
     *         this point in time. If this is not provided, the latest time with data available will be used.
     */
    public final Instant exportToTime() {
        return exportToTime;
    }

    /**
     * <p>
     * Choice of whether to output the previous item image prior to the start time of the incremental export. Valid
     * values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #exportViewType}
     * will return {@link ExportViewType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #exportViewTypeAsString}.
     * </p>
     * 
     * @return Choice of whether to output the previous item image prior to the start time of the incremental export.
     *         Valid values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
     * @see ExportViewType
     */
    public final ExportViewType exportViewType() {
        return ExportViewType.fromValue(exportViewType);
    }

    /**
     * <p>
     * Choice of whether to output the previous item image prior to the start time of the incremental export. Valid
     * values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #exportViewType}
     * will return {@link ExportViewType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #exportViewTypeAsString}.
     * </p>
     * 
     * @return Choice of whether to output the previous item image prior to the start time of the incremental export.
     *         Valid values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
     * @see ExportViewType
     */
    public final String exportViewTypeAsString() {
        return exportViewType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(exportFromTime());
        hashCode = 31 * hashCode + Objects.hashCode(exportToTime());
        hashCode = 31 * hashCode + Objects.hashCode(exportViewTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IncrementalExportSpecification)) {
            return false;
        }
        IncrementalExportSpecification other = (IncrementalExportSpecification) obj;
        return Objects.equals(exportFromTime(), other.exportFromTime()) && Objects.equals(exportToTime(), other.exportToTime())
                && Objects.equals(exportViewTypeAsString(), other.exportViewTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IncrementalExportSpecification").add("ExportFromTime", exportFromTime())
                .add("ExportToTime", exportToTime()).add("ExportViewType", exportViewTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExportFromTime":
            return Optional.ofNullable(clazz.cast(exportFromTime()));
        case "ExportToTime":
            return Optional.ofNullable(clazz.cast(exportToTime()));
        case "ExportViewType":
            return Optional.ofNullable(clazz.cast(exportViewTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IncrementalExportSpecification, T> g) {
        return obj -> g.apply((IncrementalExportSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IncrementalExportSpecification> {
        /**
         * <p>
         * Time in the past which provides the inclusive start range for the export table's data, counted in seconds
         * from the start of the Unix epoch. The incremental export will reflect the table's state including and after
         * this point in time.
         * </p>
         * 
         * @param exportFromTime
         *        Time in the past which provides the inclusive start range for the export table's data, counted in
         *        seconds from the start of the Unix epoch. The incremental export will reflect the table's state
         *        including and after this point in time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportFromTime(Instant exportFromTime);

        /**
         * <p>
         * Time in the past which provides the exclusive end range for the export table's data, counted in seconds from
         * the start of the Unix epoch. The incremental export will reflect the table's state just prior to this point
         * in time. If this is not provided, the latest time with data available will be used.
         * </p>
         * 
         * @param exportToTime
         *        Time in the past which provides the exclusive end range for the export table's data, counted in
         *        seconds from the start of the Unix epoch. The incremental export will reflect the table's state just
         *        prior to this point in time. If this is not provided, the latest time with data available will be
         *        used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder exportToTime(Instant exportToTime);

        /**
         * <p>
         * Choice of whether to output the previous item image prior to the start time of the incremental export. Valid
         * values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
         * </p>
         * 
         * @param exportViewType
         *        Choice of whether to output the previous item image prior to the start time of the incremental export.
         *        Valid values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
         * @see ExportViewType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExportViewType
         */
        Builder exportViewType(String exportViewType);

        /**
         * <p>
         * Choice of whether to output the previous item image prior to the start time of the incremental export. Valid
         * values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
         * </p>
         * 
         * @param exportViewType
         *        Choice of whether to output the previous item image prior to the start time of the incremental export.
         *        Valid values are <code>NEW_AND_OLD_IMAGES</code> and <code>NEW_IMAGES</code>.
         * @see ExportViewType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExportViewType
         */
        Builder exportViewType(ExportViewType exportViewType);
    }

    static final class BuilderImpl implements Builder {
        private Instant exportFromTime;

        private Instant exportToTime;

        private String exportViewType;

        private BuilderImpl() {
        }

        private BuilderImpl(IncrementalExportSpecification model) {
            exportFromTime(model.exportFromTime);
            exportToTime(model.exportToTime);
            exportViewType(model.exportViewType);
        }

        public final Instant getExportFromTime() {
            return exportFromTime;
        }

        public final void setExportFromTime(Instant exportFromTime) {
            this.exportFromTime = exportFromTime;
        }

        @Override
        public final Builder exportFromTime(Instant exportFromTime) {
            this.exportFromTime = exportFromTime;
            return this;
        }

        public final Instant getExportToTime() {
            return exportToTime;
        }

        public final void setExportToTime(Instant exportToTime) {
            this.exportToTime = exportToTime;
        }

        @Override
        public final Builder exportToTime(Instant exportToTime) {
            this.exportToTime = exportToTime;
            return this;
        }

        public final String getExportViewType() {
            return exportViewType;
        }

        public final void setExportViewType(String exportViewType) {
            this.exportViewType = exportViewType;
        }

        @Override
        public final Builder exportViewType(String exportViewType) {
            this.exportViewType = exportViewType;
            return this;
        }

        @Override
        public final Builder exportViewType(ExportViewType exportViewType) {
            this.exportViewType(exportViewType == null ? null : exportViewType.toString());
            return this;
        }

        @Override
        public IncrementalExportSpecification build() {
            return new IncrementalExportSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
