/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.directory.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the specified configurable setting for a directory.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SettingEntry implements SdkPojo, Serializable, ToCopyableBuilder<SettingEntry.Builder, SettingEntry> {
    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Type")
            .getter(getter(SettingEntry::type)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(SettingEntry::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ALLOWED_VALUES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AllowedValues").getter(getter(SettingEntry::allowedValues)).setter(setter(Builder::allowedValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedValues").build()).build();

    private static final SdkField<String> APPLIED_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppliedValue").getter(getter(SettingEntry::appliedValue)).setter(setter(Builder::appliedValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppliedValue").build()).build();

    private static final SdkField<String> REQUESTED_VALUE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequestedValue").getter(getter(SettingEntry::requestedValue)).setter(setter(Builder::requestedValue))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestedValue").build()).build();

    private static final SdkField<String> REQUEST_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequestStatus").getter(getter(SettingEntry::requestStatusAsString))
            .setter(setter(Builder::requestStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestStatus").build()).build();

    private static final SdkField<Map<String, String>> REQUEST_DETAILED_STATUS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("RequestDetailedStatus")
            .getter(getter(SettingEntry::requestDetailedStatusAsStrings))
            .setter(setter(Builder::requestDetailedStatusWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestDetailedStatus").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> REQUEST_STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RequestStatusMessage").getter(getter(SettingEntry::requestStatusMessage))
            .setter(setter(Builder::requestStatusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequestStatusMessage").build())
            .build();

    private static final SdkField<Instant> LAST_UPDATED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastUpdatedDateTime").getter(getter(SettingEntry::lastUpdatedDateTime))
            .setter(setter(Builder::lastUpdatedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedDateTime").build())
            .build();

    private static final SdkField<Instant> LAST_REQUESTED_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastRequestedDateTime").getter(getter(SettingEntry::lastRequestedDateTime))
            .setter(setter(Builder::lastRequestedDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastRequestedDateTime").build())
            .build();

    private static final SdkField<String> DATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DataType").getter(getter(SettingEntry::dataType)).setter(setter(Builder::dataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TYPE_FIELD, NAME_FIELD,
            ALLOWED_VALUES_FIELD, APPLIED_VALUE_FIELD, REQUESTED_VALUE_FIELD, REQUEST_STATUS_FIELD,
            REQUEST_DETAILED_STATUS_FIELD, REQUEST_STATUS_MESSAGE_FIELD, LAST_UPDATED_DATE_TIME_FIELD,
            LAST_REQUESTED_DATE_TIME_FIELD, DATA_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String type;

    private final String name;

    private final String allowedValues;

    private final String appliedValue;

    private final String requestedValue;

    private final String requestStatus;

    private final Map<String, String> requestDetailedStatus;

    private final String requestStatusMessage;

    private final Instant lastUpdatedDateTime;

    private final Instant lastRequestedDateTime;

    private final String dataType;

    private SettingEntry(BuilderImpl builder) {
        this.type = builder.type;
        this.name = builder.name;
        this.allowedValues = builder.allowedValues;
        this.appliedValue = builder.appliedValue;
        this.requestedValue = builder.requestedValue;
        this.requestStatus = builder.requestStatus;
        this.requestDetailedStatus = builder.requestDetailedStatus;
        this.requestStatusMessage = builder.requestStatusMessage;
        this.lastUpdatedDateTime = builder.lastUpdatedDateTime;
        this.lastRequestedDateTime = builder.lastRequestedDateTime;
        this.dataType = builder.dataType;
    }

    /**
     * <p>
     * The type, or category, of a directory setting. Similar settings have the same type. For example,
     * <code>Protocol</code>, <code>Cipher</code>, or <code>Certificate-Based Authentication</code>.
     * </p>
     * 
     * @return The type, or category, of a directory setting. Similar settings have the same type. For example,
     *         <code>Protocol</code>, <code>Cipher</code>, or <code>Certificate-Based Authentication</code>.
     */
    public final String type() {
        return type;
    }

    /**
     * <p>
     * The name of the directory setting. For example:
     * </p>
     * <p>
     * <code>TLS_1_0</code>
     * </p>
     * 
     * @return The name of the directory setting. For example:</p>
     *         <p>
     *         <code>TLS_1_0</code>
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The valid range of values for the directory setting. These values depend on the <code>DataType</code> of your
     * directory.
     * </p>
     * 
     * @return The valid range of values for the directory setting. These values depend on the <code>DataType</code> of
     *         your directory.
     */
    public final String allowedValues() {
        return allowedValues;
    }

    /**
     * <p>
     * The value of the directory setting that is applied to the directory.
     * </p>
     * 
     * @return The value of the directory setting that is applied to the directory.
     */
    public final String appliedValue() {
        return appliedValue;
    }

    /**
     * <p>
     * The value that was last requested for the directory setting.
     * </p>
     * 
     * @return The value that was last requested for the directory setting.
     */
    public final String requestedValue() {
        return requestedValue;
    }

    /**
     * <p>
     * The overall status of the request to update the directory setting request. If the directory setting is deployed
     * in more than one region, and the request fails in any region, the overall status is <code>Failed</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requestStatus}
     * will return {@link DirectoryConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #requestStatusAsString}.
     * </p>
     * 
     * @return The overall status of the request to update the directory setting request. If the directory setting is
     *         deployed in more than one region, and the request fails in any region, the overall status is
     *         <code>Failed</code>.
     * @see DirectoryConfigurationStatus
     */
    public final DirectoryConfigurationStatus requestStatus() {
        return DirectoryConfigurationStatus.fromValue(requestStatus);
    }

    /**
     * <p>
     * The overall status of the request to update the directory setting request. If the directory setting is deployed
     * in more than one region, and the request fails in any region, the overall status is <code>Failed</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #requestStatus}
     * will return {@link DirectoryConfigurationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #requestStatusAsString}.
     * </p>
     * 
     * @return The overall status of the request to update the directory setting request. If the directory setting is
     *         deployed in more than one region, and the request fails in any region, the overall status is
     *         <code>Failed</code>.
     * @see DirectoryConfigurationStatus
     */
    public final String requestStatusAsString() {
        return requestStatus;
    }

    /**
     * <p>
     * Details about the status of the request to update the directory setting. If the directory setting is deployed in
     * more than one region, status is returned for the request in each region where the setting is deployed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestDetailedStatus} method.
     * </p>
     * 
     * @return Details about the status of the request to update the directory setting. If the directory setting is
     *         deployed in more than one region, status is returned for the request in each region where the setting is
     *         deployed.
     */
    public final Map<String, DirectoryConfigurationStatus> requestDetailedStatus() {
        return DirectoryConfigurationSettingRequestDetailedStatusCopier.copyStringToEnum(requestDetailedStatus);
    }

    /**
     * For responses, this returns true if the service returned a value for the RequestDetailedStatus property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasRequestDetailedStatus() {
        return requestDetailedStatus != null && !(requestDetailedStatus instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Details about the status of the request to update the directory setting. If the directory setting is deployed in
     * more than one region, status is returned for the request in each region where the setting is deployed.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRequestDetailedStatus} method.
     * </p>
     * 
     * @return Details about the status of the request to update the directory setting. If the directory setting is
     *         deployed in more than one region, status is returned for the request in each region where the setting is
     *         deployed.
     */
    public final Map<String, String> requestDetailedStatusAsStrings() {
        return requestDetailedStatus;
    }

    /**
     * <p>
     * The last status message for the directory status request.
     * </p>
     * 
     * @return The last status message for the directory status request.
     */
    public final String requestStatusMessage() {
        return requestStatusMessage;
    }

    /**
     * <p>
     * The date and time when the directory setting was last updated.
     * </p>
     * 
     * @return The date and time when the directory setting was last updated.
     */
    public final Instant lastUpdatedDateTime() {
        return lastUpdatedDateTime;
    }

    /**
     * <p>
     * The date and time when the request to update a directory setting was last submitted.
     * </p>
     * 
     * @return The date and time when the request to update a directory setting was last submitted.
     */
    public final Instant lastRequestedDateTime() {
        return lastRequestedDateTime;
    }

    /**
     * <p>
     * The data type of a directory setting. This is used to define the <code>AllowedValues</code> of a setting. For
     * example a data type can be <code>Boolean</code>, <code>DurationInSeconds</code>, or <code>Enum</code>.
     * </p>
     * 
     * @return The data type of a directory setting. This is used to define the <code>AllowedValues</code> of a setting.
     *         For example a data type can be <code>Boolean</code>, <code>DurationInSeconds</code>, or <code>Enum</code>
     *         .
     */
    public final String dataType() {
        return dataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(type());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(allowedValues());
        hashCode = 31 * hashCode + Objects.hashCode(appliedValue());
        hashCode = 31 * hashCode + Objects.hashCode(requestedValue());
        hashCode = 31 * hashCode + Objects.hashCode(requestStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasRequestDetailedStatus() ? requestDetailedStatusAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(requestStatusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastRequestedDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(dataType());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SettingEntry)) {
            return false;
        }
        SettingEntry other = (SettingEntry) obj;
        return Objects.equals(type(), other.type()) && Objects.equals(name(), other.name())
                && Objects.equals(allowedValues(), other.allowedValues()) && Objects.equals(appliedValue(), other.appliedValue())
                && Objects.equals(requestedValue(), other.requestedValue())
                && Objects.equals(requestStatusAsString(), other.requestStatusAsString())
                && hasRequestDetailedStatus() == other.hasRequestDetailedStatus()
                && Objects.equals(requestDetailedStatusAsStrings(), other.requestDetailedStatusAsStrings())
                && Objects.equals(requestStatusMessage(), other.requestStatusMessage())
                && Objects.equals(lastUpdatedDateTime(), other.lastUpdatedDateTime())
                && Objects.equals(lastRequestedDateTime(), other.lastRequestedDateTime())
                && Objects.equals(dataType(), other.dataType());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SettingEntry").add("Type", type()).add("Name", name()).add("AllowedValues", allowedValues())
                .add("AppliedValue", appliedValue()).add("RequestedValue", requestedValue())
                .add("RequestStatus", requestStatusAsString())
                .add("RequestDetailedStatus", hasRequestDetailedStatus() ? requestDetailedStatusAsStrings() : null)
                .add("RequestStatusMessage", requestStatusMessage()).add("LastUpdatedDateTime", lastUpdatedDateTime())
                .add("LastRequestedDateTime", lastRequestedDateTime()).add("DataType", dataType()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Type":
            return Optional.ofNullable(clazz.cast(type()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "AllowedValues":
            return Optional.ofNullable(clazz.cast(allowedValues()));
        case "AppliedValue":
            return Optional.ofNullable(clazz.cast(appliedValue()));
        case "RequestedValue":
            return Optional.ofNullable(clazz.cast(requestedValue()));
        case "RequestStatus":
            return Optional.ofNullable(clazz.cast(requestStatusAsString()));
        case "RequestDetailedStatus":
            return Optional.ofNullable(clazz.cast(requestDetailedStatusAsStrings()));
        case "RequestStatusMessage":
            return Optional.ofNullable(clazz.cast(requestStatusMessage()));
        case "LastUpdatedDateTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedDateTime()));
        case "LastRequestedDateTime":
            return Optional.ofNullable(clazz.cast(lastRequestedDateTime()));
        case "DataType":
            return Optional.ofNullable(clazz.cast(dataType()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SettingEntry, T> g) {
        return obj -> g.apply((SettingEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SettingEntry> {
        /**
         * <p>
         * The type, or category, of a directory setting. Similar settings have the same type. For example,
         * <code>Protocol</code>, <code>Cipher</code>, or <code>Certificate-Based Authentication</code>.
         * </p>
         * 
         * @param type
         *        The type, or category, of a directory setting. Similar settings have the same type. For example,
         *        <code>Protocol</code>, <code>Cipher</code>, or <code>Certificate-Based Authentication</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder type(String type);

        /**
         * <p>
         * The name of the directory setting. For example:
         * </p>
         * <p>
         * <code>TLS_1_0</code>
         * </p>
         * 
         * @param name
         *        The name of the directory setting. For example:</p>
         *        <p>
         *        <code>TLS_1_0</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The valid range of values for the directory setting. These values depend on the <code>DataType</code> of your
         * directory.
         * </p>
         * 
         * @param allowedValues
         *        The valid range of values for the directory setting. These values depend on the <code>DataType</code>
         *        of your directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedValues(String allowedValues);

        /**
         * <p>
         * The value of the directory setting that is applied to the directory.
         * </p>
         * 
         * @param appliedValue
         *        The value of the directory setting that is applied to the directory.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appliedValue(String appliedValue);

        /**
         * <p>
         * The value that was last requested for the directory setting.
         * </p>
         * 
         * @param requestedValue
         *        The value that was last requested for the directory setting.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestedValue(String requestedValue);

        /**
         * <p>
         * The overall status of the request to update the directory setting request. If the directory setting is
         * deployed in more than one region, and the request fails in any region, the overall status is
         * <code>Failed</code>.
         * </p>
         * 
         * @param requestStatus
         *        The overall status of the request to update the directory setting request. If the directory setting is
         *        deployed in more than one region, and the request fails in any region, the overall status is
         *        <code>Failed</code>.
         * @see DirectoryConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryConfigurationStatus
         */
        Builder requestStatus(String requestStatus);

        /**
         * <p>
         * The overall status of the request to update the directory setting request. If the directory setting is
         * deployed in more than one region, and the request fails in any region, the overall status is
         * <code>Failed</code>.
         * </p>
         * 
         * @param requestStatus
         *        The overall status of the request to update the directory setting request. If the directory setting is
         *        deployed in more than one region, and the request fails in any region, the overall status is
         *        <code>Failed</code>.
         * @see DirectoryConfigurationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DirectoryConfigurationStatus
         */
        Builder requestStatus(DirectoryConfigurationStatus requestStatus);

        /**
         * <p>
         * Details about the status of the request to update the directory setting. If the directory setting is deployed
         * in more than one region, status is returned for the request in each region where the setting is deployed.
         * </p>
         * 
         * @param requestDetailedStatus
         *        Details about the status of the request to update the directory setting. If the directory setting is
         *        deployed in more than one region, status is returned for the request in each region where the setting
         *        is deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestDetailedStatusWithStrings(Map<String, String> requestDetailedStatus);

        /**
         * <p>
         * Details about the status of the request to update the directory setting. If the directory setting is deployed
         * in more than one region, status is returned for the request in each region where the setting is deployed.
         * </p>
         * 
         * @param requestDetailedStatus
         *        Details about the status of the request to update the directory setting. If the directory setting is
         *        deployed in more than one region, status is returned for the request in each region where the setting
         *        is deployed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestDetailedStatus(Map<String, DirectoryConfigurationStatus> requestDetailedStatus);

        /**
         * <p>
         * The last status message for the directory status request.
         * </p>
         * 
         * @param requestStatusMessage
         *        The last status message for the directory status request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requestStatusMessage(String requestStatusMessage);

        /**
         * <p>
         * The date and time when the directory setting was last updated.
         * </p>
         * 
         * @param lastUpdatedDateTime
         *        The date and time when the directory setting was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedDateTime(Instant lastUpdatedDateTime);

        /**
         * <p>
         * The date and time when the request to update a directory setting was last submitted.
         * </p>
         * 
         * @param lastRequestedDateTime
         *        The date and time when the request to update a directory setting was last submitted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRequestedDateTime(Instant lastRequestedDateTime);

        /**
         * <p>
         * The data type of a directory setting. This is used to define the <code>AllowedValues</code> of a setting. For
         * example a data type can be <code>Boolean</code>, <code>DurationInSeconds</code>, or <code>Enum</code>.
         * </p>
         * 
         * @param dataType
         *        The data type of a directory setting. This is used to define the <code>AllowedValues</code> of a
         *        setting. For example a data type can be <code>Boolean</code>, <code>DurationInSeconds</code>, or
         *        <code>Enum</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dataType(String dataType);
    }

    static final class BuilderImpl implements Builder {
        private String type;

        private String name;

        private String allowedValues;

        private String appliedValue;

        private String requestedValue;

        private String requestStatus;

        private Map<String, String> requestDetailedStatus = DefaultSdkAutoConstructMap.getInstance();

        private String requestStatusMessage;

        private Instant lastUpdatedDateTime;

        private Instant lastRequestedDateTime;

        private String dataType;

        private BuilderImpl() {
        }

        private BuilderImpl(SettingEntry model) {
            type(model.type);
            name(model.name);
            allowedValues(model.allowedValues);
            appliedValue(model.appliedValue);
            requestedValue(model.requestedValue);
            requestStatus(model.requestStatus);
            requestDetailedStatusWithStrings(model.requestDetailedStatus);
            requestStatusMessage(model.requestStatusMessage);
            lastUpdatedDateTime(model.lastUpdatedDateTime);
            lastRequestedDateTime(model.lastRequestedDateTime);
            dataType(model.dataType);
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getAllowedValues() {
            return allowedValues;
        }

        public final void setAllowedValues(String allowedValues) {
            this.allowedValues = allowedValues;
        }

        @Override
        public final Builder allowedValues(String allowedValues) {
            this.allowedValues = allowedValues;
            return this;
        }

        public final String getAppliedValue() {
            return appliedValue;
        }

        public final void setAppliedValue(String appliedValue) {
            this.appliedValue = appliedValue;
        }

        @Override
        public final Builder appliedValue(String appliedValue) {
            this.appliedValue = appliedValue;
            return this;
        }

        public final String getRequestedValue() {
            return requestedValue;
        }

        public final void setRequestedValue(String requestedValue) {
            this.requestedValue = requestedValue;
        }

        @Override
        public final Builder requestedValue(String requestedValue) {
            this.requestedValue = requestedValue;
            return this;
        }

        public final String getRequestStatus() {
            return requestStatus;
        }

        public final void setRequestStatus(String requestStatus) {
            this.requestStatus = requestStatus;
        }

        @Override
        public final Builder requestStatus(String requestStatus) {
            this.requestStatus = requestStatus;
            return this;
        }

        @Override
        public final Builder requestStatus(DirectoryConfigurationStatus requestStatus) {
            this.requestStatus(requestStatus == null ? null : requestStatus.toString());
            return this;
        }

        public final Map<String, String> getRequestDetailedStatus() {
            if (requestDetailedStatus instanceof SdkAutoConstructMap) {
                return null;
            }
            return requestDetailedStatus;
        }

        public final void setRequestDetailedStatus(Map<String, String> requestDetailedStatus) {
            this.requestDetailedStatus = DirectoryConfigurationSettingRequestDetailedStatusCopier.copy(requestDetailedStatus);
        }

        @Override
        public final Builder requestDetailedStatusWithStrings(Map<String, String> requestDetailedStatus) {
            this.requestDetailedStatus = DirectoryConfigurationSettingRequestDetailedStatusCopier.copy(requestDetailedStatus);
            return this;
        }

        @Override
        public final Builder requestDetailedStatus(Map<String, DirectoryConfigurationStatus> requestDetailedStatus) {
            this.requestDetailedStatus = DirectoryConfigurationSettingRequestDetailedStatusCopier
                    .copyEnumToString(requestDetailedStatus);
            return this;
        }

        public final String getRequestStatusMessage() {
            return requestStatusMessage;
        }

        public final void setRequestStatusMessage(String requestStatusMessage) {
            this.requestStatusMessage = requestStatusMessage;
        }

        @Override
        public final Builder requestStatusMessage(String requestStatusMessage) {
            this.requestStatusMessage = requestStatusMessage;
            return this;
        }

        public final Instant getLastUpdatedDateTime() {
            return lastUpdatedDateTime;
        }

        public final void setLastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
        }

        @Override
        public final Builder lastUpdatedDateTime(Instant lastUpdatedDateTime) {
            this.lastUpdatedDateTime = lastUpdatedDateTime;
            return this;
        }

        public final Instant getLastRequestedDateTime() {
            return lastRequestedDateTime;
        }

        public final void setLastRequestedDateTime(Instant lastRequestedDateTime) {
            this.lastRequestedDateTime = lastRequestedDateTime;
        }

        @Override
        public final Builder lastRequestedDateTime(Instant lastRequestedDateTime) {
            this.lastRequestedDateTime = lastRequestedDateTime;
            return this;
        }

        public final String getDataType() {
            return dataType;
        }

        public final void setDataType(String dataType) {
            this.dataType = dataType;
        }

        @Override
        public final Builder dataType(String dataType) {
            this.dataType = dataType;
            return this;
        }

        @Override
        public SettingEntry build() {
            return new SettingEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
