/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.connect.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the filter used when retrieving metrics. <code>MetricFiltersV2</code> can be used on the
 * following metrics: <code>AVG_AGENT_CONNECTING_TIME</code>, <code>CONTACTS_CREATED</code>,
 * <code>CONTACTS_HANDLED</code>, <code>SUM_CONTACTS_DISCONNECTED</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricFilterV2 implements SdkPojo, Serializable, ToCopyableBuilder<MetricFilterV2.Builder, MetricFilterV2> {
    private static final SdkField<String> METRIC_FILTER_KEY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricFilterKey").getter(getter(MetricFilterV2::metricFilterKey))
            .setter(setter(Builder::metricFilterKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricFilterKey").build()).build();

    private static final SdkField<List<String>> METRIC_FILTER_VALUES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("MetricFilterValues")
            .getter(getter(MetricFilterV2::metricFilterValues))
            .setter(setter(Builder::metricFilterValues))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricFilterValues").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> NEGATE_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Negate").getter(getter(MetricFilterV2::negate)).setter(setter(Builder::negate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Negate").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_FILTER_KEY_FIELD,
            METRIC_FILTER_VALUES_FIELD, NEGATE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricFilterKey;

    private final List<String> metricFilterValues;

    private final Boolean negate;

    private MetricFilterV2(BuilderImpl builder) {
        this.metricFilterKey = builder.metricFilterKey;
        this.metricFilterValues = builder.metricFilterValues;
        this.negate = builder.negate;
    }

    /**
     * <p>
     * The key to use for filtering data.
     * </p>
     * <p>
     * Valid metric filter keys: <code>INITIATION_METHOD</code>, <code>DISCONNECT_REASON</code>. These are the same
     * values as the <code>InitiationMethod</code> and <code>DisconnectReason</code> in the contact record. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/connect/latest/adminguide/ctr-data-model.html#ctr-ContactTraceRecord"
     * >ContactTraceRecord</a> in the <i>Amazon Connect Administrator's Guide</i>.
     * </p>
     * 
     * @return The key to use for filtering data. </p>
     *         <p>
     *         Valid metric filter keys: <code>INITIATION_METHOD</code>, <code>DISCONNECT_REASON</code>. These are the
     *         same values as the <code>InitiationMethod</code> and <code>DisconnectReason</code> in the contact record.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/connect/latest/adminguide/ctr-data-model.html#ctr-ContactTraceRecord"
     *         >ContactTraceRecord</a> in the <i>Amazon Connect Administrator's Guide</i>.
     */
    public final String metricFilterKey() {
        return metricFilterKey;
    }

    /**
     * For responses, this returns true if the service returned a value for the MetricFilterValues property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasMetricFilterValues() {
        return metricFilterValues != null && !(metricFilterValues instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The values to use for filtering data.
     * </p>
     * <p>
     * Valid metric filter values for <code>INITIATION_METHOD</code>: <code>INBOUND</code> | <code>OUTBOUND</code> |
     * <code>TRANSFER</code> | <code>QUEUE_TRANSFER</code> | <code>CALLBACK</code> | <code>API</code>
     * </p>
     * <p>
     * Valid metric filter values for <code>DISCONNECT_REASON</code>: <code>CUSTOMER_DISCONNECT</code> |
     * <code>AGENT_DISCONNECT</code> | <code>THIRD_PARTY_DISCONNECT</code> | <code>TELECOM_PROBLEM</code> |
     * <code>BARGED</code> | <code>CONTACT_FLOW_DISCONNECT</code> | <code>OTHER</code> | <code>EXPIRED</code> |
     * <code>API</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetricFilterValues} method.
     * </p>
     * 
     * @return The values to use for filtering data. </p>
     *         <p>
     *         Valid metric filter values for <code>INITIATION_METHOD</code>: <code>INBOUND</code> |
     *         <code>OUTBOUND</code> | <code>TRANSFER</code> | <code>QUEUE_TRANSFER</code> | <code>CALLBACK</code> |
     *         <code>API</code>
     *         </p>
     *         <p>
     *         Valid metric filter values for <code>DISCONNECT_REASON</code>: <code>CUSTOMER_DISCONNECT</code> |
     *         <code>AGENT_DISCONNECT</code> | <code>THIRD_PARTY_DISCONNECT</code> | <code>TELECOM_PROBLEM</code> |
     *         <code>BARGED</code> | <code>CONTACT_FLOW_DISCONNECT</code> | <code>OTHER</code> | <code>EXPIRED</code> |
     *         <code>API</code>
     */
    public final List<String> metricFilterValues() {
        return metricFilterValues;
    }

    /**
     * <p>
     * The flag to use to filter on requested metric filter values or to not filter on requested metric filter values.
     * By default the negate is <code>false</code>, which indicates to filter on the requested metric filter.
     * </p>
     * 
     * @return The flag to use to filter on requested metric filter values or to not filter on requested metric filter
     *         values. By default the negate is <code>false</code>, which indicates to filter on the requested metric
     *         filter.
     */
    public final Boolean negate() {
        return negate;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricFilterKey());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetricFilterValues() ? metricFilterValues() : null);
        hashCode = 31 * hashCode + Objects.hashCode(negate());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricFilterV2)) {
            return false;
        }
        MetricFilterV2 other = (MetricFilterV2) obj;
        return Objects.equals(metricFilterKey(), other.metricFilterKey())
                && hasMetricFilterValues() == other.hasMetricFilterValues()
                && Objects.equals(metricFilterValues(), other.metricFilterValues()) && Objects.equals(negate(), other.negate());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricFilterV2").add("MetricFilterKey", metricFilterKey())
                .add("MetricFilterValues", hasMetricFilterValues() ? metricFilterValues() : null).add("Negate", negate()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricFilterKey":
            return Optional.ofNullable(clazz.cast(metricFilterKey()));
        case "MetricFilterValues":
            return Optional.ofNullable(clazz.cast(metricFilterValues()));
        case "Negate":
            return Optional.ofNullable(clazz.cast(negate()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricFilterV2, T> g) {
        return obj -> g.apply((MetricFilterV2) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricFilterV2> {
        /**
         * <p>
         * The key to use for filtering data.
         * </p>
         * <p>
         * Valid metric filter keys: <code>INITIATION_METHOD</code>, <code>DISCONNECT_REASON</code>. These are the same
         * values as the <code>InitiationMethod</code> and <code>DisconnectReason</code> in the contact record. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/connect/latest/adminguide/ctr-data-model.html#ctr-ContactTraceRecord"
         * >ContactTraceRecord</a> in the <i>Amazon Connect Administrator's Guide</i>.
         * </p>
         * 
         * @param metricFilterKey
         *        The key to use for filtering data. </p>
         *        <p>
         *        Valid metric filter keys: <code>INITIATION_METHOD</code>, <code>DISCONNECT_REASON</code>. These are
         *        the same values as the <code>InitiationMethod</code> and <code>DisconnectReason</code> in the contact
         *        record. For more information, see <a href=
         *        "https://docs.aws.amazon.com/connect/latest/adminguide/ctr-data-model.html#ctr-ContactTraceRecord"
         *        >ContactTraceRecord</a> in the <i>Amazon Connect Administrator's Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricFilterKey(String metricFilterKey);

        /**
         * <p>
         * The values to use for filtering data.
         * </p>
         * <p>
         * Valid metric filter values for <code>INITIATION_METHOD</code>: <code>INBOUND</code> | <code>OUTBOUND</code> |
         * <code>TRANSFER</code> | <code>QUEUE_TRANSFER</code> | <code>CALLBACK</code> | <code>API</code>
         * </p>
         * <p>
         * Valid metric filter values for <code>DISCONNECT_REASON</code>: <code>CUSTOMER_DISCONNECT</code> |
         * <code>AGENT_DISCONNECT</code> | <code>THIRD_PARTY_DISCONNECT</code> | <code>TELECOM_PROBLEM</code> |
         * <code>BARGED</code> | <code>CONTACT_FLOW_DISCONNECT</code> | <code>OTHER</code> | <code>EXPIRED</code> |
         * <code>API</code>
         * </p>
         * 
         * @param metricFilterValues
         *        The values to use for filtering data. </p>
         *        <p>
         *        Valid metric filter values for <code>INITIATION_METHOD</code>: <code>INBOUND</code> |
         *        <code>OUTBOUND</code> | <code>TRANSFER</code> | <code>QUEUE_TRANSFER</code> | <code>CALLBACK</code> |
         *        <code>API</code>
         *        </p>
         *        <p>
         *        Valid metric filter values for <code>DISCONNECT_REASON</code>: <code>CUSTOMER_DISCONNECT</code> |
         *        <code>AGENT_DISCONNECT</code> | <code>THIRD_PARTY_DISCONNECT</code> | <code>TELECOM_PROBLEM</code> |
         *        <code>BARGED</code> | <code>CONTACT_FLOW_DISCONNECT</code> | <code>OTHER</code> | <code>EXPIRED</code>
         *        | <code>API</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricFilterValues(Collection<String> metricFilterValues);

        /**
         * <p>
         * The values to use for filtering data.
         * </p>
         * <p>
         * Valid metric filter values for <code>INITIATION_METHOD</code>: <code>INBOUND</code> | <code>OUTBOUND</code> |
         * <code>TRANSFER</code> | <code>QUEUE_TRANSFER</code> | <code>CALLBACK</code> | <code>API</code>
         * </p>
         * <p>
         * Valid metric filter values for <code>DISCONNECT_REASON</code>: <code>CUSTOMER_DISCONNECT</code> |
         * <code>AGENT_DISCONNECT</code> | <code>THIRD_PARTY_DISCONNECT</code> | <code>TELECOM_PROBLEM</code> |
         * <code>BARGED</code> | <code>CONTACT_FLOW_DISCONNECT</code> | <code>OTHER</code> | <code>EXPIRED</code> |
         * <code>API</code>
         * </p>
         * 
         * @param metricFilterValues
         *        The values to use for filtering data. </p>
         *        <p>
         *        Valid metric filter values for <code>INITIATION_METHOD</code>: <code>INBOUND</code> |
         *        <code>OUTBOUND</code> | <code>TRANSFER</code> | <code>QUEUE_TRANSFER</code> | <code>CALLBACK</code> |
         *        <code>API</code>
         *        </p>
         *        <p>
         *        Valid metric filter values for <code>DISCONNECT_REASON</code>: <code>CUSTOMER_DISCONNECT</code> |
         *        <code>AGENT_DISCONNECT</code> | <code>THIRD_PARTY_DISCONNECT</code> | <code>TELECOM_PROBLEM</code> |
         *        <code>BARGED</code> | <code>CONTACT_FLOW_DISCONNECT</code> | <code>OTHER</code> | <code>EXPIRED</code>
         *        | <code>API</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricFilterValues(String... metricFilterValues);

        /**
         * <p>
         * The flag to use to filter on requested metric filter values or to not filter on requested metric filter
         * values. By default the negate is <code>false</code>, which indicates to filter on the requested metric
         * filter.
         * </p>
         * 
         * @param negate
         *        The flag to use to filter on requested metric filter values or to not filter on requested metric
         *        filter values. By default the negate is <code>false</code>, which indicates to filter on the requested
         *        metric filter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder negate(Boolean negate);
    }

    static final class BuilderImpl implements Builder {
        private String metricFilterKey;

        private List<String> metricFilterValues = DefaultSdkAutoConstructList.getInstance();

        private Boolean negate;

        private BuilderImpl() {
        }

        private BuilderImpl(MetricFilterV2 model) {
            metricFilterKey(model.metricFilterKey);
            metricFilterValues(model.metricFilterValues);
            negate(model.negate);
        }

        public final String getMetricFilterKey() {
            return metricFilterKey;
        }

        public final void setMetricFilterKey(String metricFilterKey) {
            this.metricFilterKey = metricFilterKey;
        }

        @Override
        public final Builder metricFilterKey(String metricFilterKey) {
            this.metricFilterKey = metricFilterKey;
            return this;
        }

        public final Collection<String> getMetricFilterValues() {
            if (metricFilterValues instanceof SdkAutoConstructList) {
                return null;
            }
            return metricFilterValues;
        }

        public final void setMetricFilterValues(Collection<String> metricFilterValues) {
            this.metricFilterValues = MetricFilterValueListCopier.copy(metricFilterValues);
        }

        @Override
        public final Builder metricFilterValues(Collection<String> metricFilterValues) {
            this.metricFilterValues = MetricFilterValueListCopier.copy(metricFilterValues);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metricFilterValues(String... metricFilterValues) {
            metricFilterValues(Arrays.asList(metricFilterValues));
            return this;
        }

        public final Boolean getNegate() {
            return negate;
        }

        public final void setNegate(Boolean negate) {
            this.negate = negate;
        }

        @Override
        public final Builder negate(Boolean negate) {
            this.negate = negate;
            return this;
        }

        @Override
        public MetricFilterV2 build() {
            return new MetricFilterV2(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
