/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A configuration for a set of HTTP response headers that are used for cross-origin resource sharing (CORS). CloudFront
 * adds these headers to HTTP responses that it sends for CORS requests that match a cache behavior associated with this
 * response headers policy.
 * </p>
 * <p>
 * For more information about CORS, see <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/CORS">Cross-Origin
 * Resource Sharing (CORS)</a> in the MDN Web Docs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseHeadersPolicyCorsConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseHeadersPolicyCorsConfig.Builder, ResponseHeadersPolicyCorsConfig> {
    private static final SdkField<ResponseHeadersPolicyAccessControlAllowOrigins> ACCESS_CONTROL_ALLOW_ORIGINS_FIELD = SdkField
            .<ResponseHeadersPolicyAccessControlAllowOrigins> builder(MarshallingType.SDK_POJO)
            .memberName("AccessControlAllowOrigins")
            .getter(getter(ResponseHeadersPolicyCorsConfig::accessControlAllowOrigins))
            .setter(setter(Builder::accessControlAllowOrigins))
            .constructor(ResponseHeadersPolicyAccessControlAllowOrigins::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlAllowOrigins")
                    .unmarshallLocationName("AccessControlAllowOrigins").build()).build();

    private static final SdkField<ResponseHeadersPolicyAccessControlAllowHeaders> ACCESS_CONTROL_ALLOW_HEADERS_FIELD = SdkField
            .<ResponseHeadersPolicyAccessControlAllowHeaders> builder(MarshallingType.SDK_POJO)
            .memberName("AccessControlAllowHeaders")
            .getter(getter(ResponseHeadersPolicyCorsConfig::accessControlAllowHeaders))
            .setter(setter(Builder::accessControlAllowHeaders))
            .constructor(ResponseHeadersPolicyAccessControlAllowHeaders::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlAllowHeaders")
                    .unmarshallLocationName("AccessControlAllowHeaders").build()).build();

    private static final SdkField<ResponseHeadersPolicyAccessControlAllowMethods> ACCESS_CONTROL_ALLOW_METHODS_FIELD = SdkField
            .<ResponseHeadersPolicyAccessControlAllowMethods> builder(MarshallingType.SDK_POJO)
            .memberName("AccessControlAllowMethods")
            .getter(getter(ResponseHeadersPolicyCorsConfig::accessControlAllowMethods))
            .setter(setter(Builder::accessControlAllowMethods))
            .constructor(ResponseHeadersPolicyAccessControlAllowMethods::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlAllowMethods")
                    .unmarshallLocationName("AccessControlAllowMethods").build()).build();

    private static final SdkField<Boolean> ACCESS_CONTROL_ALLOW_CREDENTIALS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AccessControlAllowCredentials")
            .getter(getter(ResponseHeadersPolicyCorsConfig::accessControlAllowCredentials))
            .setter(setter(Builder::accessControlAllowCredentials))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlAllowCredentials")
                    .unmarshallLocationName("AccessControlAllowCredentials").build()).build();

    private static final SdkField<ResponseHeadersPolicyAccessControlExposeHeaders> ACCESS_CONTROL_EXPOSE_HEADERS_FIELD = SdkField
            .<ResponseHeadersPolicyAccessControlExposeHeaders> builder(MarshallingType.SDK_POJO)
            .memberName("AccessControlExposeHeaders")
            .getter(getter(ResponseHeadersPolicyCorsConfig::accessControlExposeHeaders))
            .setter(setter(Builder::accessControlExposeHeaders))
            .constructor(ResponseHeadersPolicyAccessControlExposeHeaders::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlExposeHeaders")
                    .unmarshallLocationName("AccessControlExposeHeaders").build()).build();

    private static final SdkField<Integer> ACCESS_CONTROL_MAX_AGE_SEC_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AccessControlMaxAgeSec")
            .getter(getter(ResponseHeadersPolicyCorsConfig::accessControlMaxAgeSec))
            .setter(setter(Builder::accessControlMaxAgeSec))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccessControlMaxAgeSec")
                    .unmarshallLocationName("AccessControlMaxAgeSec").build()).build();

    private static final SdkField<Boolean> ORIGIN_OVERRIDE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("OriginOverride")
            .getter(getter(ResponseHeadersPolicyCorsConfig::originOverride))
            .setter(setter(Builder::originOverride))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OriginOverride")
                    .unmarshallLocationName("OriginOverride").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ACCESS_CONTROL_ALLOW_ORIGINS_FIELD, ACCESS_CONTROL_ALLOW_HEADERS_FIELD, ACCESS_CONTROL_ALLOW_METHODS_FIELD,
            ACCESS_CONTROL_ALLOW_CREDENTIALS_FIELD, ACCESS_CONTROL_EXPOSE_HEADERS_FIELD, ACCESS_CONTROL_MAX_AGE_SEC_FIELD,
            ORIGIN_OVERRIDE_FIELD));

    private static final long serialVersionUID = 1L;

    private final ResponseHeadersPolicyAccessControlAllowOrigins accessControlAllowOrigins;

    private final ResponseHeadersPolicyAccessControlAllowHeaders accessControlAllowHeaders;

    private final ResponseHeadersPolicyAccessControlAllowMethods accessControlAllowMethods;

    private final Boolean accessControlAllowCredentials;

    private final ResponseHeadersPolicyAccessControlExposeHeaders accessControlExposeHeaders;

    private final Integer accessControlMaxAgeSec;

    private final Boolean originOverride;

    private ResponseHeadersPolicyCorsConfig(BuilderImpl builder) {
        this.accessControlAllowOrigins = builder.accessControlAllowOrigins;
        this.accessControlAllowHeaders = builder.accessControlAllowHeaders;
        this.accessControlAllowMethods = builder.accessControlAllowMethods;
        this.accessControlAllowCredentials = builder.accessControlAllowCredentials;
        this.accessControlExposeHeaders = builder.accessControlExposeHeaders;
        this.accessControlMaxAgeSec = builder.accessControlMaxAgeSec;
        this.originOverride = builder.originOverride;
    }

    /**
     * <p>
     * A list of origins (domain names) that CloudFront can use as the value for the
     * <code>Access-Control-Allow-Origin</code> HTTP response header.
     * </p>
     * <p>
     * For more information about the <code>Access-Control-Allow-Origin</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin"
     * >Access-Control-Allow-Origin</a> in the MDN Web Docs.
     * </p>
     * 
     * @return A list of origins (domain names) that CloudFront can use as the value for the
     *         <code>Access-Control-Allow-Origin</code> HTTP response header.</p>
     *         <p>
     *         For more information about the <code>Access-Control-Allow-Origin</code> HTTP response header, see <a
     *         href=
     *         "https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin">Access-Control
     *         -Allow-Origin</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyAccessControlAllowOrigins accessControlAllowOrigins() {
        return accessControlAllowOrigins;
    }

    /**
     * <p>
     * A list of HTTP header names that CloudFront includes as values for the <code>Access-Control-Allow-Headers</code>
     * HTTP response header.
     * </p>
     * <p>
     * For more information about the <code>Access-Control-Allow-Headers</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers"
     * >Access-Control-Allow-Headers</a> in the MDN Web Docs.
     * </p>
     * 
     * @return A list of HTTP header names that CloudFront includes as values for the
     *         <code>Access-Control-Allow-Headers</code> HTTP response header.</p>
     *         <p>
     *         For more information about the <code>Access-Control-Allow-Headers</code> HTTP response header, see <a
     *         href
     *         ="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers">Access-Control
     *         -Allow-Headers</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyAccessControlAllowHeaders accessControlAllowHeaders() {
        return accessControlAllowHeaders;
    }

    /**
     * <p>
     * A list of HTTP methods that CloudFront includes as values for the <code>Access-Control-Allow-Methods</code> HTTP
     * response header.
     * </p>
     * <p>
     * For more information about the <code>Access-Control-Allow-Methods</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods"
     * >Access-Control-Allow-Methods</a> in the MDN Web Docs.
     * </p>
     * 
     * @return A list of HTTP methods that CloudFront includes as values for the
     *         <code>Access-Control-Allow-Methods</code> HTTP response header.</p>
     *         <p>
     *         For more information about the <code>Access-Control-Allow-Methods</code> HTTP response header, see <a
     *         href
     *         ="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods">Access-Control
     *         -Allow-Methods</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyAccessControlAllowMethods accessControlAllowMethods() {
        return accessControlAllowMethods;
    }

    /**
     * <p>
     * A Boolean that CloudFront uses as the value for the <code>Access-Control-Allow-Credentials</code> HTTP response
     * header.
     * </p>
     * <p>
     * For more information about the <code>Access-Control-Allow-Credentials</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials"
     * >Access-Control-Allow-Credentials</a> in the MDN Web Docs.
     * </p>
     * 
     * @return A Boolean that CloudFront uses as the value for the <code>Access-Control-Allow-Credentials</code> HTTP
     *         response header.</p>
     *         <p>
     *         For more information about the <code>Access-Control-Allow-Credentials</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials">Access-
     *         Control-Allow-Credentials</a> in the MDN Web Docs.
     */
    public final Boolean accessControlAllowCredentials() {
        return accessControlAllowCredentials;
    }

    /**
     * <p>
     * A list of HTTP headers that CloudFront includes as values for the <code>Access-Control-Expose-Headers</code> HTTP
     * response header.
     * </p>
     * <p>
     * For more information about the <code>Access-Control-Expose-Headers</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers"
     * >Access-Control-Expose-Headers</a> in the MDN Web Docs.
     * </p>
     * 
     * @return A list of HTTP headers that CloudFront includes as values for the
     *         <code>Access-Control-Expose-Headers</code> HTTP response header.</p>
     *         <p>
     *         For more information about the <code>Access-Control-Expose-Headers</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers">Access-
     *         Control-Expose-Headers</a> in the MDN Web Docs.
     */
    public final ResponseHeadersPolicyAccessControlExposeHeaders accessControlExposeHeaders() {
        return accessControlExposeHeaders;
    }

    /**
     * <p>
     * A number that CloudFront uses as the value for the <code>Access-Control-Max-Age</code> HTTP response header.
     * </p>
     * <p>
     * For more information about the <code>Access-Control-Max-Age</code> HTTP response header, see <a
     * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age"
     * >Access-Control-Max-Age</a> in the MDN Web Docs.
     * </p>
     * 
     * @return A number that CloudFront uses as the value for the <code>Access-Control-Max-Age</code> HTTP response
     *         header.</p>
     *         <p>
     *         For more information about the <code>Access-Control-Max-Age</code> HTTP response header, see <a
     *         href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age"
     *         >Access-Control-Max-Age</a> in the MDN Web Docs.
     */
    public final Integer accessControlMaxAgeSec() {
        return accessControlMaxAgeSec;
    }

    /**
     * <p>
     * A Boolean that determines whether CloudFront overrides HTTP response headers received from the origin with the
     * ones specified in this response headers policy.
     * </p>
     * 
     * @return A Boolean that determines whether CloudFront overrides HTTP response headers received from the origin
     *         with the ones specified in this response headers policy.
     */
    public final Boolean originOverride() {
        return originOverride;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accessControlAllowOrigins());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlAllowHeaders());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlAllowMethods());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlAllowCredentials());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlExposeHeaders());
        hashCode = 31 * hashCode + Objects.hashCode(accessControlMaxAgeSec());
        hashCode = 31 * hashCode + Objects.hashCode(originOverride());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseHeadersPolicyCorsConfig)) {
            return false;
        }
        ResponseHeadersPolicyCorsConfig other = (ResponseHeadersPolicyCorsConfig) obj;
        return Objects.equals(accessControlAllowOrigins(), other.accessControlAllowOrigins())
                && Objects.equals(accessControlAllowHeaders(), other.accessControlAllowHeaders())
                && Objects.equals(accessControlAllowMethods(), other.accessControlAllowMethods())
                && Objects.equals(accessControlAllowCredentials(), other.accessControlAllowCredentials())
                && Objects.equals(accessControlExposeHeaders(), other.accessControlExposeHeaders())
                && Objects.equals(accessControlMaxAgeSec(), other.accessControlMaxAgeSec())
                && Objects.equals(originOverride(), other.originOverride());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResponseHeadersPolicyCorsConfig").add("AccessControlAllowOrigins", accessControlAllowOrigins())
                .add("AccessControlAllowHeaders", accessControlAllowHeaders())
                .add("AccessControlAllowMethods", accessControlAllowMethods())
                .add("AccessControlAllowCredentials", accessControlAllowCredentials())
                .add("AccessControlExposeHeaders", accessControlExposeHeaders())
                .add("AccessControlMaxAgeSec", accessControlMaxAgeSec()).add("OriginOverride", originOverride()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccessControlAllowOrigins":
            return Optional.ofNullable(clazz.cast(accessControlAllowOrigins()));
        case "AccessControlAllowHeaders":
            return Optional.ofNullable(clazz.cast(accessControlAllowHeaders()));
        case "AccessControlAllowMethods":
            return Optional.ofNullable(clazz.cast(accessControlAllowMethods()));
        case "AccessControlAllowCredentials":
            return Optional.ofNullable(clazz.cast(accessControlAllowCredentials()));
        case "AccessControlExposeHeaders":
            return Optional.ofNullable(clazz.cast(accessControlExposeHeaders()));
        case "AccessControlMaxAgeSec":
            return Optional.ofNullable(clazz.cast(accessControlMaxAgeSec()));
        case "OriginOverride":
            return Optional.ofNullable(clazz.cast(originOverride()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResponseHeadersPolicyCorsConfig, T> g) {
        return obj -> g.apply((ResponseHeadersPolicyCorsConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseHeadersPolicyCorsConfig> {
        /**
         * <p>
         * A list of origins (domain names) that CloudFront can use as the value for the
         * <code>Access-Control-Allow-Origin</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Origin</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin"
         * >Access-Control-Allow-Origin</a> in the MDN Web Docs.
         * </p>
         * 
         * @param accessControlAllowOrigins
         *        A list of origins (domain names) that CloudFront can use as the value for the
         *        <code>Access-Control-Allow-Origin</code> HTTP response header.</p>
         *        <p>
         *        For more information about the <code>Access-Control-Allow-Origin</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin">Access-
         *        Control-Allow-Origin</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlAllowOrigins(ResponseHeadersPolicyAccessControlAllowOrigins accessControlAllowOrigins);

        /**
         * <p>
         * A list of origins (domain names) that CloudFront can use as the value for the
         * <code>Access-Control-Allow-Origin</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Origin</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Origin"
         * >Access-Control-Allow-Origin</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyAccessControlAllowOrigins.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyAccessControlAllowOrigins#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyAccessControlAllowOrigins.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #accessControlAllowOrigins(ResponseHeadersPolicyAccessControlAllowOrigins)}.
         * 
         * @param accessControlAllowOrigins
         *        a consumer that will call methods on {@link ResponseHeadersPolicyAccessControlAllowOrigins.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlAllowOrigins(ResponseHeadersPolicyAccessControlAllowOrigins)
         */
        default Builder accessControlAllowOrigins(
                Consumer<ResponseHeadersPolicyAccessControlAllowOrigins.Builder> accessControlAllowOrigins) {
            return accessControlAllowOrigins(ResponseHeadersPolicyAccessControlAllowOrigins.builder()
                    .applyMutation(accessControlAllowOrigins).build());
        }

        /**
         * <p>
         * A list of HTTP header names that CloudFront includes as values for the
         * <code>Access-Control-Allow-Headers</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Headers</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers"
         * >Access-Control-Allow-Headers</a> in the MDN Web Docs.
         * </p>
         * 
         * @param accessControlAllowHeaders
         *        A list of HTTP header names that CloudFront includes as values for the
         *        <code>Access-Control-Allow-Headers</code> HTTP response header.</p>
         *        <p>
         *        For more information about the <code>Access-Control-Allow-Headers</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers">Access-
         *        Control-Allow-Headers</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlAllowHeaders(ResponseHeadersPolicyAccessControlAllowHeaders accessControlAllowHeaders);

        /**
         * <p>
         * A list of HTTP header names that CloudFront includes as values for the
         * <code>Access-Control-Allow-Headers</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Headers</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Headers"
         * >Access-Control-Allow-Headers</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyAccessControlAllowHeaders.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyAccessControlAllowHeaders#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyAccessControlAllowHeaders.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #accessControlAllowHeaders(ResponseHeadersPolicyAccessControlAllowHeaders)}.
         * 
         * @param accessControlAllowHeaders
         *        a consumer that will call methods on {@link ResponseHeadersPolicyAccessControlAllowHeaders.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlAllowHeaders(ResponseHeadersPolicyAccessControlAllowHeaders)
         */
        default Builder accessControlAllowHeaders(
                Consumer<ResponseHeadersPolicyAccessControlAllowHeaders.Builder> accessControlAllowHeaders) {
            return accessControlAllowHeaders(ResponseHeadersPolicyAccessControlAllowHeaders.builder()
                    .applyMutation(accessControlAllowHeaders).build());
        }

        /**
         * <p>
         * A list of HTTP methods that CloudFront includes as values for the <code>Access-Control-Allow-Methods</code>
         * HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Methods</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods"
         * >Access-Control-Allow-Methods</a> in the MDN Web Docs.
         * </p>
         * 
         * @param accessControlAllowMethods
         *        A list of HTTP methods that CloudFront includes as values for the
         *        <code>Access-Control-Allow-Methods</code> HTTP response header.</p>
         *        <p>
         *        For more information about the <code>Access-Control-Allow-Methods</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods">Access-
         *        Control-Allow-Methods</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlAllowMethods(ResponseHeadersPolicyAccessControlAllowMethods accessControlAllowMethods);

        /**
         * <p>
         * A list of HTTP methods that CloudFront includes as values for the <code>Access-Control-Allow-Methods</code>
         * HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Methods</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Methods"
         * >Access-Control-Allow-Methods</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyAccessControlAllowMethods.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyAccessControlAllowMethods#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyAccessControlAllowMethods.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #accessControlAllowMethods(ResponseHeadersPolicyAccessControlAllowMethods)}.
         * 
         * @param accessControlAllowMethods
         *        a consumer that will call methods on {@link ResponseHeadersPolicyAccessControlAllowMethods.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlAllowMethods(ResponseHeadersPolicyAccessControlAllowMethods)
         */
        default Builder accessControlAllowMethods(
                Consumer<ResponseHeadersPolicyAccessControlAllowMethods.Builder> accessControlAllowMethods) {
            return accessControlAllowMethods(ResponseHeadersPolicyAccessControlAllowMethods.builder()
                    .applyMutation(accessControlAllowMethods).build());
        }

        /**
         * <p>
         * A Boolean that CloudFront uses as the value for the <code>Access-Control-Allow-Credentials</code> HTTP
         * response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Allow-Credentials</code> HTTP response header, see <a
         * href
         * ="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials">Access-Control
         * -Allow-Credentials</a> in the MDN Web Docs.
         * </p>
         * 
         * @param accessControlAllowCredentials
         *        A Boolean that CloudFront uses as the value for the <code>Access-Control-Allow-Credentials</code> HTTP
         *        response header.</p>
         *        <p>
         *        For more information about the <code>Access-Control-Allow-Credentials</code> HTTP response header, see
         *        <a href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Allow-Credentials">
         *        Access-Control-Allow-Credentials</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlAllowCredentials(Boolean accessControlAllowCredentials);

        /**
         * <p>
         * A list of HTTP headers that CloudFront includes as values for the <code>Access-Control-Expose-Headers</code>
         * HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Expose-Headers</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers"
         * >Access-Control-Expose-Headers</a> in the MDN Web Docs.
         * </p>
         * 
         * @param accessControlExposeHeaders
         *        A list of HTTP headers that CloudFront includes as values for the
         *        <code>Access-Control-Expose-Headers</code> HTTP response header.</p>
         *        <p>
         *        For more information about the <code>Access-Control-Expose-Headers</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers">Access-
         *        Control-Expose-Headers</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlExposeHeaders(ResponseHeadersPolicyAccessControlExposeHeaders accessControlExposeHeaders);

        /**
         * <p>
         * A list of HTTP headers that CloudFront includes as values for the <code>Access-Control-Expose-Headers</code>
         * HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Expose-Headers</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Expose-Headers"
         * >Access-Control-Expose-Headers</a> in the MDN Web Docs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link ResponseHeadersPolicyAccessControlExposeHeaders.Builder} avoiding the need to create one manually via
         * {@link ResponseHeadersPolicyAccessControlExposeHeaders#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyAccessControlExposeHeaders.Builder#build()}
         * is called immediately and its result is passed to
         * {@link #accessControlExposeHeaders(ResponseHeadersPolicyAccessControlExposeHeaders)}.
         * 
         * @param accessControlExposeHeaders
         *        a consumer that will call methods on {@link ResponseHeadersPolicyAccessControlExposeHeaders.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accessControlExposeHeaders(ResponseHeadersPolicyAccessControlExposeHeaders)
         */
        default Builder accessControlExposeHeaders(
                Consumer<ResponseHeadersPolicyAccessControlExposeHeaders.Builder> accessControlExposeHeaders) {
            return accessControlExposeHeaders(ResponseHeadersPolicyAccessControlExposeHeaders.builder()
                    .applyMutation(accessControlExposeHeaders).build());
        }

        /**
         * <p>
         * A number that CloudFront uses as the value for the <code>Access-Control-Max-Age</code> HTTP response header.
         * </p>
         * <p>
         * For more information about the <code>Access-Control-Max-Age</code> HTTP response header, see <a
         * href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age"
         * >Access-Control-Max-Age</a> in the MDN Web Docs.
         * </p>
         * 
         * @param accessControlMaxAgeSec
         *        A number that CloudFront uses as the value for the <code>Access-Control-Max-Age</code> HTTP response
         *        header.</p>
         *        <p>
         *        For more information about the <code>Access-Control-Max-Age</code> HTTP response header, see <a
         *        href="https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Access-Control-Max-Age"
         *        >Access-Control-Max-Age</a> in the MDN Web Docs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accessControlMaxAgeSec(Integer accessControlMaxAgeSec);

        /**
         * <p>
         * A Boolean that determines whether CloudFront overrides HTTP response headers received from the origin with
         * the ones specified in this response headers policy.
         * </p>
         * 
         * @param originOverride
         *        A Boolean that determines whether CloudFront overrides HTTP response headers received from the origin
         *        with the ones specified in this response headers policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder originOverride(Boolean originOverride);
    }

    static final class BuilderImpl implements Builder {
        private ResponseHeadersPolicyAccessControlAllowOrigins accessControlAllowOrigins;

        private ResponseHeadersPolicyAccessControlAllowHeaders accessControlAllowHeaders;

        private ResponseHeadersPolicyAccessControlAllowMethods accessControlAllowMethods;

        private Boolean accessControlAllowCredentials;

        private ResponseHeadersPolicyAccessControlExposeHeaders accessControlExposeHeaders;

        private Integer accessControlMaxAgeSec;

        private Boolean originOverride;

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseHeadersPolicyCorsConfig model) {
            accessControlAllowOrigins(model.accessControlAllowOrigins);
            accessControlAllowHeaders(model.accessControlAllowHeaders);
            accessControlAllowMethods(model.accessControlAllowMethods);
            accessControlAllowCredentials(model.accessControlAllowCredentials);
            accessControlExposeHeaders(model.accessControlExposeHeaders);
            accessControlMaxAgeSec(model.accessControlMaxAgeSec);
            originOverride(model.originOverride);
        }

        public final ResponseHeadersPolicyAccessControlAllowOrigins.Builder getAccessControlAllowOrigins() {
            return accessControlAllowOrigins != null ? accessControlAllowOrigins.toBuilder() : null;
        }

        public final void setAccessControlAllowOrigins(
                ResponseHeadersPolicyAccessControlAllowOrigins.BuilderImpl accessControlAllowOrigins) {
            this.accessControlAllowOrigins = accessControlAllowOrigins != null ? accessControlAllowOrigins.build() : null;
        }

        @Override
        public final Builder accessControlAllowOrigins(ResponseHeadersPolicyAccessControlAllowOrigins accessControlAllowOrigins) {
            this.accessControlAllowOrigins = accessControlAllowOrigins;
            return this;
        }

        public final ResponseHeadersPolicyAccessControlAllowHeaders.Builder getAccessControlAllowHeaders() {
            return accessControlAllowHeaders != null ? accessControlAllowHeaders.toBuilder() : null;
        }

        public final void setAccessControlAllowHeaders(
                ResponseHeadersPolicyAccessControlAllowHeaders.BuilderImpl accessControlAllowHeaders) {
            this.accessControlAllowHeaders = accessControlAllowHeaders != null ? accessControlAllowHeaders.build() : null;
        }

        @Override
        public final Builder accessControlAllowHeaders(ResponseHeadersPolicyAccessControlAllowHeaders accessControlAllowHeaders) {
            this.accessControlAllowHeaders = accessControlAllowHeaders;
            return this;
        }

        public final ResponseHeadersPolicyAccessControlAllowMethods.Builder getAccessControlAllowMethods() {
            return accessControlAllowMethods != null ? accessControlAllowMethods.toBuilder() : null;
        }

        public final void setAccessControlAllowMethods(
                ResponseHeadersPolicyAccessControlAllowMethods.BuilderImpl accessControlAllowMethods) {
            this.accessControlAllowMethods = accessControlAllowMethods != null ? accessControlAllowMethods.build() : null;
        }

        @Override
        public final Builder accessControlAllowMethods(ResponseHeadersPolicyAccessControlAllowMethods accessControlAllowMethods) {
            this.accessControlAllowMethods = accessControlAllowMethods;
            return this;
        }

        public final Boolean getAccessControlAllowCredentials() {
            return accessControlAllowCredentials;
        }

        public final void setAccessControlAllowCredentials(Boolean accessControlAllowCredentials) {
            this.accessControlAllowCredentials = accessControlAllowCredentials;
        }

        @Override
        public final Builder accessControlAllowCredentials(Boolean accessControlAllowCredentials) {
            this.accessControlAllowCredentials = accessControlAllowCredentials;
            return this;
        }

        public final ResponseHeadersPolicyAccessControlExposeHeaders.Builder getAccessControlExposeHeaders() {
            return accessControlExposeHeaders != null ? accessControlExposeHeaders.toBuilder() : null;
        }

        public final void setAccessControlExposeHeaders(
                ResponseHeadersPolicyAccessControlExposeHeaders.BuilderImpl accessControlExposeHeaders) {
            this.accessControlExposeHeaders = accessControlExposeHeaders != null ? accessControlExposeHeaders.build() : null;
        }

        @Override
        public final Builder accessControlExposeHeaders(ResponseHeadersPolicyAccessControlExposeHeaders accessControlExposeHeaders) {
            this.accessControlExposeHeaders = accessControlExposeHeaders;
            return this;
        }

        public final Integer getAccessControlMaxAgeSec() {
            return accessControlMaxAgeSec;
        }

        public final void setAccessControlMaxAgeSec(Integer accessControlMaxAgeSec) {
            this.accessControlMaxAgeSec = accessControlMaxAgeSec;
        }

        @Override
        public final Builder accessControlMaxAgeSec(Integer accessControlMaxAgeSec) {
            this.accessControlMaxAgeSec = accessControlMaxAgeSec;
            return this;
        }

        public final Boolean getOriginOverride() {
            return originOverride;
        }

        public final void setOriginOverride(Boolean originOverride) {
            this.originOverride = originOverride;
        }

        @Override
        public final Builder originOverride(Boolean originOverride) {
            this.originOverride = originOverride;
            return this;
        }

        @Override
        public ResponseHeadersPolicyCorsConfig build() {
            return new ResponseHeadersPolicyCorsConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
