/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chime.endpoints.internal;

import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletionException;
import java.util.function.Supplier;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.signer.Aws4Signer;
import software.amazon.awssdk.auth.signer.SignerLoader;
import software.amazon.awssdk.awscore.AwsExecutionAttribute;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.awscore.endpoints.authscheme.EndpointAuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4AuthScheme;
import software.amazon.awssdk.awscore.endpoints.authscheme.SigV4aAuthScheme;
import software.amazon.awssdk.awscore.util.SignerOverrideUtils;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.SelectedAuthScheme;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.Context;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.interceptor.ExecutionInterceptor;
import software.amazon.awssdk.core.interceptor.SdkExecutionAttribute;
import software.amazon.awssdk.core.interceptor.SdkInternalExecutionAttribute;
import software.amazon.awssdk.core.signer.Signer;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.http.SdkHttpRequest;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4HttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.AwsV4aHttpSigner;
import software.amazon.awssdk.http.auth.aws.signer.RegionSet;
import software.amazon.awssdk.http.auth.spi.scheme.AuthSchemeOption;
import software.amazon.awssdk.identity.spi.Identity;
import software.amazon.awssdk.services.chime.endpoints.ChimeEndpointParams;
import software.amazon.awssdk.services.chime.endpoints.ChimeEndpointProvider;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class ChimeResolveEndpointInterceptor implements ExecutionInterceptor {
    @Override
    public SdkRequest modifyRequest(Context.ModifyRequest context, ExecutionAttributes executionAttributes) {
        SdkRequest result = context.request();
        if (AwsEndpointProviderUtils.endpointIsDiscovered(executionAttributes)) {
            return result;
        }
        ChimeEndpointProvider provider = (ChimeEndpointProvider) executionAttributes
                .getAttribute(SdkInternalExecutionAttribute.ENDPOINT_PROVIDER);
        try {
            Endpoint endpoint = provider.resolveEndpoint(ruleParams(result, executionAttributes)).join();
            if (!AwsEndpointProviderUtils.disableHostPrefixInjection(executionAttributes)) {
                Optional<String> hostPrefix = hostPrefix(executionAttributes.getAttribute(SdkExecutionAttribute.OPERATION_NAME),
                        result);
                if (hostPrefix.isPresent()) {
                    endpoint = AwsEndpointProviderUtils.addHostPrefix(endpoint, hostPrefix.get());
                }
            }
            List<EndpointAuthScheme> endpointAuthSchemes = endpoint.attribute(AwsEndpointAttribute.AUTH_SCHEMES);
            SelectedAuthScheme<?> selectedAuthScheme = executionAttributes
                    .getAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME);
            if (endpointAuthSchemes != null && selectedAuthScheme != null) {
                selectedAuthScheme = authSchemeWithEndpointSignerProperties(endpointAuthSchemes, selectedAuthScheme);
                executionAttributes.putAttribute(SdkInternalExecutionAttribute.SELECTED_AUTH_SCHEME, selectedAuthScheme);
            }
            if (endpointAuthSchemes != null) {
                EndpointAuthScheme chosenAuthScheme = AuthSchemeUtils.chooseAuthScheme(endpointAuthSchemes);
                Supplier<Signer> signerProvider = signerProvider(chosenAuthScheme);
                result = SignerOverrideUtils.overrideSignerIfNotOverridden(result, executionAttributes, signerProvider);
            }
            executionAttributes.putAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT, endpoint);
            return result;
        } catch (CompletionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof SdkClientException) {
                throw (SdkClientException) cause;
            } else {
                throw SdkClientException.create("Endpoint resolution failed", cause);
            }
        }
    }

    @Override
    public SdkHttpRequest modifyHttpRequest(Context.ModifyHttpRequest context, ExecutionAttributes executionAttributes) {
        Endpoint resolvedEndpoint = executionAttributes.getAttribute(SdkInternalExecutionAttribute.RESOLVED_ENDPOINT);
        if (resolvedEndpoint.headers().isEmpty()) {
            return context.httpRequest();
        }
        SdkHttpRequest.Builder httpRequestBuilder = context.httpRequest().toBuilder();
        resolvedEndpoint.headers().forEach((name, values) -> {
            values.forEach(v -> httpRequestBuilder.appendHeader(name, v));
        });
        return httpRequestBuilder.build();
    }

    public static ChimeEndpointParams ruleParams(SdkRequest request, ExecutionAttributes executionAttributes) {
        ChimeEndpointParams.Builder builder = ChimeEndpointParams.builder();
        builder.region(AwsEndpointProviderUtils.regionBuiltIn(executionAttributes));
        builder.useDualStack(AwsEndpointProviderUtils.dualStackEnabledBuiltIn(executionAttributes));
        builder.useFips(AwsEndpointProviderUtils.fipsEnabledBuiltIn(executionAttributes));
        builder.endpoint(AwsEndpointProviderUtils.endpointBuiltIn(executionAttributes));
        setContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME), request);
        setStaticContextParams(builder, executionAttributes.getAttribute(AwsExecutionAttribute.OPERATION_NAME));
        return builder.build();
    }

    private static void setContextParams(ChimeEndpointParams.Builder params, String operationName, SdkRequest request) {
    }

    private static void setStaticContextParams(ChimeEndpointParams.Builder params, String operationName) {
    }

    private <T extends Identity> SelectedAuthScheme<T> authSchemeWithEndpointSignerProperties(
            List<EndpointAuthScheme> endpointAuthSchemes, SelectedAuthScheme<T> selectedAuthScheme) {
        for (EndpointAuthScheme endpointAuthScheme : endpointAuthSchemes) {
            AuthSchemeOption.Builder option = selectedAuthScheme.authSchemeOption().toBuilder();
            if (endpointAuthScheme instanceof SigV4AuthScheme) {
                SigV4AuthScheme v4AuthScheme = (SigV4AuthScheme) endpointAuthScheme;
                if (v4AuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4HttpSigner.DOUBLE_URL_ENCODE, !v4AuthScheme.disableDoubleEncoding());
                }
                if (v4AuthScheme.signingRegion() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.REGION_NAME, v4AuthScheme.signingRegion());
                }
                if (v4AuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4HttpSigner.SERVICE_SIGNING_NAME, v4AuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            if (endpointAuthScheme instanceof SigV4aAuthScheme) {
                SigV4aAuthScheme v4aAuthScheme = (SigV4aAuthScheme) endpointAuthScheme;
                if (v4aAuthScheme.isDisableDoubleEncodingSet()) {
                    option.putSignerProperty(AwsV4aHttpSigner.DOUBLE_URL_ENCODE, !v4aAuthScheme.disableDoubleEncoding());
                }
                if (v4aAuthScheme.signingRegionSet() != null) {
                    RegionSet regionSet = RegionSet.create(v4aAuthScheme.signingRegionSet());
                    option.putSignerProperty(AwsV4aHttpSigner.REGION_SET, regionSet);
                }
                if (v4aAuthScheme.signingName() != null) {
                    option.putSignerProperty(AwsV4aHttpSigner.SERVICE_SIGNING_NAME, v4aAuthScheme.signingName());
                }
                return new SelectedAuthScheme<>(selectedAuthScheme.identity(), selectedAuthScheme.signer(), option.build());
            }
            throw new IllegalArgumentException("Endpoint auth scheme '" + endpointAuthScheme.name()
                    + "' cannot be mapped to the SDK auth scheme. Was it declared in the service's model?");
        }
        return selectedAuthScheme;
    }

    private static Optional<String> hostPrefix(String operationName, SdkRequest request) {
        switch (operationName) {
        case "BatchCreateChannelMembership": {
            return Optional.of("messaging-");
        }
        case "CreateAppInstance": {
            return Optional.of("identity-");
        }
        case "CreateAppInstanceAdmin": {
            return Optional.of("identity-");
        }
        case "CreateAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "CreateChannel": {
            return Optional.of("messaging-");
        }
        case "CreateChannelBan": {
            return Optional.of("messaging-");
        }
        case "CreateChannelMembership": {
            return Optional.of("messaging-");
        }
        case "CreateChannelModerator": {
            return Optional.of("messaging-");
        }
        case "DeleteAppInstance": {
            return Optional.of("identity-");
        }
        case "DeleteAppInstanceAdmin": {
            return Optional.of("identity-");
        }
        case "DeleteAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "DeleteChannel": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelBan": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelMembership": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelMessage": {
            return Optional.of("messaging-");
        }
        case "DeleteChannelModerator": {
            return Optional.of("messaging-");
        }
        case "DescribeAppInstance": {
            return Optional.of("identity-");
        }
        case "DescribeAppInstanceAdmin": {
            return Optional.of("identity-");
        }
        case "DescribeAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "DescribeChannel": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelBan": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelMembership": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelMembershipForAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelModeratedByAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "DescribeChannelModerator": {
            return Optional.of("messaging-");
        }
        case "GetAppInstanceRetentionSettings": {
            return Optional.of("identity-");
        }
        case "GetChannelMessage": {
            return Optional.of("messaging-");
        }
        case "GetMessagingSessionEndpoint": {
            return Optional.of("messaging-");
        }
        case "ListAppInstanceAdmins": {
            return Optional.of("identity-");
        }
        case "ListAppInstanceUsers": {
            return Optional.of("identity-");
        }
        case "ListAppInstances": {
            return Optional.of("identity-");
        }
        case "ListChannelBans": {
            return Optional.of("messaging-");
        }
        case "ListChannelMemberships": {
            return Optional.of("messaging-");
        }
        case "ListChannelMembershipsForAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "ListChannelMessages": {
            return Optional.of("messaging-");
        }
        case "ListChannelModerators": {
            return Optional.of("messaging-");
        }
        case "ListChannels": {
            return Optional.of("messaging-");
        }
        case "ListChannelsModeratedByAppInstanceUser": {
            return Optional.of("messaging-");
        }
        case "PutAppInstanceRetentionSettings": {
            return Optional.of("identity-");
        }
        case "RedactChannelMessage": {
            return Optional.of("messaging-");
        }
        case "SendChannelMessage": {
            return Optional.of("messaging-");
        }
        case "UpdateAppInstance": {
            return Optional.of("identity-");
        }
        case "UpdateAppInstanceUser": {
            return Optional.of("identity-");
        }
        case "UpdateChannel": {
            return Optional.of("messaging-");
        }
        case "UpdateChannelMessage": {
            return Optional.of("messaging-");
        }
        case "UpdateChannelReadMarker": {
            return Optional.of("messaging-");
        }
        default:
            return Optional.empty();
        }
    }

    private Supplier<Signer> signerProvider(EndpointAuthScheme authScheme) {
        switch (authScheme.name()) {
        case "sigv4":
            return Aws4Signer::create;
        case "sigv4a":
            return SignerLoader::getSigV4aSigner;
        default:
            break;
        }
        throw SdkClientException.create("Don't know how to create signer for auth scheme: " + authScheme.name());
    }
}
