/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.emr.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.emr.EmrClient;
import software.amazon.awssdk.services.emr.model.DescribeClusterRequest;
import software.amazon.awssdk.services.emr.model.DescribeClusterResponse;
import software.amazon.awssdk.services.emr.model.DescribeStepRequest;
import software.amazon.awssdk.services.emr.model.DescribeStepResponse;
import software.amazon.awssdk.services.emr.model.EmrRequest;
import software.amazon.awssdk.services.emr.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultEmrWaiter implements EmrWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final EmrClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeClusterResponse> clusterRunningWaiter;

    private final Waiter<DescribeStepResponse> stepCompleteWaiter;

    private final Waiter<DescribeClusterResponse> clusterTerminatedWaiter;

    private DefaultEmrWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = EmrClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.clusterRunningWaiter = Waiter.builder(DescribeClusterResponse.class).acceptors(clusterRunningWaiterAcceptors())
                .overrideConfiguration(clusterRunningWaiterConfig(builder.overrideConfiguration)).build();
        this.stepCompleteWaiter = Waiter.builder(DescribeStepResponse.class).acceptors(stepCompleteWaiterAcceptors())
                .overrideConfiguration(stepCompleteWaiterConfig(builder.overrideConfiguration)).build();
        this.clusterTerminatedWaiter = Waiter.builder(DescribeClusterResponse.class)
                .acceptors(clusterTerminatedWaiterAcceptors())
                .overrideConfiguration(clusterTerminatedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterRunning(DescribeClusterRequest describeClusterRequest) {
        return clusterRunningWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterRunning(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterRunningWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterRunningWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterTerminated(DescribeClusterRequest describeClusterRequest) {
        return clusterTerminatedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)));
    }

    @Override
    public WaiterResponse<DescribeClusterResponse> waitUntilClusterTerminated(DescribeClusterRequest describeClusterRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return clusterTerminatedWaiter.run(() -> client.describeCluster(applyWaitersUserAgent(describeClusterRequest)),
                clusterTerminatedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeStepResponse> waitUntilStepComplete(DescribeStepRequest describeStepRequest) {
        return stepCompleteWaiter.run(() -> client.describeStep(applyWaitersUserAgent(describeStepRequest)));
    }

    @Override
    public WaiterResponse<DescribeStepResponse> waitUntilStepComplete(DescribeStepRequest describeStepRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return stepCompleteWaiter.run(() -> client.describeStep(applyWaitersUserAgent(describeStepRequest)),
                stepCompleteWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "RUNNING");
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "WAITING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "TERMINATING");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "TERMINATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "TERMINATED_WITH_ERRORS");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeStepResponse>> stepCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeStepResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Step").field("Status").field("State").value(), "COMPLETED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Step").field("Status").field("State").value(), "FAILED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Step").field("Status").field("State").value(), "CANCELLED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeClusterResponse>> clusterTerminatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeClusterResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "TERMINATED");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Cluster").field("Status").field("State").value(), "TERMINATED_WITH_ERRORS");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration clusterRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration stepCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration clusterTerminatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static EmrWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends EmrRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements EmrWaiter.Builder {
        private EmrClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public EmrWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public EmrWaiter.Builder client(EmrClient client) {
            this.client = client;
            return this;
        }

        public EmrWaiter build() {
            return new DefaultEmrWaiter(this);
        }
    }
}
