/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticloadbalancing.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.elasticloadbalancing.ElasticLoadBalancingClient;
import software.amazon.awssdk.services.elasticloadbalancing.model.DescribeInstanceHealthRequest;
import software.amazon.awssdk.services.elasticloadbalancing.model.DescribeInstanceHealthResponse;
import software.amazon.awssdk.services.elasticloadbalancing.model.ElasticLoadBalancingRequest;
import software.amazon.awssdk.services.elasticloadbalancing.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElasticLoadBalancingWaiter implements ElasticLoadBalancingWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final ElasticLoadBalancingClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeInstanceHealthResponse> instanceDeregisteredWaiter;

    private final Waiter<DescribeInstanceHealthResponse> anyInstanceInServiceWaiter;

    private final Waiter<DescribeInstanceHealthResponse> instanceInServiceWaiter;

    private DefaultElasticLoadBalancingWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElasticLoadBalancingClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.instanceDeregisteredWaiter = Waiter.builder(DescribeInstanceHealthResponse.class)
                .acceptors(instanceDeregisteredWaiterAcceptors())
                .overrideConfiguration(instanceDeregisteredWaiterConfig(builder.overrideConfiguration)).build();
        this.anyInstanceInServiceWaiter = Waiter.builder(DescribeInstanceHealthResponse.class)
                .acceptors(anyInstanceInServiceWaiterAcceptors())
                .overrideConfiguration(anyInstanceInServiceWaiterConfig(builder.overrideConfiguration)).build();
        this.instanceInServiceWaiter = Waiter.builder(DescribeInstanceHealthResponse.class)
                .acceptors(instanceInServiceWaiterAcceptors())
                .overrideConfiguration(instanceInServiceWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeInstanceHealthResponse> waitUntilAnyInstanceInService(
            DescribeInstanceHealthRequest describeInstanceHealthRequest) {
        return anyInstanceInServiceWaiter.run(() -> client
                .describeInstanceHealth(applyWaitersUserAgent(describeInstanceHealthRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstanceHealthResponse> waitUntilAnyInstanceInService(
            DescribeInstanceHealthRequest describeInstanceHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return anyInstanceInServiceWaiter.run(
                () -> client.describeInstanceHealth(applyWaitersUserAgent(describeInstanceHealthRequest)),
                anyInstanceInServiceWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInstanceHealthResponse> waitUntilInstanceDeregistered(
            DescribeInstanceHealthRequest describeInstanceHealthRequest) {
        return instanceDeregisteredWaiter.run(() -> client
                .describeInstanceHealth(applyWaitersUserAgent(describeInstanceHealthRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstanceHealthResponse> waitUntilInstanceDeregistered(
            DescribeInstanceHealthRequest describeInstanceHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceDeregisteredWaiter.run(
                () -> client.describeInstanceHealth(applyWaitersUserAgent(describeInstanceHealthRequest)),
                instanceDeregisteredWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeInstanceHealthResponse> waitUntilInstanceInService(
            DescribeInstanceHealthRequest describeInstanceHealthRequest) {
        return instanceInServiceWaiter.run(() -> client
                .describeInstanceHealth(applyWaitersUserAgent(describeInstanceHealthRequest)));
    }

    @Override
    public WaiterResponse<DescribeInstanceHealthResponse> waitUntilInstanceInService(
            DescribeInstanceHealthRequest describeInstanceHealthRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceInServiceWaiter.run(
                () -> client.describeInstanceHealth(applyWaitersUserAgent(describeInstanceHealthRequest)),
                instanceInServiceWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeInstanceHealthResponse>> instanceDeregisteredWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstanceHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("InstanceStates").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "OutOfService"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidInstance")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstanceHealthResponse>> anyInstanceInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstanceHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("InstanceStates").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "InService"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstanceHealthResponse>> instanceInServiceWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstanceHealthResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("InstanceStates").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "InService"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidInstance")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration instanceDeregisteredWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration anyInstanceInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceInServiceWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElasticLoadBalancingWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElasticLoadBalancingRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElasticLoadBalancingWaiter.Builder {
        private ElasticLoadBalancingClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public ElasticLoadBalancingWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElasticLoadBalancingWaiter.Builder client(ElasticLoadBalancingClient client) {
            this.client = client;
            return this;
        }

        public ElasticLoadBalancingWaiter build() {
            return new DefaultElasticLoadBalancingWaiter(this);
        }
    }
}
