/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.costandusagereport;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.costandusagereport.model.CostAndUsageReportException;
import software.amazon.awssdk.services.costandusagereport.model.DeleteReportDefinitionRequest;
import software.amazon.awssdk.services.costandusagereport.model.DeleteReportDefinitionResponse;
import software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsRequest;
import software.amazon.awssdk.services.costandusagereport.model.DescribeReportDefinitionsResponse;
import software.amazon.awssdk.services.costandusagereport.model.DuplicateReportNameException;
import software.amazon.awssdk.services.costandusagereport.model.InternalErrorException;
import software.amazon.awssdk.services.costandusagereport.model.ModifyReportDefinitionRequest;
import software.amazon.awssdk.services.costandusagereport.model.ModifyReportDefinitionResponse;
import software.amazon.awssdk.services.costandusagereport.model.PutReportDefinitionRequest;
import software.amazon.awssdk.services.costandusagereport.model.PutReportDefinitionResponse;
import software.amazon.awssdk.services.costandusagereport.model.ReportLimitReachedException;
import software.amazon.awssdk.services.costandusagereport.model.ValidationException;
import software.amazon.awssdk.services.costandusagereport.transform.DeleteReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costandusagereport.transform.DescribeReportDefinitionsRequestMarshaller;
import software.amazon.awssdk.services.costandusagereport.transform.ModifyReportDefinitionRequestMarshaller;
import software.amazon.awssdk.services.costandusagereport.transform.PutReportDefinitionRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link CostAndUsageReportAsyncClient}.
 *
 * @see CostAndUsageReportAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCostAndUsageReportAsyncClient implements CostAndUsageReportAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultCostAndUsageReportAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final CostAndUsageReportServiceClientConfiguration serviceClientConfiguration;

    protected DefaultCostAndUsageReportAsyncClient(CostAndUsageReportServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Deletes the specified report.
     * </p>
     *
     * @param deleteReportDefinitionRequest
     *        Deletes the specified report.
     * @return A Java Future containing the result of the DeleteReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostAndUsageReportException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostAndUsageReportAsyncClient.DeleteReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/DeleteReportDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteReportDefinitionResponse> deleteReportDefinition(
            DeleteReportDefinitionRequest deleteReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost and Usage Report Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteReportDefinitionRequest, DeleteReportDefinitionResponse>()
                            .withOperationName("DeleteReportDefinition")
                            .withMarshaller(new DeleteReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteReportDefinitionRequest));
            CompletableFuture<DeleteReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the AWS Cost and Usage reports available to this account.
     * </p>
     *
     * @param describeReportDefinitionsRequest
     *        Requests a list of AWS Cost and Usage reports owned by the account.
     * @return A Java Future containing the result of the DescribeReportDefinitions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostAndUsageReportException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostAndUsageReportAsyncClient.DescribeReportDefinitions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/DescribeReportDefinitions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeReportDefinitionsResponse> describeReportDefinitions(
            DescribeReportDefinitionsRequest describeReportDefinitionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeReportDefinitionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost and Usage Report Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeReportDefinitions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeReportDefinitionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeReportDefinitionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeReportDefinitionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeReportDefinitionsRequest, DescribeReportDefinitionsResponse>()
                            .withOperationName("DescribeReportDefinitions")
                            .withMarshaller(new DescribeReportDefinitionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeReportDefinitionsRequest));
            CompletableFuture<DescribeReportDefinitionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Allows you to programatically update your report preferences.
     * </p>
     *
     * @param modifyReportDefinitionRequest
     * @return A Java Future containing the result of the ModifyReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostAndUsageReportException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostAndUsageReportAsyncClient.ModifyReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/ModifyReportDefinition" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ModifyReportDefinitionResponse> modifyReportDefinition(
            ModifyReportDefinitionRequest modifyReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost and Usage Report Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ModifyReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ModifyReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ModifyReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ModifyReportDefinitionRequest, ModifyReportDefinitionResponse>()
                            .withOperationName("ModifyReportDefinition")
                            .withMarshaller(new ModifyReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(modifyReportDefinitionRequest));
            CompletableFuture<ModifyReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new report using the description that you provide.
     * </p>
     *
     * @param putReportDefinitionRequest
     *        Creates a Cost and Usage Report.
     * @return A Java Future containing the result of the PutReportDefinition operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>DuplicateReportNameException A report with the specified name already exists in the account. Specify
     *         a different report name.</li>
     *         <li>ReportLimitReachedException This account already has five reports defined. To define a new report,
     *         you must delete an existing report.</li>
     *         <li>InternalErrorException An error on the server occurred during the processing of your request. Try
     *         again later.</li>
     *         <li>ValidationException The input fails to satisfy the constraints specified by an AWS service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>CostAndUsageReportException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample CostAndUsageReportAsyncClient.PutReportDefinition
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/cur-2017-01-06/PutReportDefinition" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutReportDefinitionResponse> putReportDefinition(
            PutReportDefinitionRequest putReportDefinitionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putReportDefinitionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Cost and Usage Report Service");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutReportDefinition");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutReportDefinitionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, PutReportDefinitionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutReportDefinitionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutReportDefinitionRequest, PutReportDefinitionResponse>()
                            .withOperationName("PutReportDefinition")
                            .withMarshaller(new PutReportDefinitionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putReportDefinitionRequest));
            CompletableFuture<PutReportDefinitionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final CostAndUsageReportServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CostAndUsageReportException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReportLimitReachedException")
                                .exceptionBuilderSupplier(ReportLimitReachedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalErrorException")
                                .exceptionBuilderSupplier(InternalErrorException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateReportNameException")
                                .exceptionBuilderSupplier(DuplicateReportNameException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
