/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An origin request policy configuration.
 * </p>
 * <p>
 * This configuration determines the values that CloudFront includes in requests that it sends to the origin. Each
 * request that CloudFront sends to the origin includes the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * The request body and the URL path (without the domain name) from the viewer request.
 * </p>
 * </li>
 * <li>
 * <p>
 * The headers that CloudFront automatically includes in every origin request, including <code>Host</code>,
 * <code>User-Agent</code>, and <code>X-Amz-Cf-Id</code>.
 * </p>
 * </li>
 * <li>
 * <p>
 * All HTTP headers, cookies, and URL query strings that are specified in the cache policy or the origin request policy.
 * These can include items from the viewer request and, in the case of headers, additional ones that are added by
 * CloudFront.
 * </p>
 * </li>
 * </ul>
 * <p>
 * CloudFront sends a request when it can't find an object in its cache that matches the request. If you want to send
 * values to the origin and also include them in the cache key, use <code>CachePolicy</code>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class OriginRequestPolicyConfig implements SdkPojo, Serializable,
        ToCopyableBuilder<OriginRequestPolicyConfig.Builder, OriginRequestPolicyConfig> {
    private static final SdkField<String> COMMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Comment")
            .getter(getter(OriginRequestPolicyConfig::comment))
            .setter(setter(Builder::comment))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Comment")
                    .unmarshallLocationName("Comment").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(OriginRequestPolicyConfig::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<OriginRequestPolicyHeadersConfig> HEADERS_CONFIG_FIELD = SdkField
            .<OriginRequestPolicyHeadersConfig> builder(MarshallingType.SDK_POJO)
            .memberName("HeadersConfig")
            .getter(getter(OriginRequestPolicyConfig::headersConfig))
            .setter(setter(Builder::headersConfig))
            .constructor(OriginRequestPolicyHeadersConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HeadersConfig")
                    .unmarshallLocationName("HeadersConfig").build()).build();

    private static final SdkField<OriginRequestPolicyCookiesConfig> COOKIES_CONFIG_FIELD = SdkField
            .<OriginRequestPolicyCookiesConfig> builder(MarshallingType.SDK_POJO)
            .memberName("CookiesConfig")
            .getter(getter(OriginRequestPolicyConfig::cookiesConfig))
            .setter(setter(Builder::cookiesConfig))
            .constructor(OriginRequestPolicyCookiesConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CookiesConfig")
                    .unmarshallLocationName("CookiesConfig").build()).build();

    private static final SdkField<OriginRequestPolicyQueryStringsConfig> QUERY_STRINGS_CONFIG_FIELD = SdkField
            .<OriginRequestPolicyQueryStringsConfig> builder(MarshallingType.SDK_POJO)
            .memberName("QueryStringsConfig")
            .getter(getter(OriginRequestPolicyConfig::queryStringsConfig))
            .setter(setter(Builder::queryStringsConfig))
            .constructor(OriginRequestPolicyQueryStringsConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("QueryStringsConfig")
                    .unmarshallLocationName("QueryStringsConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(COMMENT_FIELD, NAME_FIELD,
            HEADERS_CONFIG_FIELD, COOKIES_CONFIG_FIELD, QUERY_STRINGS_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String comment;

    private final String name;

    private final OriginRequestPolicyHeadersConfig headersConfig;

    private final OriginRequestPolicyCookiesConfig cookiesConfig;

    private final OriginRequestPolicyQueryStringsConfig queryStringsConfig;

    private OriginRequestPolicyConfig(BuilderImpl builder) {
        this.comment = builder.comment;
        this.name = builder.name;
        this.headersConfig = builder.headersConfig;
        this.cookiesConfig = builder.cookiesConfig;
        this.queryStringsConfig = builder.queryStringsConfig;
    }

    /**
     * <p>
     * A comment to describe the origin request policy. The comment cannot be longer than 128 characters.
     * </p>
     * 
     * @return A comment to describe the origin request policy. The comment cannot be longer than 128 characters.
     */
    public final String comment() {
        return comment;
    }

    /**
     * <p>
     * A unique name to identify the origin request policy.
     * </p>
     * 
     * @return A unique name to identify the origin request policy.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The HTTP headers to include in origin requests. These can include headers from viewer requests and additional
     * headers added by CloudFront.
     * </p>
     * 
     * @return The HTTP headers to include in origin requests. These can include headers from viewer requests and
     *         additional headers added by CloudFront.
     */
    public final OriginRequestPolicyHeadersConfig headersConfig() {
        return headersConfig;
    }

    /**
     * <p>
     * The cookies from viewer requests to include in origin requests.
     * </p>
     * 
     * @return The cookies from viewer requests to include in origin requests.
     */
    public final OriginRequestPolicyCookiesConfig cookiesConfig() {
        return cookiesConfig;
    }

    /**
     * <p>
     * The URL query strings from viewer requests to include in origin requests.
     * </p>
     * 
     * @return The URL query strings from viewer requests to include in origin requests.
     */
    public final OriginRequestPolicyQueryStringsConfig queryStringsConfig() {
        return queryStringsConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(comment());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(headersConfig());
        hashCode = 31 * hashCode + Objects.hashCode(cookiesConfig());
        hashCode = 31 * hashCode + Objects.hashCode(queryStringsConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof OriginRequestPolicyConfig)) {
            return false;
        }
        OriginRequestPolicyConfig other = (OriginRequestPolicyConfig) obj;
        return Objects.equals(comment(), other.comment()) && Objects.equals(name(), other.name())
                && Objects.equals(headersConfig(), other.headersConfig())
                && Objects.equals(cookiesConfig(), other.cookiesConfig())
                && Objects.equals(queryStringsConfig(), other.queryStringsConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("OriginRequestPolicyConfig").add("Comment", comment()).add("Name", name())
                .add("HeadersConfig", headersConfig()).add("CookiesConfig", cookiesConfig())
                .add("QueryStringsConfig", queryStringsConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Comment":
            return Optional.ofNullable(clazz.cast(comment()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "HeadersConfig":
            return Optional.ofNullable(clazz.cast(headersConfig()));
        case "CookiesConfig":
            return Optional.ofNullable(clazz.cast(cookiesConfig()));
        case "QueryStringsConfig":
            return Optional.ofNullable(clazz.cast(queryStringsConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<OriginRequestPolicyConfig, T> g) {
        return obj -> g.apply((OriginRequestPolicyConfig) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, OriginRequestPolicyConfig> {
        /**
         * <p>
         * A comment to describe the origin request policy. The comment cannot be longer than 128 characters.
         * </p>
         * 
         * @param comment
         *        A comment to describe the origin request policy. The comment cannot be longer than 128 characters.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder comment(String comment);

        /**
         * <p>
         * A unique name to identify the origin request policy.
         * </p>
         * 
         * @param name
         *        A unique name to identify the origin request policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The HTTP headers to include in origin requests. These can include headers from viewer requests and additional
         * headers added by CloudFront.
         * </p>
         * 
         * @param headersConfig
         *        The HTTP headers to include in origin requests. These can include headers from viewer requests and
         *        additional headers added by CloudFront.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder headersConfig(OriginRequestPolicyHeadersConfig headersConfig);

        /**
         * <p>
         * The HTTP headers to include in origin requests. These can include headers from viewer requests and additional
         * headers added by CloudFront.
         * </p>
         * This is a convenience method that creates an instance of the {@link OriginRequestPolicyHeadersConfig.Builder}
         * avoiding the need to create one manually via {@link OriginRequestPolicyHeadersConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OriginRequestPolicyHeadersConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #headersConfig(OriginRequestPolicyHeadersConfig)}.
         * 
         * @param headersConfig
         *        a consumer that will call methods on {@link OriginRequestPolicyHeadersConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #headersConfig(OriginRequestPolicyHeadersConfig)
         */
        default Builder headersConfig(Consumer<OriginRequestPolicyHeadersConfig.Builder> headersConfig) {
            return headersConfig(OriginRequestPolicyHeadersConfig.builder().applyMutation(headersConfig).build());
        }

        /**
         * <p>
         * The cookies from viewer requests to include in origin requests.
         * </p>
         * 
         * @param cookiesConfig
         *        The cookies from viewer requests to include in origin requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cookiesConfig(OriginRequestPolicyCookiesConfig cookiesConfig);

        /**
         * <p>
         * The cookies from viewer requests to include in origin requests.
         * </p>
         * This is a convenience method that creates an instance of the {@link OriginRequestPolicyCookiesConfig.Builder}
         * avoiding the need to create one manually via {@link OriginRequestPolicyCookiesConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OriginRequestPolicyCookiesConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #cookiesConfig(OriginRequestPolicyCookiesConfig)}.
         * 
         * @param cookiesConfig
         *        a consumer that will call methods on {@link OriginRequestPolicyCookiesConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #cookiesConfig(OriginRequestPolicyCookiesConfig)
         */
        default Builder cookiesConfig(Consumer<OriginRequestPolicyCookiesConfig.Builder> cookiesConfig) {
            return cookiesConfig(OriginRequestPolicyCookiesConfig.builder().applyMutation(cookiesConfig).build());
        }

        /**
         * <p>
         * The URL query strings from viewer requests to include in origin requests.
         * </p>
         * 
         * @param queryStringsConfig
         *        The URL query strings from viewer requests to include in origin requests.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder queryStringsConfig(OriginRequestPolicyQueryStringsConfig queryStringsConfig);

        /**
         * <p>
         * The URL query strings from viewer requests to include in origin requests.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link OriginRequestPolicyQueryStringsConfig.Builder} avoiding the need to create one manually via
         * {@link OriginRequestPolicyQueryStringsConfig#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OriginRequestPolicyQueryStringsConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #queryStringsConfig(OriginRequestPolicyQueryStringsConfig)}.
         * 
         * @param queryStringsConfig
         *        a consumer that will call methods on {@link OriginRequestPolicyQueryStringsConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #queryStringsConfig(OriginRequestPolicyQueryStringsConfig)
         */
        default Builder queryStringsConfig(Consumer<OriginRequestPolicyQueryStringsConfig.Builder> queryStringsConfig) {
            return queryStringsConfig(OriginRequestPolicyQueryStringsConfig.builder().applyMutation(queryStringsConfig).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String comment;

        private String name;

        private OriginRequestPolicyHeadersConfig headersConfig;

        private OriginRequestPolicyCookiesConfig cookiesConfig;

        private OriginRequestPolicyQueryStringsConfig queryStringsConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(OriginRequestPolicyConfig model) {
            comment(model.comment);
            name(model.name);
            headersConfig(model.headersConfig);
            cookiesConfig(model.cookiesConfig);
            queryStringsConfig(model.queryStringsConfig);
        }

        public final String getComment() {
            return comment;
        }

        public final void setComment(String comment) {
            this.comment = comment;
        }

        @Override
        public final Builder comment(String comment) {
            this.comment = comment;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final OriginRequestPolicyHeadersConfig.Builder getHeadersConfig() {
            return headersConfig != null ? headersConfig.toBuilder() : null;
        }

        public final void setHeadersConfig(OriginRequestPolicyHeadersConfig.BuilderImpl headersConfig) {
            this.headersConfig = headersConfig != null ? headersConfig.build() : null;
        }

        @Override
        public final Builder headersConfig(OriginRequestPolicyHeadersConfig headersConfig) {
            this.headersConfig = headersConfig;
            return this;
        }

        public final OriginRequestPolicyCookiesConfig.Builder getCookiesConfig() {
            return cookiesConfig != null ? cookiesConfig.toBuilder() : null;
        }

        public final void setCookiesConfig(OriginRequestPolicyCookiesConfig.BuilderImpl cookiesConfig) {
            this.cookiesConfig = cookiesConfig != null ? cookiesConfig.build() : null;
        }

        @Override
        public final Builder cookiesConfig(OriginRequestPolicyCookiesConfig cookiesConfig) {
            this.cookiesConfig = cookiesConfig;
            return this;
        }

        public final OriginRequestPolicyQueryStringsConfig.Builder getQueryStringsConfig() {
            return queryStringsConfig != null ? queryStringsConfig.toBuilder() : null;
        }

        public final void setQueryStringsConfig(OriginRequestPolicyQueryStringsConfig.BuilderImpl queryStringsConfig) {
            this.queryStringsConfig = queryStringsConfig != null ? queryStringsConfig.build() : null;
        }

        @Override
        public final Builder queryStringsConfig(OriginRequestPolicyQueryStringsConfig queryStringsConfig) {
            this.queryStringsConfig = queryStringsConfig;
            return this;
        }

        @Override
        public OriginRequestPolicyConfig build() {
            return new OriginRequestPolicyConfig(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
