/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.autoscaling.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <code>GetPredictiveScalingForecast</code> call returns the load forecast for a predictive scaling policy. This
 * structure includes the data points for that load forecast, along with the timestamps of those data points and the
 * metric specification.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LoadForecast implements SdkPojo, Serializable, ToCopyableBuilder<LoadForecast.Builder, LoadForecast> {
    private static final SdkField<List<Instant>> TIMESTAMPS_FIELD = SdkField
            .<List<Instant>> builder(MarshallingType.LIST)
            .memberName("Timestamps")
            .getter(getter(LoadForecast::timestamps))
            .setter(setter(Builder::timestamps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instant> builder(MarshallingType.INSTANT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> VALUES_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(LoadForecast::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<PredictiveScalingMetricSpecification> METRIC_SPECIFICATION_FIELD = SdkField
            .<PredictiveScalingMetricSpecification> builder(MarshallingType.SDK_POJO).memberName("MetricSpecification")
            .getter(getter(LoadForecast::metricSpecification)).setter(setter(Builder::metricSpecification))
            .constructor(PredictiveScalingMetricSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricSpecification").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(TIMESTAMPS_FIELD,
            VALUES_FIELD, METRIC_SPECIFICATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<Instant> timestamps;

    private final List<Double> values;

    private final PredictiveScalingMetricSpecification metricSpecification;

    private LoadForecast(BuilderImpl builder) {
        this.timestamps = builder.timestamps;
        this.values = builder.values;
        this.metricSpecification = builder.metricSpecification;
    }

    /**
     * For responses, this returns true if the service returned a value for the Timestamps property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasTimestamps() {
        return timestamps != null && !(timestamps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The timestamps for the data points, in UTC format.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTimestamps} method.
     * </p>
     * 
     * @return The timestamps for the data points, in UTC format.
     */
    public final List<Instant> timestamps() {
        return timestamps;
    }

    /**
     * For responses, this returns true if the service returned a value for the Values property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The values of the data points.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasValues} method.
     * </p>
     * 
     * @return The values of the data points.
     */
    public final List<Double> values() {
        return values;
    }

    /**
     * <p>
     * The metric specification for the load forecast.
     * </p>
     * 
     * @return The metric specification for the load forecast.
     */
    public final PredictiveScalingMetricSpecification metricSpecification() {
        return metricSpecification;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasTimestamps() ? timestamps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        hashCode = 31 * hashCode + Objects.hashCode(metricSpecification());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LoadForecast)) {
            return false;
        }
        LoadForecast other = (LoadForecast) obj;
        return hasTimestamps() == other.hasTimestamps() && Objects.equals(timestamps(), other.timestamps())
                && hasValues() == other.hasValues() && Objects.equals(values(), other.values())
                && Objects.equals(metricSpecification(), other.metricSpecification());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LoadForecast").add("Timestamps", hasTimestamps() ? timestamps() : null)
                .add("Values", hasValues() ? values() : null).add("MetricSpecification", metricSpecification()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Timestamps":
            return Optional.ofNullable(clazz.cast(timestamps()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        case "MetricSpecification":
            return Optional.ofNullable(clazz.cast(metricSpecification()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LoadForecast, T> g) {
        return obj -> g.apply((LoadForecast) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LoadForecast> {
        /**
         * <p>
         * The timestamps for the data points, in UTC format.
         * </p>
         * 
         * @param timestamps
         *        The timestamps for the data points, in UTC format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Collection<Instant> timestamps);

        /**
         * <p>
         * The timestamps for the data points, in UTC format.
         * </p>
         * 
         * @param timestamps
         *        The timestamps for the data points, in UTC format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Instant... timestamps);

        /**
         * <p>
         * The values of the data points.
         * </p>
         * 
         * @param values
         *        The values of the data points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<Double> values);

        /**
         * <p>
         * The values of the data points.
         * </p>
         * 
         * @param values
         *        The values of the data points.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Double... values);

        /**
         * <p>
         * The metric specification for the load forecast.
         * </p>
         * 
         * @param metricSpecification
         *        The metric specification for the load forecast.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricSpecification(PredictiveScalingMetricSpecification metricSpecification);

        /**
         * <p>
         * The metric specification for the load forecast.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link PredictiveScalingMetricSpecification.Builder} avoiding the need to create one manually via
         * {@link PredictiveScalingMetricSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PredictiveScalingMetricSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #metricSpecification(PredictiveScalingMetricSpecification)}.
         * 
         * @param metricSpecification
         *        a consumer that will call methods on {@link PredictiveScalingMetricSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricSpecification(PredictiveScalingMetricSpecification)
         */
        default Builder metricSpecification(Consumer<PredictiveScalingMetricSpecification.Builder> metricSpecification) {
            return metricSpecification(PredictiveScalingMetricSpecification.builder().applyMutation(metricSpecification).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private List<Instant> timestamps = DefaultSdkAutoConstructList.getInstance();

        private List<Double> values = DefaultSdkAutoConstructList.getInstance();

        private PredictiveScalingMetricSpecification metricSpecification;

        private BuilderImpl() {
        }

        private BuilderImpl(LoadForecast model) {
            timestamps(model.timestamps);
            values(model.values);
            metricSpecification(model.metricSpecification);
        }

        public final Collection<Instant> getTimestamps() {
            if (timestamps instanceof SdkAutoConstructList) {
                return null;
            }
            return timestamps;
        }

        public final void setTimestamps(Collection<Instant> timestamps) {
            this.timestamps = PredictiveScalingForecastTimestampsCopier.copy(timestamps);
        }

        @Override
        public final Builder timestamps(Collection<Instant> timestamps) {
            this.timestamps = PredictiveScalingForecastTimestampsCopier.copy(timestamps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder timestamps(Instant... timestamps) {
            timestamps(Arrays.asList(timestamps));
            return this;
        }

        public final Collection<Double> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        public final void setValues(Collection<Double> values) {
            this.values = PredictiveScalingForecastValuesCopier.copy(values);
        }

        @Override
        public final Builder values(Collection<Double> values) {
            this.values = PredictiveScalingForecastValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(Double... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final PredictiveScalingMetricSpecification.Builder getMetricSpecification() {
            return metricSpecification != null ? metricSpecification.toBuilder() : null;
        }

        public final void setMetricSpecification(PredictiveScalingMetricSpecification.BuilderImpl metricSpecification) {
            this.metricSpecification = metricSpecification != null ? metricSpecification.build() : null;
        }

        @Override
        public final Builder metricSpecification(PredictiveScalingMetricSpecification metricSpecification) {
            this.metricSpecification = metricSpecification;
            return this;
        }

        @Override
        public LoadForecast build() {
            return new LoadForecast(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
