/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appstream.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an image.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Image implements SdkPojo, Serializable, ToCopyableBuilder<Image.Builder, Image> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Image::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(Image::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Arn").build()).build();

    private static final SdkField<String> BASE_IMAGE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BaseImageArn").getter(getter(Image::baseImageArn)).setter(setter(Builder::baseImageArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaseImageArn").build()).build();

    private static final SdkField<String> DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DisplayName").getter(getter(Image::displayName)).setter(setter(Builder::displayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DisplayName").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(Image::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> VISIBILITY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Visibility").getter(getter(Image::visibilityAsString)).setter(setter(Builder::visibility))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Visibility").build()).build();

    private static final SdkField<Boolean> IMAGE_BUILDER_SUPPORTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ImageBuilderSupported").getter(getter(Image::imageBuilderSupported))
            .setter(setter(Builder::imageBuilderSupported))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageBuilderSupported").build())
            .build();

    private static final SdkField<String> IMAGE_BUILDER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ImageBuilderName").getter(getter(Image::imageBuilderName)).setter(setter(Builder::imageBuilderName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageBuilderName").build()).build();

    private static final SdkField<String> PLATFORM_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Platform").getter(getter(Image::platformAsString)).setter(setter(Builder::platform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Platform").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(Image::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<ImageStateChangeReason> STATE_CHANGE_REASON_FIELD = SdkField
            .<ImageStateChangeReason> builder(MarshallingType.SDK_POJO).memberName("StateChangeReason")
            .getter(getter(Image::stateChangeReason)).setter(setter(Builder::stateChangeReason))
            .constructor(ImageStateChangeReason::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateChangeReason").build()).build();

    private static final SdkField<List<Application>> APPLICATIONS_FIELD = SdkField
            .<List<Application>> builder(MarshallingType.LIST)
            .memberName("Applications")
            .getter(getter(Image::applications))
            .setter(setter(Builder::applications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Applications").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Application> builder(MarshallingType.SDK_POJO)
                                            .constructor(Application::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(Image::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> PUBLIC_BASE_IMAGE_RELEASED_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("PublicBaseImageReleasedDate")
            .getter(getter(Image::publicBaseImageReleasedDate))
            .setter(setter(Builder::publicBaseImageReleasedDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicBaseImageReleasedDate")
                    .build()).build();

    private static final SdkField<String> APPSTREAM_AGENT_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AppstreamAgentVersion").getter(getter(Image::appstreamAgentVersion))
            .setter(setter(Builder::appstreamAgentVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AppstreamAgentVersion").build())
            .build();

    private static final SdkField<ImagePermissions> IMAGE_PERMISSIONS_FIELD = SdkField
            .<ImagePermissions> builder(MarshallingType.SDK_POJO).memberName("ImagePermissions")
            .getter(getter(Image::imagePermissions)).setter(setter(Builder::imagePermissions))
            .constructor(ImagePermissions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImagePermissions").build()).build();

    private static final SdkField<List<ResourceError>> IMAGE_ERRORS_FIELD = SdkField
            .<List<ResourceError>> builder(MarshallingType.LIST)
            .memberName("ImageErrors")
            .getter(getter(Image::imageErrors))
            .setter(setter(Builder::imageErrors))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageErrors").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ResourceError> builder(MarshallingType.SDK_POJO)
                                            .constructor(ResourceError::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, ARN_FIELD,
            BASE_IMAGE_ARN_FIELD, DISPLAY_NAME_FIELD, STATE_FIELD, VISIBILITY_FIELD, IMAGE_BUILDER_SUPPORTED_FIELD,
            IMAGE_BUILDER_NAME_FIELD, PLATFORM_FIELD, DESCRIPTION_FIELD, STATE_CHANGE_REASON_FIELD, APPLICATIONS_FIELD,
            CREATED_TIME_FIELD, PUBLIC_BASE_IMAGE_RELEASED_DATE_FIELD, APPSTREAM_AGENT_VERSION_FIELD, IMAGE_PERMISSIONS_FIELD,
            IMAGE_ERRORS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String arn;

    private final String baseImageArn;

    private final String displayName;

    private final String state;

    private final String visibility;

    private final Boolean imageBuilderSupported;

    private final String imageBuilderName;

    private final String platform;

    private final String description;

    private final ImageStateChangeReason stateChangeReason;

    private final List<Application> applications;

    private final Instant createdTime;

    private final Instant publicBaseImageReleasedDate;

    private final String appstreamAgentVersion;

    private final ImagePermissions imagePermissions;

    private final List<ResourceError> imageErrors;

    private Image(BuilderImpl builder) {
        this.name = builder.name;
        this.arn = builder.arn;
        this.baseImageArn = builder.baseImageArn;
        this.displayName = builder.displayName;
        this.state = builder.state;
        this.visibility = builder.visibility;
        this.imageBuilderSupported = builder.imageBuilderSupported;
        this.imageBuilderName = builder.imageBuilderName;
        this.platform = builder.platform;
        this.description = builder.description;
        this.stateChangeReason = builder.stateChangeReason;
        this.applications = builder.applications;
        this.createdTime = builder.createdTime;
        this.publicBaseImageReleasedDate = builder.publicBaseImageReleasedDate;
        this.appstreamAgentVersion = builder.appstreamAgentVersion;
        this.imagePermissions = builder.imagePermissions;
        this.imageErrors = builder.imageErrors;
    }

    /**
     * <p>
     * The name of the image.
     * </p>
     * 
     * @return The name of the image.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ARN of the image.
     * </p>
     * 
     * @return The ARN of the image.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The ARN of the image from which this image was created.
     * </p>
     * 
     * @return The ARN of the image from which this image was created.
     */
    public final String baseImageArn() {
        return baseImageArn;
    }

    /**
     * <p>
     * The image name to display.
     * </p>
     * 
     * @return The image name to display.
     */
    public final String displayName() {
        return displayName;
    }

    /**
     * <p>
     * The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
     * <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ImageState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
     *         <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
     * @see ImageState
     */
    public final ImageState state() {
        return ImageState.fromValue(state);
    }

    /**
     * <p>
     * The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
     * <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ImageState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
     *         <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
     * @see ImageState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * Indicates whether the image is public or private.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #visibility} will
     * return {@link VisibilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #visibilityAsString}.
     * </p>
     * 
     * @return Indicates whether the image is public or private.
     * @see VisibilityType
     */
    public final VisibilityType visibility() {
        return VisibilityType.fromValue(visibility);
    }

    /**
     * <p>
     * Indicates whether the image is public or private.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #visibility} will
     * return {@link VisibilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #visibilityAsString}.
     * </p>
     * 
     * @return Indicates whether the image is public or private.
     * @see VisibilityType
     */
    public final String visibilityAsString() {
        return visibility;
    }

    /**
     * <p>
     * Indicates whether an image builder can be launched from this image.
     * </p>
     * 
     * @return Indicates whether an image builder can be launched from this image.
     */
    public final Boolean imageBuilderSupported() {
        return imageBuilderSupported;
    }

    /**
     * <p>
     * The name of the image builder that was used to create the private image. If the image is shared, this value is
     * null.
     * </p>
     * 
     * @return The name of the image builder that was used to create the private image. If the image is shared, this
     *         value is null.
     */
    public final String imageBuilderName() {
        return imageBuilderName;
    }

    /**
     * <p>
     * The operating system platform of the image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the image.
     * @see PlatformType
     */
    public final PlatformType platform() {
        return PlatformType.fromValue(platform);
    }

    /**
     * <p>
     * The operating system platform of the image.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformAsString}.
     * </p>
     * 
     * @return The operating system platform of the image.
     * @see PlatformType
     */
    public final String platformAsString() {
        return platform;
    }

    /**
     * <p>
     * The description to display.
     * </p>
     * 
     * @return The description to display.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The reason why the last state change occurred.
     * </p>
     * 
     * @return The reason why the last state change occurred.
     */
    public final ImageStateChangeReason stateChangeReason() {
        return stateChangeReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Applications property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasApplications() {
        return applications != null && !(applications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The applications associated with the image.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasApplications} method.
     * </p>
     * 
     * @return The applications associated with the image.
     */
    public final List<Application> applications() {
        return applications;
    }

    /**
     * <p>
     * The time the image was created.
     * </p>
     * 
     * @return The time the image was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The release date of the public base image. For private images, this date is the release date of the base image
     * from which the image was created.
     * </p>
     * 
     * @return The release date of the public base image. For private images, this date is the release date of the base
     *         image from which the image was created.
     */
    public final Instant publicBaseImageReleasedDate() {
        return publicBaseImageReleasedDate;
    }

    /**
     * <p>
     * The version of the AppStream 2.0 agent to use for instances that are launched from this image.
     * </p>
     * 
     * @return The version of the AppStream 2.0 agent to use for instances that are launched from this image.
     */
    public final String appstreamAgentVersion() {
        return appstreamAgentVersion;
    }

    /**
     * <p>
     * The permissions to provide to the destination AWS account for the specified image.
     * </p>
     * 
     * @return The permissions to provide to the destination AWS account for the specified image.
     */
    public final ImagePermissions imagePermissions() {
        return imagePermissions;
    }

    /**
     * For responses, this returns true if the service returned a value for the ImageErrors property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasImageErrors() {
        return imageErrors != null && !(imageErrors instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Describes the errors that are returned when a new image can't be created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasImageErrors} method.
     * </p>
     * 
     * @return Describes the errors that are returned when a new image can't be created.
     */
    public final List<ResourceError> imageErrors() {
        return imageErrors;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(baseImageArn());
        hashCode = 31 * hashCode + Objects.hashCode(displayName());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(visibilityAsString());
        hashCode = 31 * hashCode + Objects.hashCode(imageBuilderSupported());
        hashCode = 31 * hashCode + Objects.hashCode(imageBuilderName());
        hashCode = 31 * hashCode + Objects.hashCode(platformAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(stateChangeReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasApplications() ? applications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(publicBaseImageReleasedDate());
        hashCode = 31 * hashCode + Objects.hashCode(appstreamAgentVersion());
        hashCode = 31 * hashCode + Objects.hashCode(imagePermissions());
        hashCode = 31 * hashCode + Objects.hashCode(hasImageErrors() ? imageErrors() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Image)) {
            return false;
        }
        Image other = (Image) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(arn(), other.arn())
                && Objects.equals(baseImageArn(), other.baseImageArn()) && Objects.equals(displayName(), other.displayName())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(visibilityAsString(), other.visibilityAsString())
                && Objects.equals(imageBuilderSupported(), other.imageBuilderSupported())
                && Objects.equals(imageBuilderName(), other.imageBuilderName())
                && Objects.equals(platformAsString(), other.platformAsString())
                && Objects.equals(description(), other.description())
                && Objects.equals(stateChangeReason(), other.stateChangeReason()) && hasApplications() == other.hasApplications()
                && Objects.equals(applications(), other.applications()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(publicBaseImageReleasedDate(), other.publicBaseImageReleasedDate())
                && Objects.equals(appstreamAgentVersion(), other.appstreamAgentVersion())
                && Objects.equals(imagePermissions(), other.imagePermissions()) && hasImageErrors() == other.hasImageErrors()
                && Objects.equals(imageErrors(), other.imageErrors());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Image").add("Name", name()).add("Arn", arn()).add("BaseImageArn", baseImageArn())
                .add("DisplayName", displayName()).add("State", stateAsString()).add("Visibility", visibilityAsString())
                .add("ImageBuilderSupported", imageBuilderSupported()).add("ImageBuilderName", imageBuilderName())
                .add("Platform", platformAsString()).add("Description", description())
                .add("StateChangeReason", stateChangeReason()).add("Applications", hasApplications() ? applications() : null)
                .add("CreatedTime", createdTime()).add("PublicBaseImageReleasedDate", publicBaseImageReleasedDate())
                .add("AppstreamAgentVersion", appstreamAgentVersion()).add("ImagePermissions", imagePermissions())
                .add("ImageErrors", hasImageErrors() ? imageErrors() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "BaseImageArn":
            return Optional.ofNullable(clazz.cast(baseImageArn()));
        case "DisplayName":
            return Optional.ofNullable(clazz.cast(displayName()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Visibility":
            return Optional.ofNullable(clazz.cast(visibilityAsString()));
        case "ImageBuilderSupported":
            return Optional.ofNullable(clazz.cast(imageBuilderSupported()));
        case "ImageBuilderName":
            return Optional.ofNullable(clazz.cast(imageBuilderName()));
        case "Platform":
            return Optional.ofNullable(clazz.cast(platformAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "StateChangeReason":
            return Optional.ofNullable(clazz.cast(stateChangeReason()));
        case "Applications":
            return Optional.ofNullable(clazz.cast(applications()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "PublicBaseImageReleasedDate":
            return Optional.ofNullable(clazz.cast(publicBaseImageReleasedDate()));
        case "AppstreamAgentVersion":
            return Optional.ofNullable(clazz.cast(appstreamAgentVersion()));
        case "ImagePermissions":
            return Optional.ofNullable(clazz.cast(imagePermissions()));
        case "ImageErrors":
            return Optional.ofNullable(clazz.cast(imageErrors()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Image, T> g) {
        return obj -> g.apply((Image) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Image> {
        /**
         * <p>
         * The name of the image.
         * </p>
         * 
         * @param name
         *        The name of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ARN of the image.
         * </p>
         * 
         * @param arn
         *        The ARN of the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The ARN of the image from which this image was created.
         * </p>
         * 
         * @param baseImageArn
         *        The ARN of the image from which this image was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baseImageArn(String baseImageArn);

        /**
         * <p>
         * The image name to display.
         * </p>
         * 
         * @param displayName
         *        The image name to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder displayName(String displayName);

        /**
         * <p>
         * The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
         * <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
         * </p>
         * 
         * @param state
         *        The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
         *        <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
         * @see ImageState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageState
         */
        Builder state(String state);

        /**
         * <p>
         * The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
         * <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
         * </p>
         * 
         * @param state
         *        The image starts in the <code>PENDING</code> state. If image creation succeeds, the state is
         *        <code>AVAILABLE</code>. If image creation fails, the state is <code>FAILED</code>.
         * @see ImageState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ImageState
         */
        Builder state(ImageState state);

        /**
         * <p>
         * Indicates whether the image is public or private.
         * </p>
         * 
         * @param visibility
         *        Indicates whether the image is public or private.
         * @see VisibilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VisibilityType
         */
        Builder visibility(String visibility);

        /**
         * <p>
         * Indicates whether the image is public or private.
         * </p>
         * 
         * @param visibility
         *        Indicates whether the image is public or private.
         * @see VisibilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VisibilityType
         */
        Builder visibility(VisibilityType visibility);

        /**
         * <p>
         * Indicates whether an image builder can be launched from this image.
         * </p>
         * 
         * @param imageBuilderSupported
         *        Indicates whether an image builder can be launched from this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageBuilderSupported(Boolean imageBuilderSupported);

        /**
         * <p>
         * The name of the image builder that was used to create the private image. If the image is shared, this value
         * is null.
         * </p>
         * 
         * @param imageBuilderName
         *        The name of the image builder that was used to create the private image. If the image is shared, this
         *        value is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageBuilderName(String imageBuilderName);

        /**
         * <p>
         * The operating system platform of the image.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the image.
         * @see PlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformType
         */
        Builder platform(String platform);

        /**
         * <p>
         * The operating system platform of the image.
         * </p>
         * 
         * @param platform
         *        The operating system platform of the image.
         * @see PlatformType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformType
         */
        Builder platform(PlatformType platform);

        /**
         * <p>
         * The description to display.
         * </p>
         * 
         * @param description
         *        The description to display.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The reason why the last state change occurred.
         * </p>
         * 
         * @param stateChangeReason
         *        The reason why the last state change occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateChangeReason(ImageStateChangeReason stateChangeReason);

        /**
         * <p>
         * The reason why the last state change occurred.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImageStateChangeReason.Builder} avoiding
         * the need to create one manually via {@link ImageStateChangeReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImageStateChangeReason.Builder#build()} is called immediately and
         * its result is passed to {@link #stateChangeReason(ImageStateChangeReason)}.
         * 
         * @param stateChangeReason
         *        a consumer that will call methods on {@link ImageStateChangeReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #stateChangeReason(ImageStateChangeReason)
         */
        default Builder stateChangeReason(Consumer<ImageStateChangeReason.Builder> stateChangeReason) {
            return stateChangeReason(ImageStateChangeReason.builder().applyMutation(stateChangeReason).build());
        }

        /**
         * <p>
         * The applications associated with the image.
         * </p>
         * 
         * @param applications
         *        The applications associated with the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applications(Collection<Application> applications);

        /**
         * <p>
         * The applications associated with the image.
         * </p>
         * 
         * @param applications
         *        The applications associated with the image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applications(Application... applications);

        /**
         * <p>
         * The applications associated with the image.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.Application.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.appstream.model.Application#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.Application.Builder#build()} is called immediately and
         * its result is passed to {@link #applications(List<Application>)}.
         * 
         * @param applications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.Application.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #applications(java.util.Collection<Application>)
         */
        Builder applications(Consumer<Application.Builder>... applications);

        /**
         * <p>
         * The time the image was created.
         * </p>
         * 
         * @param createdTime
         *        The time the image was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The release date of the public base image. For private images, this date is the release date of the base
         * image from which the image was created.
         * </p>
         * 
         * @param publicBaseImageReleasedDate
         *        The release date of the public base image. For private images, this date is the release date of the
         *        base image from which the image was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publicBaseImageReleasedDate(Instant publicBaseImageReleasedDate);

        /**
         * <p>
         * The version of the AppStream 2.0 agent to use for instances that are launched from this image.
         * </p>
         * 
         * @param appstreamAgentVersion
         *        The version of the AppStream 2.0 agent to use for instances that are launched from this image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appstreamAgentVersion(String appstreamAgentVersion);

        /**
         * <p>
         * The permissions to provide to the destination AWS account for the specified image.
         * </p>
         * 
         * @param imagePermissions
         *        The permissions to provide to the destination AWS account for the specified image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imagePermissions(ImagePermissions imagePermissions);

        /**
         * <p>
         * The permissions to provide to the destination AWS account for the specified image.
         * </p>
         * This is a convenience method that creates an instance of the {@link ImagePermissions.Builder} avoiding the
         * need to create one manually via {@link ImagePermissions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ImagePermissions.Builder#build()} is called immediately and its
         * result is passed to {@link #imagePermissions(ImagePermissions)}.
         * 
         * @param imagePermissions
         *        a consumer that will call methods on {@link ImagePermissions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imagePermissions(ImagePermissions)
         */
        default Builder imagePermissions(Consumer<ImagePermissions.Builder> imagePermissions) {
            return imagePermissions(ImagePermissions.builder().applyMutation(imagePermissions).build());
        }

        /**
         * <p>
         * Describes the errors that are returned when a new image can't be created.
         * </p>
         * 
         * @param imageErrors
         *        Describes the errors that are returned when a new image can't be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageErrors(Collection<ResourceError> imageErrors);

        /**
         * <p>
         * Describes the errors that are returned when a new image can't be created.
         * </p>
         * 
         * @param imageErrors
         *        Describes the errors that are returned when a new image can't be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageErrors(ResourceError... imageErrors);

        /**
         * <p>
         * Describes the errors that are returned when a new image can't be created.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.appstream.model.ResourceError#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder#build()} is called immediately
         * and its result is passed to {@link #imageErrors(List<ResourceError>)}.
         * 
         * @param imageErrors
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.appstream.model.ResourceError.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #imageErrors(java.util.Collection<ResourceError>)
         */
        Builder imageErrors(Consumer<ResourceError.Builder>... imageErrors);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String arn;

        private String baseImageArn;

        private String displayName;

        private String state;

        private String visibility;

        private Boolean imageBuilderSupported;

        private String imageBuilderName;

        private String platform;

        private String description;

        private ImageStateChangeReason stateChangeReason;

        private List<Application> applications = DefaultSdkAutoConstructList.getInstance();

        private Instant createdTime;

        private Instant publicBaseImageReleasedDate;

        private String appstreamAgentVersion;

        private ImagePermissions imagePermissions;

        private List<ResourceError> imageErrors = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Image model) {
            name(model.name);
            arn(model.arn);
            baseImageArn(model.baseImageArn);
            displayName(model.displayName);
            state(model.state);
            visibility(model.visibility);
            imageBuilderSupported(model.imageBuilderSupported);
            imageBuilderName(model.imageBuilderName);
            platform(model.platform);
            description(model.description);
            stateChangeReason(model.stateChangeReason);
            applications(model.applications);
            createdTime(model.createdTime);
            publicBaseImageReleasedDate(model.publicBaseImageReleasedDate);
            appstreamAgentVersion(model.appstreamAgentVersion);
            imagePermissions(model.imagePermissions);
            imageErrors(model.imageErrors);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final String getBaseImageArn() {
            return baseImageArn;
        }

        public final void setBaseImageArn(String baseImageArn) {
            this.baseImageArn = baseImageArn;
        }

        @Override
        public final Builder baseImageArn(String baseImageArn) {
            this.baseImageArn = baseImageArn;
            return this;
        }

        public final String getDisplayName() {
            return displayName;
        }

        public final void setDisplayName(String displayName) {
            this.displayName = displayName;
        }

        @Override
        public final Builder displayName(String displayName) {
            this.displayName = displayName;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ImageState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getVisibility() {
            return visibility;
        }

        public final void setVisibility(String visibility) {
            this.visibility = visibility;
        }

        @Override
        public final Builder visibility(String visibility) {
            this.visibility = visibility;
            return this;
        }

        @Override
        public final Builder visibility(VisibilityType visibility) {
            this.visibility(visibility == null ? null : visibility.toString());
            return this;
        }

        public final Boolean getImageBuilderSupported() {
            return imageBuilderSupported;
        }

        public final void setImageBuilderSupported(Boolean imageBuilderSupported) {
            this.imageBuilderSupported = imageBuilderSupported;
        }

        @Override
        public final Builder imageBuilderSupported(Boolean imageBuilderSupported) {
            this.imageBuilderSupported = imageBuilderSupported;
            return this;
        }

        public final String getImageBuilderName() {
            return imageBuilderName;
        }

        public final void setImageBuilderName(String imageBuilderName) {
            this.imageBuilderName = imageBuilderName;
        }

        @Override
        public final Builder imageBuilderName(String imageBuilderName) {
            this.imageBuilderName = imageBuilderName;
            return this;
        }

        public final String getPlatform() {
            return platform;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(PlatformType platform) {
            this.platform(platform == null ? null : platform.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final ImageStateChangeReason.Builder getStateChangeReason() {
            return stateChangeReason != null ? stateChangeReason.toBuilder() : null;
        }

        public final void setStateChangeReason(ImageStateChangeReason.BuilderImpl stateChangeReason) {
            this.stateChangeReason = stateChangeReason != null ? stateChangeReason.build() : null;
        }

        @Override
        public final Builder stateChangeReason(ImageStateChangeReason stateChangeReason) {
            this.stateChangeReason = stateChangeReason;
            return this;
        }

        public final List<Application.Builder> getApplications() {
            List<Application.Builder> result = ApplicationsCopier.copyToBuilder(this.applications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setApplications(Collection<Application.BuilderImpl> applications) {
            this.applications = ApplicationsCopier.copyFromBuilder(applications);
        }

        @Override
        public final Builder applications(Collection<Application> applications) {
            this.applications = ApplicationsCopier.copy(applications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder applications(Application... applications) {
            applications(Arrays.asList(applications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder applications(Consumer<Application.Builder>... applications) {
            applications(Stream.of(applications).map(c -> Application.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getPublicBaseImageReleasedDate() {
            return publicBaseImageReleasedDate;
        }

        public final void setPublicBaseImageReleasedDate(Instant publicBaseImageReleasedDate) {
            this.publicBaseImageReleasedDate = publicBaseImageReleasedDate;
        }

        @Override
        public final Builder publicBaseImageReleasedDate(Instant publicBaseImageReleasedDate) {
            this.publicBaseImageReleasedDate = publicBaseImageReleasedDate;
            return this;
        }

        public final String getAppstreamAgentVersion() {
            return appstreamAgentVersion;
        }

        public final void setAppstreamAgentVersion(String appstreamAgentVersion) {
            this.appstreamAgentVersion = appstreamAgentVersion;
        }

        @Override
        public final Builder appstreamAgentVersion(String appstreamAgentVersion) {
            this.appstreamAgentVersion = appstreamAgentVersion;
            return this;
        }

        public final ImagePermissions.Builder getImagePermissions() {
            return imagePermissions != null ? imagePermissions.toBuilder() : null;
        }

        public final void setImagePermissions(ImagePermissions.BuilderImpl imagePermissions) {
            this.imagePermissions = imagePermissions != null ? imagePermissions.build() : null;
        }

        @Override
        public final Builder imagePermissions(ImagePermissions imagePermissions) {
            this.imagePermissions = imagePermissions;
            return this;
        }

        public final List<ResourceError.Builder> getImageErrors() {
            List<ResourceError.Builder> result = ResourceErrorsCopier.copyToBuilder(this.imageErrors);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setImageErrors(Collection<ResourceError.BuilderImpl> imageErrors) {
            this.imageErrors = ResourceErrorsCopier.copyFromBuilder(imageErrors);
        }

        @Override
        public final Builder imageErrors(Collection<ResourceError> imageErrors) {
            this.imageErrors = ResourceErrorsCopier.copy(imageErrors);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageErrors(ResourceError... imageErrors) {
            imageErrors(Arrays.asList(imageErrors));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder imageErrors(Consumer<ResourceError.Builder>... imageErrors) {
            imageErrors(Stream.of(imageErrors).map(c -> ResourceError.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Image build() {
            return new Image(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
