/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigatewayv2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Updates an Authorizer.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateAuthorizerRequest extends ApiGatewayV2Request implements
        ToCopyableBuilder<UpdateAuthorizerRequest.Builder, UpdateAuthorizerRequest> {
    private static final SdkField<String> API_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ApiId")
            .getter(getter(UpdateAuthorizerRequest::apiId)).setter(setter(Builder::apiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("apiId").build()).build();

    private static final SdkField<String> AUTHORIZER_CREDENTIALS_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthorizerCredentialsArn").getter(getter(UpdateAuthorizerRequest::authorizerCredentialsArn))
            .setter(setter(Builder::authorizerCredentialsArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerCredentialsArn").build())
            .build();

    private static final SdkField<String> AUTHORIZER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthorizerId").getter(getter(UpdateAuthorizerRequest::authorizerId))
            .setter(setter(Builder::authorizerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("authorizerId").build()).build();

    private static final SdkField<String> AUTHORIZER_PAYLOAD_FORMAT_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AuthorizerPayloadFormatVersion")
            .getter(getter(UpdateAuthorizerRequest::authorizerPayloadFormatVersion))
            .setter(setter(Builder::authorizerPayloadFormatVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerPayloadFormatVersion")
                    .build()).build();

    private static final SdkField<Integer> AUTHORIZER_RESULT_TTL_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AuthorizerResultTtlInSeconds")
            .getter(getter(UpdateAuthorizerRequest::authorizerResultTtlInSeconds))
            .setter(setter(Builder::authorizerResultTtlInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerResultTtlInSeconds")
                    .build()).build();

    private static final SdkField<String> AUTHORIZER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthorizerType").getter(getter(UpdateAuthorizerRequest::authorizerTypeAsString))
            .setter(setter(Builder::authorizerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerType").build()).build();

    private static final SdkField<String> AUTHORIZER_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AuthorizerUri").getter(getter(UpdateAuthorizerRequest::authorizerUri))
            .setter(setter(Builder::authorizerUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerUri").build()).build();

    private static final SdkField<Boolean> ENABLE_SIMPLE_RESPONSES_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableSimpleResponses").getter(getter(UpdateAuthorizerRequest::enableSimpleResponses))
            .setter(setter(Builder::enableSimpleResponses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("enableSimpleResponses").build())
            .build();

    private static final SdkField<List<String>> IDENTITY_SOURCE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("IdentitySource")
            .getter(getter(UpdateAuthorizerRequest::identitySource))
            .setter(setter(Builder::identitySource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identitySource").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IDENTITY_VALIDATION_EXPRESSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IdentityValidationExpression")
            .getter(getter(UpdateAuthorizerRequest::identityValidationExpression))
            .setter(setter(Builder::identityValidationExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identityValidationExpression")
                    .build()).build();

    private static final SdkField<JWTConfiguration> JWT_CONFIGURATION_FIELD = SdkField
            .<JWTConfiguration> builder(MarshallingType.SDK_POJO).memberName("JwtConfiguration")
            .getter(getter(UpdateAuthorizerRequest::jwtConfiguration)).setter(setter(Builder::jwtConfiguration))
            .constructor(JWTConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jwtConfiguration").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(UpdateAuthorizerRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(API_ID_FIELD,
            AUTHORIZER_CREDENTIALS_ARN_FIELD, AUTHORIZER_ID_FIELD, AUTHORIZER_PAYLOAD_FORMAT_VERSION_FIELD,
            AUTHORIZER_RESULT_TTL_IN_SECONDS_FIELD, AUTHORIZER_TYPE_FIELD, AUTHORIZER_URI_FIELD, ENABLE_SIMPLE_RESPONSES_FIELD,
            IDENTITY_SOURCE_FIELD, IDENTITY_VALIDATION_EXPRESSION_FIELD, JWT_CONFIGURATION_FIELD, NAME_FIELD));

    private final String apiId;

    private final String authorizerCredentialsArn;

    private final String authorizerId;

    private final String authorizerPayloadFormatVersion;

    private final Integer authorizerResultTtlInSeconds;

    private final String authorizerType;

    private final String authorizerUri;

    private final Boolean enableSimpleResponses;

    private final List<String> identitySource;

    private final String identityValidationExpression;

    private final JWTConfiguration jwtConfiguration;

    private final String name;

    private UpdateAuthorizerRequest(BuilderImpl builder) {
        super(builder);
        this.apiId = builder.apiId;
        this.authorizerCredentialsArn = builder.authorizerCredentialsArn;
        this.authorizerId = builder.authorizerId;
        this.authorizerPayloadFormatVersion = builder.authorizerPayloadFormatVersion;
        this.authorizerResultTtlInSeconds = builder.authorizerResultTtlInSeconds;
        this.authorizerType = builder.authorizerType;
        this.authorizerUri = builder.authorizerUri;
        this.enableSimpleResponses = builder.enableSimpleResponses;
        this.identitySource = builder.identitySource;
        this.identityValidationExpression = builder.identityValidationExpression;
        this.jwtConfiguration = builder.jwtConfiguration;
        this.name = builder.name;
    }

    /**
     * <p>
     * The API identifier.
     * </p>
     * 
     * @return The API identifier.
     */
    public final String apiId() {
        return apiId;
    }

    /**
     * <p>
     * Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM
     * role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions on
     * the Lambda function, don't specify this parameter.
     * </p>
     * 
     * @return Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an
     *         IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based
     *         permissions on the Lambda function, don't specify this parameter.
     */
    public final String authorizerCredentialsArn() {
        return authorizerCredentialsArn;
    }

    /**
     * <p>
     * The authorizer identifier.
     * </p>
     * 
     * @return The authorizer identifier.
     */
    public final String authorizerId() {
        return authorizerId;
    }

    /**
     * <p>
     * Specifies the format of the payload sent to an HTTP API Lambda authorizer. Required for HTTP API Lambda
     * authorizers. Supported values are 1.0 and 2.0. To learn more, see <a
     * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working with
     * AWS Lambda authorizers for HTTP APIs</a>.
     * </p>
     * 
     * @return Specifies the format of the payload sent to an HTTP API Lambda authorizer. Required for HTTP API Lambda
     *         authorizers. Supported values are 1.0 and 2.0. To learn more, see <a
     *         href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
     *         >Working with AWS Lambda authorizers for HTTP APIs</a>.
     */
    public final String authorizerPayloadFormatVersion() {
        return authorizerPayloadFormatVersion;
    }

    /**
     * <p>
     * The time to live (TTL) for cached authorizer results, in seconds. If it equals 0, authorization caching is
     * disabled. If it is greater than 0, API Gateway caches authorizer responses. The maximum value is 3600, or 1 hour.
     * Supported only for HTTP API Lambda authorizers.
     * </p>
     * 
     * @return The time to live (TTL) for cached authorizer results, in seconds. If it equals 0, authorization caching
     *         is disabled. If it is greater than 0, API Gateway caches authorizer responses. The maximum value is 3600,
     *         or 1 hour. Supported only for HTTP API Lambda authorizers.
     */
    public final Integer authorizerResultTtlInSeconds() {
        return authorizerResultTtlInSeconds;
    }

    /**
     * <p>
     * The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify JWT to use
     * JSON Web Tokens (supported only for HTTP APIs).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authorizerType}
     * will return {@link AuthorizerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #authorizerTypeAsString}.
     * </p>
     * 
     * @return The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify JWT
     *         to use JSON Web Tokens (supported only for HTTP APIs).
     * @see AuthorizerType
     */
    public final AuthorizerType authorizerType() {
        return AuthorizerType.fromValue(authorizerType);
    }

    /**
     * <p>
     * The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify JWT to use
     * JSON Web Tokens (supported only for HTTP APIs).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authorizerType}
     * will return {@link AuthorizerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #authorizerTypeAsString}.
     * </p>
     * 
     * @return The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify JWT
     *         to use JSON Web Tokens (supported only for HTTP APIs).
     * @see AuthorizerType
     */
    public final String authorizerTypeAsString() {
        return authorizerType;
    }

    /**
     * <p>
     * The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a well-formed Lambda
     * function URI, for example,
     * arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:
     * <replaceable>{account_id}</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations.
     * In general, the URI has this form:
     * arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>{service_api}</replaceable> ,
     * where <replaceable></replaceable>{region} is the same as the region hosting the Lambda function, path indicates
     * that the remaining substring in the URI should be treated as the path to the resource, including the initial /.
     * For Lambda functions, this is usually of the form /2015-03-31/functions/[FunctionARN]/invocations. Supported only
     * for REQUEST authorizers.
     * </p>
     * 
     * @return The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a well-formed
     *         Lambda function URI, for example,
     *         arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda
     *         :us-west-2:<replaceable>{account_id
     *         }</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations. In general, the
     *         URI has this form:
     *         arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>{service_api
     *         }</replaceable> , where <replaceable></replaceable>{region} is the same as the region hosting the Lambda
     *         function, path indicates that the remaining substring in the URI should be treated as the path to the
     *         resource, including the initial /. For Lambda functions, this is usually of the form
     *         /2015-03-31/functions/[FunctionARN]/invocations. Supported only for REQUEST authorizers.
     */
    public final String authorizerUri() {
        return authorizerUri;
    }

    /**
     * <p>
     * Specifies whether a Lambda authorizer returns a response in a simple format. By default, a Lambda authorizer must
     * return an IAM policy. If enabled, the Lambda authorizer can return a boolean value instead of an IAM policy.
     * Supported only for HTTP APIs. To learn more, see <a
     * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working with
     * AWS Lambda authorizers for HTTP APIs</a>
     * </p>
     * 
     * @return Specifies whether a Lambda authorizer returns a response in a simple format. By default, a Lambda
     *         authorizer must return an IAM policy. If enabled, the Lambda authorizer can return a boolean value
     *         instead of an IAM policy. Supported only for HTTP APIs. To learn more, see <a
     *         href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
     *         >Working with AWS Lambda authorizers for HTTP APIs</a>
     */
    public final Boolean enableSimpleResponses() {
        return enableSimpleResponses;
    }

    /**
     * For responses, this returns true if the service returned a value for the IdentitySource property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasIdentitySource() {
        return identitySource != null && !(identitySource instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identity source for which authorization is requested.
     * </p>
     * <p>
     * For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
     * specified request parameters. The identity source can be headers, query string parameters, stage variables, and
     * context parameters. For example, if an Auth header and a Name query string parameter are defined as identity
     * sources, this value is route.request.header.Auth, route.request.querystring.Name for WebSocket APIs. For HTTP
     * APIs, use selection expressions prefixed with $, for example, $request.header.Auth, $request.querystring.Name.
     * These parameters are used to perform runtime validation for Lambda-based authorizers by verifying all of the
     * identity-related request parameters are present in the request, not null, and non-empty. Only when this is true
     * does the authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401 Unauthorized response
     * without calling the Lambda function. For HTTP APIs, identity sources are also used as the cache key when caching
     * is enabled. To learn more, see <a
     * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working with
     * AWS Lambda authorizers for HTTP APIs</a>.
     * </p>
     * <p>
     * For JWT, a single entry that specifies where to extract the JSON Web Token (JWT) from inbound requests. Currently
     * only header-based and query parameter-based selections are supported, for example $request.header.Authorization.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIdentitySource} method.
     * </p>
     * 
     * @return The identity source for which authorization is requested.</p>
     *         <p>
     *         For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
     *         specified request parameters. The identity source can be headers, query string parameters, stage
     *         variables, and context parameters. For example, if an Auth header and a Name query string parameter are
     *         defined as identity sources, this value is route.request.header.Auth, route.request.querystring.Name for
     *         WebSocket APIs. For HTTP APIs, use selection expressions prefixed with $, for example,
     *         $request.header.Auth, $request.querystring.Name. These parameters are used to perform runtime validation
     *         for Lambda-based authorizers by verifying all of the identity-related request parameters are present in
     *         the request, not null, and non-empty. Only when this is true does the authorizer invoke the authorizer
     *         Lambda function. Otherwise, it returns a 401 Unauthorized response without calling the Lambda function.
     *         For HTTP APIs, identity sources are also used as the cache key when caching is enabled. To learn more,
     *         see <a
     *         href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
     *         >Working with AWS Lambda authorizers for HTTP APIs</a>.
     *         </p>
     *         <p>
     *         For JWT, a single entry that specifies where to extract the JSON Web Token (JWT) from inbound requests.
     *         Currently only header-based and query parameter-based selections are supported, for example
     *         $request.header.Authorization.
     */
    public final List<String> identitySource() {
        return identitySource;
    }

    /**
     * <p>
     * This parameter is not used.
     * </p>
     * 
     * @return This parameter is not used.
     */
    public final String identityValidationExpression() {
        return identityValidationExpression;
    }

    /**
     * <p>
     * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for HTTP
     * APIs.
     * </p>
     * 
     * @return Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only
     *         for HTTP APIs.
     */
    public final JWTConfiguration jwtConfiguration() {
        return jwtConfiguration;
    }

    /**
     * <p>
     * The name of the authorizer.
     * </p>
     * 
     * @return The name of the authorizer.
     */
    public final String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(apiId());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerCredentialsArn());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerId());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerPayloadFormatVersion());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerResultTtlInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerUri());
        hashCode = 31 * hashCode + Objects.hashCode(enableSimpleResponses());
        hashCode = 31 * hashCode + Objects.hashCode(hasIdentitySource() ? identitySource() : null);
        hashCode = 31 * hashCode + Objects.hashCode(identityValidationExpression());
        hashCode = 31 * hashCode + Objects.hashCode(jwtConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateAuthorizerRequest)) {
            return false;
        }
        UpdateAuthorizerRequest other = (UpdateAuthorizerRequest) obj;
        return Objects.equals(apiId(), other.apiId())
                && Objects.equals(authorizerCredentialsArn(), other.authorizerCredentialsArn())
                && Objects.equals(authorizerId(), other.authorizerId())
                && Objects.equals(authorizerPayloadFormatVersion(), other.authorizerPayloadFormatVersion())
                && Objects.equals(authorizerResultTtlInSeconds(), other.authorizerResultTtlInSeconds())
                && Objects.equals(authorizerTypeAsString(), other.authorizerTypeAsString())
                && Objects.equals(authorizerUri(), other.authorizerUri())
                && Objects.equals(enableSimpleResponses(), other.enableSimpleResponses())
                && hasIdentitySource() == other.hasIdentitySource() && Objects.equals(identitySource(), other.identitySource())
                && Objects.equals(identityValidationExpression(), other.identityValidationExpression())
                && Objects.equals(jwtConfiguration(), other.jwtConfiguration()) && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateAuthorizerRequest").add("ApiId", apiId())
                .add("AuthorizerCredentialsArn", authorizerCredentialsArn()).add("AuthorizerId", authorizerId())
                .add("AuthorizerPayloadFormatVersion", authorizerPayloadFormatVersion())
                .add("AuthorizerResultTtlInSeconds", authorizerResultTtlInSeconds())
                .add("AuthorizerType", authorizerTypeAsString()).add("AuthorizerUri", authorizerUri())
                .add("EnableSimpleResponses", enableSimpleResponses())
                .add("IdentitySource", hasIdentitySource() ? identitySource() : null)
                .add("IdentityValidationExpression", identityValidationExpression()).add("JwtConfiguration", jwtConfiguration())
                .add("Name", name()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApiId":
            return Optional.ofNullable(clazz.cast(apiId()));
        case "AuthorizerCredentialsArn":
            return Optional.ofNullable(clazz.cast(authorizerCredentialsArn()));
        case "AuthorizerId":
            return Optional.ofNullable(clazz.cast(authorizerId()));
        case "AuthorizerPayloadFormatVersion":
            return Optional.ofNullable(clazz.cast(authorizerPayloadFormatVersion()));
        case "AuthorizerResultTtlInSeconds":
            return Optional.ofNullable(clazz.cast(authorizerResultTtlInSeconds()));
        case "AuthorizerType":
            return Optional.ofNullable(clazz.cast(authorizerTypeAsString()));
        case "AuthorizerUri":
            return Optional.ofNullable(clazz.cast(authorizerUri()));
        case "EnableSimpleResponses":
            return Optional.ofNullable(clazz.cast(enableSimpleResponses()));
        case "IdentitySource":
            return Optional.ofNullable(clazz.cast(identitySource()));
        case "IdentityValidationExpression":
            return Optional.ofNullable(clazz.cast(identityValidationExpression()));
        case "JwtConfiguration":
            return Optional.ofNullable(clazz.cast(jwtConfiguration()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateAuthorizerRequest, T> g) {
        return obj -> g.apply((UpdateAuthorizerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayV2Request.Builder, SdkPojo, CopyableBuilder<Builder, UpdateAuthorizerRequest> {
        /**
         * <p>
         * The API identifier.
         * </p>
         * 
         * @param apiId
         *        The API identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiId(String apiId);

        /**
         * <p>
         * Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM
         * role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions
         * on the Lambda function, don't specify this parameter.
         * </p>
         * 
         * @param authorizerCredentialsArn
         *        Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify
         *        an IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use
         *        resource-based permissions on the Lambda function, don't specify this parameter.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerCredentialsArn(String authorizerCredentialsArn);

        /**
         * <p>
         * The authorizer identifier.
         * </p>
         * 
         * @param authorizerId
         *        The authorizer identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerId(String authorizerId);

        /**
         * <p>
         * Specifies the format of the payload sent to an HTTP API Lambda authorizer. Required for HTTP API Lambda
         * authorizers. Supported values are 1.0 and 2.0. To learn more, see <a
         * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working
         * with AWS Lambda authorizers for HTTP APIs</a>.
         * </p>
         * 
         * @param authorizerPayloadFormatVersion
         *        Specifies the format of the payload sent to an HTTP API Lambda authorizer. Required for HTTP API
         *        Lambda authorizers. Supported values are 1.0 and 2.0. To learn more, see <a
         *        href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
         *        >Working with AWS Lambda authorizers for HTTP APIs</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerPayloadFormatVersion(String authorizerPayloadFormatVersion);

        /**
         * <p>
         * The time to live (TTL) for cached authorizer results, in seconds. If it equals 0, authorization caching is
         * disabled. If it is greater than 0, API Gateway caches authorizer responses. The maximum value is 3600, or 1
         * hour. Supported only for HTTP API Lambda authorizers.
         * </p>
         * 
         * @param authorizerResultTtlInSeconds
         *        The time to live (TTL) for cached authorizer results, in seconds. If it equals 0, authorization
         *        caching is disabled. If it is greater than 0, API Gateway caches authorizer responses. The maximum
         *        value is 3600, or 1 hour. Supported only for HTTP API Lambda authorizers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds);

        /**
         * <p>
         * The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify JWT to
         * use JSON Web Tokens (supported only for HTTP APIs).
         * </p>
         * 
         * @param authorizerType
         *        The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify
         *        JWT to use JSON Web Tokens (supported only for HTTP APIs).
         * @see AuthorizerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerType
         */
        Builder authorizerType(String authorizerType);

        /**
         * <p>
         * The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify JWT to
         * use JSON Web Tokens (supported only for HTTP APIs).
         * </p>
         * 
         * @param authorizerType
         *        The authorizer type. Specify REQUEST for a Lambda function using incoming request parameters. Specify
         *        JWT to use JSON Web Tokens (supported only for HTTP APIs).
         * @see AuthorizerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerType
         */
        Builder authorizerType(AuthorizerType authorizerType);

        /**
         * <p>
         * The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a well-formed
         * Lambda function URI, for example,
         * arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda
         * :us-west-2:<replaceable>{account_id
         * }</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations. In general, the URI
         * has this form:
         * arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>{service_api}</replaceable> ,
         * where <replaceable></replaceable>{region} is the same as the region hosting the Lambda function, path
         * indicates that the remaining substring in the URI should be treated as the path to the resource, including
         * the initial /. For Lambda functions, this is usually of the form
         * /2015-03-31/functions/[FunctionARN]/invocations. Supported only for REQUEST authorizers.
         * </p>
         * 
         * @param authorizerUri
         *        The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a
         *        well-formed Lambda function URI, for example,
         *        arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions
         *        /arn:aws:lambda:us-west-2:<replaceable>{account_id
         *        }</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations. In general,
         *        the URI has this form:
         *        arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>
         *        {service_api}</replaceable> , where <replaceable></replaceable>{region} is the same as the region
         *        hosting the Lambda function, path indicates that the remaining substring in the URI should be treated
         *        as the path to the resource, including the initial /. For Lambda functions, this is usually of the
         *        form /2015-03-31/functions/[FunctionARN]/invocations. Supported only for REQUEST authorizers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerUri(String authorizerUri);

        /**
         * <p>
         * Specifies whether a Lambda authorizer returns a response in a simple format. By default, a Lambda authorizer
         * must return an IAM policy. If enabled, the Lambda authorizer can return a boolean value instead of an IAM
         * policy. Supported only for HTTP APIs. To learn more, see <a
         * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working
         * with AWS Lambda authorizers for HTTP APIs</a>
         * </p>
         * 
         * @param enableSimpleResponses
         *        Specifies whether a Lambda authorizer returns a response in a simple format. By default, a Lambda
         *        authorizer must return an IAM policy. If enabled, the Lambda authorizer can return a boolean value
         *        instead of an IAM policy. Supported only for HTTP APIs. To learn more, see <a
         *        href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
         *        >Working with AWS Lambda authorizers for HTTP APIs</a>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableSimpleResponses(Boolean enableSimpleResponses);

        /**
         * <p>
         * The identity source for which authorization is requested.
         * </p>
         * <p>
         * For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
         * specified request parameters. The identity source can be headers, query string parameters, stage variables,
         * and context parameters. For example, if an Auth header and a Name query string parameter are defined as
         * identity sources, this value is route.request.header.Auth, route.request.querystring.Name for WebSocket APIs.
         * For HTTP APIs, use selection expressions prefixed with $, for example, $request.header.Auth,
         * $request.querystring.Name. These parameters are used to perform runtime validation for Lambda-based
         * authorizers by verifying all of the identity-related request parameters are present in the request, not null,
         * and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda function. Otherwise,
         * it returns a 401 Unauthorized response without calling the Lambda function. For HTTP APIs, identity sources
         * are also used as the cache key when caching is enabled. To learn more, see <a
         * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working
         * with AWS Lambda authorizers for HTTP APIs</a>.
         * </p>
         * <p>
         * For JWT, a single entry that specifies where to extract the JSON Web Token (JWT) from inbound requests.
         * Currently only header-based and query parameter-based selections are supported, for example
         * $request.header.Authorization.
         * </p>
         * 
         * @param identitySource
         *        The identity source for which authorization is requested.</p>
         *        <p>
         *        For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of
         *        the specified request parameters. The identity source can be headers, query string parameters, stage
         *        variables, and context parameters. For example, if an Auth header and a Name query string parameter
         *        are defined as identity sources, this value is route.request.header.Auth,
         *        route.request.querystring.Name for WebSocket APIs. For HTTP APIs, use selection expressions prefixed
         *        with $, for example, $request.header.Auth, $request.querystring.Name. These parameters are used to
         *        perform runtime validation for Lambda-based authorizers by verifying all of the identity-related
         *        request parameters are present in the request, not null, and non-empty. Only when this is true does
         *        the authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401 Unauthorized
         *        response without calling the Lambda function. For HTTP APIs, identity sources are also used as the
         *        cache key when caching is enabled. To learn more, see <a
         *        href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
         *        >Working with AWS Lambda authorizers for HTTP APIs</a>.
         *        </p>
         *        <p>
         *        For JWT, a single entry that specifies where to extract the JSON Web Token (JWT) from inbound
         *        requests. Currently only header-based and query parameter-based selections are supported, for example
         *        $request.header.Authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identitySource(Collection<String> identitySource);

        /**
         * <p>
         * The identity source for which authorization is requested.
         * </p>
         * <p>
         * For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
         * specified request parameters. The identity source can be headers, query string parameters, stage variables,
         * and context parameters. For example, if an Auth header and a Name query string parameter are defined as
         * identity sources, this value is route.request.header.Auth, route.request.querystring.Name for WebSocket APIs.
         * For HTTP APIs, use selection expressions prefixed with $, for example, $request.header.Auth,
         * $request.querystring.Name. These parameters are used to perform runtime validation for Lambda-based
         * authorizers by verifying all of the identity-related request parameters are present in the request, not null,
         * and non-empty. Only when this is true does the authorizer invoke the authorizer Lambda function. Otherwise,
         * it returns a 401 Unauthorized response without calling the Lambda function. For HTTP APIs, identity sources
         * are also used as the cache key when caching is enabled. To learn more, see <a
         * href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html">Working
         * with AWS Lambda authorizers for HTTP APIs</a>.
         * </p>
         * <p>
         * For JWT, a single entry that specifies where to extract the JSON Web Token (JWT) from inbound requests.
         * Currently only header-based and query parameter-based selections are supported, for example
         * $request.header.Authorization.
         * </p>
         * 
         * @param identitySource
         *        The identity source for which authorization is requested.</p>
         *        <p>
         *        For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of
         *        the specified request parameters. The identity source can be headers, query string parameters, stage
         *        variables, and context parameters. For example, if an Auth header and a Name query string parameter
         *        are defined as identity sources, this value is route.request.header.Auth,
         *        route.request.querystring.Name for WebSocket APIs. For HTTP APIs, use selection expressions prefixed
         *        with $, for example, $request.header.Auth, $request.querystring.Name. These parameters are used to
         *        perform runtime validation for Lambda-based authorizers by verifying all of the identity-related
         *        request parameters are present in the request, not null, and non-empty. Only when this is true does
         *        the authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401 Unauthorized
         *        response without calling the Lambda function. For HTTP APIs, identity sources are also used as the
         *        cache key when caching is enabled. To learn more, see <a
         *        href="https://docs.aws.amazon.com/apigateway/latest/developerguide/http-api-lambda-authorizer.html"
         *        >Working with AWS Lambda authorizers for HTTP APIs</a>.
         *        </p>
         *        <p>
         *        For JWT, a single entry that specifies where to extract the JSON Web Token (JWT) from inbound
         *        requests. Currently only header-based and query parameter-based selections are supported, for example
         *        $request.header.Authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identitySource(String... identitySource);

        /**
         * <p>
         * This parameter is not used.
         * </p>
         * 
         * @param identityValidationExpression
         *        This parameter is not used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityValidationExpression(String identityValidationExpression);

        /**
         * <p>
         * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for
         * HTTP APIs.
         * </p>
         * 
         * @param jwtConfiguration
         *        Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only
         *        for HTTP APIs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jwtConfiguration(JWTConfiguration jwtConfiguration);

        /**
         * <p>
         * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for
         * HTTP APIs.
         * </p>
         * This is a convenience method that creates an instance of the {@link JWTConfiguration.Builder} avoiding the
         * need to create one manually via {@link JWTConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link JWTConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #jwtConfiguration(JWTConfiguration)}.
         * 
         * @param jwtConfiguration
         *        a consumer that will call methods on {@link JWTConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jwtConfiguration(JWTConfiguration)
         */
        default Builder jwtConfiguration(Consumer<JWTConfiguration.Builder> jwtConfiguration) {
            return jwtConfiguration(JWTConfiguration.builder().applyMutation(jwtConfiguration).build());
        }

        /**
         * <p>
         * The name of the authorizer.
         * </p>
         * 
         * @param name
         *        The name of the authorizer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayV2Request.BuilderImpl implements Builder {
        private String apiId;

        private String authorizerCredentialsArn;

        private String authorizerId;

        private String authorizerPayloadFormatVersion;

        private Integer authorizerResultTtlInSeconds;

        private String authorizerType;

        private String authorizerUri;

        private Boolean enableSimpleResponses;

        private List<String> identitySource = DefaultSdkAutoConstructList.getInstance();

        private String identityValidationExpression;

        private JWTConfiguration jwtConfiguration;

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateAuthorizerRequest model) {
            super(model);
            apiId(model.apiId);
            authorizerCredentialsArn(model.authorizerCredentialsArn);
            authorizerId(model.authorizerId);
            authorizerPayloadFormatVersion(model.authorizerPayloadFormatVersion);
            authorizerResultTtlInSeconds(model.authorizerResultTtlInSeconds);
            authorizerType(model.authorizerType);
            authorizerUri(model.authorizerUri);
            enableSimpleResponses(model.enableSimpleResponses);
            identitySource(model.identitySource);
            identityValidationExpression(model.identityValidationExpression);
            jwtConfiguration(model.jwtConfiguration);
            name(model.name);
        }

        public final String getApiId() {
            return apiId;
        }

        public final void setApiId(String apiId) {
            this.apiId = apiId;
        }

        @Override
        public final Builder apiId(String apiId) {
            this.apiId = apiId;
            return this;
        }

        public final String getAuthorizerCredentialsArn() {
            return authorizerCredentialsArn;
        }

        public final void setAuthorizerCredentialsArn(String authorizerCredentialsArn) {
            this.authorizerCredentialsArn = authorizerCredentialsArn;
        }

        @Override
        public final Builder authorizerCredentialsArn(String authorizerCredentialsArn) {
            this.authorizerCredentialsArn = authorizerCredentialsArn;
            return this;
        }

        public final String getAuthorizerId() {
            return authorizerId;
        }

        public final void setAuthorizerId(String authorizerId) {
            this.authorizerId = authorizerId;
        }

        @Override
        public final Builder authorizerId(String authorizerId) {
            this.authorizerId = authorizerId;
            return this;
        }

        public final String getAuthorizerPayloadFormatVersion() {
            return authorizerPayloadFormatVersion;
        }

        public final void setAuthorizerPayloadFormatVersion(String authorizerPayloadFormatVersion) {
            this.authorizerPayloadFormatVersion = authorizerPayloadFormatVersion;
        }

        @Override
        public final Builder authorizerPayloadFormatVersion(String authorizerPayloadFormatVersion) {
            this.authorizerPayloadFormatVersion = authorizerPayloadFormatVersion;
            return this;
        }

        public final Integer getAuthorizerResultTtlInSeconds() {
            return authorizerResultTtlInSeconds;
        }

        public final void setAuthorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds) {
            this.authorizerResultTtlInSeconds = authorizerResultTtlInSeconds;
        }

        @Override
        public final Builder authorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds) {
            this.authorizerResultTtlInSeconds = authorizerResultTtlInSeconds;
            return this;
        }

        public final String getAuthorizerType() {
            return authorizerType;
        }

        public final void setAuthorizerType(String authorizerType) {
            this.authorizerType = authorizerType;
        }

        @Override
        public final Builder authorizerType(String authorizerType) {
            this.authorizerType = authorizerType;
            return this;
        }

        @Override
        public final Builder authorizerType(AuthorizerType authorizerType) {
            this.authorizerType(authorizerType == null ? null : authorizerType.toString());
            return this;
        }

        public final String getAuthorizerUri() {
            return authorizerUri;
        }

        public final void setAuthorizerUri(String authorizerUri) {
            this.authorizerUri = authorizerUri;
        }

        @Override
        public final Builder authorizerUri(String authorizerUri) {
            this.authorizerUri = authorizerUri;
            return this;
        }

        public final Boolean getEnableSimpleResponses() {
            return enableSimpleResponses;
        }

        public final void setEnableSimpleResponses(Boolean enableSimpleResponses) {
            this.enableSimpleResponses = enableSimpleResponses;
        }

        @Override
        public final Builder enableSimpleResponses(Boolean enableSimpleResponses) {
            this.enableSimpleResponses = enableSimpleResponses;
            return this;
        }

        public final Collection<String> getIdentitySource() {
            if (identitySource instanceof SdkAutoConstructList) {
                return null;
            }
            return identitySource;
        }

        public final void setIdentitySource(Collection<String> identitySource) {
            this.identitySource = IdentitySourceListCopier.copy(identitySource);
        }

        @Override
        public final Builder identitySource(Collection<String> identitySource) {
            this.identitySource = IdentitySourceListCopier.copy(identitySource);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identitySource(String... identitySource) {
            identitySource(Arrays.asList(identitySource));
            return this;
        }

        public final String getIdentityValidationExpression() {
            return identityValidationExpression;
        }

        public final void setIdentityValidationExpression(String identityValidationExpression) {
            this.identityValidationExpression = identityValidationExpression;
        }

        @Override
        public final Builder identityValidationExpression(String identityValidationExpression) {
            this.identityValidationExpression = identityValidationExpression;
            return this;
        }

        public final JWTConfiguration.Builder getJwtConfiguration() {
            return jwtConfiguration != null ? jwtConfiguration.toBuilder() : null;
        }

        public final void setJwtConfiguration(JWTConfiguration.BuilderImpl jwtConfiguration) {
            this.jwtConfiguration = jwtConfiguration != null ? jwtConfiguration.build() : null;
        }

        @Override
        public final Builder jwtConfiguration(JWTConfiguration jwtConfiguration) {
            this.jwtConfiguration = jwtConfiguration;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateAuthorizerRequest build() {
            return new UpdateAuthorizerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
