/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockruntime.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An audio content block that contains audio data in various supported formats.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AudioBlock implements SdkPojo, Serializable, ToCopyableBuilder<AudioBlock.Builder, AudioBlock> {
    private static final SdkField<String> FORMAT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("format")
            .getter(getter(AudioBlock::formatAsString)).setter(setter(Builder::format))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("format").build()).build();

    private static final SdkField<AudioSource> SOURCE_FIELD = SdkField.<AudioSource> builder(MarshallingType.SDK_POJO)
            .memberName("source").getter(getter(AudioBlock::source)).setter(setter(Builder::source))
            .constructor(AudioSource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("source").build()).build();

    private static final SdkField<ErrorBlock> ERROR_FIELD = SdkField.<ErrorBlock> builder(MarshallingType.SDK_POJO)
            .memberName("error").getter(getter(AudioBlock::error)).setter(setter(Builder::error))
            .constructor(ErrorBlock::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("error").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FORMAT_FIELD, SOURCE_FIELD,
            ERROR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String format;

    private final AudioSource source;

    private final ErrorBlock error;

    private AudioBlock(BuilderImpl builder) {
        this.format = builder.format;
        this.source = builder.source;
        this.error = builder.error;
    }

    /**
     * <p>
     * The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link AudioFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
     * @see AudioFormat
     */
    public final AudioFormat format() {
        return AudioFormat.fromValue(format);
    }

    /**
     * <p>
     * The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #format} will
     * return {@link AudioFormat#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #formatAsString}.
     * </p>
     * 
     * @return The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
     * @see AudioFormat
     */
    public final String formatAsString() {
        return format;
    }

    /**
     * <p>
     * The source of the audio data, which can be provided as raw bytes or an S3 location.
     * </p>
     * 
     * @return The source of the audio data, which can be provided as raw bytes or an S3 location.
     */
    public final AudioSource source() {
        return source;
    }

    /**
     * <p>
     * Error information if the audio block could not be processed or contains invalid data.
     * </p>
     * 
     * @return Error information if the audio block could not be processed or contains invalid data.
     */
    public final ErrorBlock error() {
        return error;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(formatAsString());
        hashCode = 31 * hashCode + Objects.hashCode(source());
        hashCode = 31 * hashCode + Objects.hashCode(error());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AudioBlock)) {
            return false;
        }
        AudioBlock other = (AudioBlock) obj;
        return Objects.equals(formatAsString(), other.formatAsString()) && Objects.equals(source(), other.source())
                && Objects.equals(error(), other.error());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AudioBlock").add("Format", formatAsString())
                .add("Source", source() == null ? null : "*** Sensitive Data Redacted ***")
                .add("Error", error() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "format":
            return Optional.ofNullable(clazz.cast(formatAsString()));
        case "source":
            return Optional.ofNullable(clazz.cast(source()));
        case "error":
            return Optional.ofNullable(clazz.cast(error()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("format", FORMAT_FIELD);
        map.put("source", SOURCE_FIELD);
        map.put("error", ERROR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AudioBlock, T> g) {
        return obj -> g.apply((AudioBlock) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AudioBlock> {
        /**
         * <p>
         * The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
         * </p>
         * 
         * @param format
         *        The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
         * @see AudioFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioFormat
         */
        Builder format(String format);

        /**
         * <p>
         * The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
         * </p>
         * 
         * @param format
         *        The format of the audio data, such as MP3, WAV, FLAC, or other supported audio formats.
         * @see AudioFormat
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AudioFormat
         */
        Builder format(AudioFormat format);

        /**
         * <p>
         * The source of the audio data, which can be provided as raw bytes or an S3 location.
         * </p>
         * 
         * @param source
         *        The source of the audio data, which can be provided as raw bytes or an S3 location.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder source(AudioSource source);

        /**
         * <p>
         * The source of the audio data, which can be provided as raw bytes or an S3 location.
         * </p>
         * This is a convenience method that creates an instance of the {@link AudioSource.Builder} avoiding the need to
         * create one manually via {@link AudioSource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AudioSource.Builder#build()} is called immediately and its result
         * is passed to {@link #source(AudioSource)}.
         * 
         * @param source
         *        a consumer that will call methods on {@link AudioSource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #source(AudioSource)
         */
        default Builder source(Consumer<AudioSource.Builder> source) {
            return source(AudioSource.builder().applyMutation(source).build());
        }

        /**
         * <p>
         * Error information if the audio block could not be processed or contains invalid data.
         * </p>
         * 
         * @param error
         *        Error information if the audio block could not be processed or contains invalid data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder error(ErrorBlock error);

        /**
         * <p>
         * Error information if the audio block could not be processed or contains invalid data.
         * </p>
         * This is a convenience method that creates an instance of the {@link ErrorBlock.Builder} avoiding the need to
         * create one manually via {@link ErrorBlock#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ErrorBlock.Builder#build()} is called immediately and its result
         * is passed to {@link #error(ErrorBlock)}.
         * 
         * @param error
         *        a consumer that will call methods on {@link ErrorBlock.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #error(ErrorBlock)
         */
        default Builder error(Consumer<ErrorBlock.Builder> error) {
            return error(ErrorBlock.builder().applyMutation(error).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String format;

        private AudioSource source;

        private ErrorBlock error;

        private BuilderImpl() {
        }

        private BuilderImpl(AudioBlock model) {
            format(model.format);
            source(model.source);
            error(model.error);
        }

        public final String getFormat() {
            return format;
        }

        public final void setFormat(String format) {
            this.format = format;
        }

        @Override
        public final Builder format(String format) {
            this.format = format;
            return this;
        }

        @Override
        public final Builder format(AudioFormat format) {
            this.format(format == null ? null : format.toString());
            return this;
        }

        public final AudioSource.Builder getSource() {
            return source != null ? source.toBuilder() : null;
        }

        public final void setSource(AudioSource.BuilderImpl source) {
            this.source = source != null ? source.build() : null;
        }

        @Override
        public final Builder source(AudioSource source) {
            this.source = source;
            return this;
        }

        public final ErrorBlock.Builder getError() {
            return error != null ? error.toBuilder() : null;
        }

        public final void setError(ErrorBlock.BuilderImpl error) {
            this.error = error != null ? error.build() : null;
        }

        @Override
        public final Builder error(ErrorBlock error) {
            this.error = error;
            return this;
        }

        @Override
        public AudioBlock build() {
            return new AudioBlock(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
