/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bedrockagent.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.document.Document;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about a variant of the prompt.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PromptVariant implements SdkPojo, Serializable, ToCopyableBuilder<PromptVariant.Builder, PromptVariant> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(PromptVariant::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> TEMPLATE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("templateType").getter(getter(PromptVariant::templateTypeAsString)).setter(setter(Builder::templateType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("templateType").build()).build();

    private static final SdkField<PromptTemplateConfiguration> TEMPLATE_CONFIGURATION_FIELD = SdkField
            .<PromptTemplateConfiguration> builder(MarshallingType.SDK_POJO).memberName("templateConfiguration")
            .getter(getter(PromptVariant::templateConfiguration)).setter(setter(Builder::templateConfiguration))
            .constructor(PromptTemplateConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("templateConfiguration").build())
            .build();

    private static final SdkField<String> MODEL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("modelId").getter(getter(PromptVariant::modelId)).setter(setter(Builder::modelId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("modelId").build()).build();

    private static final SdkField<PromptInferenceConfiguration> INFERENCE_CONFIGURATION_FIELD = SdkField
            .<PromptInferenceConfiguration> builder(MarshallingType.SDK_POJO).memberName("inferenceConfiguration")
            .getter(getter(PromptVariant::inferenceConfiguration)).setter(setter(Builder::inferenceConfiguration))
            .constructor(PromptInferenceConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("inferenceConfiguration").build())
            .build();

    private static final SdkField<List<PromptMetadataEntry>> METADATA_FIELD = SdkField
            .<List<PromptMetadataEntry>> builder(MarshallingType.LIST)
            .memberName("metadata")
            .getter(getter(PromptVariant::metadata))
            .setter(setter(Builder::metadata))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metadata").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PromptMetadataEntry> builder(MarshallingType.SDK_POJO)
                                            .constructor(PromptMetadataEntry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Document> ADDITIONAL_MODEL_REQUEST_FIELDS_FIELD = SdkField
            .<Document> builder(MarshallingType.DOCUMENT)
            .memberName("additionalModelRequestFields")
            .getter(getter(PromptVariant::additionalModelRequestFields))
            .setter(setter(Builder::additionalModelRequestFields))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("additionalModelRequestFields")
                    .build()).build();

    private static final SdkField<PromptGenAiResource> GEN_AI_RESOURCE_FIELD = SdkField
            .<PromptGenAiResource> builder(MarshallingType.SDK_POJO).memberName("genAiResource")
            .getter(getter(PromptVariant::genAiResource)).setter(setter(Builder::genAiResource))
            .constructor(PromptGenAiResource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("genAiResource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD,
            TEMPLATE_TYPE_FIELD, TEMPLATE_CONFIGURATION_FIELD, MODEL_ID_FIELD, INFERENCE_CONFIGURATION_FIELD, METADATA_FIELD,
            ADDITIONAL_MODEL_REQUEST_FIELDS_FIELD, GEN_AI_RESOURCE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String templateType;

    private final PromptTemplateConfiguration templateConfiguration;

    private final String modelId;

    private final PromptInferenceConfiguration inferenceConfiguration;

    private final List<PromptMetadataEntry> metadata;

    private final Document additionalModelRequestFields;

    private final PromptGenAiResource genAiResource;

    private PromptVariant(BuilderImpl builder) {
        this.name = builder.name;
        this.templateType = builder.templateType;
        this.templateConfiguration = builder.templateConfiguration;
        this.modelId = builder.modelId;
        this.inferenceConfiguration = builder.inferenceConfiguration;
        this.metadata = builder.metadata;
        this.additionalModelRequestFields = builder.additionalModelRequestFields;
        this.genAiResource = builder.genAiResource;
    }

    /**
     * <p>
     * The name of the prompt variant.
     * </p>
     * 
     * @return The name of the prompt variant.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The type of prompt template to use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #templateType} will
     * return {@link PromptTemplateType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #templateTypeAsString}.
     * </p>
     * 
     * @return The type of prompt template to use.
     * @see PromptTemplateType
     */
    public final PromptTemplateType templateType() {
        return PromptTemplateType.fromValue(templateType);
    }

    /**
     * <p>
     * The type of prompt template to use.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #templateType} will
     * return {@link PromptTemplateType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #templateTypeAsString}.
     * </p>
     * 
     * @return The type of prompt template to use.
     * @see PromptTemplateType
     */
    public final String templateTypeAsString() {
        return templateType;
    }

    /**
     * <p>
     * Contains configurations for the prompt template.
     * </p>
     * 
     * @return Contains configurations for the prompt template.
     */
    public final PromptTemplateConfiguration templateConfiguration() {
        return templateConfiguration;
    }

    /**
     * <p>
     * The unique identifier of the model or <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference profile</a>
     * with which to run inference on the prompt.
     * </p>
     * 
     * @return The unique identifier of the model or <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference
     *         profile</a> with which to run inference on the prompt.
     */
    public final String modelId() {
        return modelId;
    }

    /**
     * <p>
     * Contains inference configurations for the prompt variant.
     * </p>
     * 
     * @return Contains inference configurations for the prompt variant.
     */
    public final PromptInferenceConfiguration inferenceConfiguration() {
        return inferenceConfiguration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Metadata property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMetadata() {
        return metadata != null && !(metadata instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects, each containing a key-value pair that defines a metadata tag and value to attach to a prompt
     * variant.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMetadata} method.
     * </p>
     * 
     * @return An array of objects, each containing a key-value pair that defines a metadata tag and value to attach to
     *         a prompt variant.
     */
    public final List<PromptMetadataEntry> metadata() {
        return metadata;
    }

    /**
     * <p>
     * Contains model-specific inference configurations that aren't in the <code>inferenceConfiguration</code> field. To
     * see model-specific inference parameters, see <a
     * href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference request parameters
     * and response fields for foundation models</a>.
     * </p>
     * 
     * @return Contains model-specific inference configurations that aren't in the <code>inferenceConfiguration</code>
     *         field. To see model-specific inference parameters, see <a
     *         href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference request
     *         parameters and response fields for foundation models</a>.
     */
    public final Document additionalModelRequestFields() {
        return additionalModelRequestFields;
    }

    /**
     * <p>
     * Specifies a generative AI resource with which to use the prompt.
     * </p>
     * 
     * @return Specifies a generative AI resource with which to use the prompt.
     */
    public final PromptGenAiResource genAiResource() {
        return genAiResource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(templateTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(templateConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(modelId());
        hashCode = 31 * hashCode + Objects.hashCode(inferenceConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(hasMetadata() ? metadata() : null);
        hashCode = 31 * hashCode + Objects.hashCode(additionalModelRequestFields());
        hashCode = 31 * hashCode + Objects.hashCode(genAiResource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PromptVariant)) {
            return false;
        }
        PromptVariant other = (PromptVariant) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(templateTypeAsString(), other.templateTypeAsString())
                && Objects.equals(templateConfiguration(), other.templateConfiguration())
                && Objects.equals(modelId(), other.modelId())
                && Objects.equals(inferenceConfiguration(), other.inferenceConfiguration())
                && hasMetadata() == other.hasMetadata() && Objects.equals(metadata(), other.metadata())
                && Objects.equals(additionalModelRequestFields(), other.additionalModelRequestFields())
                && Objects.equals(genAiResource(), other.genAiResource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PromptVariant").add("Name", name()).add("TemplateType", templateTypeAsString())
                .add("TemplateConfiguration", templateConfiguration()).add("ModelId", modelId())
                .add("InferenceConfiguration", inferenceConfiguration())
                .add("Metadata", metadata() == null ? null : "*** Sensitive Data Redacted ***")
                .add("AdditionalModelRequestFields", additionalModelRequestFields())
                .add("GenAiResource", genAiResource() == null ? null : "*** Sensitive Data Redacted ***").build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "templateType":
            return Optional.ofNullable(clazz.cast(templateTypeAsString()));
        case "templateConfiguration":
            return Optional.ofNullable(clazz.cast(templateConfiguration()));
        case "modelId":
            return Optional.ofNullable(clazz.cast(modelId()));
        case "inferenceConfiguration":
            return Optional.ofNullable(clazz.cast(inferenceConfiguration()));
        case "metadata":
            return Optional.ofNullable(clazz.cast(metadata()));
        case "additionalModelRequestFields":
            return Optional.ofNullable(clazz.cast(additionalModelRequestFields()));
        case "genAiResource":
            return Optional.ofNullable(clazz.cast(genAiResource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("templateType", TEMPLATE_TYPE_FIELD);
        map.put("templateConfiguration", TEMPLATE_CONFIGURATION_FIELD);
        map.put("modelId", MODEL_ID_FIELD);
        map.put("inferenceConfiguration", INFERENCE_CONFIGURATION_FIELD);
        map.put("metadata", METADATA_FIELD);
        map.put("additionalModelRequestFields", ADDITIONAL_MODEL_REQUEST_FIELDS_FIELD);
        map.put("genAiResource", GEN_AI_RESOURCE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<PromptVariant, T> g) {
        return obj -> g.apply((PromptVariant) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PromptVariant> {
        /**
         * <p>
         * The name of the prompt variant.
         * </p>
         * 
         * @param name
         *        The name of the prompt variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The type of prompt template to use.
         * </p>
         * 
         * @param templateType
         *        The type of prompt template to use.
         * @see PromptTemplateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PromptTemplateType
         */
        Builder templateType(String templateType);

        /**
         * <p>
         * The type of prompt template to use.
         * </p>
         * 
         * @param templateType
         *        The type of prompt template to use.
         * @see PromptTemplateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PromptTemplateType
         */
        Builder templateType(PromptTemplateType templateType);

        /**
         * <p>
         * Contains configurations for the prompt template.
         * </p>
         * 
         * @param templateConfiguration
         *        Contains configurations for the prompt template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder templateConfiguration(PromptTemplateConfiguration templateConfiguration);

        /**
         * <p>
         * Contains configurations for the prompt template.
         * </p>
         * This is a convenience method that creates an instance of the {@link PromptTemplateConfiguration.Builder}
         * avoiding the need to create one manually via {@link PromptTemplateConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PromptTemplateConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #templateConfiguration(PromptTemplateConfiguration)}.
         * 
         * @param templateConfiguration
         *        a consumer that will call methods on {@link PromptTemplateConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #templateConfiguration(PromptTemplateConfiguration)
         */
        default Builder templateConfiguration(Consumer<PromptTemplateConfiguration.Builder> templateConfiguration) {
            return templateConfiguration(PromptTemplateConfiguration.builder().applyMutation(templateConfiguration).build());
        }

        /**
         * <p>
         * The unique identifier of the model or <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference profile</a>
         * with which to run inference on the prompt.
         * </p>
         * 
         * @param modelId
         *        The unique identifier of the model or <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/cross-region-inference.html">inference
         *        profile</a> with which to run inference on the prompt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder modelId(String modelId);

        /**
         * <p>
         * Contains inference configurations for the prompt variant.
         * </p>
         * 
         * @param inferenceConfiguration
         *        Contains inference configurations for the prompt variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder inferenceConfiguration(PromptInferenceConfiguration inferenceConfiguration);

        /**
         * <p>
         * Contains inference configurations for the prompt variant.
         * </p>
         * This is a convenience method that creates an instance of the {@link PromptInferenceConfiguration.Builder}
         * avoiding the need to create one manually via {@link PromptInferenceConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PromptInferenceConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #inferenceConfiguration(PromptInferenceConfiguration)}.
         * 
         * @param inferenceConfiguration
         *        a consumer that will call methods on {@link PromptInferenceConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #inferenceConfiguration(PromptInferenceConfiguration)
         */
        default Builder inferenceConfiguration(Consumer<PromptInferenceConfiguration.Builder> inferenceConfiguration) {
            return inferenceConfiguration(PromptInferenceConfiguration.builder().applyMutation(inferenceConfiguration).build());
        }

        /**
         * <p>
         * An array of objects, each containing a key-value pair that defines a metadata tag and value to attach to a
         * prompt variant.
         * </p>
         * 
         * @param metadata
         *        An array of objects, each containing a key-value pair that defines a metadata tag and value to attach
         *        to a prompt variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(Collection<PromptMetadataEntry> metadata);

        /**
         * <p>
         * An array of objects, each containing a key-value pair that defines a metadata tag and value to attach to a
         * prompt variant.
         * </p>
         * 
         * @param metadata
         *        An array of objects, each containing a key-value pair that defines a metadata tag and value to attach
         *        to a prompt variant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metadata(PromptMetadataEntry... metadata);

        /**
         * <p>
         * An array of objects, each containing a key-value pair that defines a metadata tag and value to attach to a
         * prompt variant.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.bedrockagent.model.PromptMetadataEntry.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.bedrockagent.model.PromptMetadataEntry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.bedrockagent.model.PromptMetadataEntry.Builder#build()} is called
         * immediately and its result is passed to {@link #metadata(List<PromptMetadataEntry>)}.
         * 
         * @param metadata
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.bedrockagent.model.PromptMetadataEntry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metadata(java.util.Collection<PromptMetadataEntry>)
         */
        Builder metadata(Consumer<PromptMetadataEntry.Builder>... metadata);

        /**
         * <p>
         * Contains model-specific inference configurations that aren't in the <code>inferenceConfiguration</code>
         * field. To see model-specific inference parameters, see <a
         * href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference request
         * parameters and response fields for foundation models</a>.
         * </p>
         * 
         * @param additionalModelRequestFields
         *        Contains model-specific inference configurations that aren't in the
         *        <code>inferenceConfiguration</code> field. To see model-specific inference parameters, see <a
         *        href="https://docs.aws.amazon.com/bedrock/latest/userguide/model-parameters.html">Inference request
         *        parameters and response fields for foundation models</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalModelRequestFields(Document additionalModelRequestFields);

        /**
         * <p>
         * Specifies a generative AI resource with which to use the prompt.
         * </p>
         * 
         * @param genAiResource
         *        Specifies a generative AI resource with which to use the prompt.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder genAiResource(PromptGenAiResource genAiResource);

        /**
         * <p>
         * Specifies a generative AI resource with which to use the prompt.
         * </p>
         * This is a convenience method that creates an instance of the {@link PromptGenAiResource.Builder} avoiding the
         * need to create one manually via {@link PromptGenAiResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PromptGenAiResource.Builder#build()} is called immediately and
         * its result is passed to {@link #genAiResource(PromptGenAiResource)}.
         * 
         * @param genAiResource
         *        a consumer that will call methods on {@link PromptGenAiResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #genAiResource(PromptGenAiResource)
         */
        default Builder genAiResource(Consumer<PromptGenAiResource.Builder> genAiResource) {
            return genAiResource(PromptGenAiResource.builder().applyMutation(genAiResource).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String templateType;

        private PromptTemplateConfiguration templateConfiguration;

        private String modelId;

        private PromptInferenceConfiguration inferenceConfiguration;

        private List<PromptMetadataEntry> metadata = DefaultSdkAutoConstructList.getInstance();

        private Document additionalModelRequestFields;

        private PromptGenAiResource genAiResource;

        private BuilderImpl() {
        }

        private BuilderImpl(PromptVariant model) {
            name(model.name);
            templateType(model.templateType);
            templateConfiguration(model.templateConfiguration);
            modelId(model.modelId);
            inferenceConfiguration(model.inferenceConfiguration);
            metadata(model.metadata);
            additionalModelRequestFields(model.additionalModelRequestFields);
            genAiResource(model.genAiResource);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getTemplateType() {
            return templateType;
        }

        public final void setTemplateType(String templateType) {
            this.templateType = templateType;
        }

        @Override
        public final Builder templateType(String templateType) {
            this.templateType = templateType;
            return this;
        }

        @Override
        public final Builder templateType(PromptTemplateType templateType) {
            this.templateType(templateType == null ? null : templateType.toString());
            return this;
        }

        public final PromptTemplateConfiguration.Builder getTemplateConfiguration() {
            return templateConfiguration != null ? templateConfiguration.toBuilder() : null;
        }

        public final void setTemplateConfiguration(PromptTemplateConfiguration.BuilderImpl templateConfiguration) {
            this.templateConfiguration = templateConfiguration != null ? templateConfiguration.build() : null;
        }

        @Override
        public final Builder templateConfiguration(PromptTemplateConfiguration templateConfiguration) {
            this.templateConfiguration = templateConfiguration;
            return this;
        }

        public final String getModelId() {
            return modelId;
        }

        public final void setModelId(String modelId) {
            this.modelId = modelId;
        }

        @Override
        public final Builder modelId(String modelId) {
            this.modelId = modelId;
            return this;
        }

        public final PromptInferenceConfiguration.Builder getInferenceConfiguration() {
            return inferenceConfiguration != null ? inferenceConfiguration.toBuilder() : null;
        }

        public final void setInferenceConfiguration(PromptInferenceConfiguration.BuilderImpl inferenceConfiguration) {
            this.inferenceConfiguration = inferenceConfiguration != null ? inferenceConfiguration.build() : null;
        }

        @Override
        public final Builder inferenceConfiguration(PromptInferenceConfiguration inferenceConfiguration) {
            this.inferenceConfiguration = inferenceConfiguration;
            return this;
        }

        public final List<PromptMetadataEntry.Builder> getMetadata() {
            List<PromptMetadataEntry.Builder> result = PromptMetadataListCopier.copyToBuilder(this.metadata);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMetadata(Collection<PromptMetadataEntry.BuilderImpl> metadata) {
            this.metadata = PromptMetadataListCopier.copyFromBuilder(metadata);
        }

        @Override
        public final Builder metadata(Collection<PromptMetadataEntry> metadata) {
            this.metadata = PromptMetadataListCopier.copy(metadata);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metadata(PromptMetadataEntry... metadata) {
            metadata(Arrays.asList(metadata));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder metadata(Consumer<PromptMetadataEntry.Builder>... metadata) {
            metadata(Stream.of(metadata).map(c -> PromptMetadataEntry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Document getAdditionalModelRequestFields() {
            return additionalModelRequestFields;
        }

        public final void setAdditionalModelRequestFields(Document additionalModelRequestFields) {
            this.additionalModelRequestFields = additionalModelRequestFields;
        }

        @Override
        public final Builder additionalModelRequestFields(Document additionalModelRequestFields) {
            this.additionalModelRequestFields = additionalModelRequestFields;
            return this;
        }

        public final PromptGenAiResource.Builder getGenAiResource() {
            return genAiResource != null ? genAiResource.toBuilder() : null;
        }

        public final void setGenAiResource(PromptGenAiResource.BuilderImpl genAiResource) {
            this.genAiResource = genAiResource != null ? genAiResource.build() : null;
        }

        @Override
        public final Builder genAiResource(PromptGenAiResource genAiResource) {
            this.genAiResource = genAiResource;
            return this;
        }

        @Override
        public PromptVariant build() {
            return new PromptVariant(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
