/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.bcmdashboards;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.bcmdashboards.model.AccessDeniedException;
import software.amazon.awssdk.services.bcmdashboards.model.BcmDashboardsException;
import software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.GetDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.GetDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.bcmdashboards.model.InternalServerException;
import software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest;
import software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ResourceNotFoundException;
import software.amazon.awssdk.services.bcmdashboards.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.bcmdashboards.model.TagResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.TagResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ThrottlingException;
import software.amazon.awssdk.services.bcmdashboards.model.UntagResourceRequest;
import software.amazon.awssdk.services.bcmdashboards.model.UntagResourceResponse;
import software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardRequest;
import software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardResponse;
import software.amazon.awssdk.services.bcmdashboards.model.ValidationException;
import software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable;

/**
 * Service client for accessing AWS Billing and Cost Management Dashboards. This can be created using the static
 * {@link #builder()} method.
 *
 * <p>
 * Amazon Web Services Billing and Cost Management Dashboards is a service that enables you to create, manage, and share
 * dashboards that combine multiple visualizations of your Amazon Web Services cost and usage data. You can combine
 * multiple data sources including Cost Explorer, Savings Plans, and Reserved Instance metrics into unified dashboards,
 * helping you analyze spending patterns and share cost insights across your organization.
 * </p>
 * <p>
 * You can use the Amazon Web Services Billing and Cost Management Dashboards API to programmatically create, manage,
 * and share dashboards. This includes creating custom dashboards, configuring widgets, managing dashboard permissions,
 * and sharing dashboards across accounts in your organization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface BcmDashboardsClient extends AwsClient {
    String SERVICE_NAME = "bcm-dashboards";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "bcm-dashboards";

    /**
     * <p>
     * Creates a new dashboard that can contain multiple widgets displaying cost and usage data. You can add custom
     * widgets or use predefined widgets, arranging them in your preferred layout.
     * </p>
     *
     * @param createDashboardRequest
     * @return Result of the CreateDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ServiceQuotaExceededException
     *         The request would exceed service quotas. For example, attempting to create more than 20 widgets in a
     *         dashboard or exceeding the maximum number of dashboards per account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.CreateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/CreateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateDashboardResponse createDashboard(CreateDashboardRequest createDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ServiceQuotaExceededException,
            AwsServiceException, SdkClientException, BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Creates a new dashboard that can contain multiple widgets displaying cost and usage data. You can add custom
     * widgets or use predefined widgets, arranging them in your preferred layout.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link CreateDashboardRequest.Builder} avoiding the need
     * to create one manually via {@link CreateDashboardRequest#builder()}
     * </p>
     *
     * @param createDashboardRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.CreateDashboardRequest.Builder} to create a
     *        request.
     * @return Result of the CreateDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ServiceQuotaExceededException
     *         The request would exceed service quotas. For example, attempting to create more than 20 widgets in a
     *         dashboard or exceeding the maximum number of dashboards per account.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.CreateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/CreateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    default CreateDashboardResponse createDashboard(Consumer<CreateDashboardRequest.Builder> createDashboardRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ServiceQuotaExceededException, AwsServiceException, SdkClientException, BcmDashboardsException {
        return createDashboard(CreateDashboardRequest.builder().applyMutation(createDashboardRequest).build());
    }

    /**
     * <p>
     * Deletes a specified dashboard. This action cannot be undone.
     * </p>
     *
     * @param deleteDashboardRequest
     * @return Result of the DeleteDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.DeleteDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/DeleteDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteDashboardResponse deleteDashboard(DeleteDashboardRequest deleteDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes a specified dashboard. This action cannot be undone.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteDashboardRequest.Builder} avoiding the need
     * to create one manually via {@link DeleteDashboardRequest#builder()}
     * </p>
     *
     * @param deleteDashboardRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.DeleteDashboardRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.DeleteDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/DeleteDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteDashboardResponse deleteDashboard(Consumer<DeleteDashboardRequest.Builder> deleteDashboardRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        return deleteDashboard(DeleteDashboardRequest.builder().applyMutation(deleteDashboardRequest).build());
    }

    /**
     * <p>
     * Retrieves the configuration and metadata of a specified dashboard, including its widgets and layout settings.
     * </p>
     *
     * @param getDashboardRequest
     * @return Result of the GetDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.GetDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetDashboard" target="_top">AWS
     *      API Documentation</a>
     */
    default GetDashboardResponse getDashboard(GetDashboardRequest getDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the configuration and metadata of a specified dashboard, including its widgets and layout settings.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetDashboardRequest.Builder} avoiding the need to
     * create one manually via {@link GetDashboardRequest#builder()}
     * </p>
     *
     * @param getDashboardRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.GetDashboardRequest.Builder} to create a
     *        request.
     * @return Result of the GetDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.GetDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetDashboard" target="_top">AWS
     *      API Documentation</a>
     */
    default GetDashboardResponse getDashboard(Consumer<GetDashboardRequest.Builder> getDashboardRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BcmDashboardsException {
        return getDashboard(GetDashboardRequest.builder().applyMutation(getDashboardRequest).build());
    }

    /**
     * <p>
     * Retrieves the resource-based policy attached to a dashboard, showing sharing configurations and permissions.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the resource-based policy attached to a dashboard, showing sharing configurations and permissions.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetResourcePolicyRequest.Builder} avoiding the need
     * to create one manually via {@link GetResourcePolicyRequest#builder()}
     * </p>
     *
     * @param getResourcePolicyRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.GetResourcePolicyRequest.Builder} to create a
     *        request.
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/GetResourcePolicy"
     *      target="_top">AWS API Documentation</a>
     */
    default GetResourcePolicyResponse getResourcePolicy(Consumer<GetResourcePolicyRequest.Builder> getResourcePolicyRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BcmDashboardsException {
        return getResourcePolicy(GetResourcePolicyRequest.builder().applyMutation(getResourcePolicyRequest).build());
    }

    /**
     * <p>
     * Returns a list of all dashboards in your account.
     * </p>
     *
     * @param listDashboardsRequest
     * @return Result of the ListDashboards operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDashboardsResponse listDashboards(ListDashboardsRequest listDashboardsRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all dashboards in your account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDashboardsRequest.Builder} avoiding the need to
     * create one manually via {@link ListDashboardsRequest#builder()}
     * </p>
     *
     * @param listDashboardsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest.Builder} to create a
     *        request.
     * @return Result of the ListDashboards operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDashboardsResponse listDashboards(Consumer<ListDashboardsRequest.Builder> listDashboardsRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        return listDashboards(ListDashboardsRequest.builder().applyMutation(listDashboardsRequest).build());
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDashboards(software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable responses = client.listDashboardsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable responses = client
     *             .listDashboardsPaginator(request);
     *     for (software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable responses = client.listDashboardsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDashboards(software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest)} operation.</b>
     * </p>
     *
     * @param listDashboardsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDashboardsIterable listDashboardsPaginator(ListDashboardsRequest listDashboardsRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        return new ListDashboardsIterable(this, listDashboardsRequest);
    }

    /**
     * <p>
     * This is a variant of
     * {@link #listDashboards(software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable responses = client.listDashboardsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable responses = client
     *             .listDashboardsPaginator(request);
     *     for (software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.bcmdashboards.paginators.ListDashboardsIterable responses = client.listDashboardsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of maxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDashboards(software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListDashboardsRequest.Builder} avoiding the need to
     * create one manually via {@link ListDashboardsRequest#builder()}
     * </p>
     *
     * @param listDashboardsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.ListDashboardsRequest.Builder} to create a
     *        request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    default ListDashboardsIterable listDashboardsPaginator(Consumer<ListDashboardsRequest.Builder> listDashboardsRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        return listDashboardsPaginator(ListDashboardsRequest.builder().applyMutation(listDashboardsRequest).build());
    }

    /**
     * <p>
     * Returns a list of all tags associated with a specified dashboard resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ThrottlingException, InternalServerException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Returns a list of all tags associated with a specified dashboard resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListTagsForResourceRequest.Builder} avoiding the
     * need to create one manually via {@link ListTagsForResourceRequest#builder()}
     * </p>
     *
     * @param listTagsForResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.ListTagsForResourceRequest.Builder} to create a
     *        request.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    default ListTagsForResourceResponse listTagsForResource(
            Consumer<ListTagsForResourceRequest.Builder> listTagsForResourceRequest) throws ThrottlingException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        return listTagsForResource(ListTagsForResourceRequest.builder().applyMutation(listTagsForResourceRequest).build());
    }

    /**
     * <p>
     * Adds or updates tags for a specified dashboard resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ThrottlingException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Adds or updates tags for a specified dashboard resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link TagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link TagResourceRequest#builder()}
     * </p>
     *
     * @param tagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.TagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the TagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/TagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default TagResourceResponse tagResource(Consumer<TagResourceRequest.Builder> tagResourceRequest) throws ThrottlingException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        return tagResource(TagResourceRequest.builder().applyMutation(tagResourceRequest).build());
    }

    /**
     * <p>
     * Removes specified tags from a dashboard resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ThrottlingException,
            InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException, SdkClientException,
            BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Removes specified tags from a dashboard resource.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UntagResourceRequest.Builder} avoiding the need to
     * create one manually via {@link UntagResourceRequest#builder()}
     * </p>
     *
     * @param untagResourceRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.UntagResourceRequest.Builder} to create a
     *        request.
     * @return Result of the UntagResource operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UntagResource" target="_top">AWS
     *      API Documentation</a>
     */
    default UntagResourceResponse untagResource(Consumer<UntagResourceRequest.Builder> untagResourceRequest)
            throws ThrottlingException, InternalServerException, ValidationException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, BcmDashboardsException {
        return untagResource(UntagResourceRequest.builder().applyMutation(untagResourceRequest).build());
    }

    /**
     * <p>
     * Updates an existing dashboard's properties, including its name, description, and widget configurations.
     * </p>
     *
     * @param updateDashboardRequest
     * @return Result of the UpdateDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.UpdateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UpdateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateDashboardResponse updateDashboard(UpdateDashboardRequest updateDashboardRequest) throws ThrottlingException,
            AccessDeniedException, InternalServerException, ValidationException, ResourceNotFoundException, AwsServiceException,
            SdkClientException, BcmDashboardsException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates an existing dashboard's properties, including its name, description, and widget configurations.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link UpdateDashboardRequest.Builder} avoiding the need
     * to create one manually via {@link UpdateDashboardRequest#builder()}
     * </p>
     *
     * @param updateDashboardRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.bcmdashboards.model.UpdateDashboardRequest.Builder} to create a
     *        request.
     * @return Result of the UpdateDashboard operation returned by the service.
     * @throws ThrottlingException
     *         The request was denied due to request throttling. Reduce the frequency of requests and use exponential
     *         backoff.
     * @throws AccessDeniedException
     *         You do not have sufficient permissions to perform this action. Verify your IAM permissions and any
     *         resource policies.
     * @throws InternalServerException
     *         An internal error occurred while processing the request. Retry your request. If the problem persists,
     *         contact Amazon Web Services Support.
     * @throws ValidationException
     *         The input parameters do not satisfy the requirements. Check the error message for specific validation
     *         details.
     * @throws ResourceNotFoundException
     *         The specified resource (dashboard, policy, or widget) was not found. Verify the ARN and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws BcmDashboardsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample BcmDashboardsClient.UpdateDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/bcm-dashboards-2025-08-18/UpdateDashboard"
     *      target="_top">AWS API Documentation</a>
     */
    default UpdateDashboardResponse updateDashboard(Consumer<UpdateDashboardRequest.Builder> updateDashboardRequest)
            throws ThrottlingException, AccessDeniedException, InternalServerException, ValidationException,
            ResourceNotFoundException, AwsServiceException, SdkClientException, BcmDashboardsException {
        return updateDashboard(UpdateDashboardRequest.builder().applyMutation(updateDashboardRequest).build());
    }

    /**
     * Create a {@link BcmDashboardsClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static BcmDashboardsClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link BcmDashboardsClient}.
     */
    static BcmDashboardsClientBuilder builder() {
        return new DefaultBcmDashboardsClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default BcmDashboardsServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
