/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An object that represents a launch template that's associated with a compute resource. You must specify either the
 * launch template ID or launch template name in the request, but not both.
 * </p>
 * <p>
 * If security groups are specified using both the <code>securityGroupIds</code> parameter of
 * <code>CreateComputeEnvironment</code> and the launch template, the values in the <code>securityGroupIds</code>
 * parameter of <code>CreateComputeEnvironment</code> will be used.
 * </p>
 * <note>
 * <p>
 * This object isn't applicable to jobs that are running on Fargate resources.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LaunchTemplateSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<LaunchTemplateSpecification.Builder, LaunchTemplateSpecification> {
    private static final SdkField<String> LAUNCH_TEMPLATE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchTemplateId").getter(getter(LaunchTemplateSpecification::launchTemplateId))
            .setter(setter(Builder::launchTemplateId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchTemplateId").build()).build();

    private static final SdkField<String> LAUNCH_TEMPLATE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("launchTemplateName").getter(getter(LaunchTemplateSpecification::launchTemplateName))
            .setter(setter(Builder::launchTemplateName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("launchTemplateName").build())
            .build();

    private static final SdkField<String> VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("version")
            .getter(getter(LaunchTemplateSpecification::version)).setter(setter(Builder::version))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("version").build()).build();

    private static final SdkField<List<LaunchTemplateSpecificationOverride>> OVERRIDES_FIELD = SdkField
            .<List<LaunchTemplateSpecificationOverride>> builder(MarshallingType.LIST)
            .memberName("overrides")
            .getter(getter(LaunchTemplateSpecification::overrides))
            .setter(setter(Builder::overrides))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("overrides").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LaunchTemplateSpecificationOverride> builder(MarshallingType.SDK_POJO)
                                            .constructor(LaunchTemplateSpecificationOverride::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> USERDATA_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("userdataType").getter(getter(LaunchTemplateSpecification::userdataTypeAsString))
            .setter(setter(Builder::userdataType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("userdataType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LAUNCH_TEMPLATE_ID_FIELD,
            LAUNCH_TEMPLATE_NAME_FIELD, VERSION_FIELD, OVERRIDES_FIELD, USERDATA_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String launchTemplateId;

    private final String launchTemplateName;

    private final String version;

    private final List<LaunchTemplateSpecificationOverride> overrides;

    private final String userdataType;

    private LaunchTemplateSpecification(BuilderImpl builder) {
        this.launchTemplateId = builder.launchTemplateId;
        this.launchTemplateName = builder.launchTemplateName;
        this.version = builder.version;
        this.overrides = builder.overrides;
        this.userdataType = builder.userdataType;
    }

    /**
     * <p>
     * The ID of the launch template.
     * </p>
     * 
     * @return The ID of the launch template.
     */
    public final String launchTemplateId() {
        return launchTemplateId;
    }

    /**
     * <p>
     * The name of the launch template.
     * </p>
     * 
     * @return The name of the launch template.
     */
    public final String launchTemplateName() {
        return launchTemplateName;
    }

    /**
     * <p>
     * The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.
     * </p>
     * <p>
     * If the value is <code>$Default</code>, the default version of the launch template is used. If the value is
     * <code>$Latest</code>, the latest version of the launch template is used.
     * </p>
     * <important>
     * <p>
     * If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when the
     * compute environment is updated. It's only changed if the <code>updateToLatestImageVersion</code> parameter for
     * the compute environment is set to <code>true</code>. During an infrastructure update, if either
     * <code>$Default</code> or <code>$Latest</code> is specified, Batch re-evaluates the launch template version, and
     * it might use a different version of the launch template. This is the case even if the launch template isn't
     * specified in the update. When updating a compute environment, changing the launch template requires an
     * infrastructure update of the compute environment. For more information, see <a
     * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
     * environments</a> in the <i>Batch User Guide</i>.
     * </p>
     * </important>
     * <p>
     * Default: <code>$Default</code>
     * </p>
     * <p>
     * Latest: <code>$Latest</code>
     * </p>
     * 
     * @return The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.</p>
     *         <p>
     *         If the value is <code>$Default</code>, the default version of the launch template is used. If the value
     *         is <code>$Latest</code>, the latest version of the launch template is used.
     *         </p>
     *         <important>
     *         <p>
     *         If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed
     *         when the compute environment is updated. It's only changed if the <code>updateToLatestImageVersion</code>
     *         parameter for the compute environment is set to <code>true</code>. During an infrastructure update, if
     *         either <code>$Default</code> or <code>$Latest</code> is specified, Batch re-evaluates the launch template
     *         version, and it might use a different version of the launch template. This is the case even if the launch
     *         template isn't specified in the update. When updating a compute environment, changing the launch template
     *         requires an infrastructure update of the compute environment. For more information, see <a
     *         href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating
     *         compute environments</a> in the <i>Batch User Guide</i>.
     *         </p>
     *         </important>
     *         <p>
     *         Default: <code>$Default</code>
     *         </p>
     *         <p>
     *         Latest: <code>$Latest</code>
     */
    public final String version() {
        return version;
    }

    /**
     * For responses, this returns true if the service returned a value for the Overrides property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasOverrides() {
        return overrides != null && !(overrides instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A launch template to use in place of the default launch template. You must specify either the launch template ID
     * or launch template name in the request, but not both.
     * </p>
     * <p>
     * You can specify up to ten (10) launch template overrides that are associated to unique instance types or families
     * for each compute environment.
     * </p>
     * <note>
     * <p>
     * To unset all override templates for a compute environment, you can pass an empty array to the <a
     * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
     * >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter when
     * submitting the <code>UpdateComputeEnvironment</code> API operation.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasOverrides} method.
     * </p>
     * 
     * @return A launch template to use in place of the default launch template. You must specify either the launch
     *         template ID or launch template name in the request, but not both.</p>
     *         <p>
     *         You can specify up to ten (10) launch template overrides that are associated to unique instance types or
     *         families for each compute environment.
     *         </p>
     *         <note>
     *         <p>
     *         To unset all override templates for a compute environment, you can pass an empty array to the <a
     *         href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
     *         >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter
     *         when submitting the <code>UpdateComputeEnvironment</code> API operation.
     *         </p>
     */
    public final List<LaunchTemplateSpecificationOverride> overrides() {
        return overrides;
    }

    /**
     * <p>
     * The EKS node initialization process to use. You only need to specify this value if you are using a custom AMI.
     * The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI based on EKS_AL2023 or
     * EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #userdataType} will
     * return {@link UserdataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #userdataTypeAsString}.
     * </p>
     * 
     * @return The EKS node initialization process to use. You only need to specify this value if you are using a custom
     *         AMI. The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI based on
     *         EKS_AL2023 or EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
     * @see UserdataType
     */
    public final UserdataType userdataType() {
        return UserdataType.fromValue(userdataType);
    }

    /**
     * <p>
     * The EKS node initialization process to use. You only need to specify this value if you are using a custom AMI.
     * The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI based on EKS_AL2023 or
     * EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #userdataType} will
     * return {@link UserdataType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #userdataTypeAsString}.
     * </p>
     * 
     * @return The EKS node initialization process to use. You only need to specify this value if you are using a custom
     *         AMI. The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI based on
     *         EKS_AL2023 or EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
     * @see UserdataType
     */
    public final String userdataTypeAsString() {
        return userdataType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(launchTemplateId());
        hashCode = 31 * hashCode + Objects.hashCode(launchTemplateName());
        hashCode = 31 * hashCode + Objects.hashCode(version());
        hashCode = 31 * hashCode + Objects.hashCode(hasOverrides() ? overrides() : null);
        hashCode = 31 * hashCode + Objects.hashCode(userdataTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LaunchTemplateSpecification)) {
            return false;
        }
        LaunchTemplateSpecification other = (LaunchTemplateSpecification) obj;
        return Objects.equals(launchTemplateId(), other.launchTemplateId())
                && Objects.equals(launchTemplateName(), other.launchTemplateName()) && Objects.equals(version(), other.version())
                && hasOverrides() == other.hasOverrides() && Objects.equals(overrides(), other.overrides())
                && Objects.equals(userdataTypeAsString(), other.userdataTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LaunchTemplateSpecification").add("LaunchTemplateId", launchTemplateId())
                .add("LaunchTemplateName", launchTemplateName()).add("Version", version())
                .add("Overrides", hasOverrides() ? overrides() : null).add("UserdataType", userdataTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "launchTemplateId":
            return Optional.ofNullable(clazz.cast(launchTemplateId()));
        case "launchTemplateName":
            return Optional.ofNullable(clazz.cast(launchTemplateName()));
        case "version":
            return Optional.ofNullable(clazz.cast(version()));
        case "overrides":
            return Optional.ofNullable(clazz.cast(overrides()));
        case "userdataType":
            return Optional.ofNullable(clazz.cast(userdataTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("launchTemplateId", LAUNCH_TEMPLATE_ID_FIELD);
        map.put("launchTemplateName", LAUNCH_TEMPLATE_NAME_FIELD);
        map.put("version", VERSION_FIELD);
        map.put("overrides", OVERRIDES_FIELD);
        map.put("userdataType", USERDATA_TYPE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LaunchTemplateSpecification, T> g) {
        return obj -> g.apply((LaunchTemplateSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LaunchTemplateSpecification> {
        /**
         * <p>
         * The ID of the launch template.
         * </p>
         * 
         * @param launchTemplateId
         *        The ID of the launch template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTemplateId(String launchTemplateId);

        /**
         * <p>
         * The name of the launch template.
         * </p>
         * 
         * @param launchTemplateName
         *        The name of the launch template.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchTemplateName(String launchTemplateName);

        /**
         * <p>
         * The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.
         * </p>
         * <p>
         * If the value is <code>$Default</code>, the default version of the launch template is used. If the value is
         * <code>$Latest</code>, the latest version of the launch template is used.
         * </p>
         * <important>
         * <p>
         * If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed when
         * the compute environment is updated. It's only changed if the <code>updateToLatestImageVersion</code>
         * parameter for the compute environment is set to <code>true</code>. During an infrastructure update, if either
         * <code>$Default</code> or <code>$Latest</code> is specified, Batch re-evaluates the launch template version,
         * and it might use a different version of the launch template. This is the case even if the launch template
         * isn't specified in the update. When updating a compute environment, changing the launch template requires an
         * infrastructure update of the compute environment. For more information, see <a
         * href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating compute
         * environments</a> in the <i>Batch User Guide</i>.
         * </p>
         * </important>
         * <p>
         * Default: <code>$Default</code>
         * </p>
         * <p>
         * Latest: <code>$Latest</code>
         * </p>
         * 
         * @param version
         *        The version number of the launch template, <code>$Default</code>, or <code>$Latest</code>.</p>
         *        <p>
         *        If the value is <code>$Default</code>, the default version of the launch template is used. If the
         *        value is <code>$Latest</code>, the latest version of the launch template is used.
         *        </p>
         *        <important>
         *        <p>
         *        If the AMI ID that's used in a compute environment is from the launch template, the AMI isn't changed
         *        when the compute environment is updated. It's only changed if the
         *        <code>updateToLatestImageVersion</code> parameter for the compute environment is set to
         *        <code>true</code>. During an infrastructure update, if either <code>$Default</code> or
         *        <code>$Latest</code> is specified, Batch re-evaluates the launch template version, and it might use a
         *        different version of the launch template. This is the case even if the launch template isn't specified
         *        in the update. When updating a compute environment, changing the launch template requires an
         *        infrastructure update of the compute environment. For more information, see <a
         *        href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">Updating
         *        compute environments</a> in the <i>Batch User Guide</i>.
         *        </p>
         *        </important>
         *        <p>
         *        Default: <code>$Default</code>
         *        </p>
         *        <p>
         *        Latest: <code>$Latest</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder version(String version);

        /**
         * <p>
         * A launch template to use in place of the default launch template. You must specify either the launch template
         * ID or launch template name in the request, but not both.
         * </p>
         * <p>
         * You can specify up to ten (10) launch template overrides that are associated to unique instance types or
         * families for each compute environment.
         * </p>
         * <note>
         * <p>
         * To unset all override templates for a compute environment, you can pass an empty array to the <a
         * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
         * >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter when
         * submitting the <code>UpdateComputeEnvironment</code> API operation.
         * </p>
         * </note>
         * 
         * @param overrides
         *        A launch template to use in place of the default launch template. You must specify either the launch
         *        template ID or launch template name in the request, but not both.</p>
         *        <p>
         *        You can specify up to ten (10) launch template overrides that are associated to unique instance types
         *        or families for each compute environment.
         *        </p>
         *        <note>
         *        <p>
         *        To unset all override templates for a compute environment, you can pass an empty array to the <a
         *        href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
         *        >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter
         *        when submitting the <code>UpdateComputeEnvironment</code> API operation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(Collection<LaunchTemplateSpecificationOverride> overrides);

        /**
         * <p>
         * A launch template to use in place of the default launch template. You must specify either the launch template
         * ID or launch template name in the request, but not both.
         * </p>
         * <p>
         * You can specify up to ten (10) launch template overrides that are associated to unique instance types or
         * families for each compute environment.
         * </p>
         * <note>
         * <p>
         * To unset all override templates for a compute environment, you can pass an empty array to the <a
         * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
         * >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter when
         * submitting the <code>UpdateComputeEnvironment</code> API operation.
         * </p>
         * </note>
         * 
         * @param overrides
         *        A launch template to use in place of the default launch template. You must specify either the launch
         *        template ID or launch template name in the request, but not both.</p>
         *        <p>
         *        You can specify up to ten (10) launch template overrides that are associated to unique instance types
         *        or families for each compute environment.
         *        </p>
         *        <note>
         *        <p>
         *        To unset all override templates for a compute environment, you can pass an empty array to the <a
         *        href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
         *        >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter
         *        when submitting the <code>UpdateComputeEnvironment</code> API operation.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder overrides(LaunchTemplateSpecificationOverride... overrides);

        /**
         * <p>
         * A launch template to use in place of the default launch template. You must specify either the launch template
         * ID or launch template name in the request, but not both.
         * </p>
         * <p>
         * You can specify up to ten (10) launch template overrides that are associated to unique instance types or
         * families for each compute environment.
         * </p>
         * <note>
         * <p>
         * To unset all override templates for a compute environment, you can pass an empty array to the <a
         * href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html"
         * >UpdateComputeEnvironment.overrides</a> parameter, or not include the <code>overrides</code> parameter when
         * submitting the <code>UpdateComputeEnvironment</code> API operation.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.LaunchTemplateSpecificationOverride.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.batch.model.LaunchTemplateSpecificationOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.LaunchTemplateSpecificationOverride.Builder#build()} is
         * called immediately and its result is passed to {@link #overrides(List<LaunchTemplateSpecificationOverride>)}.
         * 
         * @param overrides
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.LaunchTemplateSpecificationOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #overrides(java.util.Collection<LaunchTemplateSpecificationOverride>)
         */
        Builder overrides(Consumer<LaunchTemplateSpecificationOverride.Builder>... overrides);

        /**
         * <p>
         * The EKS node initialization process to use. You only need to specify this value if you are using a custom
         * AMI. The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI based on
         * EKS_AL2023 or EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
         * </p>
         * 
         * @param userdataType
         *        The EKS node initialization process to use. You only need to specify this value if you are using a
         *        custom AMI. The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI
         *        based on EKS_AL2023 or EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
         * @see UserdataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserdataType
         */
        Builder userdataType(String userdataType);

        /**
         * <p>
         * The EKS node initialization process to use. You only need to specify this value if you are using a custom
         * AMI. The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI based on
         * EKS_AL2023 or EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
         * </p>
         * 
         * @param userdataType
         *        The EKS node initialization process to use. You only need to specify this value if you are using a
         *        custom AMI. The default value is <code>EKS_BOOTSTRAP_SH</code>. If <i>imageType</i> is a custom AMI
         *        based on EKS_AL2023 or EKS_AL2023_NVIDIA then you must choose <code>EKS_NODEADM</code>.
         * @see UserdataType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see UserdataType
         */
        Builder userdataType(UserdataType userdataType);
    }

    static final class BuilderImpl implements Builder {
        private String launchTemplateId;

        private String launchTemplateName;

        private String version;

        private List<LaunchTemplateSpecificationOverride> overrides = DefaultSdkAutoConstructList.getInstance();

        private String userdataType;

        private BuilderImpl() {
        }

        private BuilderImpl(LaunchTemplateSpecification model) {
            launchTemplateId(model.launchTemplateId);
            launchTemplateName(model.launchTemplateName);
            version(model.version);
            overrides(model.overrides);
            userdataType(model.userdataType);
        }

        public final String getLaunchTemplateId() {
            return launchTemplateId;
        }

        public final void setLaunchTemplateId(String launchTemplateId) {
            this.launchTemplateId = launchTemplateId;
        }

        @Override
        public final Builder launchTemplateId(String launchTemplateId) {
            this.launchTemplateId = launchTemplateId;
            return this;
        }

        public final String getLaunchTemplateName() {
            return launchTemplateName;
        }

        public final void setLaunchTemplateName(String launchTemplateName) {
            this.launchTemplateName = launchTemplateName;
        }

        @Override
        public final Builder launchTemplateName(String launchTemplateName) {
            this.launchTemplateName = launchTemplateName;
            return this;
        }

        public final String getVersion() {
            return version;
        }

        public final void setVersion(String version) {
            this.version = version;
        }

        @Override
        public final Builder version(String version) {
            this.version = version;
            return this;
        }

        public final List<LaunchTemplateSpecificationOverride.Builder> getOverrides() {
            List<LaunchTemplateSpecificationOverride.Builder> result = LaunchTemplateSpecificationOverrideListCopier
                    .copyToBuilder(this.overrides);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setOverrides(Collection<LaunchTemplateSpecificationOverride.BuilderImpl> overrides) {
            this.overrides = LaunchTemplateSpecificationOverrideListCopier.copyFromBuilder(overrides);
        }

        @Override
        public final Builder overrides(Collection<LaunchTemplateSpecificationOverride> overrides) {
            this.overrides = LaunchTemplateSpecificationOverrideListCopier.copy(overrides);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder overrides(LaunchTemplateSpecificationOverride... overrides) {
            overrides(Arrays.asList(overrides));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder overrides(Consumer<LaunchTemplateSpecificationOverride.Builder>... overrides) {
            overrides(Stream.of(overrides).map(c -> LaunchTemplateSpecificationOverride.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getUserdataType() {
            return userdataType;
        }

        public final void setUserdataType(String userdataType) {
            this.userdataType = userdataType;
        }

        @Override
        public final Builder userdataType(String userdataType) {
            this.userdataType = userdataType;
            return this;
        }

        @Override
        public final Builder userdataType(UserdataType userdataType) {
            this.userdataType(userdataType == null ? null : userdataType.toString());
            return this;
        }

        @Override
        public LaunchTemplateSpecification build() {
            return new LaunchTemplateSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
