/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies an action that Batch will take after the job has remained at the head of the queue in the specified state
 * for longer than the specified time.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JobStateTimeLimitAction implements SdkPojo, Serializable,
        ToCopyableBuilder<JobStateTimeLimitAction.Builder, JobStateTimeLimitAction> {
    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("reason")
            .getter(getter(JobStateTimeLimitAction::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("state")
            .getter(getter(JobStateTimeLimitAction::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("state").build()).build();

    private static final SdkField<Integer> MAX_TIME_SECONDS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxTimeSeconds").getter(getter(JobStateTimeLimitAction::maxTimeSeconds))
            .setter(setter(Builder::maxTimeSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxTimeSeconds").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("action")
            .getter(getter(JobStateTimeLimitAction::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("action").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REASON_FIELD, STATE_FIELD,
            MAX_TIME_SECONDS_FIELD, ACTION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String reason;

    private final String state;

    private final Integer maxTimeSeconds;

    private final String action;

    private JobStateTimeLimitAction(BuilderImpl builder) {
        this.reason = builder.reason;
        this.state = builder.state;
        this.maxTimeSeconds = builder.maxTimeSeconds;
        this.action = builder.action;
    }

    /**
     * <p>
     * The reason to log for the action being taken.
     * </p>
     * 
     * @return The reason to log for the action being taken.
     */
    public final String reason() {
        return reason;
    }

    /**
     * <p>
     * The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobStateTimeLimitActionsState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
     * @see JobStateTimeLimitActionsState
     */
    public final JobStateTimeLimitActionsState state() {
        return JobStateTimeLimitActionsState.fromValue(state);
    }

    /**
     * <p>
     * The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link JobStateTimeLimitActionsState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
     * @see JobStateTimeLimitActionsState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The approximate amount of time, in seconds, that must pass with the job in the specified state before the action
     * is taken. The minimum value is 600 (10 minutes) and the maximum value is 86,400 (24 hours).
     * </p>
     * 
     * @return The approximate amount of time, in seconds, that must pass with the job in the specified state before the
     *         action is taken. The minimum value is 600 (10 minutes) and the maximum value is 86,400 (24 hours).
     */
    public final Integer maxTimeSeconds() {
        return maxTimeSeconds;
    }

    /**
     * <p>
     * The action to take when a job is at the head of the job queue in the specified state for the specified period of
     * time. The only supported value is <code>CANCEL</code>, which will cancel the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link JobStateTimeLimitActionsAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #actionAsString}.
     * </p>
     * 
     * @return The action to take when a job is at the head of the job queue in the specified state for the specified
     *         period of time. The only supported value is <code>CANCEL</code>, which will cancel the job.
     * @see JobStateTimeLimitActionsAction
     */
    public final JobStateTimeLimitActionsAction action() {
        return JobStateTimeLimitActionsAction.fromValue(action);
    }

    /**
     * <p>
     * The action to take when a job is at the head of the job queue in the specified state for the specified period of
     * time. The only supported value is <code>CANCEL</code>, which will cancel the job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link JobStateTimeLimitActionsAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #actionAsString}.
     * </p>
     * 
     * @return The action to take when a job is at the head of the job queue in the specified state for the specified
     *         period of time. The only supported value is <code>CANCEL</code>, which will cancel the job.
     * @see JobStateTimeLimitActionsAction
     */
    public final String actionAsString() {
        return action;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxTimeSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JobStateTimeLimitAction)) {
            return false;
        }
        JobStateTimeLimitAction other = (JobStateTimeLimitAction) obj;
        return Objects.equals(reason(), other.reason()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(maxTimeSeconds(), other.maxTimeSeconds())
                && Objects.equals(actionAsString(), other.actionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("JobStateTimeLimitAction").add("Reason", reason()).add("State", stateAsString())
                .add("MaxTimeSeconds", maxTimeSeconds()).add("Action", actionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        case "state":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "maxTimeSeconds":
            return Optional.ofNullable(clazz.cast(maxTimeSeconds()));
        case "action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("reason", REASON_FIELD);
        map.put("state", STATE_FIELD);
        map.put("maxTimeSeconds", MAX_TIME_SECONDS_FIELD);
        map.put("action", ACTION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<JobStateTimeLimitAction, T> g) {
        return obj -> g.apply((JobStateTimeLimitAction) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JobStateTimeLimitAction> {
        /**
         * <p>
         * The reason to log for the action being taken.
         * </p>
         * 
         * @param reason
         *        The reason to log for the action being taken.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);

        /**
         * <p>
         * The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
         * </p>
         * 
         * @param state
         *        The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
         * @see JobStateTimeLimitActionsState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStateTimeLimitActionsState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
         * </p>
         * 
         * @param state
         *        The state of the job needed to trigger the action. The only supported value is <code>RUNNABLE</code>.
         * @see JobStateTimeLimitActionsState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStateTimeLimitActionsState
         */
        Builder state(JobStateTimeLimitActionsState state);

        /**
         * <p>
         * The approximate amount of time, in seconds, that must pass with the job in the specified state before the
         * action is taken. The minimum value is 600 (10 minutes) and the maximum value is 86,400 (24 hours).
         * </p>
         * 
         * @param maxTimeSeconds
         *        The approximate amount of time, in seconds, that must pass with the job in the specified state before
         *        the action is taken. The minimum value is 600 (10 minutes) and the maximum value is 86,400 (24 hours).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTimeSeconds(Integer maxTimeSeconds);

        /**
         * <p>
         * The action to take when a job is at the head of the job queue in the specified state for the specified period
         * of time. The only supported value is <code>CANCEL</code>, which will cancel the job.
         * </p>
         * 
         * @param action
         *        The action to take when a job is at the head of the job queue in the specified state for the specified
         *        period of time. The only supported value is <code>CANCEL</code>, which will cancel the job.
         * @see JobStateTimeLimitActionsAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStateTimeLimitActionsAction
         */
        Builder action(String action);

        /**
         * <p>
         * The action to take when a job is at the head of the job queue in the specified state for the specified period
         * of time. The only supported value is <code>CANCEL</code>, which will cancel the job.
         * </p>
         * 
         * @param action
         *        The action to take when a job is at the head of the job queue in the specified state for the specified
         *        period of time. The only supported value is <code>CANCEL</code>, which will cancel the job.
         * @see JobStateTimeLimitActionsAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see JobStateTimeLimitActionsAction
         */
        Builder action(JobStateTimeLimitActionsAction action);
    }

    static final class BuilderImpl implements Builder {
        private String reason;

        private String state;

        private Integer maxTimeSeconds;

        private String action;

        private BuilderImpl() {
        }

        private BuilderImpl(JobStateTimeLimitAction model) {
            reason(model.reason);
            state(model.state);
            maxTimeSeconds(model.maxTimeSeconds);
            action(model.action);
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(JobStateTimeLimitActionsState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Integer getMaxTimeSeconds() {
            return maxTimeSeconds;
        }

        public final void setMaxTimeSeconds(Integer maxTimeSeconds) {
            this.maxTimeSeconds = maxTimeSeconds;
        }

        @Override
        public final Builder maxTimeSeconds(Integer maxTimeSeconds) {
            this.maxTimeSeconds = maxTimeSeconds;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        public final Builder action(JobStateTimeLimitActionsAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        @Override
        public JobStateTimeLimitAction build() {
            return new JobStateTimeLimitAction(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
