/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.batch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * EKS container properties are used in job definitions for Amazon EKS based job definitions to describe the properties
 * for a container node in the pod that's launched as part of a job. This can't be specified for Amazon ECS based job
 * definitions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EksContainer implements SdkPojo, Serializable, ToCopyableBuilder<EksContainer.Builder, EksContainer> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(EksContainer::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> IMAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("image")
            .getter(getter(EksContainer::image)).setter(setter(Builder::image))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("image").build()).build();

    private static final SdkField<String> IMAGE_PULL_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("imagePullPolicy").getter(getter(EksContainer::imagePullPolicy)).setter(setter(Builder::imagePullPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("imagePullPolicy").build()).build();

    private static final SdkField<List<String>> COMMAND_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("command")
            .getter(getter(EksContainer::command))
            .setter(setter(Builder::command))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("command").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> ARGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("args")
            .getter(getter(EksContainer::args))
            .setter(setter(Builder::args))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("args").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<EksContainerEnvironmentVariable>> ENV_FIELD = SdkField
            .<List<EksContainerEnvironmentVariable>> builder(MarshallingType.LIST)
            .memberName("env")
            .getter(getter(EksContainer::env))
            .setter(setter(Builder::env))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("env").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EksContainerEnvironmentVariable> builder(MarshallingType.SDK_POJO)
                                            .constructor(EksContainerEnvironmentVariable::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EksContainerResourceRequirements> RESOURCES_FIELD = SdkField
            .<EksContainerResourceRequirements> builder(MarshallingType.SDK_POJO).memberName("resources")
            .getter(getter(EksContainer::resources)).setter(setter(Builder::resources))
            .constructor(EksContainerResourceRequirements::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("resources").build()).build();

    private static final SdkField<List<EksContainerVolumeMount>> VOLUME_MOUNTS_FIELD = SdkField
            .<List<EksContainerVolumeMount>> builder(MarshallingType.LIST)
            .memberName("volumeMounts")
            .getter(getter(EksContainer::volumeMounts))
            .setter(setter(Builder::volumeMounts))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeMounts").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EksContainerVolumeMount> builder(MarshallingType.SDK_POJO)
                                            .constructor(EksContainerVolumeMount::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<EksContainerSecurityContext> SECURITY_CONTEXT_FIELD = SdkField
            .<EksContainerSecurityContext> builder(MarshallingType.SDK_POJO).memberName("securityContext")
            .getter(getter(EksContainer::securityContext)).setter(setter(Builder::securityContext))
            .constructor(EksContainerSecurityContext::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("securityContext").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, IMAGE_FIELD,
            IMAGE_PULL_POLICY_FIELD, COMMAND_FIELD, ARGS_FIELD, ENV_FIELD, RESOURCES_FIELD, VOLUME_MOUNTS_FIELD,
            SECURITY_CONTEXT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String image;

    private final String imagePullPolicy;

    private final List<String> command;

    private final List<String> args;

    private final List<EksContainerEnvironmentVariable> env;

    private final EksContainerResourceRequirements resources;

    private final List<EksContainerVolumeMount> volumeMounts;

    private final EksContainerSecurityContext securityContext;

    private EksContainer(BuilderImpl builder) {
        this.name = builder.name;
        this.image = builder.image;
        this.imagePullPolicy = builder.imagePullPolicy;
        this.command = builder.command;
        this.args = builder.args;
        this.env = builder.env;
        this.resources = builder.resources;
        this.volumeMounts = builder.volumeMounts;
        this.securityContext = builder.securityContext;
    }

    /**
     * <p>
     * The name of the container. If the name isn't specified, the default name "<code>Default</code>" is used. Each
     * container in a pod must have a unique name.
     * </p>
     * 
     * @return The name of the container. If the name isn't specified, the default name "<code>Default</code>" is used.
     *         Each container in a pod must have a unique name.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The Docker image used to start the container.
     * </p>
     * 
     * @return The Docker image used to start the container.
     */
    public final String image() {
        return image;
    }

    /**
     * <p>
     * The image pull policy for the container. Supported values are <code>Always</code>, <code>IfNotPresent</code>, and
     * <code>Never</code>. This parameter defaults to <code>IfNotPresent</code>. However, if the <code>:latest</code>
     * tag is specified, it defaults to <code>Always</code>. For more information, see <a
     * href="https://kubernetes.io/docs/concepts/containers/images/#updating-images">Updating images</a> in the
     * <i>Kubernetes documentation</i>.
     * </p>
     * 
     * @return The image pull policy for the container. Supported values are <code>Always</code>,
     *         <code>IfNotPresent</code>, and <code>Never</code>. This parameter defaults to <code>IfNotPresent</code>.
     *         However, if the <code>:latest</code> tag is specified, it defaults to <code>Always</code>. For more
     *         information, see <a
     *         href="https://kubernetes.io/docs/concepts/containers/images/#updating-images">Updating images</a> in the
     *         <i>Kubernetes documentation</i>.
     */
    public final String imagePullPolicy() {
        return imagePullPolicy;
    }

    /**
     * For responses, this returns true if the service returned a value for the Command property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasCommand() {
        return command != null && !(command instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The entrypoint for the container. This isn't run within a shell. If this isn't specified, the
     * <code>ENTRYPOINT</code> of the container image is used. Environment variable references are expanded using the
     * container's environment.
     * </p>
     * <p>
     * If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if
     * the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable doesn't exist, the
     * command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with <code>$</code> and the
     * resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> will be passed as
     * <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists. The entrypoint
     * can't be updated. For more information, see <a
     * href="https://docs.docker.com/engine/reference/builder/#entrypoint">ENTRYPOINT</a> in the <i>Dockerfile
     * reference</i> and <a
     * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
     * command and arguments for a container</a> and <a
     * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint">Entrypoint</a>
     * in the <i>Kubernetes documentation</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCommand} method.
     * </p>
     * 
     * @return The entrypoint for the container. This isn't run within a shell. If this isn't specified, the
     *         <code>ENTRYPOINT</code> of the container image is used. Environment variable references are expanded
     *         using the container's environment.</p>
     *         <p>
     *         If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
     *         example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable
     *         doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with
     *         <code>$</code> and the resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> will be
     *         passed as <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists.
     *         The entrypoint can't be updated. For more information, see <a
     *         href="https://docs.docker.com/engine/reference/builder/#entrypoint">ENTRYPOINT</a> in the <i>Dockerfile
     *         reference</i> and <a
     *         href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define
     *         a command and arguments for a container</a> and <a
     *         href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
     *         >Entrypoint</a> in the <i>Kubernetes documentation</i>.
     */
    public final List<String> command() {
        return command;
    }

    /**
     * For responses, this returns true if the service returned a value for the Args property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasArgs() {
        return args != null && !(args instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of arguments to the entrypoint. If this isn't specified, the <code>CMD</code> of the container image is
     * used. This corresponds to the <code>args</code> member in the <a
     * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint">Entrypoint</a>
     * portion of the <a href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/">Pod</a>
     * in Kubernetes. Environment variable references are expanded using the container's environment.
     * </p>
     * <p>
     * If the referenced environment variable doesn't exist, the reference in the command isn't changed. For example, if
     * the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable doesn't exist, the
     * command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with <code>$</code>, and the
     * resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> is passed as <code>$(VAR_NAME)</code>
     * whether or not the <code>VAR_NAME</code> environment variable exists. For more information, see <a
     * href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
     * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
     * command and arguments for a pod</a> in the <i>Kubernetes documentation</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasArgs} method.
     * </p>
     * 
     * @return An array of arguments to the entrypoint. If this isn't specified, the <code>CMD</code> of the container
     *         image is used. This corresponds to the <code>args</code> member in the <a
     *         href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
     *         >Entrypoint</a> portion of the <a
     *         href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/">Pod</a> in
     *         Kubernetes. Environment variable references are expanded using the container's environment.</p>
     *         <p>
     *         If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
     *         example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable
     *         doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with
     *         <code>$</code>, and the resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> is passed
     *         as <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists. For
     *         more information, see <a href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile
     *         reference: CMD</a> and <a
     *         href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define
     *         a command and arguments for a pod</a> in the <i>Kubernetes documentation</i>.
     */
    public final List<String> args() {
        return args;
    }

    /**
     * For responses, this returns true if the service returned a value for the Env property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasEnv() {
        return env != null && !(env instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The environment variables to pass to a container.
     * </p>
     * <note>
     * <p>
     * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
     * variables that Batch sets.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEnv} method.
     * </p>
     * 
     * @return The environment variables to pass to a container.</p> <note>
     *         <p>
     *         Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
     *         variables that Batch sets.
     *         </p>
     */
    public final List<EksContainerEnvironmentVariable> env() {
        return env;
    }

    /**
     * <p>
     * The type and amount of resources to assign to a container. The supported resources include <code>memory</code>,
     * <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more information, see <a
     * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management for
     * pods and containers</a> in the <i>Kubernetes documentation</i>.
     * </p>
     * 
     * @return The type and amount of resources to assign to a container. The supported resources include
     *         <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more information, see <a
     *         href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management
     *         for pods and containers</a> in the <i>Kubernetes documentation</i>.
     */
    public final EksContainerResourceRequirements resources() {
        return resources;
    }

    /**
     * For responses, this returns true if the service returned a value for the VolumeMounts property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVolumeMounts() {
        return volumeMounts != null && !(volumeMounts instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
     * <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes, see <a
     * href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes documentation</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVolumeMounts} method.
     * </p>
     * 
     * @return The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
     *         <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes, see
     *         <a href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes
     *         documentation</i>.
     */
    public final List<EksContainerVolumeMount> volumeMounts() {
        return volumeMounts;
    }

    /**
     * <p>
     * The security context for a job. For more information, see <a
     * href="https://kubernetes.io/docs/tasks/configure-pod-container/security-context/">Configure a security context
     * for a pod or container</a> in the <i>Kubernetes documentation</i>.
     * </p>
     * 
     * @return The security context for a job. For more information, see <a
     *         href="https://kubernetes.io/docs/tasks/configure-pod-container/security-context/">Configure a security
     *         context for a pod or container</a> in the <i>Kubernetes documentation</i>.
     */
    public final EksContainerSecurityContext securityContext() {
        return securityContext;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(image());
        hashCode = 31 * hashCode + Objects.hashCode(imagePullPolicy());
        hashCode = 31 * hashCode + Objects.hashCode(hasCommand() ? command() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasArgs() ? args() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasEnv() ? env() : null);
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumeMounts() ? volumeMounts() : null);
        hashCode = 31 * hashCode + Objects.hashCode(securityContext());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EksContainer)) {
            return false;
        }
        EksContainer other = (EksContainer) obj;
        return Objects.equals(name(), other.name()) && Objects.equals(image(), other.image())
                && Objects.equals(imagePullPolicy(), other.imagePullPolicy()) && hasCommand() == other.hasCommand()
                && Objects.equals(command(), other.command()) && hasArgs() == other.hasArgs()
                && Objects.equals(args(), other.args()) && hasEnv() == other.hasEnv() && Objects.equals(env(), other.env())
                && Objects.equals(resources(), other.resources()) && hasVolumeMounts() == other.hasVolumeMounts()
                && Objects.equals(volumeMounts(), other.volumeMounts())
                && Objects.equals(securityContext(), other.securityContext());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EksContainer").add("Name", name()).add("Image", image())
                .add("ImagePullPolicy", imagePullPolicy()).add("Command", hasCommand() ? command() : null)
                .add("Args", hasArgs() ? args() : null).add("Env", hasEnv() ? env() : null).add("Resources", resources())
                .add("VolumeMounts", hasVolumeMounts() ? volumeMounts() : null).add("SecurityContext", securityContext()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(name()));
        case "image":
            return Optional.ofNullable(clazz.cast(image()));
        case "imagePullPolicy":
            return Optional.ofNullable(clazz.cast(imagePullPolicy()));
        case "command":
            return Optional.ofNullable(clazz.cast(command()));
        case "args":
            return Optional.ofNullable(clazz.cast(args()));
        case "env":
            return Optional.ofNullable(clazz.cast(env()));
        case "resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "volumeMounts":
            return Optional.ofNullable(clazz.cast(volumeMounts()));
        case "securityContext":
            return Optional.ofNullable(clazz.cast(securityContext()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("name", NAME_FIELD);
        map.put("image", IMAGE_FIELD);
        map.put("imagePullPolicy", IMAGE_PULL_POLICY_FIELD);
        map.put("command", COMMAND_FIELD);
        map.put("args", ARGS_FIELD);
        map.put("env", ENV_FIELD);
        map.put("resources", RESOURCES_FIELD);
        map.put("volumeMounts", VOLUME_MOUNTS_FIELD);
        map.put("securityContext", SECURITY_CONTEXT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EksContainer, T> g) {
        return obj -> g.apply((EksContainer) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EksContainer> {
        /**
         * <p>
         * The name of the container. If the name isn't specified, the default name "<code>Default</code>" is used. Each
         * container in a pod must have a unique name.
         * </p>
         * 
         * @param name
         *        The name of the container. If the name isn't specified, the default name "<code>Default</code>" is
         *        used. Each container in a pod must have a unique name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The Docker image used to start the container.
         * </p>
         * 
         * @param image
         *        The Docker image used to start the container.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder image(String image);

        /**
         * <p>
         * The image pull policy for the container. Supported values are <code>Always</code>, <code>IfNotPresent</code>,
         * and <code>Never</code>. This parameter defaults to <code>IfNotPresent</code>. However, if the
         * <code>:latest</code> tag is specified, it defaults to <code>Always</code>. For more information, see <a
         * href="https://kubernetes.io/docs/concepts/containers/images/#updating-images">Updating images</a> in the
         * <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param imagePullPolicy
         *        The image pull policy for the container. Supported values are <code>Always</code>,
         *        <code>IfNotPresent</code>, and <code>Never</code>. This parameter defaults to
         *        <code>IfNotPresent</code>. However, if the <code>:latest</code> tag is specified, it defaults to
         *        <code>Always</code>. For more information, see <a
         *        href="https://kubernetes.io/docs/concepts/containers/images/#updating-images">Updating images</a> in
         *        the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imagePullPolicy(String imagePullPolicy);

        /**
         * <p>
         * The entrypoint for the container. This isn't run within a shell. If this isn't specified, the
         * <code>ENTRYPOINT</code> of the container image is used. Environment variable references are expanded using
         * the container's environment.
         * </p>
         * <p>
         * If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         * example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable
         * doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with
         * <code>$</code> and the resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> will be passed
         * as <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists. The
         * entrypoint can't be updated. For more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#entrypoint">ENTRYPOINT</a> in the <i>Dockerfile
         * reference</i> and <a
         * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         * command and arguments for a container</a> and <a
         * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         * >Entrypoint</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param command
         *        The entrypoint for the container. This isn't run within a shell. If this isn't specified, the
         *        <code>ENTRYPOINT</code> of the container image is used. Environment variable references are expanded
         *        using the container's environment.</p>
         *        <p>
         *        If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         *        example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment
         *        variable doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is
         *        replaced with <code>$</code> and the resulting string isn't expanded. For example,
         *        <code>$$(VAR_NAME)</code> will be passed as <code>$(VAR_NAME)</code> whether or not the
         *        <code>VAR_NAME</code> environment variable exists. The entrypoint can't be updated. For more
         *        information, see <a href="https://docs.docker.com/engine/reference/builder/#entrypoint">ENTRYPOINT</a>
         *        in the <i>Dockerfile reference</i> and <a
         *        href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/"
         *        >Define a command and arguments for a container</a> and <a
         *        href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         *        >Entrypoint</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(Collection<String> command);

        /**
         * <p>
         * The entrypoint for the container. This isn't run within a shell. If this isn't specified, the
         * <code>ENTRYPOINT</code> of the container image is used. Environment variable references are expanded using
         * the container's environment.
         * </p>
         * <p>
         * If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         * example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable
         * doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with
         * <code>$</code> and the resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> will be passed
         * as <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists. The
         * entrypoint can't be updated. For more information, see <a
         * href="https://docs.docker.com/engine/reference/builder/#entrypoint">ENTRYPOINT</a> in the <i>Dockerfile
         * reference</i> and <a
         * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         * command and arguments for a container</a> and <a
         * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         * >Entrypoint</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param command
         *        The entrypoint for the container. This isn't run within a shell. If this isn't specified, the
         *        <code>ENTRYPOINT</code> of the container image is used. Environment variable references are expanded
         *        using the container's environment.</p>
         *        <p>
         *        If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         *        example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment
         *        variable doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is
         *        replaced with <code>$</code> and the resulting string isn't expanded. For example,
         *        <code>$$(VAR_NAME)</code> will be passed as <code>$(VAR_NAME)</code> whether or not the
         *        <code>VAR_NAME</code> environment variable exists. The entrypoint can't be updated. For more
         *        information, see <a href="https://docs.docker.com/engine/reference/builder/#entrypoint">ENTRYPOINT</a>
         *        in the <i>Dockerfile reference</i> and <a
         *        href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/"
         *        >Define a command and arguments for a container</a> and <a
         *        href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         *        >Entrypoint</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder command(String... command);

        /**
         * <p>
         * An array of arguments to the entrypoint. If this isn't specified, the <code>CMD</code> of the container image
         * is used. This corresponds to the <code>args</code> member in the <a
         * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         * >Entrypoint</a> portion of the <a
         * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/">Pod</a> in Kubernetes.
         * Environment variable references are expanded using the container's environment.
         * </p>
         * <p>
         * If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         * example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable
         * doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with
         * <code>$</code>, and the resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> is passed as
         * <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists. For more
         * information, see <a href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference:
         * CMD</a> and <a
         * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         * command and arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param args
         *        An array of arguments to the entrypoint. If this isn't specified, the <code>CMD</code> of the
         *        container image is used. This corresponds to the <code>args</code> member in the <a
         *        href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         *        >Entrypoint</a> portion of the <a
         *        href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/">Pod</a> in
         *        Kubernetes. Environment variable references are expanded using the container's environment.</p>
         *        <p>
         *        If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         *        example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment
         *        variable doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is
         *        replaced with <code>$</code>, and the resulting string isn't expanded. For example,
         *        <code>$$(VAR_NAME)</code> is passed as <code>$(VAR_NAME)</code> whether or not the
         *        <code>VAR_NAME</code> environment variable exists. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
         *        href=
         *        "https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         *        command and arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(Collection<String> args);

        /**
         * <p>
         * An array of arguments to the entrypoint. If this isn't specified, the <code>CMD</code> of the container image
         * is used. This corresponds to the <code>args</code> member in the <a
         * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         * >Entrypoint</a> portion of the <a
         * href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/">Pod</a> in Kubernetes.
         * Environment variable references are expanded using the container's environment.
         * </p>
         * <p>
         * If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         * example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment variable
         * doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is replaced with
         * <code>$</code>, and the resulting string isn't expanded. For example, <code>$$(VAR_NAME)</code> is passed as
         * <code>$(VAR_NAME)</code> whether or not the <code>VAR_NAME</code> environment variable exists. For more
         * information, see <a href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference:
         * CMD</a> and <a
         * href="https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         * command and arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param args
         *        An array of arguments to the entrypoint. If this isn't specified, the <code>CMD</code> of the
         *        container image is used. This corresponds to the <code>args</code> member in the <a
         *        href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/#entrypoint"
         *        >Entrypoint</a> portion of the <a
         *        href="https://kubernetes.io/docs/reference/kubernetes-api/workload-resources/pod-v1/">Pod</a> in
         *        Kubernetes. Environment variable references are expanded using the container's environment.</p>
         *        <p>
         *        If the referenced environment variable doesn't exist, the reference in the command isn't changed. For
         *        example, if the reference is to "<code>$(NAME1)</code>" and the <code>NAME1</code> environment
         *        variable doesn't exist, the command string will remain "<code>$(NAME1)</code>." <code>$$</code> is
         *        replaced with <code>$</code>, and the resulting string isn't expanded. For example,
         *        <code>$$(VAR_NAME)</code> is passed as <code>$(VAR_NAME)</code> whether or not the
         *        <code>VAR_NAME</code> environment variable exists. For more information, see <a
         *        href="https://docs.docker.com/engine/reference/builder/#cmd">Dockerfile reference: CMD</a> and <a
         *        href=
         *        "https://kubernetes.io/docs/tasks/inject-data-application/define-command-argument-container/">Define a
         *        command and arguments for a pod</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder args(String... args);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param env
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved
         *        for variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder env(Collection<EksContainerEnvironmentVariable> env);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note>
         * 
         * @param env
         *        The environment variables to pass to a container.</p> <note>
         *        <p>
         *        Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved
         *        for variables that Batch sets.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder env(EksContainerEnvironmentVariable... env);

        /**
         * <p>
         * The environment variables to pass to a container.
         * </p>
         * <note>
         * <p>
         * Environment variables cannot start with "<code>AWS_BATCH</code>". This naming convention is reserved for
         * variables that Batch sets.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable.Builder#build()} is called
         * immediately and its result is passed to {@link #env(List<EksContainerEnvironmentVariable>)}.
         * 
         * @param env
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.EksContainerEnvironmentVariable.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #env(java.util.Collection<EksContainerEnvironmentVariable>)
         */
        Builder env(Consumer<EksContainerEnvironmentVariable.Builder>... env);

        /**
         * <p>
         * The type and amount of resources to assign to a container. The supported resources include
         * <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more information, see <a
         * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management for
         * pods and containers</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param resources
         *        The type and amount of resources to assign to a container. The supported resources include
         *        <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more information, see <a
         *        href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource
         *        management for pods and containers</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(EksContainerResourceRequirements resources);

        /**
         * <p>
         * The type and amount of resources to assign to a container. The supported resources include
         * <code>memory</code>, <code>cpu</code>, and <code>nvidia.com/gpu</code>. For more information, see <a
         * href="https://kubernetes.io/docs/concepts/configuration/manage-resources-containers/">Resource management for
         * pods and containers</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksContainerResourceRequirements.Builder}
         * avoiding the need to create one manually via {@link EksContainerResourceRequirements#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksContainerResourceRequirements.Builder#build()} is called
         * immediately and its result is passed to {@link #resources(EksContainerResourceRequirements)}.
         * 
         * @param resources
         *        a consumer that will call methods on {@link EksContainerResourceRequirements.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resources(EksContainerResourceRequirements)
         */
        default Builder resources(Consumer<EksContainerResourceRequirements.Builder> resources) {
            return resources(EksContainerResourceRequirements.builder().applyMutation(resources).build());
        }

        /**
         * <p>
         * The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
         * <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes, see <a
         * href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes
         * documentation</i>.
         * </p>
         * 
         * @param volumeMounts
         *        The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
         *        <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes,
         *        see <a href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes
         *        documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeMounts(Collection<EksContainerVolumeMount> volumeMounts);

        /**
         * <p>
         * The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
         * <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes, see <a
         * href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes
         * documentation</i>.
         * </p>
         * 
         * @param volumeMounts
         *        The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
         *        <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes,
         *        see <a href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes
         *        documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeMounts(EksContainerVolumeMount... volumeMounts);

        /**
         * <p>
         * The volume mounts for the container. Batch supports <code>emptyDir</code>, <code>hostPath</code>, and
         * <code>secret</code> volume types. For more information about volumes and volume mounts in Kubernetes, see <a
         * href="https://kubernetes.io/docs/concepts/storage/volumes/">Volumes</a> in the <i>Kubernetes
         * documentation</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.batch.model.EksContainerVolumeMount.Builder} avoiding the need to
         * create one manually via {@link software.amazon.awssdk.services.batch.model.EksContainerVolumeMount#builder()}
         * .
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.batch.model.EksContainerVolumeMount.Builder#build()} is called
         * immediately and its result is passed to {@link #volumeMounts(List<EksContainerVolumeMount>)}.
         * 
         * @param volumeMounts
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.batch.model.EksContainerVolumeMount.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumeMounts(java.util.Collection<EksContainerVolumeMount>)
         */
        Builder volumeMounts(Consumer<EksContainerVolumeMount.Builder>... volumeMounts);

        /**
         * <p>
         * The security context for a job. For more information, see <a
         * href="https://kubernetes.io/docs/tasks/configure-pod-container/security-context/">Configure a security
         * context for a pod or container</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * 
         * @param securityContext
         *        The security context for a job. For more information, see <a
         *        href="https://kubernetes.io/docs/tasks/configure-pod-container/security-context/">Configure a security
         *        context for a pod or container</a> in the <i>Kubernetes documentation</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityContext(EksContainerSecurityContext securityContext);

        /**
         * <p>
         * The security context for a job. For more information, see <a
         * href="https://kubernetes.io/docs/tasks/configure-pod-container/security-context/">Configure a security
         * context for a pod or container</a> in the <i>Kubernetes documentation</i>.
         * </p>
         * This is a convenience method that creates an instance of the {@link EksContainerSecurityContext.Builder}
         * avoiding the need to create one manually via {@link EksContainerSecurityContext#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link EksContainerSecurityContext.Builder#build()} is called
         * immediately and its result is passed to {@link #securityContext(EksContainerSecurityContext)}.
         * 
         * @param securityContext
         *        a consumer that will call methods on {@link EksContainerSecurityContext.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #securityContext(EksContainerSecurityContext)
         */
        default Builder securityContext(Consumer<EksContainerSecurityContext.Builder> securityContext) {
            return securityContext(EksContainerSecurityContext.builder().applyMutation(securityContext).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String image;

        private String imagePullPolicy;

        private List<String> command = DefaultSdkAutoConstructList.getInstance();

        private List<String> args = DefaultSdkAutoConstructList.getInstance();

        private List<EksContainerEnvironmentVariable> env = DefaultSdkAutoConstructList.getInstance();

        private EksContainerResourceRequirements resources;

        private List<EksContainerVolumeMount> volumeMounts = DefaultSdkAutoConstructList.getInstance();

        private EksContainerSecurityContext securityContext;

        private BuilderImpl() {
        }

        private BuilderImpl(EksContainer model) {
            name(model.name);
            image(model.image);
            imagePullPolicy(model.imagePullPolicy);
            command(model.command);
            args(model.args);
            env(model.env);
            resources(model.resources);
            volumeMounts(model.volumeMounts);
            securityContext(model.securityContext);
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getImage() {
            return image;
        }

        public final void setImage(String image) {
            this.image = image;
        }

        @Override
        public final Builder image(String image) {
            this.image = image;
            return this;
        }

        public final String getImagePullPolicy() {
            return imagePullPolicy;
        }

        public final void setImagePullPolicy(String imagePullPolicy) {
            this.imagePullPolicy = imagePullPolicy;
        }

        @Override
        public final Builder imagePullPolicy(String imagePullPolicy) {
            this.imagePullPolicy = imagePullPolicy;
            return this;
        }

        public final Collection<String> getCommand() {
            if (command instanceof SdkAutoConstructList) {
                return null;
            }
            return command;
        }

        public final void setCommand(Collection<String> command) {
            this.command = StringListCopier.copy(command);
        }

        @Override
        public final Builder command(Collection<String> command) {
            this.command = StringListCopier.copy(command);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder command(String... command) {
            command(Arrays.asList(command));
            return this;
        }

        public final Collection<String> getArgs() {
            if (args instanceof SdkAutoConstructList) {
                return null;
            }
            return args;
        }

        public final void setArgs(Collection<String> args) {
            this.args = StringListCopier.copy(args);
        }

        @Override
        public final Builder args(Collection<String> args) {
            this.args = StringListCopier.copy(args);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder args(String... args) {
            args(Arrays.asList(args));
            return this;
        }

        public final List<EksContainerEnvironmentVariable.Builder> getEnv() {
            List<EksContainerEnvironmentVariable.Builder> result = EksContainerEnvironmentVariablesCopier.copyToBuilder(this.env);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEnv(Collection<EksContainerEnvironmentVariable.BuilderImpl> env) {
            this.env = EksContainerEnvironmentVariablesCopier.copyFromBuilder(env);
        }

        @Override
        public final Builder env(Collection<EksContainerEnvironmentVariable> env) {
            this.env = EksContainerEnvironmentVariablesCopier.copy(env);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder env(EksContainerEnvironmentVariable... env) {
            env(Arrays.asList(env));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder env(Consumer<EksContainerEnvironmentVariable.Builder>... env) {
            env(Stream.of(env).map(c -> EksContainerEnvironmentVariable.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final EksContainerResourceRequirements.Builder getResources() {
            return resources != null ? resources.toBuilder() : null;
        }

        public final void setResources(EksContainerResourceRequirements.BuilderImpl resources) {
            this.resources = resources != null ? resources.build() : null;
        }

        @Override
        public final Builder resources(EksContainerResourceRequirements resources) {
            this.resources = resources;
            return this;
        }

        public final List<EksContainerVolumeMount.Builder> getVolumeMounts() {
            List<EksContainerVolumeMount.Builder> result = EksContainerVolumeMountsCopier.copyToBuilder(this.volumeMounts);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVolumeMounts(Collection<EksContainerVolumeMount.BuilderImpl> volumeMounts) {
            this.volumeMounts = EksContainerVolumeMountsCopier.copyFromBuilder(volumeMounts);
        }

        @Override
        public final Builder volumeMounts(Collection<EksContainerVolumeMount> volumeMounts) {
            this.volumeMounts = EksContainerVolumeMountsCopier.copy(volumeMounts);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeMounts(EksContainerVolumeMount... volumeMounts) {
            volumeMounts(Arrays.asList(volumeMounts));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeMounts(Consumer<EksContainerVolumeMount.Builder>... volumeMounts) {
            volumeMounts(Stream.of(volumeMounts).map(c -> EksContainerVolumeMount.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final EksContainerSecurityContext.Builder getSecurityContext() {
            return securityContext != null ? securityContext.toBuilder() : null;
        }

        public final void setSecurityContext(EksContainerSecurityContext.BuilderImpl securityContext) {
            this.securityContext = securityContext != null ? securityContext.build() : null;
        }

        @Override
        public final Builder securityContext(EksContainerSecurityContext securityContext) {
            this.securityContext = securityContext;
            return this;
        }

        @Override
        public EksContainer build() {
            return new EksContainer(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
