/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains detailed information about a backup job.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BackupJob implements SdkPojo, Serializable, ToCopyableBuilder<BackupJob.Builder, BackupJob> {
    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AccountId").getter(getter(BackupJob::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId").build()).build();

    private static final SdkField<String> BACKUP_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupJobId").getter(getter(BackupJob::backupJobId)).setter(setter(Builder::backupJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupJobId").build()).build();

    private static final SdkField<String> BACKUP_VAULT_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultName").getter(getter(BackupJob::backupVaultName)).setter(setter(Builder::backupVaultName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultName").build()).build();

    private static final SdkField<String> BACKUP_VAULT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupVaultArn").getter(getter(BackupJob::backupVaultArn)).setter(setter(Builder::backupVaultArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupVaultArn").build()).build();

    private static final SdkField<String> VAULT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VaultType").getter(getter(BackupJob::vaultType)).setter(setter(Builder::vaultType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VaultType").build()).build();

    private static final SdkField<String> VAULT_LOCK_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("VaultLockState").getter(getter(BackupJob::vaultLockState)).setter(setter(Builder::vaultLockState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VaultLockState").build()).build();

    private static final SdkField<String> RECOVERY_POINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RecoveryPointArn").getter(getter(BackupJob::recoveryPointArn)).setter(setter(Builder::recoveryPointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointArn").build()).build();

    private static final SdkField<Lifecycle> RECOVERY_POINT_LIFECYCLE_FIELD = SdkField
            .<Lifecycle> builder(MarshallingType.SDK_POJO).memberName("RecoveryPointLifecycle")
            .getter(getter(BackupJob::recoveryPointLifecycle)).setter(setter(Builder::recoveryPointLifecycle))
            .constructor(Lifecycle::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecoveryPointLifecycle").build())
            .build();

    private static final SdkField<String> ENCRYPTION_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionKeyArn").getter(getter(BackupJob::encryptionKeyArn)).setter(setter(Builder::encryptionKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EncryptionKeyArn").build()).build();

    private static final SdkField<Boolean> IS_ENCRYPTED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsEncrypted").getter(getter(BackupJob::isEncrypted)).setter(setter(Builder::isEncrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsEncrypted").build()).build();

    private static final SdkField<String> RESOURCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceArn").getter(getter(BackupJob::resourceArn)).setter(setter(Builder::resourceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceArn").build()).build();

    private static final SdkField<Instant> CREATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationDate").getter(getter(BackupJob::creationDate)).setter(setter(Builder::creationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationDate").build()).build();

    private static final SdkField<Instant> COMPLETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CompletionDate").getter(getter(BackupJob::completionDate)).setter(setter(Builder::completionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CompletionDate").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("State")
            .getter(getter(BackupJob::stateAsString)).setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(BackupJob::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> PERCENT_DONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PercentDone").getter(getter(BackupJob::percentDone)).setter(setter(Builder::percentDone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PercentDone").build()).build();

    private static final SdkField<Long> BACKUP_SIZE_IN_BYTES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BackupSizeInBytes").getter(getter(BackupJob::backupSizeInBytes))
            .setter(setter(Builder::backupSizeInBytes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupSizeInBytes").build()).build();

    private static final SdkField<String> IAM_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IamRoleArn").getter(getter(BackupJob::iamRoleArn)).setter(setter(Builder::iamRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IamRoleArn").build()).build();

    private static final SdkField<RecoveryPointCreator> CREATED_BY_FIELD = SdkField
            .<RecoveryPointCreator> builder(MarshallingType.SDK_POJO).memberName("CreatedBy")
            .getter(getter(BackupJob::createdBy)).setter(setter(Builder::createdBy)).constructor(RecoveryPointCreator::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedBy").build()).build();

    private static final SdkField<Instant> EXPECTED_COMPLETION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ExpectedCompletionDate").getter(getter(BackupJob::expectedCompletionDate))
            .setter(setter(Builder::expectedCompletionDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExpectedCompletionDate").build())
            .build();

    private static final SdkField<Instant> START_BY_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartBy").getter(getter(BackupJob::startBy)).setter(setter(Builder::startBy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartBy").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceType").getter(getter(BackupJob::resourceType)).setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType").build()).build();

    private static final SdkField<Long> BYTES_TRANSFERRED_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("BytesTransferred").getter(getter(BackupJob::bytesTransferred)).setter(setter(Builder::bytesTransferred))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BytesTransferred").build()).build();

    private static final SdkField<Map<String, String>> BACKUP_OPTIONS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("BackupOptions")
            .getter(getter(BackupJob::backupOptions))
            .setter(setter(Builder::backupOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupOptions").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> BACKUP_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("BackupType").getter(getter(BackupJob::backupType)).setter(setter(Builder::backupType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BackupType").build()).build();

    private static final SdkField<String> PARENT_JOB_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ParentJobId").getter(getter(BackupJob::parentJobId)).setter(setter(Builder::parentJobId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ParentJobId").build()).build();

    private static final SdkField<Boolean> IS_PARENT_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsParent").getter(getter(BackupJob::isParent)).setter(setter(Builder::isParent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsParent").build()).build();

    private static final SdkField<String> RESOURCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceName").getter(getter(BackupJob::resourceName)).setter(setter(Builder::resourceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceName").build()).build();

    private static final SdkField<Instant> INITIATION_DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("InitiationDate").getter(getter(BackupJob::initiationDate)).setter(setter(Builder::initiationDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InitiationDate").build()).build();

    private static final SdkField<String> MESSAGE_CATEGORY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MessageCategory").getter(getter(BackupJob::messageCategory)).setter(setter(Builder::messageCategory))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MessageCategory").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCOUNT_ID_FIELD,
            BACKUP_JOB_ID_FIELD, BACKUP_VAULT_NAME_FIELD, BACKUP_VAULT_ARN_FIELD, VAULT_TYPE_FIELD, VAULT_LOCK_STATE_FIELD,
            RECOVERY_POINT_ARN_FIELD, RECOVERY_POINT_LIFECYCLE_FIELD, ENCRYPTION_KEY_ARN_FIELD, IS_ENCRYPTED_FIELD,
            RESOURCE_ARN_FIELD, CREATION_DATE_FIELD, COMPLETION_DATE_FIELD, STATE_FIELD, STATUS_MESSAGE_FIELD,
            PERCENT_DONE_FIELD, BACKUP_SIZE_IN_BYTES_FIELD, IAM_ROLE_ARN_FIELD, CREATED_BY_FIELD, EXPECTED_COMPLETION_DATE_FIELD,
            START_BY_FIELD, RESOURCE_TYPE_FIELD, BYTES_TRANSFERRED_FIELD, BACKUP_OPTIONS_FIELD, BACKUP_TYPE_FIELD,
            PARENT_JOB_ID_FIELD, IS_PARENT_FIELD, RESOURCE_NAME_FIELD, INITIATION_DATE_FIELD, MESSAGE_CATEGORY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String accountId;

    private final String backupJobId;

    private final String backupVaultName;

    private final String backupVaultArn;

    private final String vaultType;

    private final String vaultLockState;

    private final String recoveryPointArn;

    private final Lifecycle recoveryPointLifecycle;

    private final String encryptionKeyArn;

    private final Boolean isEncrypted;

    private final String resourceArn;

    private final Instant creationDate;

    private final Instant completionDate;

    private final String state;

    private final String statusMessage;

    private final String percentDone;

    private final Long backupSizeInBytes;

    private final String iamRoleArn;

    private final RecoveryPointCreator createdBy;

    private final Instant expectedCompletionDate;

    private final Instant startBy;

    private final String resourceType;

    private final Long bytesTransferred;

    private final Map<String, String> backupOptions;

    private final String backupType;

    private final String parentJobId;

    private final Boolean isParent;

    private final String resourceName;

    private final Instant initiationDate;

    private final String messageCategory;

    private BackupJob(BuilderImpl builder) {
        this.accountId = builder.accountId;
        this.backupJobId = builder.backupJobId;
        this.backupVaultName = builder.backupVaultName;
        this.backupVaultArn = builder.backupVaultArn;
        this.vaultType = builder.vaultType;
        this.vaultLockState = builder.vaultLockState;
        this.recoveryPointArn = builder.recoveryPointArn;
        this.recoveryPointLifecycle = builder.recoveryPointLifecycle;
        this.encryptionKeyArn = builder.encryptionKeyArn;
        this.isEncrypted = builder.isEncrypted;
        this.resourceArn = builder.resourceArn;
        this.creationDate = builder.creationDate;
        this.completionDate = builder.completionDate;
        this.state = builder.state;
        this.statusMessage = builder.statusMessage;
        this.percentDone = builder.percentDone;
        this.backupSizeInBytes = builder.backupSizeInBytes;
        this.iamRoleArn = builder.iamRoleArn;
        this.createdBy = builder.createdBy;
        this.expectedCompletionDate = builder.expectedCompletionDate;
        this.startBy = builder.startBy;
        this.resourceType = builder.resourceType;
        this.bytesTransferred = builder.bytesTransferred;
        this.backupOptions = builder.backupOptions;
        this.backupType = builder.backupType;
        this.parentJobId = builder.parentJobId;
        this.isParent = builder.isParent;
        this.resourceName = builder.resourceName;
        this.initiationDate = builder.initiationDate;
        this.messageCategory = builder.messageCategory;
    }

    /**
     * <p>
     * The account ID that owns the backup job.
     * </p>
     * 
     * @return The account ID that owns the backup job.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * Uniquely identifies a request to Backup to back up a resource.
     * </p>
     * 
     * @return Uniquely identifies a request to Backup to back up a resource.
     */
    public final String backupJobId() {
        return backupJobId;
    }

    /**
     * <p>
     * The name of a logical container where backups are stored. Backup vaults are identified by names that are unique
     * to the account used to create them and the Amazon Web Services Region where they are created.
     * </p>
     * 
     * @return The name of a logical container where backups are stored. Backup vaults are identified by names that are
     *         unique to the account used to create them and the Amazon Web Services Region where they are created.
     */
    public final String backupVaultName() {
        return backupVaultName;
    }

    /**
     * <p>
     * An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
     * </p>
     * 
     * @return An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
     */
    public final String backupVaultArn() {
        return backupVaultArn;
    }

    /**
     * <p>
     * The type of backup vault where the recovery point is stored. Valid values are <code>BACKUP_VAULT</code> for
     * standard backup vaults and <code>LOGICALLY_AIR_GAPPED_BACKUP_VAULT</code> for logically air-gapped vaults.
     * </p>
     * 
     * @return The type of backup vault where the recovery point is stored. Valid values are <code>BACKUP_VAULT</code>
     *         for standard backup vaults and <code>LOGICALLY_AIR_GAPPED_BACKUP_VAULT</code> for logically air-gapped
     *         vaults.
     */
    public final String vaultType() {
        return vaultType;
    }

    /**
     * <p>
     * The lock state of the backup vault. For logically air-gapped vaults, this indicates whether the vault is locked
     * in compliance mode. Valid values include <code>LOCKED</code> and <code>UNLOCKED</code>.
     * </p>
     * 
     * @return The lock state of the backup vault. For logically air-gapped vaults, this indicates whether the vault is
     *         locked in compliance mode. Valid values include <code>LOCKED</code> and <code>UNLOCKED</code>.
     */
    public final String vaultLockState() {
        return vaultLockState;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a recovery point; for example,
     * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     * </p>
     * 
     * @return An ARN that uniquely identifies a recovery point; for example,
     *         <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
     */
    public final String recoveryPointArn() {
        return recoveryPointArn;
    }

    /**
     * Returns the value of the RecoveryPointLifecycle property for this object.
     * 
     * @return The value of the RecoveryPointLifecycle property for this object.
     */
    public final Lifecycle recoveryPointLifecycle() {
        return recoveryPointLifecycle;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the KMS key used to encrypt the backup. This can be a customer-managed key or
     * an Amazon Web Services managed key, depending on the vault configuration.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the KMS key used to encrypt the backup. This can be a customer-managed
     *         key or an Amazon Web Services managed key, depending on the vault configuration.
     */
    public final String encryptionKeyArn() {
        return encryptionKeyArn;
    }

    /**
     * <p>
     * A boolean value indicating whether the backup is encrypted. All backups in Backup are encrypted, but this field
     * indicates the encryption status for transparency.
     * </p>
     * 
     * @return A boolean value indicating whether the backup is encrypted. All backups in Backup are encrypted, but this
     *         field indicates the encryption status for transparency.
     */
    public final Boolean isEncrypted() {
        return isEncrypted;
    }

    /**
     * <p>
     * An ARN that uniquely identifies a resource. The format of the ARN depends on the resource type.
     * </p>
     * 
     * @return An ARN that uniquely identifies a resource. The format of the ARN depends on the resource type.
     */
    public final String resourceArn() {
        return resourceArn;
    }

    /**
     * <p>
     * The date and time a backup job is created, in Unix format and Coordinated Universal Time (UTC). The value of
     * <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents Friday,
     * January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time a backup job is created, in Unix format and Coordinated Universal Time (UTC). The value
     *         of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
     *         represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant creationDate() {
        return creationDate;
    }

    /**
     * <p>
     * The date and time a job to create a backup job is completed, in Unix format and Coordinated Universal Time (UTC).
     * The value of <code>CompletionDate</code> is accurate to milliseconds. For example, the value 1516925490.087
     * represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time a job to create a backup job is completed, in Unix format and Coordinated Universal
     *         Time (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For example, the value
     *         1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant completionDate() {
        return completionDate;
    }

    /**
     * <p>
     * The current state of a backup job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link BackupJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of a backup job.
     * @see BackupJobState
     */
    public final BackupJobState state() {
        return BackupJobState.fromValue(state);
    }

    /**
     * <p>
     * The current state of a backup job.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link BackupJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of a backup job.
     * @see BackupJobState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * A detailed message explaining the status of the job to back up a resource.
     * </p>
     * 
     * @return A detailed message explaining the status of the job to back up a resource.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * Contains an estimated percentage complete of a job at the time the job status was queried.
     * </p>
     * 
     * @return Contains an estimated percentage complete of a job at the time the job status was queried.
     */
    public final String percentDone() {
        return percentDone;
    }

    /**
     * <p>
     * The size, in bytes, of a backup (recovery point).
     * </p>
     * <p>
     * This value can render differently depending on the resource type as Backup pulls in data information from other
     * Amazon Web Services services. For example, the value returned may show a value of <code>0</code>, which may
     * differ from the anticipated value.
     * </p>
     * <p>
     * The expected behavior for values by resource type are described as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon Aurora, Amazon DocumentDB, and Amazon Neptune do not have this value populate from the operation
     * <code>GetBackupJobStatus</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon DynamoDB with advanced features, this value refers to the size of the recovery point (backup).
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon EC2 and Amazon EBS show volume size (provisioned storage) returned as part of this value. Amazon EBS does
     * not return backup size information; snapshot size will have the same value as the original resource that was
     * backed up.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon EFS, this value refers to the delta bytes transferred during a backup.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx does not populate this value from the operation <code>GetBackupJobStatus</code> for FSx file systems.
     * </p>
     * </li>
     * <li>
     * <p>
     * An Amazon RDS instance will show as <code>0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For virtual machines running VMware, this value is passed to Backup through an asynchronous workflow, which can
     * mean this displayed value can under-represent the actual backup size.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The size, in bytes, of a backup (recovery point).</p>
     *         <p>
     *         This value can render differently depending on the resource type as Backup pulls in data information from
     *         other Amazon Web Services services. For example, the value returned may show a value of <code>0</code>,
     *         which may differ from the anticipated value.
     *         </p>
     *         <p>
     *         The expected behavior for values by resource type are described as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Amazon Aurora, Amazon DocumentDB, and Amazon Neptune do not have this value populate from the operation
     *         <code>GetBackupJobStatus</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon DynamoDB with advanced features, this value refers to the size of the recovery point (backup).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon EC2 and Amazon EBS show volume size (provisioned storage) returned as part of this value. Amazon
     *         EBS does not return backup size information; snapshot size will have the same value as the original
     *         resource that was backed up.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Amazon EFS, this value refers to the delta bytes transferred during a backup.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon FSx does not populate this value from the operation <code>GetBackupJobStatus</code> for FSx file
     *         systems.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An Amazon RDS instance will show as <code>0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For virtual machines running VMware, this value is passed to Backup through an asynchronous workflow,
     *         which can mean this displayed value can under-represent the actual backup size.
     *         </p>
     *         </li>
     */
    public final Long backupSizeInBytes() {
        return backupSizeInBytes;
    }

    /**
     * <p>
     * Specifies the IAM role ARN used to create the target recovery point. IAM roles other than the default role must
     * include either <code>AWSBackup</code> or <code>AwsBackup</code> in the role name. For example,
     * <code>arn:aws:iam::123456789012:role/AWSBackupRDSAccess</code>. Role names without those strings lack permissions
     * to perform backup jobs.
     * </p>
     * 
     * @return Specifies the IAM role ARN used to create the target recovery point. IAM roles other than the default
     *         role must include either <code>AWSBackup</code> or <code>AwsBackup</code> in the role name. For example,
     *         <code>arn:aws:iam::123456789012:role/AWSBackupRDSAccess</code>. Role names without those strings lack
     *         permissions to perform backup jobs.
     */
    public final String iamRoleArn() {
        return iamRoleArn;
    }

    /**
     * <p>
     * Contains identifying information about the creation of a backup job, including the <code>BackupPlanArn</code>,
     * <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and <code>BackupRuleId</code> of the backup plan used
     * to create it.
     * </p>
     * 
     * @return Contains identifying information about the creation of a backup job, including the
     *         <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and
     *         <code>BackupRuleId</code> of the backup plan used to create it.
     */
    public final RecoveryPointCreator createdBy() {
        return createdBy;
    }

    /**
     * <p>
     * The date and time a job to back up resources is expected to be completed, in Unix format and Coordinated
     * Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to milliseconds. For example,
     * the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return The date and time a job to back up resources is expected to be completed, in Unix format and Coordinated
     *         Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to milliseconds. For
     *         example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant expectedCompletionDate() {
        return expectedCompletionDate;
    }

    /**
     * <p>
     * Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be started before
     * it is canceled. The value is calculated by adding the start window to the scheduled time. So if the scheduled
     * time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code> time would be 8:00 PM on the date
     * specified. The value of <code>StartBy</code> is accurate to milliseconds. For example, the value 1516925490.087
     * represents Friday, January 26, 2018 12:11:30.087 AM.
     * </p>
     * 
     * @return Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be started
     *         before it is canceled. The value is calculated by adding the start window to the scheduled time. So if
     *         the scheduled time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code> time would be
     *         8:00 PM on the date specified. The value of <code>StartBy</code> is accurate to milliseconds. For
     *         example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
     */
    public final Instant startBy() {
        return startBy;
    }

    /**
     * <p>
     * The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store (Amazon EBS)
     * volume or an Amazon Relational Database Service (Amazon RDS) database. For Windows Volume Shadow Copy Service
     * (VSS) backups, the only supported resource type is Amazon EC2.
     * </p>
     * 
     * @return The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store
     *         (Amazon EBS) volume or an Amazon Relational Database Service (Amazon RDS) database. For Windows Volume
     *         Shadow Copy Service (VSS) backups, the only supported resource type is Amazon EC2.
     */
    public final String resourceType() {
        return resourceType;
    }

    /**
     * <p>
     * The size in bytes transferred to a backup vault at the time that the job status was queried.
     * </p>
     * 
     * @return The size in bytes transferred to a backup vault at the time that the job status was queried.
     */
    public final Long bytesTransferred() {
        return bytesTransferred;
    }

    /**
     * For responses, this returns true if the service returned a value for the BackupOptions property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasBackupOptions() {
        return backupOptions != null && !(backupOptions instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Specifies the backup option for a selected resource. This option is only available for Windows Volume Shadow Copy
     * Service (VSS) backup jobs.
     * </p>
     * <p>
     * Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option and
     * create a Windows VSS backup. Set to <code>"WindowsVSS":"disabled"</code> to create a regular backup. If you
     * specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBackupOptions} method.
     * </p>
     * 
     * @return Specifies the backup option for a selected resource. This option is only available for Windows Volume
     *         Shadow Copy Service (VSS) backup jobs.</p>
     *         <p>
     *         Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup
     *         option and create a Windows VSS backup. Set to <code>"WindowsVSS":"disabled"</code> to create a regular
     *         backup. If you specify an invalid option, you get an <code>InvalidParameterValueException</code>
     *         exception.
     */
    public final Map<String, String> backupOptions() {
        return backupOptions;
    }

    /**
     * <p>
     * Represents the type of backup for a backup job.
     * </p>
     * 
     * @return Represents the type of backup for a backup job.
     */
    public final String backupType() {
        return backupType;
    }

    /**
     * <p>
     * This uniquely identifies a request to Backup to back up a resource. The return will be the parent (composite) job
     * ID.
     * </p>
     * 
     * @return This uniquely identifies a request to Backup to back up a resource. The return will be the parent
     *         (composite) job ID.
     */
    public final String parentJobId() {
        return parentJobId;
    }

    /**
     * <p>
     * This is a boolean value indicating this is a parent (composite) backup job.
     * </p>
     * 
     * @return This is a boolean value indicating this is a parent (composite) backup job.
     */
    public final Boolean isParent() {
        return isParent;
    }

    /**
     * <p>
     * The non-unique name of the resource that belongs to the specified backup.
     * </p>
     * 
     * @return The non-unique name of the resource that belongs to the specified backup.
     */
    public final String resourceName() {
        return resourceName;
    }

    /**
     * <p>
     * The date on which the backup job was initiated.
     * </p>
     * 
     * @return The date on which the backup job was initiated.
     */
    public final Instant initiationDate() {
        return initiationDate;
    }

    /**
     * <p>
     * This parameter is the job count for the specified message category.
     * </p>
     * <p>
     * Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>, and
     * <code>INVALIDPARAMETERS</code>. See <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list of
     * MessageCategory strings.
     * </p>
     * <p>
     * The the value ANY returns count of all message categories.
     * </p>
     * <p>
     * <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum.
     * </p>
     * 
     * @return This parameter is the job count for the specified message category.</p>
     *         <p>
     *         Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>,
     *         and <code>INVALIDPARAMETERS</code>. See <a
     *         href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list
     *         of MessageCategory strings.
     *         </p>
     *         <p>
     *         The the value ANY returns count of all message categories.
     *         </p>
     *         <p>
     *         <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum.
     */
    public final String messageCategory() {
        return messageCategory;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(backupJobId());
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultName());
        hashCode = 31 * hashCode + Objects.hashCode(backupVaultArn());
        hashCode = 31 * hashCode + Objects.hashCode(vaultType());
        hashCode = 31 * hashCode + Objects.hashCode(vaultLockState());
        hashCode = 31 * hashCode + Objects.hashCode(recoveryPointArn());
        hashCode = 31 * hashCode + Objects.hashCode(recoveryPointLifecycle());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(isEncrypted());
        hashCode = 31 * hashCode + Objects.hashCode(resourceArn());
        hashCode = 31 * hashCode + Objects.hashCode(creationDate());
        hashCode = 31 * hashCode + Objects.hashCode(completionDate());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(percentDone());
        hashCode = 31 * hashCode + Objects.hashCode(backupSizeInBytes());
        hashCode = 31 * hashCode + Objects.hashCode(iamRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(createdBy());
        hashCode = 31 * hashCode + Objects.hashCode(expectedCompletionDate());
        hashCode = 31 * hashCode + Objects.hashCode(startBy());
        hashCode = 31 * hashCode + Objects.hashCode(resourceType());
        hashCode = 31 * hashCode + Objects.hashCode(bytesTransferred());
        hashCode = 31 * hashCode + Objects.hashCode(hasBackupOptions() ? backupOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(backupType());
        hashCode = 31 * hashCode + Objects.hashCode(parentJobId());
        hashCode = 31 * hashCode + Objects.hashCode(isParent());
        hashCode = 31 * hashCode + Objects.hashCode(resourceName());
        hashCode = 31 * hashCode + Objects.hashCode(initiationDate());
        hashCode = 31 * hashCode + Objects.hashCode(messageCategory());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BackupJob)) {
            return false;
        }
        BackupJob other = (BackupJob) obj;
        return Objects.equals(accountId(), other.accountId()) && Objects.equals(backupJobId(), other.backupJobId())
                && Objects.equals(backupVaultName(), other.backupVaultName())
                && Objects.equals(backupVaultArn(), other.backupVaultArn()) && Objects.equals(vaultType(), other.vaultType())
                && Objects.equals(vaultLockState(), other.vaultLockState())
                && Objects.equals(recoveryPointArn(), other.recoveryPointArn())
                && Objects.equals(recoveryPointLifecycle(), other.recoveryPointLifecycle())
                && Objects.equals(encryptionKeyArn(), other.encryptionKeyArn())
                && Objects.equals(isEncrypted(), other.isEncrypted()) && Objects.equals(resourceArn(), other.resourceArn())
                && Objects.equals(creationDate(), other.creationDate())
                && Objects.equals(completionDate(), other.completionDate())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(percentDone(), other.percentDone())
                && Objects.equals(backupSizeInBytes(), other.backupSizeInBytes())
                && Objects.equals(iamRoleArn(), other.iamRoleArn()) && Objects.equals(createdBy(), other.createdBy())
                && Objects.equals(expectedCompletionDate(), other.expectedCompletionDate())
                && Objects.equals(startBy(), other.startBy()) && Objects.equals(resourceType(), other.resourceType())
                && Objects.equals(bytesTransferred(), other.bytesTransferred()) && hasBackupOptions() == other.hasBackupOptions()
                && Objects.equals(backupOptions(), other.backupOptions()) && Objects.equals(backupType(), other.backupType())
                && Objects.equals(parentJobId(), other.parentJobId()) && Objects.equals(isParent(), other.isParent())
                && Objects.equals(resourceName(), other.resourceName())
                && Objects.equals(initiationDate(), other.initiationDate())
                && Objects.equals(messageCategory(), other.messageCategory());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BackupJob").add("AccountId", accountId()).add("BackupJobId", backupJobId())
                .add("BackupVaultName", backupVaultName()).add("BackupVaultArn", backupVaultArn()).add("VaultType", vaultType())
                .add("VaultLockState", vaultLockState()).add("RecoveryPointArn", recoveryPointArn())
                .add("RecoveryPointLifecycle", recoveryPointLifecycle()).add("EncryptionKeyArn", encryptionKeyArn())
                .add("IsEncrypted", isEncrypted()).add("ResourceArn", resourceArn()).add("CreationDate", creationDate())
                .add("CompletionDate", completionDate()).add("State", stateAsString()).add("StatusMessage", statusMessage())
                .add("PercentDone", percentDone()).add("BackupSizeInBytes", backupSizeInBytes()).add("IamRoleArn", iamRoleArn())
                .add("CreatedBy", createdBy()).add("ExpectedCompletionDate", expectedCompletionDate()).add("StartBy", startBy())
                .add("ResourceType", resourceType()).add("BytesTransferred", bytesTransferred())
                .add("BackupOptions", hasBackupOptions() ? backupOptions() : null).add("BackupType", backupType())
                .add("ParentJobId", parentJobId()).add("IsParent", isParent()).add("ResourceName", resourceName())
                .add("InitiationDate", initiationDate()).add("MessageCategory", messageCategory()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "BackupJobId":
            return Optional.ofNullable(clazz.cast(backupJobId()));
        case "BackupVaultName":
            return Optional.ofNullable(clazz.cast(backupVaultName()));
        case "BackupVaultArn":
            return Optional.ofNullable(clazz.cast(backupVaultArn()));
        case "VaultType":
            return Optional.ofNullable(clazz.cast(vaultType()));
        case "VaultLockState":
            return Optional.ofNullable(clazz.cast(vaultLockState()));
        case "RecoveryPointArn":
            return Optional.ofNullable(clazz.cast(recoveryPointArn()));
        case "RecoveryPointLifecycle":
            return Optional.ofNullable(clazz.cast(recoveryPointLifecycle()));
        case "EncryptionKeyArn":
            return Optional.ofNullable(clazz.cast(encryptionKeyArn()));
        case "IsEncrypted":
            return Optional.ofNullable(clazz.cast(isEncrypted()));
        case "ResourceArn":
            return Optional.ofNullable(clazz.cast(resourceArn()));
        case "CreationDate":
            return Optional.ofNullable(clazz.cast(creationDate()));
        case "CompletionDate":
            return Optional.ofNullable(clazz.cast(completionDate()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "PercentDone":
            return Optional.ofNullable(clazz.cast(percentDone()));
        case "BackupSizeInBytes":
            return Optional.ofNullable(clazz.cast(backupSizeInBytes()));
        case "IamRoleArn":
            return Optional.ofNullable(clazz.cast(iamRoleArn()));
        case "CreatedBy":
            return Optional.ofNullable(clazz.cast(createdBy()));
        case "ExpectedCompletionDate":
            return Optional.ofNullable(clazz.cast(expectedCompletionDate()));
        case "StartBy":
            return Optional.ofNullable(clazz.cast(startBy()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceType()));
        case "BytesTransferred":
            return Optional.ofNullable(clazz.cast(bytesTransferred()));
        case "BackupOptions":
            return Optional.ofNullable(clazz.cast(backupOptions()));
        case "BackupType":
            return Optional.ofNullable(clazz.cast(backupType()));
        case "ParentJobId":
            return Optional.ofNullable(clazz.cast(parentJobId()));
        case "IsParent":
            return Optional.ofNullable(clazz.cast(isParent()));
        case "ResourceName":
            return Optional.ofNullable(clazz.cast(resourceName()));
        case "InitiationDate":
            return Optional.ofNullable(clazz.cast(initiationDate()));
        case "MessageCategory":
            return Optional.ofNullable(clazz.cast(messageCategory()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AccountId", ACCOUNT_ID_FIELD);
        map.put("BackupJobId", BACKUP_JOB_ID_FIELD);
        map.put("BackupVaultName", BACKUP_VAULT_NAME_FIELD);
        map.put("BackupVaultArn", BACKUP_VAULT_ARN_FIELD);
        map.put("VaultType", VAULT_TYPE_FIELD);
        map.put("VaultLockState", VAULT_LOCK_STATE_FIELD);
        map.put("RecoveryPointArn", RECOVERY_POINT_ARN_FIELD);
        map.put("RecoveryPointLifecycle", RECOVERY_POINT_LIFECYCLE_FIELD);
        map.put("EncryptionKeyArn", ENCRYPTION_KEY_ARN_FIELD);
        map.put("IsEncrypted", IS_ENCRYPTED_FIELD);
        map.put("ResourceArn", RESOURCE_ARN_FIELD);
        map.put("CreationDate", CREATION_DATE_FIELD);
        map.put("CompletionDate", COMPLETION_DATE_FIELD);
        map.put("State", STATE_FIELD);
        map.put("StatusMessage", STATUS_MESSAGE_FIELD);
        map.put("PercentDone", PERCENT_DONE_FIELD);
        map.put("BackupSizeInBytes", BACKUP_SIZE_IN_BYTES_FIELD);
        map.put("IamRoleArn", IAM_ROLE_ARN_FIELD);
        map.put("CreatedBy", CREATED_BY_FIELD);
        map.put("ExpectedCompletionDate", EXPECTED_COMPLETION_DATE_FIELD);
        map.put("StartBy", START_BY_FIELD);
        map.put("ResourceType", RESOURCE_TYPE_FIELD);
        map.put("BytesTransferred", BYTES_TRANSFERRED_FIELD);
        map.put("BackupOptions", BACKUP_OPTIONS_FIELD);
        map.put("BackupType", BACKUP_TYPE_FIELD);
        map.put("ParentJobId", PARENT_JOB_ID_FIELD);
        map.put("IsParent", IS_PARENT_FIELD);
        map.put("ResourceName", RESOURCE_NAME_FIELD);
        map.put("InitiationDate", INITIATION_DATE_FIELD);
        map.put("MessageCategory", MESSAGE_CATEGORY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<BackupJob, T> g) {
        return obj -> g.apply((BackupJob) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BackupJob> {
        /**
         * <p>
         * The account ID that owns the backup job.
         * </p>
         * 
         * @param accountId
         *        The account ID that owns the backup job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * Uniquely identifies a request to Backup to back up a resource.
         * </p>
         * 
         * @param backupJobId
         *        Uniquely identifies a request to Backup to back up a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupJobId(String backupJobId);

        /**
         * <p>
         * The name of a logical container where backups are stored. Backup vaults are identified by names that are
         * unique to the account used to create them and the Amazon Web Services Region where they are created.
         * </p>
         * 
         * @param backupVaultName
         *        The name of a logical container where backups are stored. Backup vaults are identified by names that
         *        are unique to the account used to create them and the Amazon Web Services Region where they are
         *        created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultName(String backupVaultName);

        /**
         * <p>
         * An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
         * </p>
         * 
         * @param backupVaultArn
         *        An Amazon Resource Name (ARN) that uniquely identifies a backup vault; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:backup-vault:aBackupVault</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupVaultArn(String backupVaultArn);

        /**
         * <p>
         * The type of backup vault where the recovery point is stored. Valid values are <code>BACKUP_VAULT</code> for
         * standard backup vaults and <code>LOGICALLY_AIR_GAPPED_BACKUP_VAULT</code> for logically air-gapped vaults.
         * </p>
         * 
         * @param vaultType
         *        The type of backup vault where the recovery point is stored. Valid values are
         *        <code>BACKUP_VAULT</code> for standard backup vaults and
         *        <code>LOGICALLY_AIR_GAPPED_BACKUP_VAULT</code> for logically air-gapped vaults.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vaultType(String vaultType);

        /**
         * <p>
         * The lock state of the backup vault. For logically air-gapped vaults, this indicates whether the vault is
         * locked in compliance mode. Valid values include <code>LOCKED</code> and <code>UNLOCKED</code>.
         * </p>
         * 
         * @param vaultLockState
         *        The lock state of the backup vault. For logically air-gapped vaults, this indicates whether the vault
         *        is locked in compliance mode. Valid values include <code>LOCKED</code> and <code>UNLOCKED</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vaultLockState(String vaultLockState);

        /**
         * <p>
         * An ARN that uniquely identifies a recovery point; for example,
         * <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>.
         * </p>
         * 
         * @param recoveryPointArn
         *        An ARN that uniquely identifies a recovery point; for example,
         *        <code>arn:aws:backup:us-east-1:123456789012:recovery-point:1EB3B5E7-9EB0-435A-A80B-108B488B0D45</code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointArn(String recoveryPointArn);

        /**
         * Sets the value of the RecoveryPointLifecycle property for this object.
         *
         * @param recoveryPointLifecycle
         *        The new value for the RecoveryPointLifecycle property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recoveryPointLifecycle(Lifecycle recoveryPointLifecycle);

        /**
         * Sets the value of the RecoveryPointLifecycle property for this object.
         *
         * This is a convenience method that creates an instance of the {@link Lifecycle.Builder} avoiding the need to
         * create one manually via {@link Lifecycle#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Lifecycle.Builder#build()} is called immediately and its result
         * is passed to {@link #recoveryPointLifecycle(Lifecycle)}.
         * 
         * @param recoveryPointLifecycle
         *        a consumer that will call methods on {@link Lifecycle.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recoveryPointLifecycle(Lifecycle)
         */
        default Builder recoveryPointLifecycle(Consumer<Lifecycle.Builder> recoveryPointLifecycle) {
            return recoveryPointLifecycle(Lifecycle.builder().applyMutation(recoveryPointLifecycle).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the KMS key used to encrypt the backup. This can be a customer-managed key
         * or an Amazon Web Services managed key, depending on the vault configuration.
         * </p>
         * 
         * @param encryptionKeyArn
         *        The Amazon Resource Name (ARN) of the KMS key used to encrypt the backup. This can be a
         *        customer-managed key or an Amazon Web Services managed key, depending on the vault configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKeyArn(String encryptionKeyArn);

        /**
         * <p>
         * A boolean value indicating whether the backup is encrypted. All backups in Backup are encrypted, but this
         * field indicates the encryption status for transparency.
         * </p>
         * 
         * @param isEncrypted
         *        A boolean value indicating whether the backup is encrypted. All backups in Backup are encrypted, but
         *        this field indicates the encryption status for transparency.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isEncrypted(Boolean isEncrypted);

        /**
         * <p>
         * An ARN that uniquely identifies a resource. The format of the ARN depends on the resource type.
         * </p>
         * 
         * @param resourceArn
         *        An ARN that uniquely identifies a resource. The format of the ARN depends on the resource type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceArn(String resourceArn);

        /**
         * <p>
         * The date and time a backup job is created, in Unix format and Coordinated Universal Time (UTC). The value of
         * <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087 represents
         * Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param creationDate
         *        The date and time a backup job is created, in Unix format and Coordinated Universal Time (UTC). The
         *        value of <code>CreationDate</code> is accurate to milliseconds. For example, the value 1516925490.087
         *        represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationDate(Instant creationDate);

        /**
         * <p>
         * The date and time a job to create a backup job is completed, in Unix format and Coordinated Universal Time
         * (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For example, the value
         * 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param completionDate
         *        The date and time a job to create a backup job is completed, in Unix format and Coordinated Universal
         *        Time (UTC). The value of <code>CompletionDate</code> is accurate to milliseconds. For example, the
         *        value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder completionDate(Instant completionDate);

        /**
         * <p>
         * The current state of a backup job.
         * </p>
         * 
         * @param state
         *        The current state of a backup job.
         * @see BackupJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupJobState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of a backup job.
         * </p>
         * 
         * @param state
         *        The current state of a backup job.
         * @see BackupJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BackupJobState
         */
        Builder state(BackupJobState state);

        /**
         * <p>
         * A detailed message explaining the status of the job to back up a resource.
         * </p>
         * 
         * @param statusMessage
         *        A detailed message explaining the status of the job to back up a resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * Contains an estimated percentage complete of a job at the time the job status was queried.
         * </p>
         * 
         * @param percentDone
         *        Contains an estimated percentage complete of a job at the time the job status was queried.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder percentDone(String percentDone);

        /**
         * <p>
         * The size, in bytes, of a backup (recovery point).
         * </p>
         * <p>
         * This value can render differently depending on the resource type as Backup pulls in data information from
         * other Amazon Web Services services. For example, the value returned may show a value of <code>0</code>, which
         * may differ from the anticipated value.
         * </p>
         * <p>
         * The expected behavior for values by resource type are described as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Amazon Aurora, Amazon DocumentDB, and Amazon Neptune do not have this value populate from the operation
         * <code>GetBackupJobStatus</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon DynamoDB with advanced features, this value refers to the size of the recovery point (backup).
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon EC2 and Amazon EBS show volume size (provisioned storage) returned as part of this value. Amazon EBS
         * does not return backup size information; snapshot size will have the same value as the original resource that
         * was backed up.
         * </p>
         * </li>
         * <li>
         * <p>
         * For Amazon EFS, this value refers to the delta bytes transferred during a backup.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon FSx does not populate this value from the operation <code>GetBackupJobStatus</code> for FSx file
         * systems.
         * </p>
         * </li>
         * <li>
         * <p>
         * An Amazon RDS instance will show as <code>0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For virtual machines running VMware, this value is passed to Backup through an asynchronous workflow, which
         * can mean this displayed value can under-represent the actual backup size.
         * </p>
         * </li>
         * </ul>
         * 
         * @param backupSizeInBytes
         *        The size, in bytes, of a backup (recovery point).</p>
         *        <p>
         *        This value can render differently depending on the resource type as Backup pulls in data information
         *        from other Amazon Web Services services. For example, the value returned may show a value of
         *        <code>0</code>, which may differ from the anticipated value.
         *        </p>
         *        <p>
         *        The expected behavior for values by resource type are described as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Amazon Aurora, Amazon DocumentDB, and Amazon Neptune do not have this value populate from the
         *        operation <code>GetBackupJobStatus</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon DynamoDB with advanced features, this value refers to the size of the recovery point
         *        (backup).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon EC2 and Amazon EBS show volume size (provisioned storage) returned as part of this value.
         *        Amazon EBS does not return backup size information; snapshot size will have the same value as the
         *        original resource that was backed up.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Amazon EFS, this value refers to the delta bytes transferred during a backup.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon FSx does not populate this value from the operation <code>GetBackupJobStatus</code> for FSx
         *        file systems.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        An Amazon RDS instance will show as <code>0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For virtual machines running VMware, this value is passed to Backup through an asynchronous workflow,
         *        which can mean this displayed value can under-represent the actual backup size.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupSizeInBytes(Long backupSizeInBytes);

        /**
         * <p>
         * Specifies the IAM role ARN used to create the target recovery point. IAM roles other than the default role
         * must include either <code>AWSBackup</code> or <code>AwsBackup</code> in the role name. For example,
         * <code>arn:aws:iam::123456789012:role/AWSBackupRDSAccess</code>. Role names without those strings lack
         * permissions to perform backup jobs.
         * </p>
         * 
         * @param iamRoleArn
         *        Specifies the IAM role ARN used to create the target recovery point. IAM roles other than the default
         *        role must include either <code>AWSBackup</code> or <code>AwsBackup</code> in the role name. For
         *        example, <code>arn:aws:iam::123456789012:role/AWSBackupRDSAccess</code>. Role names without those
         *        strings lack permissions to perform backup jobs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamRoleArn(String iamRoleArn);

        /**
         * <p>
         * Contains identifying information about the creation of a backup job, including the <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and <code>BackupRuleId</code> of the backup plan
         * used to create it.
         * </p>
         * 
         * @param createdBy
         *        Contains identifying information about the creation of a backup job, including the
         *        <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and
         *        <code>BackupRuleId</code> of the backup plan used to create it.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdBy(RecoveryPointCreator createdBy);

        /**
         * <p>
         * Contains identifying information about the creation of a backup job, including the <code>BackupPlanArn</code>, <code>BackupPlanId</code>, <code>BackupPlanVersion</code>, and <code>BackupRuleId</code> of the backup plan
         * used to create it.
         * </p>
         * This is a convenience method that creates an instance of the {@link RecoveryPointCreator.Builder} avoiding
         * the need to create one manually via {@link RecoveryPointCreator#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RecoveryPointCreator.Builder#build()} is called immediately and
         * its result is passed to {@link #createdBy(RecoveryPointCreator)}.
         * 
         * @param createdBy
         *        a consumer that will call methods on {@link RecoveryPointCreator.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #createdBy(RecoveryPointCreator)
         */
        default Builder createdBy(Consumer<RecoveryPointCreator.Builder> createdBy) {
            return createdBy(RecoveryPointCreator.builder().applyMutation(createdBy).build());
        }

        /**
         * <p>
         * The date and time a job to back up resources is expected to be completed, in Unix format and Coordinated
         * Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to milliseconds. For
         * example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param expectedCompletionDate
         *        The date and time a job to back up resources is expected to be completed, in Unix format and
         *        Coordinated Universal Time (UTC). The value of <code>ExpectedCompletionDate</code> is accurate to
         *        milliseconds. For example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087
         *        AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expectedCompletionDate(Instant expectedCompletionDate);

        /**
         * <p>
         * Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be started
         * before it is canceled. The value is calculated by adding the start window to the scheduled time. So if the
         * scheduled time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code> time would be 8:00 PM
         * on the date specified. The value of <code>StartBy</code> is accurate to milliseconds. For example, the value
         * 1516925490.087 represents Friday, January 26, 2018 12:11:30.087 AM.
         * </p>
         * 
         * @param startBy
         *        Specifies the time in Unix format and Coordinated Universal Time (UTC) when a backup job must be
         *        started before it is canceled. The value is calculated by adding the start window to the scheduled
         *        time. So if the scheduled time were 6:00 PM and the start window is 2 hours, the <code>StartBy</code>
         *        time would be 8:00 PM on the date specified. The value of <code>StartBy</code> is accurate to
         *        milliseconds. For example, the value 1516925490.087 represents Friday, January 26, 2018 12:11:30.087
         *        AM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startBy(Instant startBy);

        /**
         * <p>
         * The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store (Amazon
         * EBS) volume or an Amazon Relational Database Service (Amazon RDS) database. For Windows Volume Shadow Copy
         * Service (VSS) backups, the only supported resource type is Amazon EC2.
         * </p>
         * 
         * @param resourceType
         *        The type of Amazon Web Services resource to be backed up; for example, an Amazon Elastic Block Store
         *        (Amazon EBS) volume or an Amazon Relational Database Service (Amazon RDS) database. For Windows Volume
         *        Shadow Copy Service (VSS) backups, the only supported resource type is Amazon EC2.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The size in bytes transferred to a backup vault at the time that the job status was queried.
         * </p>
         * 
         * @param bytesTransferred
         *        The size in bytes transferred to a backup vault at the time that the job status was queried.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bytesTransferred(Long bytesTransferred);

        /**
         * <p>
         * Specifies the backup option for a selected resource. This option is only available for Windows Volume Shadow
         * Copy Service (VSS) backup jobs.
         * </p>
         * <p>
         * Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup option
         * and create a Windows VSS backup. Set to <code>"WindowsVSS":"disabled"</code> to create a regular backup. If
         * you specify an invalid option, you get an <code>InvalidParameterValueException</code> exception.
         * </p>
         * 
         * @param backupOptions
         *        Specifies the backup option for a selected resource. This option is only available for Windows Volume
         *        Shadow Copy Service (VSS) backup jobs.</p>
         *        <p>
         *        Valid values: Set to <code>"WindowsVSS":"enabled"</code> to enable the <code>WindowsVSS</code> backup
         *        option and create a Windows VSS backup. Set to <code>"WindowsVSS":"disabled"</code> to create a
         *        regular backup. If you specify an invalid option, you get an
         *        <code>InvalidParameterValueException</code> exception.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupOptions(Map<String, String> backupOptions);

        /**
         * <p>
         * Represents the type of backup for a backup job.
         * </p>
         * 
         * @param backupType
         *        Represents the type of backup for a backup job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder backupType(String backupType);

        /**
         * <p>
         * This uniquely identifies a request to Backup to back up a resource. The return will be the parent (composite)
         * job ID.
         * </p>
         * 
         * @param parentJobId
         *        This uniquely identifies a request to Backup to back up a resource. The return will be the parent
         *        (composite) job ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parentJobId(String parentJobId);

        /**
         * <p>
         * This is a boolean value indicating this is a parent (composite) backup job.
         * </p>
         * 
         * @param isParent
         *        This is a boolean value indicating this is a parent (composite) backup job.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isParent(Boolean isParent);

        /**
         * <p>
         * The non-unique name of the resource that belongs to the specified backup.
         * </p>
         * 
         * @param resourceName
         *        The non-unique name of the resource that belongs to the specified backup.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceName(String resourceName);

        /**
         * <p>
         * The date on which the backup job was initiated.
         * </p>
         * 
         * @param initiationDate
         *        The date on which the backup job was initiated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initiationDate(Instant initiationDate);

        /**
         * <p>
         * This parameter is the job count for the specified message category.
         * </p>
         * <p>
         * Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>, <code>AGGREGATE_ALL</code>, and
         * <code>INVALIDPARAMETERS</code>. See <a
         * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a list of
         * MessageCategory strings.
         * </p>
         * <p>
         * The the value ANY returns count of all message categories.
         * </p>
         * <p>
         * <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum.
         * </p>
         * 
         * @param messageCategory
         *        This parameter is the job count for the specified message category.</p>
         *        <p>
         *        Example strings may include <code>AccessDenied</code>, <code>SUCCESS</code>,
         *        <code>AGGREGATE_ALL</code>, and <code>INVALIDPARAMETERS</code>. See <a
         *        href="https://docs.aws.amazon.com/aws-backup/latest/devguide/monitoring.html">Monitoring</a> for a
         *        list of MessageCategory strings.
         *        </p>
         *        <p>
         *        The the value ANY returns count of all message categories.
         *        </p>
         *        <p>
         *        <code>AGGREGATE_ALL</code> aggregates job counts for all message categories and returns the sum.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messageCategory(String messageCategory);
    }

    static final class BuilderImpl implements Builder {
        private String accountId;

        private String backupJobId;

        private String backupVaultName;

        private String backupVaultArn;

        private String vaultType;

        private String vaultLockState;

        private String recoveryPointArn;

        private Lifecycle recoveryPointLifecycle;

        private String encryptionKeyArn;

        private Boolean isEncrypted;

        private String resourceArn;

        private Instant creationDate;

        private Instant completionDate;

        private String state;

        private String statusMessage;

        private String percentDone;

        private Long backupSizeInBytes;

        private String iamRoleArn;

        private RecoveryPointCreator createdBy;

        private Instant expectedCompletionDate;

        private Instant startBy;

        private String resourceType;

        private Long bytesTransferred;

        private Map<String, String> backupOptions = DefaultSdkAutoConstructMap.getInstance();

        private String backupType;

        private String parentJobId;

        private Boolean isParent;

        private String resourceName;

        private Instant initiationDate;

        private String messageCategory;

        private BuilderImpl() {
        }

        private BuilderImpl(BackupJob model) {
            accountId(model.accountId);
            backupJobId(model.backupJobId);
            backupVaultName(model.backupVaultName);
            backupVaultArn(model.backupVaultArn);
            vaultType(model.vaultType);
            vaultLockState(model.vaultLockState);
            recoveryPointArn(model.recoveryPointArn);
            recoveryPointLifecycle(model.recoveryPointLifecycle);
            encryptionKeyArn(model.encryptionKeyArn);
            isEncrypted(model.isEncrypted);
            resourceArn(model.resourceArn);
            creationDate(model.creationDate);
            completionDate(model.completionDate);
            state(model.state);
            statusMessage(model.statusMessage);
            percentDone(model.percentDone);
            backupSizeInBytes(model.backupSizeInBytes);
            iamRoleArn(model.iamRoleArn);
            createdBy(model.createdBy);
            expectedCompletionDate(model.expectedCompletionDate);
            startBy(model.startBy);
            resourceType(model.resourceType);
            bytesTransferred(model.bytesTransferred);
            backupOptions(model.backupOptions);
            backupType(model.backupType);
            parentJobId(model.parentJobId);
            isParent(model.isParent);
            resourceName(model.resourceName);
            initiationDate(model.initiationDate);
            messageCategory(model.messageCategory);
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getBackupJobId() {
            return backupJobId;
        }

        public final void setBackupJobId(String backupJobId) {
            this.backupJobId = backupJobId;
        }

        @Override
        public final Builder backupJobId(String backupJobId) {
            this.backupJobId = backupJobId;
            return this;
        }

        public final String getBackupVaultName() {
            return backupVaultName;
        }

        public final void setBackupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
        }

        @Override
        public final Builder backupVaultName(String backupVaultName) {
            this.backupVaultName = backupVaultName;
            return this;
        }

        public final String getBackupVaultArn() {
            return backupVaultArn;
        }

        public final void setBackupVaultArn(String backupVaultArn) {
            this.backupVaultArn = backupVaultArn;
        }

        @Override
        public final Builder backupVaultArn(String backupVaultArn) {
            this.backupVaultArn = backupVaultArn;
            return this;
        }

        public final String getVaultType() {
            return vaultType;
        }

        public final void setVaultType(String vaultType) {
            this.vaultType = vaultType;
        }

        @Override
        public final Builder vaultType(String vaultType) {
            this.vaultType = vaultType;
            return this;
        }

        public final String getVaultLockState() {
            return vaultLockState;
        }

        public final void setVaultLockState(String vaultLockState) {
            this.vaultLockState = vaultLockState;
        }

        @Override
        public final Builder vaultLockState(String vaultLockState) {
            this.vaultLockState = vaultLockState;
            return this;
        }

        public final String getRecoveryPointArn() {
            return recoveryPointArn;
        }

        public final void setRecoveryPointArn(String recoveryPointArn) {
            this.recoveryPointArn = recoveryPointArn;
        }

        @Override
        public final Builder recoveryPointArn(String recoveryPointArn) {
            this.recoveryPointArn = recoveryPointArn;
            return this;
        }

        public final Lifecycle.Builder getRecoveryPointLifecycle() {
            return recoveryPointLifecycle != null ? recoveryPointLifecycle.toBuilder() : null;
        }

        public final void setRecoveryPointLifecycle(Lifecycle.BuilderImpl recoveryPointLifecycle) {
            this.recoveryPointLifecycle = recoveryPointLifecycle != null ? recoveryPointLifecycle.build() : null;
        }

        @Override
        public final Builder recoveryPointLifecycle(Lifecycle recoveryPointLifecycle) {
            this.recoveryPointLifecycle = recoveryPointLifecycle;
            return this;
        }

        public final String getEncryptionKeyArn() {
            return encryptionKeyArn;
        }

        public final void setEncryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
        }

        @Override
        public final Builder encryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
            return this;
        }

        public final Boolean getIsEncrypted() {
            return isEncrypted;
        }

        public final void setIsEncrypted(Boolean isEncrypted) {
            this.isEncrypted = isEncrypted;
        }

        @Override
        public final Builder isEncrypted(Boolean isEncrypted) {
            this.isEncrypted = isEncrypted;
            return this;
        }

        public final String getResourceArn() {
            return resourceArn;
        }

        public final void setResourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
        }

        @Override
        public final Builder resourceArn(String resourceArn) {
            this.resourceArn = resourceArn;
            return this;
        }

        public final Instant getCreationDate() {
            return creationDate;
        }

        public final void setCreationDate(Instant creationDate) {
            this.creationDate = creationDate;
        }

        @Override
        public final Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public final Instant getCompletionDate() {
            return completionDate;
        }

        public final void setCompletionDate(Instant completionDate) {
            this.completionDate = completionDate;
        }

        @Override
        public final Builder completionDate(Instant completionDate) {
            this.completionDate = completionDate;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(BackupJobState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final String getPercentDone() {
            return percentDone;
        }

        public final void setPercentDone(String percentDone) {
            this.percentDone = percentDone;
        }

        @Override
        public final Builder percentDone(String percentDone) {
            this.percentDone = percentDone;
            return this;
        }

        public final Long getBackupSizeInBytes() {
            return backupSizeInBytes;
        }

        public final void setBackupSizeInBytes(Long backupSizeInBytes) {
            this.backupSizeInBytes = backupSizeInBytes;
        }

        @Override
        public final Builder backupSizeInBytes(Long backupSizeInBytes) {
            this.backupSizeInBytes = backupSizeInBytes;
            return this;
        }

        public final String getIamRoleArn() {
            return iamRoleArn;
        }

        public final void setIamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
        }

        @Override
        public final Builder iamRoleArn(String iamRoleArn) {
            this.iamRoleArn = iamRoleArn;
            return this;
        }

        public final RecoveryPointCreator.Builder getCreatedBy() {
            return createdBy != null ? createdBy.toBuilder() : null;
        }

        public final void setCreatedBy(RecoveryPointCreator.BuilderImpl createdBy) {
            this.createdBy = createdBy != null ? createdBy.build() : null;
        }

        @Override
        public final Builder createdBy(RecoveryPointCreator createdBy) {
            this.createdBy = createdBy;
            return this;
        }

        public final Instant getExpectedCompletionDate() {
            return expectedCompletionDate;
        }

        public final void setExpectedCompletionDate(Instant expectedCompletionDate) {
            this.expectedCompletionDate = expectedCompletionDate;
        }

        @Override
        public final Builder expectedCompletionDate(Instant expectedCompletionDate) {
            this.expectedCompletionDate = expectedCompletionDate;
            return this;
        }

        public final Instant getStartBy() {
            return startBy;
        }

        public final void setStartBy(Instant startBy) {
            this.startBy = startBy;
        }

        @Override
        public final Builder startBy(Instant startBy) {
            this.startBy = startBy;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public final Long getBytesTransferred() {
            return bytesTransferred;
        }

        public final void setBytesTransferred(Long bytesTransferred) {
            this.bytesTransferred = bytesTransferred;
        }

        @Override
        public final Builder bytesTransferred(Long bytesTransferred) {
            this.bytesTransferred = bytesTransferred;
            return this;
        }

        public final Map<String, String> getBackupOptions() {
            if (backupOptions instanceof SdkAutoConstructMap) {
                return null;
            }
            return backupOptions;
        }

        public final void setBackupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
        }

        @Override
        public final Builder backupOptions(Map<String, String> backupOptions) {
            this.backupOptions = BackupOptionsCopier.copy(backupOptions);
            return this;
        }

        public final String getBackupType() {
            return backupType;
        }

        public final void setBackupType(String backupType) {
            this.backupType = backupType;
        }

        @Override
        public final Builder backupType(String backupType) {
            this.backupType = backupType;
            return this;
        }

        public final String getParentJobId() {
            return parentJobId;
        }

        public final void setParentJobId(String parentJobId) {
            this.parentJobId = parentJobId;
        }

        @Override
        public final Builder parentJobId(String parentJobId) {
            this.parentJobId = parentJobId;
            return this;
        }

        public final Boolean getIsParent() {
            return isParent;
        }

        public final void setIsParent(Boolean isParent) {
            this.isParent = isParent;
        }

        @Override
        public final Builder isParent(Boolean isParent) {
            this.isParent = isParent;
            return this;
        }

        public final String getResourceName() {
            return resourceName;
        }

        public final void setResourceName(String resourceName) {
            this.resourceName = resourceName;
        }

        @Override
        public final Builder resourceName(String resourceName) {
            this.resourceName = resourceName;
            return this;
        }

        public final Instant getInitiationDate() {
            return initiationDate;
        }

        public final void setInitiationDate(Instant initiationDate) {
            this.initiationDate = initiationDate;
        }

        @Override
        public final Builder initiationDate(Instant initiationDate) {
            this.initiationDate = initiationDate;
            return this;
        }

        public final String getMessageCategory() {
            return messageCategory;
        }

        public final void setMessageCategory(String messageCategory) {
            this.messageCategory = messageCategory;
        }

        @Override
        public final Builder messageCategory(String messageCategory) {
            this.messageCategory = messageCategory;
            return this;
        }

        @Override
        public BackupJob build() {
            return new BackupJob(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
