/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the results of a security scan, including scanner information, scan state, and any findings discovered.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScanResult implements SdkPojo, Serializable, ToCopyableBuilder<ScanResult.Builder, ScanResult> {
    private static final SdkField<String> MALWARE_SCANNER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MalwareScanner").getter(getter(ScanResult::malwareScannerAsString))
            .setter(setter(Builder::malwareScanner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MalwareScanner").build()).build();

    private static final SdkField<String> SCAN_JOB_STATE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScanJobState").getter(getter(ScanResult::scanJobStateAsString)).setter(setter(Builder::scanJobState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScanJobState").build()).build();

    private static final SdkField<Instant> LAST_SCAN_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastScanTimestamp").getter(getter(ScanResult::lastScanTimestamp))
            .setter(setter(Builder::lastScanTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastScanTimestamp").build()).build();

    private static final SdkField<List<String>> FINDINGS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Findings")
            .getter(getter(ScanResult::findingsAsStrings))
            .setter(setter(Builder::findingsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Findings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MALWARE_SCANNER_FIELD,
            SCAN_JOB_STATE_FIELD, LAST_SCAN_TIMESTAMP_FIELD, FINDINGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String malwareScanner;

    private final String scanJobState;

    private final Instant lastScanTimestamp;

    private final List<String> findings;

    private ScanResult(BuilderImpl builder) {
        this.malwareScanner = builder.malwareScanner;
        this.scanJobState = builder.scanJobState;
        this.lastScanTimestamp = builder.lastScanTimestamp;
        this.findings = builder.findings;
    }

    /**
     * <p>
     * The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #malwareScanner}
     * will return {@link MalwareScanner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #malwareScannerAsString}.
     * </p>
     * 
     * @return The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
     * @see MalwareScanner
     */
    public final MalwareScanner malwareScanner() {
        return MalwareScanner.fromValue(malwareScanner);
    }

    /**
     * <p>
     * The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #malwareScanner}
     * will return {@link MalwareScanner#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #malwareScannerAsString}.
     * </p>
     * 
     * @return The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
     * @see MalwareScanner
     */
    public final String malwareScannerAsString() {
        return malwareScanner;
    }

    /**
     * <p>
     * The final state of the scan job.
     * </p>
     * <p>
     * Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanJobState} will
     * return {@link ScanJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanJobStateAsString}.
     * </p>
     * 
     * @return The final state of the scan job.</p>
     *         <p>
     *         Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
     * @see ScanJobState
     */
    public final ScanJobState scanJobState() {
        return ScanJobState.fromValue(scanJobState);
    }

    /**
     * <p>
     * The final state of the scan job.
     * </p>
     * <p>
     * Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanJobState} will
     * return {@link ScanJobState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanJobStateAsString}.
     * </p>
     * 
     * @return The final state of the scan job.</p>
     *         <p>
     *         Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
     * @see ScanJobState
     */
    public final String scanJobStateAsString() {
        return scanJobState;
    }

    /**
     * <p>
     * The timestamp of when the last scan was performed, in Unix format and Coordinated Universal Time (UTC).
     * </p>
     * 
     * @return The timestamp of when the last scan was performed, in Unix format and Coordinated Universal Time (UTC).
     */
    public final Instant lastScanTimestamp() {
        return lastScanTimestamp;
    }

    /**
     * <p>
     * An array of findings discovered during the scan.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindings} method.
     * </p>
     * 
     * @return An array of findings discovered during the scan.
     */
    public final List<ScanFinding> findings() {
        return ScanFindingsCopier.copyStringToEnum(findings);
    }

    /**
     * For responses, this returns true if the service returned a value for the Findings property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFindings() {
        return findings != null && !(findings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of findings discovered during the scan.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFindings} method.
     * </p>
     * 
     * @return An array of findings discovered during the scan.
     */
    public final List<String> findingsAsStrings() {
        return findings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(malwareScannerAsString());
        hashCode = 31 * hashCode + Objects.hashCode(scanJobStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastScanTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindings() ? findingsAsStrings() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScanResult)) {
            return false;
        }
        ScanResult other = (ScanResult) obj;
        return Objects.equals(malwareScannerAsString(), other.malwareScannerAsString())
                && Objects.equals(scanJobStateAsString(), other.scanJobStateAsString())
                && Objects.equals(lastScanTimestamp(), other.lastScanTimestamp()) && hasFindings() == other.hasFindings()
                && Objects.equals(findingsAsStrings(), other.findingsAsStrings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScanResult").add("MalwareScanner", malwareScannerAsString())
                .add("ScanJobState", scanJobStateAsString()).add("LastScanTimestamp", lastScanTimestamp())
                .add("Findings", hasFindings() ? findingsAsStrings() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MalwareScanner":
            return Optional.ofNullable(clazz.cast(malwareScannerAsString()));
        case "ScanJobState":
            return Optional.ofNullable(clazz.cast(scanJobStateAsString()));
        case "LastScanTimestamp":
            return Optional.ofNullable(clazz.cast(lastScanTimestamp()));
        case "Findings":
            return Optional.ofNullable(clazz.cast(findingsAsStrings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MalwareScanner", MALWARE_SCANNER_FIELD);
        map.put("ScanJobState", SCAN_JOB_STATE_FIELD);
        map.put("LastScanTimestamp", LAST_SCAN_TIMESTAMP_FIELD);
        map.put("Findings", FINDINGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ScanResult, T> g) {
        return obj -> g.apply((ScanResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScanResult> {
        /**
         * <p>
         * The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
         * </p>
         * 
         * @param malwareScanner
         *        The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
         * @see MalwareScanner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MalwareScanner
         */
        Builder malwareScanner(String malwareScanner);

        /**
         * <p>
         * The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
         * </p>
         * 
         * @param malwareScanner
         *        The malware scanner used to perform the scan. Currently only <code>GUARDDUTY</code> is supported.
         * @see MalwareScanner
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MalwareScanner
         */
        Builder malwareScanner(MalwareScanner malwareScanner);

        /**
         * <p>
         * The final state of the scan job.
         * </p>
         * <p>
         * Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
         * </p>
         * 
         * @param scanJobState
         *        The final state of the scan job.</p>
         *        <p>
         *        Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
         * @see ScanJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanJobState
         */
        Builder scanJobState(String scanJobState);

        /**
         * <p>
         * The final state of the scan job.
         * </p>
         * <p>
         * Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
         * </p>
         * 
         * @param scanJobState
         *        The final state of the scan job.</p>
         *        <p>
         *        Valid values: <code>COMPLETED</code> | <code>FAILED</code> | <code>CANCELED</code>.
         * @see ScanJobState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanJobState
         */
        Builder scanJobState(ScanJobState scanJobState);

        /**
         * <p>
         * The timestamp of when the last scan was performed, in Unix format and Coordinated Universal Time (UTC).
         * </p>
         * 
         * @param lastScanTimestamp
         *        The timestamp of when the last scan was performed, in Unix format and Coordinated Universal Time
         *        (UTC).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastScanTimestamp(Instant lastScanTimestamp);

        /**
         * <p>
         * An array of findings discovered during the scan.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered during the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingsWithStrings(Collection<String> findings);

        /**
         * <p>
         * An array of findings discovered during the scan.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered during the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingsWithStrings(String... findings);

        /**
         * <p>
         * An array of findings discovered during the scan.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered during the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findings(Collection<ScanFinding> findings);

        /**
         * <p>
         * An array of findings discovered during the scan.
         * </p>
         * 
         * @param findings
         *        An array of findings discovered during the scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findings(ScanFinding... findings);
    }

    static final class BuilderImpl implements Builder {
        private String malwareScanner;

        private String scanJobState;

        private Instant lastScanTimestamp;

        private List<String> findings = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ScanResult model) {
            malwareScanner(model.malwareScanner);
            scanJobState(model.scanJobState);
            lastScanTimestamp(model.lastScanTimestamp);
            findingsWithStrings(model.findings);
        }

        public final String getMalwareScanner() {
            return malwareScanner;
        }

        public final void setMalwareScanner(String malwareScanner) {
            this.malwareScanner = malwareScanner;
        }

        @Override
        public final Builder malwareScanner(String malwareScanner) {
            this.malwareScanner = malwareScanner;
            return this;
        }

        @Override
        public final Builder malwareScanner(MalwareScanner malwareScanner) {
            this.malwareScanner(malwareScanner == null ? null : malwareScanner.toString());
            return this;
        }

        public final String getScanJobState() {
            return scanJobState;
        }

        public final void setScanJobState(String scanJobState) {
            this.scanJobState = scanJobState;
        }

        @Override
        public final Builder scanJobState(String scanJobState) {
            this.scanJobState = scanJobState;
            return this;
        }

        @Override
        public final Builder scanJobState(ScanJobState scanJobState) {
            this.scanJobState(scanJobState == null ? null : scanJobState.toString());
            return this;
        }

        public final Instant getLastScanTimestamp() {
            return lastScanTimestamp;
        }

        public final void setLastScanTimestamp(Instant lastScanTimestamp) {
            this.lastScanTimestamp = lastScanTimestamp;
        }

        @Override
        public final Builder lastScanTimestamp(Instant lastScanTimestamp) {
            this.lastScanTimestamp = lastScanTimestamp;
            return this;
        }

        public final Collection<String> getFindings() {
            if (findings instanceof SdkAutoConstructList) {
                return null;
            }
            return findings;
        }

        public final void setFindings(Collection<String> findings) {
            this.findings = ScanFindingsCopier.copy(findings);
        }

        @Override
        public final Builder findingsWithStrings(Collection<String> findings) {
            this.findings = ScanFindingsCopier.copy(findings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingsWithStrings(String... findings) {
            findingsWithStrings(Arrays.asList(findings));
            return this;
        }

        @Override
        public final Builder findings(Collection<ScanFinding> findings) {
            this.findings = ScanFindingsCopier.copyEnumToString(findings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findings(ScanFinding... findings) {
            findings(Arrays.asList(findings));
            return this;
        }

        @Override
        public ScanResult build() {
            return new ScanResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
