/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.backup.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains <code>DeleteAt</code> and <code>MoveToColdStorageAt</code> timestamps, which are used to specify a lifecycle
 * for a recovery point.
 * </p>
 * <p>
 * The lifecycle defines when a protected resource is transitioned to cold storage and when it expires. Backup
 * transitions and expires backups automatically according to the lifecycle that you define.
 * </p>
 * <p>
 * Backups transitioned to cold storage must be stored in cold storage for a minimum of 90 days. Therefore, the
 * “retention” setting must be 90 days greater than the “transition to cold after days” setting. The “transition to cold
 * after days” setting cannot be changed after a backup has been transitioned to cold.
 * </p>
 * <p>
 * Resource types that can transition to cold storage are listed in the <a
 * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/backup-feature-availability.html#features-by-resource"
 * >Feature availability by resource</a> table. Backup ignores this expression for other resource types.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CalculatedLifecycle implements SdkPojo, Serializable,
        ToCopyableBuilder<CalculatedLifecycle.Builder, CalculatedLifecycle> {
    private static final SdkField<Instant> MOVE_TO_COLD_STORAGE_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("MoveToColdStorageAt").getter(getter(CalculatedLifecycle::moveToColdStorageAt))
            .setter(setter(Builder::moveToColdStorageAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MoveToColdStorageAt").build())
            .build();

    private static final SdkField<Instant> DELETE_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("DeleteAt").getter(getter(CalculatedLifecycle::deleteAt)).setter(setter(Builder::deleteAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteAt").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(MOVE_TO_COLD_STORAGE_AT_FIELD,
            DELETE_AT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Instant moveToColdStorageAt;

    private final Instant deleteAt;

    private CalculatedLifecycle(BuilderImpl builder) {
        this.moveToColdStorageAt = builder.moveToColdStorageAt;
        this.deleteAt = builder.deleteAt;
    }

    /**
     * <p>
     * A timestamp that specifies when to transition a recovery point to cold storage.
     * </p>
     * 
     * @return A timestamp that specifies when to transition a recovery point to cold storage.
     */
    public final Instant moveToColdStorageAt() {
        return moveToColdStorageAt;
    }

    /**
     * <p>
     * A timestamp that specifies when to delete a recovery point.
     * </p>
     * 
     * @return A timestamp that specifies when to delete a recovery point.
     */
    public final Instant deleteAt() {
        return deleteAt;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(moveToColdStorageAt());
        hashCode = 31 * hashCode + Objects.hashCode(deleteAt());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CalculatedLifecycle)) {
            return false;
        }
        CalculatedLifecycle other = (CalculatedLifecycle) obj;
        return Objects.equals(moveToColdStorageAt(), other.moveToColdStorageAt()) && Objects.equals(deleteAt(), other.deleteAt());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CalculatedLifecycle").add("MoveToColdStorageAt", moveToColdStorageAt())
                .add("DeleteAt", deleteAt()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MoveToColdStorageAt":
            return Optional.ofNullable(clazz.cast(moveToColdStorageAt()));
        case "DeleteAt":
            return Optional.ofNullable(clazz.cast(deleteAt()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("MoveToColdStorageAt", MOVE_TO_COLD_STORAGE_AT_FIELD);
        map.put("DeleteAt", DELETE_AT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CalculatedLifecycle, T> g) {
        return obj -> g.apply((CalculatedLifecycle) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CalculatedLifecycle> {
        /**
         * <p>
         * A timestamp that specifies when to transition a recovery point to cold storage.
         * </p>
         * 
         * @param moveToColdStorageAt
         *        A timestamp that specifies when to transition a recovery point to cold storage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder moveToColdStorageAt(Instant moveToColdStorageAt);

        /**
         * <p>
         * A timestamp that specifies when to delete a recovery point.
         * </p>
         * 
         * @param deleteAt
         *        A timestamp that specifies when to delete a recovery point.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteAt(Instant deleteAt);
    }

    static final class BuilderImpl implements Builder {
        private Instant moveToColdStorageAt;

        private Instant deleteAt;

        private BuilderImpl() {
        }

        private BuilderImpl(CalculatedLifecycle model) {
            moveToColdStorageAt(model.moveToColdStorageAt);
            deleteAt(model.deleteAt);
        }

        public final Instant getMoveToColdStorageAt() {
            return moveToColdStorageAt;
        }

        public final void setMoveToColdStorageAt(Instant moveToColdStorageAt) {
            this.moveToColdStorageAt = moveToColdStorageAt;
        }

        @Override
        public final Builder moveToColdStorageAt(Instant moveToColdStorageAt) {
            this.moveToColdStorageAt = moveToColdStorageAt;
            return this;
        }

        public final Instant getDeleteAt() {
            return deleteAt;
        }

        public final void setDeleteAt(Instant deleteAt) {
            this.deleteAt = deleteAt;
        }

        @Override
        public final Builder deleteAt(Instant deleteAt) {
            this.deleteAt = deleteAt;
            return this;
        }

        @Override
        public CalculatedLifecycle build() {
            return new CalculatedLifecycle(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
