/*
 * Decompiled with CFR 0.152.
 */
package software.amazon.awssdk.auth.signer.internal;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;
import software.amazon.awssdk.auth.signer.Aws4Signer;
import software.amazon.awssdk.auth.signer.AwsSignerExecutionAttribute;
import software.amazon.awssdk.auth.signer.internal.AbstractAwsSigner;
import software.amazon.awssdk.auth.signer.internal.Aws4SignerRequestParams;
import software.amazon.awssdk.auth.signer.internal.Aws4SignerUtils;
import software.amazon.awssdk.auth.signer.internal.FifoCache;
import software.amazon.awssdk.auth.signer.internal.SignerKey;
import software.amazon.awssdk.auth.signer.internal.SigningAlgorithm;
import software.amazon.awssdk.auth.signer.params.Aws4PresignerParams;
import software.amazon.awssdk.auth.signer.params.Aws4SignerParams;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.interceptor.ExecutionAttributes;
import software.amazon.awssdk.core.signer.Presigner;
import software.amazon.awssdk.http.SdkHttpFullRequest;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.utils.BinaryUtils;
import software.amazon.awssdk.utils.DateUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.StringUtils;
import software.amazon.awssdk.utils.http.SdkHttpUtils;

@SdkInternalApi
public abstract class AbstractAws4Signer<T extends Aws4SignerParams, U extends Aws4PresignerParams>
extends AbstractAwsSigner
implements Presigner {
    public static final String EMPTY_STRING_SHA256_HEX = BinaryUtils.toHex((byte[])AbstractAws4Signer.hash(""));
    private static final Logger LOG = Logger.loggerFor(Aws4Signer.class);
    private static final int SIGNER_CACHE_MAX_SIZE = 300;
    private static final FifoCache<SignerKey> SIGNER_CACHE = new FifoCache(300);
    private static final List<String> LIST_OF_HEADERS_TO_IGNORE_IN_LOWER_CASE = Arrays.asList("connection", "x-amzn-trace-id", "user-agent", "expect");

    protected SdkHttpFullRequest.Builder doSign(SdkHttpFullRequest request, Aws4SignerRequestParams requestParams, T signingParams) {
        SdkHttpFullRequest.Builder mutableRequest = request.toBuilder();
        AwsCredentials sanitizedCredentials = this.sanitizeCredentials(((Aws4SignerParams)signingParams).awsCredentials());
        if (sanitizedCredentials instanceof AwsSessionCredentials) {
            this.addSessionCredentials(mutableRequest, (AwsSessionCredentials)sanitizedCredentials);
        }
        this.addHostHeader(mutableRequest);
        this.addDateHeader(mutableRequest, requestParams.getFormattedRequestSigningDateTime());
        String contentSha256 = this.calculateContentHash(mutableRequest, signingParams);
        mutableRequest.firstMatchingHeader("x-amz-content-sha256").filter(h -> h.equals("required")).ifPresent(h -> mutableRequest.putHeader("x-amz-content-sha256", contentSha256));
        String canonicalRequest = this.createCanonicalRequest(mutableRequest, contentSha256, ((Aws4SignerParams)signingParams).doubleUrlEncode());
        String stringToSign = this.createStringToSign(canonicalRequest, requestParams);
        byte[] signingKey = this.deriveSigningKey(sanitizedCredentials, requestParams);
        byte[] signature = this.computeSignature(stringToSign, signingKey);
        mutableRequest.putHeader("Authorization", this.buildAuthorizationHeader(signature, sanitizedCredentials, requestParams, mutableRequest));
        this.processRequestPayload(mutableRequest, signature, signingKey, requestParams, signingParams);
        return mutableRequest;
    }

    protected SdkHttpFullRequest.Builder doPresign(SdkHttpFullRequest request, Aws4SignerRequestParams requestParams, U signingParams) {
        SdkHttpFullRequest.Builder mutableRequest = request.toBuilder();
        long expirationInSeconds = this.generateExpirationTime(signingParams);
        this.addHostHeader(mutableRequest);
        AwsCredentials sanitizedCredentials = this.sanitizeCredentials(((Aws4SignerParams)signingParams).awsCredentials());
        if (sanitizedCredentials instanceof AwsSessionCredentials) {
            mutableRequest.putRawQueryParameter("X-Amz-Security-Token", ((AwsSessionCredentials)sanitizedCredentials).sessionToken());
        }
        String timeStamp = requestParams.getFormattedRequestSigningDateTime();
        this.addPreSignInformationToRequest(mutableRequest, sanitizedCredentials, requestParams, timeStamp, expirationInSeconds);
        String contentSha256 = this.calculateContentHashPresign(mutableRequest, signingParams);
        String canonicalRequest = this.createCanonicalRequest(mutableRequest, contentSha256, ((Aws4SignerParams)signingParams).doubleUrlEncode());
        String stringToSign = this.createStringToSign(canonicalRequest, requestParams);
        byte[] signingKey = this.deriveSigningKey(sanitizedCredentials, requestParams);
        byte[] signature = this.computeSignature(stringToSign, signingKey);
        mutableRequest.putRawQueryParameter("X-Amz-Signature", BinaryUtils.toHex((byte[])signature));
        return mutableRequest;
    }

    @Override
    protected void addSessionCredentials(SdkHttpFullRequest.Builder mutableRequest, AwsSessionCredentials credentials) {
        mutableRequest.putHeader("X-Amz-Security-Token", credentials.sessionToken());
    }

    protected String calculateContentHash(SdkHttpFullRequest.Builder mutableRequest, T signerParams) {
        InputStream payloadStream = this.getBinaryRequestPayloadStream(mutableRequest.contentStreamProvider());
        return BinaryUtils.toHex((byte[])this.hash(payloadStream));
    }

    protected abstract void processRequestPayload(SdkHttpFullRequest.Builder var1, byte[] var2, byte[] var3, Aws4SignerRequestParams var4, T var5);

    protected abstract String calculateContentHashPresign(SdkHttpFullRequest.Builder var1, U var2);

    protected byte[] deriveSigningKey(AwsCredentials credentials, Aws4SignerRequestParams signerRequestParams) {
        String cacheKey = this.computeSigningCacheKeyName(credentials, signerRequestParams);
        long daysSinceEpochSigningDate = DateUtils.numberOfDaysSinceEpoch((long)signerRequestParams.getRequestSigningDateTimeMilli());
        SignerKey signerKey = SIGNER_CACHE.get(cacheKey);
        if (signerKey != null && daysSinceEpochSigningDate == signerKey.getNumberOfDaysSinceEpoch()) {
            return signerKey.getSigningKey();
        }
        LOG.trace(() -> "Generating a new signing key as the signing key not available in the cache for the date: " + TimeUnit.DAYS.toMillis(daysSinceEpochSigningDate));
        byte[] signingKey = this.newSigningKey(credentials, signerRequestParams.getFormattedRequestSigningDate(), signerRequestParams.getRegionName(), signerRequestParams.getServiceSigningName());
        SIGNER_CACHE.add(cacheKey, new SignerKey(daysSinceEpochSigningDate, signingKey));
        return signingKey;
    }

    private String createCanonicalRequest(SdkHttpFullRequest.Builder request, String contentSha256, boolean doubleUrlEncode) {
        String canonicalRequest = request.method().toString() + "\n" + this.getCanonicalizedResourcePath(request.encodedPath(), doubleUrlEncode) + "\n" + this.getCanonicalizedQueryString(request.rawQueryParameters()) + "\n" + this.getCanonicalizedHeaderString(request.headers()) + "\n" + this.getSignedHeadersString(request.headers()) + "\n" + contentSha256;
        LOG.trace(() -> "AWS4 Canonical Request: " + canonicalRequest);
        return canonicalRequest;
    }

    private String createStringToSign(String canonicalRequest, Aws4SignerRequestParams requestParams) {
        String stringToSign = requestParams.getSigningAlgorithm() + "\n" + requestParams.getFormattedRequestSigningDateTime() + "\n" + requestParams.getScope() + "\n" + BinaryUtils.toHex((byte[])AbstractAws4Signer.hash(canonicalRequest));
        LOG.debug(() -> "AWS4 String to sign: " + stringToSign);
        return stringToSign;
    }

    private String computeSigningCacheKeyName(AwsCredentials credentials, Aws4SignerRequestParams signerRequestParams) {
        return credentials.secretAccessKey() + "-" + signerRequestParams.getRegionName() + "-" + signerRequestParams.getServiceSigningName();
    }

    private byte[] computeSignature(String stringToSign, byte[] signingKey) {
        return this.sign(stringToSign.getBytes(Charset.forName("UTF-8")), signingKey, SigningAlgorithm.HmacSHA256);
    }

    private String buildAuthorizationHeader(byte[] signature, AwsCredentials credentials, Aws4SignerRequestParams signerParams, SdkHttpFullRequest.Builder mutableRequest) {
        String signingCredentials = credentials.accessKeyId() + "/" + signerParams.getScope();
        String credential = "Credential=" + signingCredentials;
        String signerHeaders = "SignedHeaders=" + this.getSignedHeadersString(mutableRequest.headers());
        String signatureHeader = "Signature=" + BinaryUtils.toHex((byte[])signature);
        return "AWS4-HMAC-SHA256 " + credential + ", " + signerHeaders + ", " + signatureHeader;
    }

    private void addPreSignInformationToRequest(SdkHttpFullRequest.Builder mutableRequest, AwsCredentials sanitizedCredentials, Aws4SignerRequestParams signerParams, String timeStamp, long expirationInSeconds) {
        String signingCredentials = sanitizedCredentials.accessKeyId() + "/" + signerParams.getScope();
        mutableRequest.putRawQueryParameter("X-Amz-Algorithm", "AWS4-HMAC-SHA256");
        mutableRequest.putRawQueryParameter("X-Amz-Date", timeStamp);
        mutableRequest.putRawQueryParameter("X-Amz-SignedHeaders", this.getSignedHeadersString(mutableRequest.headers()));
        mutableRequest.putRawQueryParameter("X-Amz-Expires", Long.toString(expirationInSeconds));
        mutableRequest.putRawQueryParameter("X-Amz-Credential", signingCredentials);
    }

    private String getCanonicalizedHeaderString(Map<String, List<String>> headers) {
        ArrayList<String> sortedHeaders = new ArrayList<String>(headers.keySet());
        sortedHeaders.sort(String.CASE_INSENSITIVE_ORDER);
        StringBuilder buffer = new StringBuilder();
        for (String header : sortedHeaders) {
            if (this.shouldExcludeHeaderFromSigning(header)) continue;
            String key = StringUtils.lowerCase((String)header);
            for (String headerValue : headers.get(header)) {
                this.appendCompactedString(buffer, key);
                buffer.append(":");
                if (headerValue != null) {
                    this.appendCompactedString(buffer, headerValue);
                }
                buffer.append("\n");
            }
        }
        return buffer.toString();
    }

    private void appendCompactedString(StringBuilder destination, String source) {
        boolean previousIsWhiteSpace = false;
        int length = source.length();
        for (int i = 0; i < length; ++i) {
            char ch = source.charAt(i);
            if (this.isWhiteSpace(ch)) {
                if (previousIsWhiteSpace) continue;
                destination.append(' ');
                previousIsWhiteSpace = true;
                continue;
            }
            destination.append(ch);
            previousIsWhiteSpace = false;
        }
    }

    private boolean isWhiteSpace(char ch) {
        return ch == ' ' || ch == '\t' || ch == '\n' || ch == '\u000b' || ch == '\r' || ch == '\f';
    }

    private String getSignedHeadersString(Map<String, List<String>> headers) {
        ArrayList<String> sortedHeaders = new ArrayList<String>(headers.keySet());
        sortedHeaders.sort(String.CASE_INSENSITIVE_ORDER);
        StringBuilder buffer = new StringBuilder();
        for (String header : sortedHeaders) {
            if (this.shouldExcludeHeaderFromSigning(header)) continue;
            if (buffer.length() > 0) {
                buffer.append(";");
            }
            buffer.append(StringUtils.lowerCase((String)header));
        }
        return buffer.toString();
    }

    private boolean shouldExcludeHeaderFromSigning(String header) {
        return LIST_OF_HEADERS_TO_IGNORE_IN_LOWER_CASE.contains(StringUtils.lowerCase((String)header));
    }

    private void addHostHeader(SdkHttpFullRequest.Builder mutableRequest) {
        StringBuilder hostHeaderBuilder = new StringBuilder(mutableRequest.host());
        if (!SdkHttpUtils.isUsingStandardPort((String)mutableRequest.protocol(), (Integer)mutableRequest.port())) {
            hostHeaderBuilder.append(":").append(mutableRequest.port());
        }
        mutableRequest.putHeader("Host", hostHeaderBuilder.toString());
    }

    private void addDateHeader(SdkHttpFullRequest.Builder mutableRequest, String dateTime) {
        mutableRequest.putHeader("X-Amz-Date", dateTime);
    }

    private long generateExpirationTime(U signingParams) {
        long expirationInSeconds = ((Aws4PresignerParams)signingParams).expirationTime().map(Instant::getEpochSecond).orElse(604800L);
        if (expirationInSeconds > 604800L) {
            throw SdkClientException.builder().message("Requests that are pre-signed by SigV4 algorithm are valid for at most 7 days. The expiration date set on the current request [" + Aws4SignerUtils.formatTimestamp(expirationInSeconds * 1000L) + "] + has exceeded this limit.").build();
        }
        return expirationInSeconds;
    }

    private byte[] newSigningKey(AwsCredentials credentials, String dateStamp, String regionName, String serviceName) {
        byte[] kSecret = ("AWS4" + credentials.secretAccessKey()).getBytes(Charset.forName("UTF-8"));
        byte[] kDate = this.sign(dateStamp, kSecret, SigningAlgorithm.HmacSHA256);
        byte[] kRegion = this.sign(regionName, kDate, SigningAlgorithm.HmacSHA256);
        byte[] kService = this.sign(serviceName, kRegion, SigningAlgorithm.HmacSHA256);
        return this.sign("aws4_request", kService, SigningAlgorithm.HmacSHA256);
    }

    protected <B extends Aws4PresignerParams.Builder> B extractPresignerParams(B builder, ExecutionAttributes executionAttributes) {
        builder = this.extractSignerParams(builder, executionAttributes);
        builder.expirationTime((Instant)executionAttributes.getAttribute(AwsSignerExecutionAttribute.PRESIGNER_EXPIRATION));
        return builder;
    }

    protected <B extends Aws4SignerParams.Builder> B extractSignerParams(B paramsBuilder, ExecutionAttributes executionAttributes) {
        paramsBuilder.awsCredentials((AwsCredentials)executionAttributes.getAttribute(AwsSignerExecutionAttribute.AWS_CREDENTIALS)).signingName((String)executionAttributes.getAttribute(AwsSignerExecutionAttribute.SERVICE_SIGNING_NAME)).signingRegion((Region)executionAttributes.getAttribute(AwsSignerExecutionAttribute.SIGNING_REGION)).timeOffset((Integer)executionAttributes.getAttribute(AwsSignerExecutionAttribute.TIME_OFFSET));
        if (executionAttributes.getAttribute(AwsSignerExecutionAttribute.SIGNER_DOUBLE_URL_ENCODE) != null) {
            paramsBuilder.doubleUrlEncode((Boolean)executionAttributes.getAttribute(AwsSignerExecutionAttribute.SIGNER_DOUBLE_URL_ENCODE));
        }
        return paramsBuilder;
    }
}

