/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appsync.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an API key.
 * </p>
 * <p>
 * Customers invoke AWS AppSync GraphQL API operations with API keys as an identity mechanism. There are two key
 * versions:
 * </p>
 * <p>
 * <b>da1</b>: This version was introduced at launch in November 2017. These keys always expire after 7 days. Key
 * expiration is managed by Amazon DynamoDB TTL. The keys ceased to be valid after February 21, 2018 and should not be
 * used after that date.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>ListApiKeys</code> returns the expiration time in milliseconds.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CreateApiKey</code> returns the expiration time in milliseconds.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UpdateApiKey</code> is not available for this key version.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>DeleteApiKey</code> deletes the item from the table.
 * </p>
 * </li>
 * <li>
 * <p>
 * Expiration is stored in Amazon DynamoDB as milliseconds. This results in a bug where keys are not automatically
 * deleted because DynamoDB expects the TTL to be stored in seconds. As a one-time action, we will delete these keys
 * from the table after February 21, 2018.
 * </p>
 * </li>
 * </ul>
 * <p>
 * <b>da2</b>: This version was introduced in February 2018 when AppSync added support to extend key expiration.
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>ListApiKeys</code> returns the expiration time in seconds.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>CreateApiKey</code> returns the expiration time in seconds and accepts a user-provided expiration time in
 * seconds.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>UpdateApiKey</code> returns the expiration time in seconds and accepts a user-provided expiration time in
 * seconds. Key expiration can only be updated while the key has not expired.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>DeleteApiKey</code> deletes the item from the table.
 * </p>
 * </li>
 * <li>
 * <p>
 * Expiration is stored in Amazon DynamoDB as seconds.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ApiKey implements SdkPojo, Serializable, ToCopyableBuilder<ApiKey.Builder, ApiKey> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).getter(getter(ApiKey::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("id").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ApiKey::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("description").build()).build();

    private static final SdkField<Long> EXPIRES_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .getter(getter(ApiKey::expires)).setter(setter(Builder::expires))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expires").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, DESCRIPTION_FIELD,
            EXPIRES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String description;

    private final Long expires;

    private ApiKey(BuilderImpl builder) {
        this.id = builder.id;
        this.description = builder.description;
        this.expires = builder.expires;
    }

    /**
     * <p>
     * The API key ID.
     * </p>
     * 
     * @return The API key ID.
     */
    public String id() {
        return id;
    }

    /**
     * <p>
     * A description of the purpose of the API key.
     * </p>
     * 
     * @return A description of the purpose of the API key.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The time after which the API key expires. The date is represented as seconds since the epoch, rounded down to the
     * nearest hour.
     * </p>
     * 
     * @return The time after which the API key expires. The date is represented as seconds since the epoch, rounded
     *         down to the nearest hour.
     */
    public Long expires() {
        return expires;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(expires());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ApiKey)) {
            return false;
        }
        ApiKey other = (ApiKey) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(description(), other.description())
                && Objects.equals(expires(), other.expires());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ApiKey").add("Id", id()).add("Description", description()).add("Expires", expires()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "id":
            return Optional.ofNullable(clazz.cast(id()));
        case "description":
            return Optional.ofNullable(clazz.cast(description()));
        case "expires":
            return Optional.ofNullable(clazz.cast(expires()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ApiKey, T> g) {
        return obj -> g.apply((ApiKey) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ApiKey> {
        /**
         * <p>
         * The API key ID.
         * </p>
         * 
         * @param id
         *        The API key ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * A description of the purpose of the API key.
         * </p>
         * 
         * @param description
         *        A description of the purpose of the API key.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The time after which the API key expires. The date is represented as seconds since the epoch, rounded down to
         * the nearest hour.
         * </p>
         * 
         * @param expires
         *        The time after which the API key expires. The date is represented as seconds since the epoch, rounded
         *        down to the nearest hour.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expires(Long expires);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String description;

        private Long expires;

        private BuilderImpl() {
        }

        private BuilderImpl(ApiKey model) {
            id(model.id);
            description(model.description);
            expires(model.expires);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Long getExpires() {
            return expires;
        }

        @Override
        public final Builder expires(Long expires) {
            this.expires = expires;
            return this;
        }

        public final void setExpires(Long expires) {
            this.expires = expires;
        }

        @Override
        public ApiKey build() {
            return new ApiKey(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
