/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apprunner.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a custom domain that's associated with an App Runner service.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CustomDomain implements SdkPojo, Serializable, ToCopyableBuilder<CustomDomain.Builder, CustomDomain> {
    private static final SdkField<String> DOMAIN_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DomainName").getter(getter(CustomDomain::domainName)).setter(setter(Builder::domainName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainName").build()).build();

    private static final SdkField<Boolean> ENABLE_WWW_SUBDOMAIN_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("EnableWWWSubdomain").getter(getter(CustomDomain::enableWWWSubdomain))
            .setter(setter(Builder::enableWWWSubdomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EnableWWWSubdomain").build())
            .build();

    private static final SdkField<List<CertificateValidationRecord>> CERTIFICATE_VALIDATION_RECORDS_FIELD = SdkField
            .<List<CertificateValidationRecord>> builder(MarshallingType.LIST)
            .memberName("CertificateValidationRecords")
            .getter(getter(CustomDomain::certificateValidationRecords))
            .setter(setter(Builder::certificateValidationRecords))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateValidationRecords")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<CertificateValidationRecord> builder(MarshallingType.SDK_POJO)
                                            .constructor(CertificateValidationRecord::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(CustomDomain::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DOMAIN_NAME_FIELD,
            ENABLE_WWW_SUBDOMAIN_FIELD, CERTIFICATE_VALIDATION_RECORDS_FIELD, STATUS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String domainName;

    private final Boolean enableWWWSubdomain;

    private final List<CertificateValidationRecord> certificateValidationRecords;

    private final String status;

    private CustomDomain(BuilderImpl builder) {
        this.domainName = builder.domainName;
        this.enableWWWSubdomain = builder.enableWWWSubdomain;
        this.certificateValidationRecords = builder.certificateValidationRecords;
        this.status = builder.status;
    }

    /**
     * <p>
     * An associated custom domain endpoint. It can be a root domain (for example, <code>example.com</code>), a
     * subdomain (for example, <code>login.example.com</code> or <code>admin.login.example.com</code>), or a wildcard
     * (for example, <code>*.example.com</code>).
     * </p>
     * 
     * @return An associated custom domain endpoint. It can be a root domain (for example, <code>example.com</code>), a
     *         subdomain (for example, <code>login.example.com</code> or <code>admin.login.example.com</code>), or a
     *         wildcard (for example, <code>*.example.com</code>).
     */
    public final String domainName() {
        return domainName;
    }

    /**
     * <p>
     * When <code>true</code>, the subdomain <code>www.<i>DomainName</i> </code> is associated with the App Runner
     * service in addition to the base domain.
     * </p>
     * 
     * @return When <code>true</code>, the subdomain <code>www.<i>DomainName</i> </code> is associated with the App
     *         Runner service in addition to the base domain.
     */
    public final Boolean enableWWWSubdomain() {
        return enableWWWSubdomain;
    }

    /**
     * For responses, this returns true if the service returned a value for the CertificateValidationRecords property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasCertificateValidationRecords() {
        return certificateValidationRecords != null && !(certificateValidationRecords instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of certificate CNAME records that's used for this domain name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCertificateValidationRecords} method.
     * </p>
     * 
     * @return A list of certificate CNAME records that's used for this domain name.
     */
    public final List<CertificateValidationRecord> certificateValidationRecords() {
        return certificateValidationRecords;
    }

    /**
     * <p>
     * The current state of the domain name association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CustomDomainAssociationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the domain name association.
     * @see CustomDomainAssociationStatus
     */
    public final CustomDomainAssociationStatus status() {
        return CustomDomainAssociationStatus.fromValue(status);
    }

    /**
     * <p>
     * The current state of the domain name association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link CustomDomainAssociationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusAsString}.
     * </p>
     * 
     * @return The current state of the domain name association.
     * @see CustomDomainAssociationStatus
     */
    public final String statusAsString() {
        return status;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(domainName());
        hashCode = 31 * hashCode + Objects.hashCode(enableWWWSubdomain());
        hashCode = 31 * hashCode + Objects.hashCode(hasCertificateValidationRecords() ? certificateValidationRecords() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CustomDomain)) {
            return false;
        }
        CustomDomain other = (CustomDomain) obj;
        return Objects.equals(domainName(), other.domainName())
                && Objects.equals(enableWWWSubdomain(), other.enableWWWSubdomain())
                && hasCertificateValidationRecords() == other.hasCertificateValidationRecords()
                && Objects.equals(certificateValidationRecords(), other.certificateValidationRecords())
                && Objects.equals(statusAsString(), other.statusAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CustomDomain").add("DomainName", domainName()).add("EnableWWWSubdomain", enableWWWSubdomain())
                .add("CertificateValidationRecords", hasCertificateValidationRecords() ? certificateValidationRecords() : null)
                .add("Status", statusAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DomainName":
            return Optional.ofNullable(clazz.cast(domainName()));
        case "EnableWWWSubdomain":
            return Optional.ofNullable(clazz.cast(enableWWWSubdomain()));
        case "CertificateValidationRecords":
            return Optional.ofNullable(clazz.cast(certificateValidationRecords()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DomainName", DOMAIN_NAME_FIELD);
        map.put("EnableWWWSubdomain", ENABLE_WWW_SUBDOMAIN_FIELD);
        map.put("CertificateValidationRecords", CERTIFICATE_VALIDATION_RECORDS_FIELD);
        map.put("Status", STATUS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CustomDomain, T> g) {
        return obj -> g.apply((CustomDomain) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CustomDomain> {
        /**
         * <p>
         * An associated custom domain endpoint. It can be a root domain (for example, <code>example.com</code>), a
         * subdomain (for example, <code>login.example.com</code> or <code>admin.login.example.com</code>), or a
         * wildcard (for example, <code>*.example.com</code>).
         * </p>
         * 
         * @param domainName
         *        An associated custom domain endpoint. It can be a root domain (for example, <code>example.com</code>),
         *        a subdomain (for example, <code>login.example.com</code> or <code>admin.login.example.com</code>), or
         *        a wildcard (for example, <code>*.example.com</code>).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainName(String domainName);

        /**
         * <p>
         * When <code>true</code>, the subdomain <code>www.<i>DomainName</i> </code> is associated with the App Runner
         * service in addition to the base domain.
         * </p>
         * 
         * @param enableWWWSubdomain
         *        When <code>true</code>, the subdomain <code>www.<i>DomainName</i> </code> is associated with the App
         *        Runner service in addition to the base domain.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enableWWWSubdomain(Boolean enableWWWSubdomain);

        /**
         * <p>
         * A list of certificate CNAME records that's used for this domain name.
         * </p>
         * 
         * @param certificateValidationRecords
         *        A list of certificate CNAME records that's used for this domain name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateValidationRecords(Collection<CertificateValidationRecord> certificateValidationRecords);

        /**
         * <p>
         * A list of certificate CNAME records that's used for this domain name.
         * </p>
         * 
         * @param certificateValidationRecords
         *        A list of certificate CNAME records that's used for this domain name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateValidationRecords(CertificateValidationRecord... certificateValidationRecords);

        /**
         * <p>
         * A list of certificate CNAME records that's used for this domain name.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.apprunner.model.CertificateValidationRecord.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.apprunner.model.CertificateValidationRecord#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.apprunner.model.CertificateValidationRecord.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #certificateValidationRecords(List<CertificateValidationRecord>)}.
         * 
         * @param certificateValidationRecords
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.apprunner.model.CertificateValidationRecord.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #certificateValidationRecords(java.util.Collection<CertificateValidationRecord>)
         */
        Builder certificateValidationRecords(Consumer<CertificateValidationRecord.Builder>... certificateValidationRecords);

        /**
         * <p>
         * The current state of the domain name association.
         * </p>
         * 
         * @param status
         *        The current state of the domain name association.
         * @see CustomDomainAssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomDomainAssociationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current state of the domain name association.
         * </p>
         * 
         * @param status
         *        The current state of the domain name association.
         * @see CustomDomainAssociationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CustomDomainAssociationStatus
         */
        Builder status(CustomDomainAssociationStatus status);
    }

    static final class BuilderImpl implements Builder {
        private String domainName;

        private Boolean enableWWWSubdomain;

        private List<CertificateValidationRecord> certificateValidationRecords = DefaultSdkAutoConstructList.getInstance();

        private String status;

        private BuilderImpl() {
        }

        private BuilderImpl(CustomDomain model) {
            domainName(model.domainName);
            enableWWWSubdomain(model.enableWWWSubdomain);
            certificateValidationRecords(model.certificateValidationRecords);
            status(model.status);
        }

        public final String getDomainName() {
            return domainName;
        }

        public final void setDomainName(String domainName) {
            this.domainName = domainName;
        }

        @Override
        public final Builder domainName(String domainName) {
            this.domainName = domainName;
            return this;
        }

        public final Boolean getEnableWWWSubdomain() {
            return enableWWWSubdomain;
        }

        public final void setEnableWWWSubdomain(Boolean enableWWWSubdomain) {
            this.enableWWWSubdomain = enableWWWSubdomain;
        }

        @Override
        public final Builder enableWWWSubdomain(Boolean enableWWWSubdomain) {
            this.enableWWWSubdomain = enableWWWSubdomain;
            return this;
        }

        public final List<CertificateValidationRecord.Builder> getCertificateValidationRecords() {
            List<CertificateValidationRecord.Builder> result = CertificateValidationRecordListCopier
                    .copyToBuilder(this.certificateValidationRecords);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setCertificateValidationRecords(
                Collection<CertificateValidationRecord.BuilderImpl> certificateValidationRecords) {
            this.certificateValidationRecords = CertificateValidationRecordListCopier
                    .copyFromBuilder(certificateValidationRecords);
        }

        @Override
        public final Builder certificateValidationRecords(Collection<CertificateValidationRecord> certificateValidationRecords) {
            this.certificateValidationRecords = CertificateValidationRecordListCopier.copy(certificateValidationRecords);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder certificateValidationRecords(CertificateValidationRecord... certificateValidationRecords) {
            certificateValidationRecords(Arrays.asList(certificateValidationRecords));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder certificateValidationRecords(
                Consumer<CertificateValidationRecord.Builder>... certificateValidationRecords) {
            certificateValidationRecords(Stream.of(certificateValidationRecords)
                    .map(c -> CertificateValidationRecord.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(CustomDomainAssociationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        @Override
        public CustomDomain build() {
            return new CustomDomain(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
