/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationinsights.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a problem that is detected by correlating observations.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Problem implements SdkPojo, Serializable, ToCopyableBuilder<Problem.Builder, Problem> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(Problem::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> TITLE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Title")
            .getter(getter(Problem::title)).setter(setter(Builder::title))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Title").build()).build();

    private static final SdkField<String> INSIGHTS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Insights").getter(getter(Problem::insights)).setter(setter(Builder::insights))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Insights").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Problem::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> AFFECTED_RESOURCE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AffectedResource").getter(getter(Problem::affectedResource)).setter(setter(Builder::affectedResource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AffectedResource").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(Problem::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(Problem::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> SEVERITY_LEVEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SeverityLevel").getter(getter(Problem::severityLevelAsString)).setter(setter(Builder::severityLevel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SeverityLevel").build()).build();

    private static final SdkField<String> RESOURCE_GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceGroupName").getter(getter(Problem::resourceGroupName))
            .setter(setter(Builder::resourceGroupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceGroupName").build()).build();

    private static final SdkField<Map<String, String>> FEEDBACK_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("Feedback")
            .getter(getter(Problem::feedbackAsStrings))
            .setter(setter(Builder::feedbackWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Feedback").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<Long> RECURRING_COUNT_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("RecurringCount").getter(getter(Problem::recurringCount)).setter(setter(Builder::recurringCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RecurringCount").build()).build();

    private static final SdkField<Instant> LAST_RECURRENCE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastRecurrenceTime").getter(getter(Problem::lastRecurrenceTime))
            .setter(setter(Builder::lastRecurrenceTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastRecurrenceTime").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, TITLE_FIELD,
            INSIGHTS_FIELD, STATUS_FIELD, AFFECTED_RESOURCE_FIELD, START_TIME_FIELD, END_TIME_FIELD, SEVERITY_LEVEL_FIELD,
            RESOURCE_GROUP_NAME_FIELD, FEEDBACK_FIELD, RECURRING_COUNT_FIELD, LAST_RECURRENCE_TIME_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String title;

    private final String insights;

    private final String status;

    private final String affectedResource;

    private final Instant startTime;

    private final Instant endTime;

    private final String severityLevel;

    private final String resourceGroupName;

    private final Map<String, String> feedback;

    private final Long recurringCount;

    private final Instant lastRecurrenceTime;

    private Problem(BuilderImpl builder) {
        this.id = builder.id;
        this.title = builder.title;
        this.insights = builder.insights;
        this.status = builder.status;
        this.affectedResource = builder.affectedResource;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.severityLevel = builder.severityLevel;
        this.resourceGroupName = builder.resourceGroupName;
        this.feedback = builder.feedback;
        this.recurringCount = builder.recurringCount;
        this.lastRecurrenceTime = builder.lastRecurrenceTime;
    }

    /**
     * <p>
     * The ID of the problem.
     * </p>
     * 
     * @return The ID of the problem.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The name of the problem.
     * </p>
     * 
     * @return The name of the problem.
     */
    public final String title() {
        return title;
    }

    /**
     * <p>
     * A detailed analysis of the problem using machine learning.
     * </p>
     * 
     * @return A detailed analysis of the problem using machine learning.
     */
    public final String insights() {
        return insights;
    }

    /**
     * <p>
     * The status of the problem.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the problem.
     * @see Status
     */
    public final Status status() {
        return Status.fromValue(status);
    }

    /**
     * <p>
     * The status of the problem.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link Status#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the problem.
     * @see Status
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The resource affected by the problem.
     * </p>
     * 
     * @return The resource affected by the problem.
     */
    public final String affectedResource() {
        return affectedResource;
    }

    /**
     * <p>
     * The time when the problem started, in epoch seconds.
     * </p>
     * 
     * @return The time when the problem started, in epoch seconds.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time when the problem ended, in epoch seconds.
     * </p>
     * 
     * @return The time when the problem ended, in epoch seconds.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * A measure of the level of impact of the problem.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityLevel}
     * will return {@link SeverityLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityLevelAsString}.
     * </p>
     * 
     * @return A measure of the level of impact of the problem.
     * @see SeverityLevel
     */
    public final SeverityLevel severityLevel() {
        return SeverityLevel.fromValue(severityLevel);
    }

    /**
     * <p>
     * A measure of the level of impact of the problem.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #severityLevel}
     * will return {@link SeverityLevel#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #severityLevelAsString}.
     * </p>
     * 
     * @return A measure of the level of impact of the problem.
     * @see SeverityLevel
     */
    public final String severityLevelAsString() {
        return severityLevel;
    }

    /**
     * <p>
     * The name of the resource group affected by the problem.
     * </p>
     * 
     * @return The name of the resource group affected by the problem.
     */
    public final String resourceGroupName() {
        return resourceGroupName;
    }

    /**
     * <p>
     * Feedback provided by the user about the problem.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeedback} method.
     * </p>
     * 
     * @return Feedback provided by the user about the problem.
     */
    public final Map<FeedbackKey, FeedbackValue> feedback() {
        return FeedbackCopier.copyStringToEnum(feedback);
    }

    /**
     * For responses, this returns true if the service returned a value for the Feedback property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFeedback() {
        return feedback != null && !(feedback instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * Feedback provided by the user about the problem.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFeedback} method.
     * </p>
     * 
     * @return Feedback provided by the user about the problem.
     */
    public final Map<String, String> feedbackAsStrings() {
        return feedback;
    }

    /**
     * <p>
     * The number of times that the same problem reoccurred after the first time it was resolved.
     * </p>
     * 
     * @return The number of times that the same problem reoccurred after the first time it was resolved.
     */
    public final Long recurringCount() {
        return recurringCount;
    }

    /**
     * <p>
     * The last time that the problem reoccurred after its last resolution.
     * </p>
     * 
     * @return The last time that the problem reoccurred after its last resolution.
     */
    public final Instant lastRecurrenceTime() {
        return lastRecurrenceTime;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(title());
        hashCode = 31 * hashCode + Objects.hashCode(insights());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(affectedResource());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(severityLevelAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceGroupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasFeedback() ? feedbackAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(recurringCount());
        hashCode = 31 * hashCode + Objects.hashCode(lastRecurrenceTime());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Problem)) {
            return false;
        }
        Problem other = (Problem) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(title(), other.title())
                && Objects.equals(insights(), other.insights()) && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(affectedResource(), other.affectedResource()) && Objects.equals(startTime(), other.startTime())
                && Objects.equals(endTime(), other.endTime())
                && Objects.equals(severityLevelAsString(), other.severityLevelAsString())
                && Objects.equals(resourceGroupName(), other.resourceGroupName()) && hasFeedback() == other.hasFeedback()
                && Objects.equals(feedbackAsStrings(), other.feedbackAsStrings())
                && Objects.equals(recurringCount(), other.recurringCount())
                && Objects.equals(lastRecurrenceTime(), other.lastRecurrenceTime());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Problem").add("Id", id()).add("Title", title()).add("Insights", insights())
                .add("Status", statusAsString()).add("AffectedResource", affectedResource()).add("StartTime", startTime())
                .add("EndTime", endTime()).add("SeverityLevel", severityLevelAsString())
                .add("ResourceGroupName", resourceGroupName()).add("Feedback", hasFeedback() ? feedbackAsStrings() : null)
                .add("RecurringCount", recurringCount()).add("LastRecurrenceTime", lastRecurrenceTime()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Title":
            return Optional.ofNullable(clazz.cast(title()));
        case "Insights":
            return Optional.ofNullable(clazz.cast(insights()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "AffectedResource":
            return Optional.ofNullable(clazz.cast(affectedResource()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "SeverityLevel":
            return Optional.ofNullable(clazz.cast(severityLevelAsString()));
        case "ResourceGroupName":
            return Optional.ofNullable(clazz.cast(resourceGroupName()));
        case "Feedback":
            return Optional.ofNullable(clazz.cast(feedbackAsStrings()));
        case "RecurringCount":
            return Optional.ofNullable(clazz.cast(recurringCount()));
        case "LastRecurrenceTime":
            return Optional.ofNullable(clazz.cast(lastRecurrenceTime()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Problem, T> g) {
        return obj -> g.apply((Problem) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Problem> {
        /**
         * <p>
         * The ID of the problem.
         * </p>
         * 
         * @param id
         *        The ID of the problem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The name of the problem.
         * </p>
         * 
         * @param title
         *        The name of the problem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder title(String title);

        /**
         * <p>
         * A detailed analysis of the problem using machine learning.
         * </p>
         * 
         * @param insights
         *        A detailed analysis of the problem using machine learning.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder insights(String insights);

        /**
         * <p>
         * The status of the problem.
         * </p>
         * 
         * @param status
         *        The status of the problem.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the problem.
         * </p>
         * 
         * @param status
         *        The status of the problem.
         * @see Status
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Status
         */
        Builder status(Status status);

        /**
         * <p>
         * The resource affected by the problem.
         * </p>
         * 
         * @param affectedResource
         *        The resource affected by the problem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder affectedResource(String affectedResource);

        /**
         * <p>
         * The time when the problem started, in epoch seconds.
         * </p>
         * 
         * @param startTime
         *        The time when the problem started, in epoch seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time when the problem ended, in epoch seconds.
         * </p>
         * 
         * @param endTime
         *        The time when the problem ended, in epoch seconds.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * A measure of the level of impact of the problem.
         * </p>
         * 
         * @param severityLevel
         *        A measure of the level of impact of the problem.
         * @see SeverityLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityLevel
         */
        Builder severityLevel(String severityLevel);

        /**
         * <p>
         * A measure of the level of impact of the problem.
         * </p>
         * 
         * @param severityLevel
         *        A measure of the level of impact of the problem.
         * @see SeverityLevel
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SeverityLevel
         */
        Builder severityLevel(SeverityLevel severityLevel);

        /**
         * <p>
         * The name of the resource group affected by the problem.
         * </p>
         * 
         * @param resourceGroupName
         *        The name of the resource group affected by the problem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceGroupName(String resourceGroupName);

        /**
         * <p>
         * Feedback provided by the user about the problem.
         * </p>
         * 
         * @param feedback
         *        Feedback provided by the user about the problem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder feedbackWithStrings(Map<String, String> feedback);

        /**
         * <p>
         * Feedback provided by the user about the problem.
         * </p>
         * 
         * @param feedback
         *        Feedback provided by the user about the problem.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder feedback(Map<FeedbackKey, FeedbackValue> feedback);

        /**
         * <p>
         * The number of times that the same problem reoccurred after the first time it was resolved.
         * </p>
         * 
         * @param recurringCount
         *        The number of times that the same problem reoccurred after the first time it was resolved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recurringCount(Long recurringCount);

        /**
         * <p>
         * The last time that the problem reoccurred after its last resolution.
         * </p>
         * 
         * @param lastRecurrenceTime
         *        The last time that the problem reoccurred after its last resolution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRecurrenceTime(Instant lastRecurrenceTime);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String title;

        private String insights;

        private String status;

        private String affectedResource;

        private Instant startTime;

        private Instant endTime;

        private String severityLevel;

        private String resourceGroupName;

        private Map<String, String> feedback = DefaultSdkAutoConstructMap.getInstance();

        private Long recurringCount;

        private Instant lastRecurrenceTime;

        private BuilderImpl() {
        }

        private BuilderImpl(Problem model) {
            id(model.id);
            title(model.title);
            insights(model.insights);
            status(model.status);
            affectedResource(model.affectedResource);
            startTime(model.startTime);
            endTime(model.endTime);
            severityLevel(model.severityLevel);
            resourceGroupName(model.resourceGroupName);
            feedbackWithStrings(model.feedback);
            recurringCount(model.recurringCount);
            lastRecurrenceTime(model.lastRecurrenceTime);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final String getTitle() {
            return title;
        }

        public final void setTitle(String title) {
            this.title = title;
        }

        @Override
        public final Builder title(String title) {
            this.title = title;
            return this;
        }

        public final String getInsights() {
            return insights;
        }

        public final void setInsights(String insights) {
            this.insights = insights;
        }

        @Override
        public final Builder insights(String insights) {
            this.insights = insights;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(Status status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getAffectedResource() {
            return affectedResource;
        }

        public final void setAffectedResource(String affectedResource) {
            this.affectedResource = affectedResource;
        }

        @Override
        public final Builder affectedResource(String affectedResource) {
            this.affectedResource = affectedResource;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getSeverityLevel() {
            return severityLevel;
        }

        public final void setSeverityLevel(String severityLevel) {
            this.severityLevel = severityLevel;
        }

        @Override
        public final Builder severityLevel(String severityLevel) {
            this.severityLevel = severityLevel;
            return this;
        }

        @Override
        public final Builder severityLevel(SeverityLevel severityLevel) {
            this.severityLevel(severityLevel == null ? null : severityLevel.toString());
            return this;
        }

        public final String getResourceGroupName() {
            return resourceGroupName;
        }

        public final void setResourceGroupName(String resourceGroupName) {
            this.resourceGroupName = resourceGroupName;
        }

        @Override
        public final Builder resourceGroupName(String resourceGroupName) {
            this.resourceGroupName = resourceGroupName;
            return this;
        }

        public final Map<String, String> getFeedback() {
            if (feedback instanceof SdkAutoConstructMap) {
                return null;
            }
            return feedback;
        }

        public final void setFeedback(Map<String, String> feedback) {
            this.feedback = FeedbackCopier.copy(feedback);
        }

        @Override
        public final Builder feedbackWithStrings(Map<String, String> feedback) {
            this.feedback = FeedbackCopier.copy(feedback);
            return this;
        }

        @Override
        public final Builder feedback(Map<FeedbackKey, FeedbackValue> feedback) {
            this.feedback = FeedbackCopier.copyEnumToString(feedback);
            return this;
        }

        public final Long getRecurringCount() {
            return recurringCount;
        }

        public final void setRecurringCount(Long recurringCount) {
            this.recurringCount = recurringCount;
        }

        @Override
        public final Builder recurringCount(Long recurringCount) {
            this.recurringCount = recurringCount;
            return this;
        }

        public final Instant getLastRecurrenceTime() {
            return lastRecurrenceTime;
        }

        public final void setLastRecurrenceTime(Instant lastRecurrenceTime) {
            this.lastRecurrenceTime = lastRecurrenceTime;
        }

        @Override
        public final Builder lastRecurrenceTime(Instant lastRecurrenceTime) {
            this.lastRecurrenceTime = lastRecurrenceTime;
            return this;
        }

        @Override
        public Problem build() {
            return new Problem(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
