/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.applicationautoscaling.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a scaling activity.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ScalingActivity implements SdkPojo, Serializable, ToCopyableBuilder<ScalingActivity.Builder, ScalingActivity> {
    private static final SdkField<String> ACTIVITY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ActivityId").getter(getter(ScalingActivity::activityId)).setter(setter(Builder::activityId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ActivityId").build()).build();

    private static final SdkField<String> SERVICE_NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ServiceNamespace").getter(getter(ScalingActivity::serviceNamespaceAsString))
            .setter(setter(Builder::serviceNamespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceNamespace").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ResourceId").getter(getter(ScalingActivity::resourceId)).setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId").build()).build();

    private static final SdkField<String> SCALABLE_DIMENSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScalableDimension").getter(getter(ScalingActivity::scalableDimensionAsString))
            .setter(setter(Builder::scalableDimension))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ScalableDimension").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Description").getter(getter(ScalingActivity::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<String> CAUSE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Cause")
            .getter(getter(ScalingActivity::cause)).setter(setter(Builder::cause))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cause").build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(ScalingActivity::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(ScalingActivity::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusCode").getter(getter(ScalingActivity::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage").getter(getter(ScalingActivity::statusMessage)).setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage").build()).build();

    private static final SdkField<String> DETAILS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Details")
            .getter(getter(ScalingActivity::details)).setter(setter(Builder::details))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Details").build()).build();

    private static final SdkField<List<NotScaledReason>> NOT_SCALED_REASONS_FIELD = SdkField
            .<List<NotScaledReason>> builder(MarshallingType.LIST)
            .memberName("NotScaledReasons")
            .getter(getter(ScalingActivity::notScaledReasons))
            .setter(setter(Builder::notScaledReasons))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotScaledReasons").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<NotScaledReason> builder(MarshallingType.SDK_POJO)
                                            .constructor(NotScaledReason::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACTIVITY_ID_FIELD,
            SERVICE_NAMESPACE_FIELD, RESOURCE_ID_FIELD, SCALABLE_DIMENSION_FIELD, DESCRIPTION_FIELD, CAUSE_FIELD,
            START_TIME_FIELD, END_TIME_FIELD, STATUS_CODE_FIELD, STATUS_MESSAGE_FIELD, DETAILS_FIELD, NOT_SCALED_REASONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String activityId;

    private final String serviceNamespace;

    private final String resourceId;

    private final String scalableDimension;

    private final String description;

    private final String cause;

    private final Instant startTime;

    private final Instant endTime;

    private final String statusCode;

    private final String statusMessage;

    private final String details;

    private final List<NotScaledReason> notScaledReasons;

    private ScalingActivity(BuilderImpl builder) {
        this.activityId = builder.activityId;
        this.serviceNamespace = builder.serviceNamespace;
        this.resourceId = builder.resourceId;
        this.scalableDimension = builder.scalableDimension;
        this.description = builder.description;
        this.cause = builder.cause;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.statusCode = builder.statusCode;
        this.statusMessage = builder.statusMessage;
        this.details = builder.details;
        this.notScaledReasons = builder.notScaledReasons;
    }

    /**
     * <p>
     * The unique identifier of the scaling activity.
     * </p>
     * 
     * @return The unique identifier of the scaling activity.
     */
    public final String activityId() {
        return activityId;
    }

    /**
     * <p>
     * The namespace of the Amazon Web Services service that provides the resource, or a <code>custom-resource</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the Amazon Web Services service that provides the resource, or a
     *         <code>custom-resource</code>.
     * @see ServiceNamespace
     */
    public final ServiceNamespace serviceNamespace() {
        return ServiceNamespace.fromValue(serviceNamespace);
    }

    /**
     * <p>
     * The namespace of the Amazon Web Services service that provides the resource, or a <code>custom-resource</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #serviceNamespace}
     * will return {@link ServiceNamespace#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #serviceNamespaceAsString}.
     * </p>
     * 
     * @return The namespace of the Amazon Web Services service that provides the resource, or a
     *         <code>custom-resource</code>.
     * @see ServiceNamespace
     */
    public final String serviceNamespaceAsString() {
        return serviceNamespace;
    }

    /**
     * <p>
     * The identifier of the resource associated with the scaling activity. This string consists of the resource type
     * and unique identifier.
     * </p>
     * <ul>
     * <li>
     * <p>
     * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and service
     * name. Example: <code>service/default/sample-webapp</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Spot Fleet - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot Fleet
     * request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID and
     * instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet name.
     * Example: <code>fleet/sample-fleet</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name. Example:
     * <code>table/my-table</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the index
     * name. Example: <code>table/my-table/index/my-table-index</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
     * Example: <code>cluster:my-db-cluster</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is the resource
     * ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Custom resources are not supported with a resource type. This parameter must specify the <code>OutputValue</code>
     * from the CloudFormation template stack used to access the resources. The unique identifier is defined by the
     * service provider. More information is available in our <a
     * href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Comprehend document classification endpoint - The resource type and unique identifier are specified using
     * the endpoint ARN. Example:
     * <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Comprehend entity recognizer endpoint - The resource type and unique identifier are specified using the
     * endpoint ARN. Example: <code>arn:aws:comprehend:us-west-2:123456789012:entity-recognizer-endpoint/EXAMPLE</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is the
     * function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
     * <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the table name.
     * Example: <code>keyspace/mykeyspace/table/mytable</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon MSK cluster - The resource type and unique identifier are specified using the cluster ARN. Example:
     * <code>arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon ElastiCache replication group - The resource type is <code>replication-group</code> and the unique
     * identifier is the replication group name. Example: <code>replication-group/mycluster</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Neptune cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
     * Example: <code>cluster:mycluster</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * SageMaker Serverless endpoint - The resource type is <code>variant</code> and the unique identifier is the
     * resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The identifier of the resource associated with the scaling activity. This string consists of the resource
     *         type and unique identifier.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
     *         service name. Example: <code>service/default/sample-webapp</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Spot Fleet - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot
     *         Fleet request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID
     *         and instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet
     *         name. Example: <code>fleet/sample-fleet</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
     *         Example: <code>table/my-table</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
     *         the index name. Example: <code>table/my-table/index/my-table-index</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
     *         name. Example: <code>cluster:my-db-cluster</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is the
     *         resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Custom resources are not supported with a resource type. This parameter must specify the
     *         <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The unique
     *         identifier is defined by the service provider. More information is available in our <a
     *         href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Comprehend document classification endpoint - The resource type and unique identifier are
     *         specified using the endpoint ARN. Example:
     *         <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Comprehend entity recognizer endpoint - The resource type and unique identifier are specified
     *         using the endpoint ARN. Example:
     *         <code>arn:aws:comprehend:us-west-2:123456789012:entity-recognizer-endpoint/EXAMPLE</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is
     *         the function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
     *         <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the table
     *         name. Example: <code>keyspace/mykeyspace/table/mytable</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon MSK cluster - The resource type and unique identifier are specified using the cluster ARN.
     *         Example:
     *         <code>arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5</code>
     *         .
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Amazon ElastiCache replication group - The resource type is <code>replication-group</code> and the unique
     *         identifier is the replication group name. Example: <code>replication-group/mycluster</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Neptune cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
     *         name. Example: <code>cluster:mycluster</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         SageMaker Serverless endpoint - The resource type is <code>variant</code> and the unique identifier is
     *         the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
     *         </p>
     *         </li>
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker model endpoint
     * variant.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
     * your own application or service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend document classification endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend entity recognizer endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an Amazon MSK
     * cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon ElastiCache
     * replication group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an Amazon
     * ElastiCache replication group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a SageMaker
     * Serverless endpoint.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension. This string consists of the service namespace, resource type, and scaling
     *         property.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker model
     *         endpoint variant.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
     *         provided by your own application or service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
     *         units for an Amazon Comprehend document classification endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference units
     *         for an Amazon Comprehend entity recognizer endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an
     *         Amazon MSK cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon
     *         ElastiCache replication group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an Amazon
     *         ElastiCache replication group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB
     *         cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a
     *         SageMaker Serverless endpoint.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public final ScalableDimension scalableDimension() {
        return ScalableDimension.fromValue(scalableDimension);
    }

    /**
     * <p>
     * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global secondary
     * index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available for
     * Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker model endpoint
     * variant.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
     * your own application or service.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend document classification endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference units for an
     * Amazon Comprehend entity recognizer endpoint.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an Amazon MSK
     * cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon ElastiCache
     * replication group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an Amazon
     * ElastiCache replication group.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB cluster.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a SageMaker
     * Serverless endpoint.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scalableDimension}
     * will return {@link ScalableDimension#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #scalableDimensionAsString}.
     * </p>
     * 
     * @return The scalable dimension. This string consists of the service namespace, resource type, and scaling
     *         property.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
     *         secondary index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
     *         Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker model
     *         endpoint variant.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
     *         provided by your own application or service.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
     *         units for an Amazon Comprehend document classification endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference units
     *         for an Amazon Comprehend entity recognizer endpoint.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
     *         table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an
     *         Amazon MSK cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon
     *         ElastiCache replication group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an Amazon
     *         ElastiCache replication group.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB
     *         cluster.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a
     *         SageMaker Serverless endpoint.
     *         </p>
     *         </li>
     * @see ScalableDimension
     */
    public final String scalableDimensionAsString() {
        return scalableDimension;
    }

    /**
     * <p>
     * A simple description of what action the scaling activity intends to accomplish.
     * </p>
     * 
     * @return A simple description of what action the scaling activity intends to accomplish.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * A simple description of what caused the scaling activity to happen.
     * </p>
     * 
     * @return A simple description of what caused the scaling activity to happen.
     */
    public final String cause() {
        return cause;
    }

    /**
     * <p>
     * The Unix timestamp for when the scaling activity began.
     * </p>
     * 
     * @return The Unix timestamp for when the scaling activity began.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The Unix timestamp for when the scaling activity ended.
     * </p>
     * 
     * @return The Unix timestamp for when the scaling activity ended.
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * Indicates the status of the scaling activity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link ScalingActivityStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Indicates the status of the scaling activity.
     * @see ScalingActivityStatusCode
     */
    public final ScalingActivityStatusCode statusCode() {
        return ScalingActivityStatusCode.fromValue(statusCode);
    }

    /**
     * <p>
     * Indicates the status of the scaling activity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link ScalingActivityStatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #statusCodeAsString}.
     * </p>
     * 
     * @return Indicates the status of the scaling activity.
     * @see ScalingActivityStatusCode
     */
    public final String statusCodeAsString() {
        return statusCode;
    }

    /**
     * <p>
     * A simple message about the current status of the scaling activity.
     * </p>
     * 
     * @return A simple message about the current status of the scaling activity.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * The details about the scaling activity.
     * </p>
     * 
     * @return The details about the scaling activity.
     */
    public final String details() {
        return details;
    }

    /**
     * For responses, this returns true if the service returned a value for the NotScaledReasons property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNotScaledReasons() {
        return notScaledReasons != null && !(notScaledReasons instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Machine-readable data that describes the reason for a not scaled activity. Only available when <a
     * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
     * >DescribeScalingActivities</a> includes not scaled activities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNotScaledReasons} method.
     * </p>
     * 
     * @return Machine-readable data that describes the reason for a not scaled activity. Only available when <a
     *         href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
     *         >DescribeScalingActivities</a> includes not scaled activities.
     */
    public final List<NotScaledReason> notScaledReasons() {
        return notScaledReasons;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(activityId());
        hashCode = 31 * hashCode + Objects.hashCode(serviceNamespaceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(scalableDimensionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(cause());
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(details());
        hashCode = 31 * hashCode + Objects.hashCode(hasNotScaledReasons() ? notScaledReasons() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ScalingActivity)) {
            return false;
        }
        ScalingActivity other = (ScalingActivity) obj;
        return Objects.equals(activityId(), other.activityId())
                && Objects.equals(serviceNamespaceAsString(), other.serviceNamespaceAsString())
                && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(scalableDimensionAsString(), other.scalableDimensionAsString())
                && Objects.equals(description(), other.description()) && Objects.equals(cause(), other.cause())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString())
                && Objects.equals(statusMessage(), other.statusMessage()) && Objects.equals(details(), other.details())
                && hasNotScaledReasons() == other.hasNotScaledReasons()
                && Objects.equals(notScaledReasons(), other.notScaledReasons());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ScalingActivity").add("ActivityId", activityId())
                .add("ServiceNamespace", serviceNamespaceAsString()).add("ResourceId", resourceId())
                .add("ScalableDimension", scalableDimensionAsString()).add("Description", description()).add("Cause", cause())
                .add("StartTime", startTime()).add("EndTime", endTime()).add("StatusCode", statusCodeAsString())
                .add("StatusMessage", statusMessage()).add("Details", details())
                .add("NotScaledReasons", hasNotScaledReasons() ? notScaledReasons() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ActivityId":
            return Optional.ofNullable(clazz.cast(activityId()));
        case "ServiceNamespace":
            return Optional.ofNullable(clazz.cast(serviceNamespaceAsString()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ScalableDimension":
            return Optional.ofNullable(clazz.cast(scalableDimensionAsString()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Cause":
            return Optional.ofNullable(clazz.cast(cause()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "Details":
            return Optional.ofNullable(clazz.cast(details()));
        case "NotScaledReasons":
            return Optional.ofNullable(clazz.cast(notScaledReasons()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ScalingActivity, T> g) {
        return obj -> g.apply((ScalingActivity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ScalingActivity> {
        /**
         * <p>
         * The unique identifier of the scaling activity.
         * </p>
         * 
         * @param activityId
         *        The unique identifier of the scaling activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder activityId(String activityId);

        /**
         * <p>
         * The namespace of the Amazon Web Services service that provides the resource, or a
         * <code>custom-resource</code>.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the Amazon Web Services service that provides the resource, or a
         *        <code>custom-resource</code>.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(String serviceNamespace);

        /**
         * <p>
         * The namespace of the Amazon Web Services service that provides the resource, or a
         * <code>custom-resource</code>.
         * </p>
         * 
         * @param serviceNamespace
         *        The namespace of the Amazon Web Services service that provides the resource, or a
         *        <code>custom-resource</code>.
         * @see ServiceNamespace
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ServiceNamespace
         */
        Builder serviceNamespace(ServiceNamespace serviceNamespace);

        /**
         * <p>
         * The identifier of the resource associated with the scaling activity. This string consists of the resource
         * type and unique identifier.
         * </p>
         * <ul>
         * <li>
         * <p>
         * ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name and
         * service name. Example: <code>service/default/sample-webapp</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Spot Fleet - The resource type is <code>spot-fleet-request</code> and the unique identifier is the Spot Fleet
         * request ID. Example: <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster ID and
         * instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet name.
         * Example: <code>fleet/sample-fleet</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
         * Example: <code>table/my-table</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is the
         * index name. Example: <code>table/my-table/index/my-table-index</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
         * Example: <code>cluster:my-db-cluster</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is the
         * resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Custom resources are not supported with a resource type. This parameter must specify the
         * <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The unique
         * identifier is defined by the service provider. More information is available in our <a
         * href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Comprehend document classification endpoint - The resource type and unique identifier are specified
         * using the endpoint ARN. Example:
         * <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Comprehend entity recognizer endpoint - The resource type and unique identifier are specified using
         * the endpoint ARN. Example:
         * <code>arn:aws:comprehend:us-west-2:123456789012:entity-recognizer-endpoint/EXAMPLE</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier is the
         * function name with a function version or alias name suffix that is not <code>$LATEST</code>. Example:
         * <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the table name.
         * Example: <code>keyspace/mykeyspace/table/mytable</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon MSK cluster - The resource type and unique identifier are specified using the cluster ARN. Example:
         * <code>arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5</code>
         * .
         * </p>
         * </li>
         * <li>
         * <p>
         * Amazon ElastiCache replication group - The resource type is <code>replication-group</code> and the unique
         * identifier is the replication group name. Example: <code>replication-group/mycluster</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Neptune cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster name.
         * Example: <code>cluster:mycluster</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * SageMaker Serverless endpoint - The resource type is <code>variant</code> and the unique identifier is the
         * resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         * </p>
         * </li>
         * </ul>
         * 
         * @param resourceId
         *        The identifier of the resource associated with the scaling activity. This string consists of the
         *        resource type and unique identifier.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        ECS service - The resource type is <code>service</code> and the unique identifier is the cluster name
         *        and service name. Example: <code>service/default/sample-webapp</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Spot Fleet - The resource type is <code>spot-fleet-request</code> and the unique identifier is the
         *        Spot Fleet request ID. Example:
         *        <code>spot-fleet-request/sfr-73fbd2ce-aa30-494c-8788-1cee4EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        EMR cluster - The resource type is <code>instancegroup</code> and the unique identifier is the cluster
         *        ID and instance group ID. Example: <code>instancegroup/j-2EEZNYKUA1NTV/ig-1791Y4E1L8YI0</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        AppStream 2.0 fleet - The resource type is <code>fleet</code> and the unique identifier is the fleet
         *        name. Example: <code>fleet/sample-fleet</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB table - The resource type is <code>table</code> and the unique identifier is the table name.
         *        Example: <code>table/my-table</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        DynamoDB global secondary index - The resource type is <code>index</code> and the unique identifier is
         *        the index name. Example: <code>table/my-table/index/my-table-index</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Aurora DB cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
         *        name. Example: <code>cluster:my-db-cluster</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SageMaker endpoint variant - The resource type is <code>variant</code> and the unique identifier is
         *        the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Custom resources are not supported with a resource type. This parameter must specify the
         *        <code>OutputValue</code> from the CloudFormation template stack used to access the resources. The
         *        unique identifier is defined by the service provider. More information is available in our <a
         *        href="https://github.com/aws/aws-auto-scaling-custom-resource">GitHub repository</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Comprehend document classification endpoint - The resource type and unique identifier are
         *        specified using the endpoint ARN. Example:
         *        <code>arn:aws:comprehend:us-west-2:123456789012:document-classifier-endpoint/EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Comprehend entity recognizer endpoint - The resource type and unique identifier are specified
         *        using the endpoint ARN. Example:
         *        <code>arn:aws:comprehend:us-west-2:123456789012:entity-recognizer-endpoint/EXAMPLE</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Lambda provisioned concurrency - The resource type is <code>function</code> and the unique identifier
         *        is the function name with a function version or alias name suffix that is not <code>$LATEST</code>.
         *        Example: <code>function:my-function:prod</code> or <code>function:my-function:1</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon Keyspaces table - The resource type is <code>table</code> and the unique identifier is the
         *        table name. Example: <code>keyspace/mykeyspace/table/mytable</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon MSK cluster - The resource type and unique identifier are specified using the cluster ARN.
         *        Example:
         *        <code>arn:aws:kafka:us-east-1:123456789012:cluster/demo-cluster-1/6357e0b2-0e6a-4b86-a0b4-70df934c2e31-5</code>
         *        .
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Amazon ElastiCache replication group - The resource type is <code>replication-group</code> and the
         *        unique identifier is the replication group name. Example: <code>replication-group/mycluster</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Neptune cluster - The resource type is <code>cluster</code> and the unique identifier is the cluster
         *        name. Example: <code>cluster:mycluster</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        SageMaker Serverless endpoint - The resource type is <code>variant</code> and the unique identifier is
         *        the resource ID. Example: <code>endpoint/my-end-point/variant/KMeansClustering</code>.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker model
         * endpoint variant.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
         * your own application or service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units
         * for an Amazon Comprehend document classification endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference units for
         * an Amazon Comprehend entity recognizer endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an Amazon
         * MSK cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon ElastiCache
         * replication group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an Amazon
         * ElastiCache replication group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a SageMaker
         * Serverless endpoint.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension. This string consists of the service namespace, resource type, and scaling
         *        property.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance
         *        Group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker
         *        model endpoint variant.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
         *        provided by your own application or service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend document classification endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend entity recognizer endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda
         *        function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
         *        table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon
         *        Keyspaces table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an
         *        Amazon MSK cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon
         *        ElastiCache replication group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an
         *        Amazon ElastiCache replication group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB
         *        cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a
         *        SageMaker Serverless endpoint.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(String scalableDimension);

        /**
         * <p>
         * The scalable dimension. This string consists of the service namespace, resource type, and scaling property.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance Group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global secondary
         * index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         * secondary index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster. Available
         * for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker model
         * endpoint variant.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource provided by
         * your own application or service.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference units
         * for an Amazon Comprehend document classification endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference units for
         * an Amazon Comprehend entity recognizer endpoint.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda function.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon Keyspaces
         * table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an Amazon
         * MSK cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon ElastiCache
         * replication group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an Amazon
         * ElastiCache replication group.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB cluster.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a SageMaker
         * Serverless endpoint.
         * </p>
         * </li>
         * </ul>
         * 
         * @param scalableDimension
         *        The scalable dimension. This string consists of the service namespace, resource type, and scaling
         *        property.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ecs:service:DesiredCount</code> - The desired task count of an ECS service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticmapreduce:instancegroup:InstanceCount</code> - The instance count of an EMR Instance
         *        Group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ec2:spot-fleet-request:TargetCapacity</code> - The target capacity of a Spot Fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>appstream:fleet:DesiredCapacity</code> - The desired capacity of an AppStream 2.0 fleet.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:table:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:ReadCapacityUnits</code> - The provisioned read capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dynamodb:index:WriteCapacityUnits</code> - The provisioned write capacity for a DynamoDB global
         *        secondary index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>rds:cluster:ReadReplicaCount</code> - The count of Aurora Replicas in an Aurora DB cluster.
         *        Available for Aurora MySQL-compatible edition and Aurora PostgreSQL-compatible edition.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredInstanceCount</code> - The number of EC2 instances for a SageMaker
         *        model endpoint variant.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>custom-resource:ResourceType:Property</code> - The scalable dimension for a custom resource
         *        provided by your own application or service.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:document-classifier-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend document classification endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>comprehend:entity-recognizer-endpoint:DesiredInferenceUnits</code> - The number of inference
         *        units for an Amazon Comprehend entity recognizer endpoint.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>lambda:function:ProvisionedConcurrency</code> - The provisioned concurrency for a Lambda
         *        function.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:ReadCapacityUnits</code> - The provisioned read capacity for an Amazon Keyspaces
         *        table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cassandra:table:WriteCapacityUnits</code> - The provisioned write capacity for an Amazon
         *        Keyspaces table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>kafka:broker-storage:VolumeSize</code> - The provisioned volume size (in GiB) for brokers in an
         *        Amazon MSK cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticache:replication-group:NodeGroups</code> - The number of node groups for an Amazon
         *        ElastiCache replication group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>elasticache:replication-group:Replicas</code> - The number of replicas per node group for an
         *        Amazon ElastiCache replication group.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>neptune:cluster:ReadReplicaCount</code> - The count of read replicas in an Amazon Neptune DB
         *        cluster.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>sagemaker:variant:DesiredProvisionedConcurrency</code> - The provisioned concurrency for a
         *        SageMaker Serverless endpoint.
         *        </p>
         *        </li>
         * @see ScalableDimension
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalableDimension
         */
        Builder scalableDimension(ScalableDimension scalableDimension);

        /**
         * <p>
         * A simple description of what action the scaling activity intends to accomplish.
         * </p>
         * 
         * @param description
         *        A simple description of what action the scaling activity intends to accomplish.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A simple description of what caused the scaling activity to happen.
         * </p>
         * 
         * @param cause
         *        A simple description of what caused the scaling activity to happen.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cause(String cause);

        /**
         * <p>
         * The Unix timestamp for when the scaling activity began.
         * </p>
         * 
         * @param startTime
         *        The Unix timestamp for when the scaling activity began.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The Unix timestamp for when the scaling activity ended.
         * </p>
         * 
         * @param endTime
         *        The Unix timestamp for when the scaling activity ended.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * Indicates the status of the scaling activity.
         * </p>
         * 
         * @param statusCode
         *        Indicates the status of the scaling activity.
         * @see ScalingActivityStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingActivityStatusCode
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * Indicates the status of the scaling activity.
         * </p>
         * 
         * @param statusCode
         *        Indicates the status of the scaling activity.
         * @see ScalingActivityStatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScalingActivityStatusCode
         */
        Builder statusCode(ScalingActivityStatusCode statusCode);

        /**
         * <p>
         * A simple message about the current status of the scaling activity.
         * </p>
         * 
         * @param statusMessage
         *        A simple message about the current status of the scaling activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The details about the scaling activity.
         * </p>
         * 
         * @param details
         *        The details about the scaling activity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder details(String details);

        /**
         * <p>
         * Machine-readable data that describes the reason for a not scaled activity. Only available when <a
         * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
         * >DescribeScalingActivities</a> includes not scaled activities.
         * </p>
         * 
         * @param notScaledReasons
         *        Machine-readable data that describes the reason for a not scaled activity. Only available when <a
         *        href=
         *        "https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
         *        >DescribeScalingActivities</a> includes not scaled activities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notScaledReasons(Collection<NotScaledReason> notScaledReasons);

        /**
         * <p>
         * Machine-readable data that describes the reason for a not scaled activity. Only available when <a
         * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
         * >DescribeScalingActivities</a> includes not scaled activities.
         * </p>
         * 
         * @param notScaledReasons
         *        Machine-readable data that describes the reason for a not scaled activity. Only available when <a
         *        href=
         *        "https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
         *        >DescribeScalingActivities</a> includes not scaled activities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notScaledReasons(NotScaledReason... notScaledReasons);

        /**
         * <p>
         * Machine-readable data that describes the reason for a not scaled activity. Only available when <a
         * href="https://docs.aws.amazon.com/autoscaling/application/APIReference/API_DescribeScalingActivities.html"
         * >DescribeScalingActivities</a> includes not scaled activities.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.applicationautoscaling.model.NotScaledReason.Builder} avoiding the
         * need to create one manually via
         * {@link software.amazon.awssdk.services.applicationautoscaling.model.NotScaledReason#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.applicationautoscaling.model.NotScaledReason.Builder#build()} is
         * called immediately and its result is passed to {@link #notScaledReasons(List<NotScaledReason>)}.
         * 
         * @param notScaledReasons
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.applicationautoscaling.model.NotScaledReason.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #notScaledReasons(java.util.Collection<NotScaledReason>)
         */
        Builder notScaledReasons(Consumer<NotScaledReason.Builder>... notScaledReasons);
    }

    static final class BuilderImpl implements Builder {
        private String activityId;

        private String serviceNamespace;

        private String resourceId;

        private String scalableDimension;

        private String description;

        private String cause;

        private Instant startTime;

        private Instant endTime;

        private String statusCode;

        private String statusMessage;

        private String details;

        private List<NotScaledReason> notScaledReasons = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ScalingActivity model) {
            activityId(model.activityId);
            serviceNamespace(model.serviceNamespace);
            resourceId(model.resourceId);
            scalableDimension(model.scalableDimension);
            description(model.description);
            cause(model.cause);
            startTime(model.startTime);
            endTime(model.endTime);
            statusCode(model.statusCode);
            statusMessage(model.statusMessage);
            details(model.details);
            notScaledReasons(model.notScaledReasons);
        }

        public final String getActivityId() {
            return activityId;
        }

        public final void setActivityId(String activityId) {
            this.activityId = activityId;
        }

        @Override
        public final Builder activityId(String activityId) {
            this.activityId = activityId;
            return this;
        }

        public final String getServiceNamespace() {
            return serviceNamespace;
        }

        public final void setServiceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
        }

        @Override
        public final Builder serviceNamespace(String serviceNamespace) {
            this.serviceNamespace = serviceNamespace;
            return this;
        }

        @Override
        public final Builder serviceNamespace(ServiceNamespace serviceNamespace) {
            this.serviceNamespace(serviceNamespace == null ? null : serviceNamespace.toString());
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getScalableDimension() {
            return scalableDimension;
        }

        public final void setScalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
        }

        @Override
        public final Builder scalableDimension(String scalableDimension) {
            this.scalableDimension = scalableDimension;
            return this;
        }

        @Override
        public final Builder scalableDimension(ScalableDimension scalableDimension) {
            this.scalableDimension(scalableDimension == null ? null : scalableDimension.toString());
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getCause() {
            return cause;
        }

        public final void setCause(String cause) {
            this.cause = cause;
        }

        @Override
        public final Builder cause(String cause) {
            this.cause = cause;
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final String getStatusCode() {
            return statusCode;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        @Override
        public final Builder statusCode(ScalingActivityStatusCode statusCode) {
            this.statusCode(statusCode == null ? null : statusCode.toString());
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final String getDetails() {
            return details;
        }

        public final void setDetails(String details) {
            this.details = details;
        }

        @Override
        public final Builder details(String details) {
            this.details = details;
            return this;
        }

        public final List<NotScaledReason.Builder> getNotScaledReasons() {
            List<NotScaledReason.Builder> result = NotScaledReasonsCopier.copyToBuilder(this.notScaledReasons);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setNotScaledReasons(Collection<NotScaledReason.BuilderImpl> notScaledReasons) {
            this.notScaledReasons = NotScaledReasonsCopier.copyFromBuilder(notScaledReasons);
        }

        @Override
        public final Builder notScaledReasons(Collection<NotScaledReason> notScaledReasons) {
            this.notScaledReasons = NotScaledReasonsCopier.copy(notScaledReasons);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notScaledReasons(NotScaledReason... notScaledReasons) {
            notScaledReasons(Arrays.asList(notScaledReasons));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notScaledReasons(Consumer<NotScaledReason.Builder>... notScaledReasons) {
            notScaledReasons(Stream.of(notScaledReasons).map(c -> NotScaledReason.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ScalingActivity build() {
            return new ScalingActivity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
