/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.appconfig.AppConfigAsyncClient;
import software.amazon.awssdk.services.appconfig.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.appconfig.model.AppConfigRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAppConfigAsyncWaiter implements AppConfigAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final AppConfigAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetEnvironmentResponse> environmentReadyForDeploymentWaiter;

    private final AsyncWaiter<GetDeploymentResponse> deploymentCompleteWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultAppConfigAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AppConfigAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.environmentReadyForDeploymentWaiter = AsyncWaiter.builder(GetEnvironmentResponse.class)
                .acceptors(environmentReadyForDeploymentWaiterAcceptors())
                .overrideConfiguration(environmentReadyForDeploymentWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.deploymentCompleteWaiter = AsyncWaiter.builder(GetDeploymentResponse.class)
                .acceptors(deploymentCompleteWaiterAcceptors())
                .overrideConfiguration(deploymentCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<GetDeploymentResponse>> waitUntilDeploymentComplete(
            GetDeploymentRequest getDeploymentRequest) {
        return deploymentCompleteWaiter.runAsync(() -> client.getDeployment(applyWaitersUserAgent(getDeploymentRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetDeploymentResponse>> waitUntilDeploymentComplete(
            GetDeploymentRequest getDeploymentRequest, WaiterOverrideConfiguration overrideConfig) {
        return deploymentCompleteWaiter.runAsync(() -> client.getDeployment(applyWaitersUserAgent(getDeploymentRequest)),
                deploymentCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetEnvironmentResponse>> waitUntilEnvironmentReadyForDeployment(
            GetEnvironmentRequest getEnvironmentRequest) {
        return environmentReadyForDeploymentWaiter.runAsync(() -> client
                .getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetEnvironmentResponse>> waitUntilEnvironmentReadyForDeployment(
            GetEnvironmentRequest getEnvironmentRequest, WaiterOverrideConfiguration overrideConfig) {
        return environmentReadyForDeploymentWaiter.runAsync(
                () -> client.getEnvironment(applyWaitersUserAgent(getEnvironmentRequest)),
                environmentReadyForDeploymentWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetEnvironmentResponse>> environmentReadyForDeploymentWaiterAcceptors() {
        List<WaiterAcceptor<? super GetEnvironmentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "ReadyForDeployment");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "RolledBack");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=RolledBack) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "Reverted");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=Reverted) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetDeploymentResponse>> deploymentCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super GetDeploymentResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("State").value(), "COMPLETE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "ROLLED_BACK");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=ROLLED_BACK) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("State").value(), "REVERTED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (State=REVERTED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration environmentReadyForDeploymentWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration deploymentCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(999);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(30)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AppConfigAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AppConfigRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AppConfigAsyncWaiter.Builder {
        private AppConfigAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public AppConfigAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public AppConfigAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AppConfigAsyncWaiter.Builder client(AppConfigAsyncClient client) {
            this.client = client;
            return this;
        }

        public AppConfigAsyncWaiter build() {
            return new DefaultAppConfigAsyncWaiter(this);
        }
    }
}
