/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appconfig;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.appconfig.model.AppConfigException;
import software.amazon.awssdk.services.appconfig.model.BadRequestException;
import software.amazon.awssdk.services.appconfig.model.ConflictException;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.CreateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.CreateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.CreateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.CreateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.CreateExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.CreateHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.DeleteHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.GetApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.GetApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.GetConfigurationResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.GetDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.GetEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.GetExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.GetExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.GetExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.GetExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionRequest;
import software.amazon.awssdk.services.appconfig.model.GetHostedConfigurationVersionResponse;
import software.amazon.awssdk.services.appconfig.model.InternalServerException;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsRequest;
import software.amazon.awssdk.services.appconfig.model.ListApplicationsResponse;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesRequest;
import software.amazon.awssdk.services.appconfig.model.ListConfigurationProfilesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentStrategiesResponse;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListDeploymentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.appconfig.model.ListEnvironmentsResponse;
import software.amazon.awssdk.services.appconfig.model.ListExtensionAssociationsRequest;
import software.amazon.awssdk.services.appconfig.model.ListExtensionAssociationsResponse;
import software.amazon.awssdk.services.appconfig.model.ListExtensionsRequest;
import software.amazon.awssdk.services.appconfig.model.ListExtensionsResponse;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsRequest;
import software.amazon.awssdk.services.appconfig.model.ListHostedConfigurationVersionsResponse;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.appconfig.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.appconfig.model.PayloadTooLargeException;
import software.amazon.awssdk.services.appconfig.model.ResourceNotFoundException;
import software.amazon.awssdk.services.appconfig.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StartDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentRequest;
import software.amazon.awssdk.services.appconfig.model.StopDeploymentResponse;
import software.amazon.awssdk.services.appconfig.model.TagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.TagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UntagResourceRequest;
import software.amazon.awssdk.services.appconfig.model.UntagResourceResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateApplicationResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateConfigurationProfileResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateDeploymentStrategyResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateEnvironmentResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionAssociationRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionAssociationResponse;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionRequest;
import software.amazon.awssdk.services.appconfig.model.UpdateExtensionResponse;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationRequest;
import software.amazon.awssdk.services.appconfig.model.ValidateConfigurationResponse;
import software.amazon.awssdk.services.appconfig.transform.CreateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.CreateHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.DeleteHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetConfigurationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.GetHostedConfigurationVersionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListApplicationsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListConfigurationProfilesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentStrategiesRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListDeploymentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListEnvironmentsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListExtensionAssociationsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListExtensionsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListHostedConfigurationVersionsRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StartDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.StopDeploymentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateApplicationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateConfigurationProfileRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateDeploymentStrategyRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateEnvironmentRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateExtensionAssociationRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.UpdateExtensionRequestMarshaller;
import software.amazon.awssdk.services.appconfig.transform.ValidateConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link AppConfigClient}.
 *
 * @see AppConfigClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAppConfigClient implements AppConfigClient {
    private static final Logger log = Logger.loggerFor(DefaultAppConfigClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private final AppConfigServiceClientConfiguration serviceClientConfiguration;

    protected DefaultAppConfigClient(AppConfigServiceClientConfiguration serviceClientConfiguration,
            SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.serviceClientConfiguration = serviceClientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Creates an application. In AppConfig, an application is simply an organizational construct like a folder. This
     * organizational construct has a relationship with some unit of executable code. For example, you could create an
     * application called MyMobileApp to organize and manage configuration data for a mobile application installed by
     * your users.
     * </p>
     *
     * @param createApplicationRequest
     * @return Result of the CreateApplication operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateApplicationResponse createApplication(CreateApplicationRequest createApplicationRequest)
            throws BadRequestException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateApplication");

            return clientHandler.execute(new ClientExecutionParams<CreateApplicationRequest, CreateApplicationResponse>()
                    .withOperationName("CreateApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a configuration profile, which is information that enables AppConfig to access the configuration source.
     * Valid configuration sources include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Configuration data in YAML, JSON, and other formats stored in the AppConfig hosted configuration store
     * </p>
     * </li>
     * <li>
     * <p>
     * Configuration data stored as objects in an Amazon Simple Storage Service (Amazon S3) bucket
     * </p>
     * </li>
     * <li>
     * <p>
     * Pipelines stored in CodePipeline
     * </p>
     * </li>
     * <li>
     * <p>
     * Secrets stored in Secrets Manager
     * </p>
     * </li>
     * <li>
     * <p>
     * Standard and secure string parameters stored in Amazon Web Services Systems Manager Parameter Store
     * </p>
     * </li>
     * <li>
     * <p>
     * Configuration data in SSM documents stored in the Systems Manager document store
     * </p>
     * </li>
     * </ul>
     * <p>
     * A configuration profile includes the following information:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The URI location of the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Identity and Access Management (IAM) role that provides access to the configuration data.
     * </p>
     * </li>
     * <li>
     * <p>
     * A validator for the configuration data. Available validators include either a JSON Schema or an Amazon Web
     * Services Lambda function.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="http://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-configuration-and-profile.html"
     * >Create a Configuration and a Configuration Profile</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param createConfigurationProfileRequest
     * @return Result of the CreateConfigurationProfile operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateConfigurationProfileResponse createConfigurationProfile(
            CreateConfigurationProfileRequest createConfigurationProfileRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateConfigurationProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateConfigurationProfileRequest, CreateConfigurationProfileResponse>()
                            .withOperationName("CreateConfigurationProfile").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createConfigurationProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateConfigurationProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a deployment strategy that defines important criteria for rolling out your configuration to the
     * designated targets. A deployment strategy includes the overall duration required, a percentage of targets to
     * receive the deployment during each interval, an algorithm that defines how percentage grows, and bake time.
     * </p>
     *
     * @param createDeploymentStrategyRequest
     * @return Result of the CreateDeploymentStrategy operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDeploymentStrategyResponse createDeploymentStrategy(
            CreateDeploymentStrategyRequest createDeploymentStrategyRequest) throws InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDeploymentStrategy");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDeploymentStrategyRequest, CreateDeploymentStrategyResponse>()
                            .withOperationName("CreateDeploymentStrategy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createDeploymentStrategyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDeploymentStrategyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an environment. For each application, you define one or more environments. An environment is a deployment
     * group of AppConfig targets, such as applications in a <code>Beta</code> or <code>Production</code> environment.
     * You can also define environments for application subcomponents such as the <code>Web</code>, <code>Mobile</code>
     * and <code>Back-end</code> components for your application. You can configure Amazon CloudWatch alarms for each
     * environment. The system monitors alarms during a configuration deployment. If an alarm is triggered, the system
     * rolls back the configuration.
     * </p>
     *
     * @param createEnvironmentRequest
     * @return Result of the CreateEnvironment operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateEnvironmentResponse createEnvironment(CreateEnvironmentRequest createEnvironmentRequest)
            throws InternalServerException, ResourceNotFoundException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateEnvironment");

            return clientHandler.execute(new ClientExecutionParams<CreateEnvironmentRequest, CreateEnvironmentResponse>()
                    .withOperationName("CreateEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an AppConfig extension. An extension augments your ability to inject logic or behavior at different
     * points during the AppConfig workflow of creating or deploying a configuration.
     * </p>
     * <p>
     * You can create your own extensions or use the Amazon Web Services authored extensions provided by AppConfig. For
     * most use cases, to create your own extension, you must create an Lambda function to perform any computation and
     * processing defined in the extension. For more information about extensions, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param createExtensionRequest
     * @return Result of the CreateExtension operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws ServiceQuotaExceededException
     *         The number of hosted configuration versions exceeds the limit for the AppConfig hosted configuration
     *         store. Delete one or more versions and try again.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateExtensionResponse createExtension(CreateExtensionRequest createExtensionRequest) throws BadRequestException,
            ConflictException, ServiceQuotaExceededException, InternalServerException, AwsServiceException, SdkClientException,
            AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateExtensionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExtension");

            return clientHandler.execute(new ClientExecutionParams<CreateExtensionRequest, CreateExtensionResponse>()
                    .withOperationName("CreateExtension").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createExtensionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateExtensionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * When you create an extension or configure an Amazon Web Services authored extension, you associate the extension
     * with an AppConfig application, environment, or configuration profile. For example, you can choose to run the
     * <code>AppConfig deployment events to Amazon SNS</code> Amazon Web Services authored extension and receive
     * notifications on an Amazon SNS topic anytime a configuration deployment is started for a specific application.
     * Defining which extension to associate with an AppConfig resource is called an <i>extension association</i>. An
     * extension association is a specified relationship between an extension and an AppConfig resource, such as an
     * application or a configuration profile. For more information about extensions and associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param createExtensionAssociationRequest
     * @return Result of the CreateExtensionAssociation operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws ServiceQuotaExceededException
     *         The number of hosted configuration versions exceeds the limit for the AppConfig hosted configuration
     *         store. Delete one or more versions and try again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateExtensionAssociationResponse createExtensionAssociation(
            CreateExtensionAssociationRequest createExtensionAssociationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, ServiceQuotaExceededException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateExtensionAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateExtensionAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateExtensionAssociationRequest, CreateExtensionAssociationResponse>()
                            .withOperationName("CreateExtensionAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createExtensionAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateExtensionAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new configuration in the AppConfig hosted configuration store.
     * </p>
     *
     * @param createHostedConfigurationVersionRequest
     * @return Result of the CreateHostedConfigurationVersion operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ServiceQuotaExceededException
     *         The number of hosted configuration versions exceeds the limit for the AppConfig hosted configuration
     *         store. Delete one or more versions and try again.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws PayloadTooLargeException
     *         The configuration size is too large.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.CreateHostedConfigurationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/CreateHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateHostedConfigurationVersionResponse createHostedConfigurationVersion(
            CreateHostedConfigurationVersionRequest createHostedConfigurationVersionRequest) throws BadRequestException,
            ServiceQuotaExceededException, ResourceNotFoundException, ConflictException, PayloadTooLargeException,
            InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<CreateHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateHostedConfigurationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHostedConfigurationVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateHostedConfigurationVersionRequest, CreateHostedConfigurationVersionResponse>()
                            .withOperationName("CreateHostedConfigurationVersion").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(createHostedConfigurationVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateHostedConfigurationVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an application. Deleting an application does not delete a configuration from a host.
     * </p>
     *
     * @param deleteApplicationRequest
     * @return Result of the DeleteApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteApplicationResponse deleteApplication(DeleteApplicationRequest deleteApplicationRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteApplication");

            return clientHandler.execute(new ClientExecutionParams<DeleteApplicationRequest, DeleteApplicationResponse>()
                    .withOperationName("DeleteApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a configuration profile. Deleting a configuration profile does not delete a configuration from a host.
     * </p>
     *
     * @param deleteConfigurationProfileRequest
     * @return Result of the DeleteConfigurationProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteConfigurationProfileResponse deleteConfigurationProfile(
            DeleteConfigurationProfileRequest deleteConfigurationProfileRequest) throws ResourceNotFoundException,
            ConflictException, InternalServerException, BadRequestException, AwsServiceException, SdkClientException,
            AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteConfigurationProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteConfigurationProfileRequest, DeleteConfigurationProfileResponse>()
                            .withOperationName("DeleteConfigurationProfile").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteConfigurationProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteConfigurationProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a deployment strategy. Deleting a deployment strategy does not delete a configuration from a host.
     * </p>
     *
     * @param deleteDeploymentStrategyRequest
     * @return Result of the DeleteDeploymentStrategy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDeploymentStrategyResponse deleteDeploymentStrategy(
            DeleteDeploymentStrategyRequest deleteDeploymentStrategyRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDeploymentStrategy");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDeploymentStrategyRequest, DeleteDeploymentStrategyResponse>()
                            .withOperationName("DeleteDeploymentStrategy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteDeploymentStrategyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDeploymentStrategyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an environment. Deleting an environment does not delete a configuration from a host.
     * </p>
     *
     * @param deleteEnvironmentRequest
     * @return Result of the DeleteEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteEnvironmentResponse deleteEnvironment(DeleteEnvironmentRequest deleteEnvironmentRequest)
            throws ResourceNotFoundException, ConflictException, InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteEnvironment");

            return clientHandler.execute(new ClientExecutionParams<DeleteEnvironmentRequest, DeleteEnvironmentResponse>()
                    .withOperationName("DeleteEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an AppConfig extension. You must delete all associations to an extension before you delete the extension.
     * </p>
     *
     * @param deleteExtensionRequest
     * @return Result of the DeleteExtension operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteExtensionResponse deleteExtension(DeleteExtensionRequest deleteExtensionRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteExtensionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExtension");

            return clientHandler.execute(new ClientExecutionParams<DeleteExtensionRequest, DeleteExtensionResponse>()
                    .withOperationName("DeleteExtension").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteExtensionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteExtensionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an extension association. This action doesn't delete extensions defined in the association.
     * </p>
     *
     * @param deleteExtensionAssociationRequest
     * @return Result of the DeleteExtensionAssociation operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteExtensionAssociationResponse deleteExtensionAssociation(
            DeleteExtensionAssociationRequest deleteExtensionAssociationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteExtensionAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteExtensionAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteExtensionAssociationRequest, DeleteExtensionAssociationResponse>()
                            .withOperationName("DeleteExtensionAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteExtensionAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteExtensionAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a version of a configuration from the AppConfig hosted configuration store.
     * </p>
     *
     * @param deleteHostedConfigurationVersionRequest
     * @return Result of the DeleteHostedConfigurationVersion operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.DeleteHostedConfigurationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/DeleteHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteHostedConfigurationVersionResponse deleteHostedConfigurationVersion(
            DeleteHostedConfigurationVersionRequest deleteHostedConfigurationVersionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteHostedConfigurationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHostedConfigurationVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteHostedConfigurationVersionRequest, DeleteHostedConfigurationVersionResponse>()
                            .withOperationName("DeleteHostedConfigurationVersion").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(deleteHostedConfigurationVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteHostedConfigurationVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an application.
     * </p>
     *
     * @param getApplicationRequest
     * @return Result of the GetApplication operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetApplication" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetApplicationResponse getApplication(GetApplicationRequest getApplicationRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetApplication");

            return clientHandler.execute(new ClientExecutionParams<GetApplicationRequest, GetApplicationResponse>()
                    .withOperationName("GetApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * (Deprecated) Retrieves the latest deployed configuration.
     * </p>
     * <important>
     * <p>
     * Note the following important information.
     * </p>
     * <ul>
     * <li>
     * <p>
     * This API action is deprecated. Calls to receive configuration data should use the <a href=
     * "https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_appconfigdata_StartConfigurationSession.html"
     * >StartConfigurationSession</a> and <a href=
     * "https://docs.aws.amazon.com/appconfig/2019-10-09/APIReference/API_appconfigdata_GetLatestConfiguration.html"
     * >GetLatestConfiguration</a> APIs instead.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>GetConfiguration</code> is a priced call. For more information, see <a
     * href="https://aws.amazon.com/systems-manager/pricing/">Pricing</a>.
     * </p>
     * </li>
     * </ul>
     * </important>
     *
     * @param getConfigurationRequest
     * @return Result of the GetConfiguration operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfiguration" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetConfigurationResponse getConfiguration(GetConfigurationRequest getConfigurationRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfiguration");

            return clientHandler.execute(new ClientExecutionParams<GetConfigurationRequest, GetConfigurationResponse>()
                    .withOperationName("GetConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a configuration profile.
     * </p>
     *
     * @param getConfigurationProfileRequest
     * @return Result of the GetConfigurationProfile operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetConfigurationProfileResponse getConfigurationProfile(GetConfigurationProfileRequest getConfigurationProfileRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetConfigurationProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<GetConfigurationProfileRequest, GetConfigurationProfileResponse>()
                            .withOperationName("GetConfigurationProfile").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getConfigurationProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetConfigurationProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a configuration deployment.
     * </p>
     *
     * @param getDeploymentRequest
     * @return Result of the GetDeployment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDeploymentResponse getDeployment(GetDeploymentRequest getDeploymentRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeployment");

            return clientHandler.execute(new ClientExecutionParams<GetDeploymentRequest, GetDeploymentResponse>()
                    .withOperationName("GetDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a deployment strategy. A deployment strategy defines important criteria for rolling
     * out your configuration to the designated targets. A deployment strategy includes the overall duration required, a
     * percentage of targets to receive the deployment during each interval, an algorithm that defines how percentage
     * grows, and bake time.
     * </p>
     *
     * @param getDeploymentStrategyRequest
     * @return Result of the GetDeploymentStrategy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetDeploymentStrategyResponse getDeploymentStrategy(GetDeploymentStrategyRequest getDeploymentStrategyRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDeploymentStrategy");

            return clientHandler.execute(new ClientExecutionParams<GetDeploymentStrategyRequest, GetDeploymentStrategyResponse>()
                    .withOperationName("GetDeploymentStrategy").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDeploymentStrategyRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDeploymentStrategyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about an environment. An environment is a deployment group of AppConfig applications, such
     * as applications in a <code>Production</code> environment or in an <code>EU_Region</code> environment. Each
     * configuration deployment targets an environment. You can enable one or more Amazon CloudWatch alarms for an
     * environment. If an alarm is triggered during a deployment, AppConfig roles back the configuration.
     * </p>
     *
     * @param getEnvironmentRequest
     * @return Result of the GetEnvironment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetEnvironment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetEnvironmentResponse getEnvironment(GetEnvironmentRequest getEnvironmentRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnvironment");

            return clientHandler.execute(new ClientExecutionParams<GetEnvironmentRequest, GetEnvironmentResponse>()
                    .withOperationName("GetEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an AppConfig extension.
     * </p>
     *
     * @param getExtensionRequest
     * @return Result of the GetExtension operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetExtensionResponse getExtension(GetExtensionRequest getExtensionRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetExtensionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtension");

            return clientHandler.execute(new ClientExecutionParams<GetExtensionRequest, GetExtensionResponse>()
                    .withOperationName("GetExtension").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getExtensionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetExtensionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about an AppConfig extension association. For more information about extensions and
     * associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param getExtensionAssociationRequest
     * @return Result of the GetExtensionAssociation operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetExtensionAssociationResponse getExtensionAssociation(GetExtensionAssociationRequest getExtensionAssociationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetExtensionAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetExtensionAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<GetExtensionAssociationRequest, GetExtensionAssociationResponse>()
                            .withOperationName("GetExtensionAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getExtensionAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetExtensionAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves information about a specific configuration version.
     * </p>
     *
     * @param getHostedConfigurationVersionRequest
     * @return Result of the GetHostedConfigurationVersion operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.GetHostedConfigurationVersion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/GetHostedConfigurationVersion"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetHostedConfigurationVersionResponse getHostedConfigurationVersion(
            GetHostedConfigurationVersionRequest getHostedConfigurationVersionRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(false).build();

        HttpResponseHandler<GetHostedConfigurationVersionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetHostedConfigurationVersionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getHostedConfigurationVersionRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetHostedConfigurationVersion");

            return clientHandler
                    .execute(new ClientExecutionParams<GetHostedConfigurationVersionRequest, GetHostedConfigurationVersionResponse>()
                            .withOperationName("GetHostedConfigurationVersion").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getHostedConfigurationVersionRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetHostedConfigurationVersionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all applications in your Amazon Web Services account.
     * </p>
     *
     * @param listApplicationsRequest
     * @return Result of the ListApplications operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListApplications
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListApplications" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListApplicationsResponse listApplications(ListApplicationsRequest listApplicationsRequest)
            throws InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListApplicationsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListApplicationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listApplicationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListApplications");

            return clientHandler.execute(new ClientExecutionParams<ListApplicationsRequest, ListApplicationsResponse>()
                    .withOperationName("ListApplications").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listApplicationsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListApplicationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the configuration profiles for an application.
     * </p>
     *
     * @param listConfigurationProfilesRequest
     * @return Result of the ListConfigurationProfiles operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListConfigurationProfiles
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListConfigurationProfiles"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListConfigurationProfilesResponse listConfigurationProfiles(
            ListConfigurationProfilesRequest listConfigurationProfilesRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListConfigurationProfilesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListConfigurationProfilesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listConfigurationProfilesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListConfigurationProfiles");

            return clientHandler
                    .execute(new ClientExecutionParams<ListConfigurationProfilesRequest, ListConfigurationProfilesResponse>()
                            .withOperationName("ListConfigurationProfiles").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listConfigurationProfilesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListConfigurationProfilesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists deployment strategies.
     * </p>
     *
     * @param listDeploymentStrategiesRequest
     * @return Result of the ListDeploymentStrategies operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListDeploymentStrategies
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeploymentStrategies"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListDeploymentStrategiesResponse listDeploymentStrategies(
            ListDeploymentStrategiesRequest listDeploymentStrategiesRequest) throws InternalServerException, BadRequestException,
            AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeploymentStrategiesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListDeploymentStrategiesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentStrategiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeploymentStrategies");

            return clientHandler
                    .execute(new ClientExecutionParams<ListDeploymentStrategiesRequest, ListDeploymentStrategiesResponse>()
                            .withOperationName("ListDeploymentStrategies").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listDeploymentStrategiesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListDeploymentStrategiesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the deployments for an environment in descending deployment number order.
     * </p>
     *
     * @param listDeploymentsRequest
     * @return Result of the ListDeployments operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListDeployments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListDeployments" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDeploymentsResponse listDeployments(ListDeploymentsRequest listDeploymentsRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListDeploymentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListDeploymentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDeploymentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDeployments");

            return clientHandler.execute(new ClientExecutionParams<ListDeploymentsRequest, ListDeploymentsResponse>()
                    .withOperationName("ListDeployments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDeploymentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDeploymentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the environments for an application.
     * </p>
     *
     * @param listEnvironmentsRequest
     * @return Result of the ListEnvironments operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListEnvironments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListEnvironments" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListEnvironmentsResponse listEnvironments(ListEnvironmentsRequest listEnvironmentsRequest)
            throws ResourceNotFoundException, InternalServerException, BadRequestException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListEnvironmentsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListEnvironmentsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEnvironmentsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEnvironments");

            return clientHandler.execute(new ClientExecutionParams<ListEnvironmentsRequest, ListEnvironmentsResponse>()
                    .withOperationName("ListEnvironments").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listEnvironmentsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListEnvironmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all AppConfig extension associations in the account. For more information about extensions and
     * associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param listExtensionAssociationsRequest
     * @return Result of the ListExtensionAssociations operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListExtensionAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListExtensionAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListExtensionAssociationsResponse listExtensionAssociations(
            ListExtensionAssociationsRequest listExtensionAssociationsRequest) throws InternalServerException,
            BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExtensionAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListExtensionAssociationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensionAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<ListExtensionAssociationsRequest, ListExtensionAssociationsResponse>()
                            .withOperationName("ListExtensionAssociations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listExtensionAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListExtensionAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all custom and Amazon Web Services authored AppConfig extensions in the account. For more information about
     * extensions, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param listExtensionsRequest
     * @return Result of the ListExtensions operation returned by the service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListExtensions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListExtensions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExtensionsResponse listExtensions(ListExtensionsRequest listExtensionsRequest) throws InternalServerException,
            BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExtensionsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExtensionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExtensionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExtensions");

            return clientHandler.execute(new ClientExecutionParams<ListExtensionsRequest, ListExtensionsResponse>()
                    .withOperationName("ListExtensions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listExtensionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExtensionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists configurations stored in the AppConfig hosted configuration store by version.
     * </p>
     *
     * @param listHostedConfigurationVersionsRequest
     * @return Result of the ListHostedConfigurationVersions operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListHostedConfigurationVersions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListHostedConfigurationVersions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListHostedConfigurationVersionsResponse listHostedConfigurationVersions(
            ListHostedConfigurationVersionsRequest listHostedConfigurationVersionsRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListHostedConfigurationVersionsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListHostedConfigurationVersionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listHostedConfigurationVersionsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListHostedConfigurationVersions");

            return clientHandler
                    .execute(new ClientExecutionParams<ListHostedConfigurationVersionsRequest, ListHostedConfigurationVersionsResponse>()
                            .withOperationName("ListHostedConfigurationVersions").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(listHostedConfigurationVersionsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListHostedConfigurationVersionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the list of key-value tags assigned to the resource.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws ResourceNotFoundException, BadRequestException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts a deployment.
     * </p>
     *
     * @param startDeploymentRequest
     * @return Result of the StartDeployment operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.StartDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StartDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StartDeploymentResponse startDeployment(StartDeploymentRequest startDeploymentRequest) throws BadRequestException,
            ResourceNotFoundException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StartDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartDeployment");

            return clientHandler.execute(new ClientExecutionParams<StartDeploymentRequest, StartDeploymentResponse>()
                    .withOperationName("StartDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops a deployment. This API action works only on deployments that have a status of <code>DEPLOYING</code>. This
     * action moves the deployment to a status of <code>ROLLED_BACK</code>.
     * </p>
     *
     * @param stopDeploymentRequest
     * @return Result of the StopDeployment operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.StopDeployment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/StopDeployment" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public StopDeploymentResponse stopDeployment(StopDeploymentRequest stopDeploymentRequest) throws ResourceNotFoundException,
            InternalServerException, BadRequestException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StopDeploymentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                StopDeploymentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopDeploymentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopDeployment");

            return clientHandler.execute(new ClientExecutionParams<StopDeploymentRequest, StopDeploymentResponse>()
                    .withOperationName("StopDeployment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopDeploymentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopDeploymentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns metadata to an AppConfig resource. Tags help organize and categorize your AppConfig resources. Each tag
     * consists of a key and an optional value, both of which you define. You can specify a maximum of 50 tags for a
     * resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws ResourceNotFoundException,
            BadRequestException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a tag key and value from an AppConfig resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws ResourceNotFoundException,
            BadRequestException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an application.
     * </p>
     *
     * @param updateApplicationRequest
     * @return Result of the UpdateApplication operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateApplication
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateApplication" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateApplicationResponse updateApplication(UpdateApplicationRequest updateApplicationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateApplicationResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateApplicationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateApplicationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateApplication");

            return clientHandler.execute(new ClientExecutionParams<UpdateApplicationRequest, UpdateApplicationResponse>()
                    .withOperationName("UpdateApplication").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateApplicationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateApplicationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a configuration profile.
     * </p>
     *
     * @param updateConfigurationProfileRequest
     * @return Result of the UpdateConfigurationProfile operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateConfigurationProfile
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateConfigurationProfile"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateConfigurationProfileResponse updateConfigurationProfile(
            UpdateConfigurationProfileRequest updateConfigurationProfileRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateConfigurationProfileResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateConfigurationProfileResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateConfigurationProfileRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateConfigurationProfile");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateConfigurationProfileRequest, UpdateConfigurationProfileResponse>()
                            .withOperationName("UpdateConfigurationProfile").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateConfigurationProfileRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateConfigurationProfileRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates a deployment strategy.
     * </p>
     *
     * @param updateDeploymentStrategyRequest
     * @return Result of the UpdateDeploymentStrategy operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateDeploymentStrategy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateDeploymentStrategy"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDeploymentStrategyResponse updateDeploymentStrategy(
            UpdateDeploymentStrategyRequest updateDeploymentStrategyRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDeploymentStrategyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDeploymentStrategyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateDeploymentStrategyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDeploymentStrategy");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDeploymentStrategyRequest, UpdateDeploymentStrategyResponse>()
                            .withOperationName("UpdateDeploymentStrategy").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateDeploymentStrategyRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDeploymentStrategyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an environment.
     * </p>
     *
     * @param updateEnvironmentRequest
     * @return Result of the UpdateEnvironment operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateEnvironment
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateEnvironment" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateEnvironmentResponse updateEnvironment(UpdateEnvironmentRequest updateEnvironmentRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnvironmentResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateEnvironmentResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnvironmentRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnvironment");

            return clientHandler.execute(new ClientExecutionParams<UpdateEnvironmentRequest, UpdateEnvironmentResponse>()
                    .withOperationName("UpdateEnvironment").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateEnvironmentRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateEnvironmentRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an AppConfig extension. For more information about extensions, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param updateExtensionRequest
     * @return Result of the UpdateExtension operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws ConflictException
     *         The request could not be processed because of conflict in the current state of the resource.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateExtension
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateExtension" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateExtensionResponse updateExtension(UpdateExtensionRequest updateExtensionRequest) throws BadRequestException,
            ResourceNotFoundException, ConflictException, InternalServerException, AwsServiceException, SdkClientException,
            AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateExtensionResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateExtensionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExtensionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExtension");

            return clientHandler.execute(new ClientExecutionParams<UpdateExtensionRequest, UpdateExtensionResponse>()
                    .withOperationName("UpdateExtension").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(updateExtensionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateExtensionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an association. For more information about extensions and associations, see <a
     * href="https://docs.aws.amazon.com/appconfig/latest/userguide/working-with-appconfig-extensions.html">Working with
     * AppConfig extensions</a> in the <i>AppConfig User Guide</i>.
     * </p>
     *
     * @param updateExtensionAssociationRequest
     * @return Result of the UpdateExtensionAssociation operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.UpdateExtensionAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/UpdateExtensionAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateExtensionAssociationResponse updateExtensionAssociation(
            UpdateExtensionAssociationRequest updateExtensionAssociationRequest) throws BadRequestException,
            ResourceNotFoundException, InternalServerException, AwsServiceException, SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateExtensionAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateExtensionAssociationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateExtensionAssociationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateExtensionAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateExtensionAssociationRequest, UpdateExtensionAssociationResponse>()
                            .withOperationName("UpdateExtensionAssociation").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateExtensionAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateExtensionAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Uses the validators in a configuration profile to validate a configuration.
     * </p>
     *
     * @param validateConfigurationRequest
     * @return Result of the ValidateConfiguration operation returned by the service.
     * @throws BadRequestException
     *         The input fails to satisfy the constraints specified by an Amazon Web Services service.
     * @throws ResourceNotFoundException
     *         The requested resource could not be found.
     * @throws InternalServerException
     *         There was an internal failure in the AppConfig service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AppConfigException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AppConfigClient.ValidateConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/appconfig-2019-10-09/ValidateConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ValidateConfigurationResponse validateConfiguration(ValidateConfigurationRequest validateConfigurationRequest)
            throws BadRequestException, ResourceNotFoundException, InternalServerException, AwsServiceException,
            SdkClientException, AppConfigException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ValidateConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ValidateConfigurationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, validateConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "AppConfig");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ValidateConfiguration");

            return clientHandler.execute(new ClientExecutionParams<ValidateConfigurationRequest, ValidateConfigurationResponse>()
                    .withOperationName("ValidateConfiguration").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(validateConfigurationRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ValidateConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AppConfigException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PayloadTooLargeException")
                                .exceptionBuilderSupplier(PayloadTooLargeException::builder).httpStatusCode(413).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BadRequestException")
                                .exceptionBuilderSupplier(BadRequestException::builder).httpStatusCode(400).build());
    }

    @Override
    public final AppConfigServiceClientConfiguration serviceClientConfiguration() {
        return this.serviceClientConfiguration;
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
