/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.amp.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.Waiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.amp.AmpClient;
import software.amazon.awssdk.services.amp.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.amp.model.AmpRequest;
import software.amazon.awssdk.services.amp.model.DescribeAnomalyDetectorRequest;
import software.amazon.awssdk.services.amp.model.DescribeAnomalyDetectorResponse;
import software.amazon.awssdk.services.amp.model.DescribeScraperRequest;
import software.amazon.awssdk.services.amp.model.DescribeScraperResponse;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceRequest;
import software.amazon.awssdk.services.amp.model.DescribeWorkspaceResponse;
import software.amazon.awssdk.services.amp.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAmpWaiter implements AmpWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private final AmpClient client;

    private final AttributeMap managedResources;

    private final Waiter<DescribeAnomalyDetectorResponse> anomalyDetectorActiveWaiter;

    private final Waiter<DescribeAnomalyDetectorResponse> anomalyDetectorDeletedWaiter;

    private final Waiter<DescribeScraperResponse> scraperActiveWaiter;

    private final Waiter<DescribeScraperResponse> scraperDeletedWaiter;

    private final Waiter<DescribeWorkspaceResponse> workspaceActiveWaiter;

    private final Waiter<DescribeWorkspaceResponse> workspaceDeletedWaiter;

    private DefaultAmpWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AmpClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        managedResources = attributeMapBuilder.build();
        this.anomalyDetectorActiveWaiter = Waiter.builder(DescribeAnomalyDetectorResponse.class)
                .acceptors(anomalyDetectorActiveWaiterAcceptors())
                .overrideConfiguration(anomalyDetectorActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.anomalyDetectorDeletedWaiter = Waiter.builder(DescribeAnomalyDetectorResponse.class)
                .acceptors(anomalyDetectorDeletedWaiterAcceptors())
                .overrideConfiguration(anomalyDetectorDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.scraperActiveWaiter = Waiter.builder(DescribeScraperResponse.class).acceptors(scraperActiveWaiterAcceptors())
                .overrideConfiguration(scraperActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.scraperDeletedWaiter = Waiter.builder(DescribeScraperResponse.class).acceptors(scraperDeletedWaiterAcceptors())
                .overrideConfiguration(scraperDeletedWaiterConfig(builder.overrideConfiguration)).build();
        this.workspaceActiveWaiter = Waiter.builder(DescribeWorkspaceResponse.class).acceptors(workspaceActiveWaiterAcceptors())
                .overrideConfiguration(workspaceActiveWaiterConfig(builder.overrideConfiguration)).build();
        this.workspaceDeletedWaiter = Waiter.builder(DescribeWorkspaceResponse.class)
                .acceptors(workspaceDeletedWaiterAcceptors())
                .overrideConfiguration(workspaceDeletedWaiterConfig(builder.overrideConfiguration)).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public WaiterResponse<DescribeAnomalyDetectorResponse> waitUntilAnomalyDetectorActive(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest) {
        return anomalyDetectorActiveWaiter.run(() -> client
                .describeAnomalyDetector(applyWaitersUserAgent(describeAnomalyDetectorRequest)));
    }

    @Override
    public WaiterResponse<DescribeAnomalyDetectorResponse> waitUntilAnomalyDetectorActive(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest, WaiterOverrideConfiguration overrideConfig) {
        return anomalyDetectorActiveWaiter.run(
                () -> client.describeAnomalyDetector(applyWaitersUserAgent(describeAnomalyDetectorRequest)),
                anomalyDetectorActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeAnomalyDetectorResponse> waitUntilAnomalyDetectorDeleted(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest) {
        return anomalyDetectorDeletedWaiter.run(() -> client
                .describeAnomalyDetector(applyWaitersUserAgent(describeAnomalyDetectorRequest)));
    }

    @Override
    public WaiterResponse<DescribeAnomalyDetectorResponse> waitUntilAnomalyDetectorDeleted(
            DescribeAnomalyDetectorRequest describeAnomalyDetectorRequest, WaiterOverrideConfiguration overrideConfig) {
        return anomalyDetectorDeletedWaiter.run(
                () -> client.describeAnomalyDetector(applyWaitersUserAgent(describeAnomalyDetectorRequest)),
                anomalyDetectorDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeScraperResponse> waitUntilScraperActive(DescribeScraperRequest describeScraperRequest) {
        return scraperActiveWaiter.run(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)));
    }

    @Override
    public WaiterResponse<DescribeScraperResponse> waitUntilScraperActive(DescribeScraperRequest describeScraperRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return scraperActiveWaiter.run(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)),
                scraperActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeScraperResponse> waitUntilScraperDeleted(DescribeScraperRequest describeScraperRequest) {
        return scraperDeletedWaiter.run(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)));
    }

    @Override
    public WaiterResponse<DescribeScraperResponse> waitUntilScraperDeleted(DescribeScraperRequest describeScraperRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return scraperDeletedWaiter.run(() -> client.describeScraper(applyWaitersUserAgent(describeScraperRequest)),
                scraperDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeWorkspaceResponse> waitUntilWorkspaceActive(DescribeWorkspaceRequest describeWorkspaceRequest) {
        return workspaceActiveWaiter.run(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)));
    }

    @Override
    public WaiterResponse<DescribeWorkspaceResponse> waitUntilWorkspaceActive(DescribeWorkspaceRequest describeWorkspaceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return workspaceActiveWaiter.run(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)),
                workspaceActiveWaiterConfig(overrideConfig));
    }

    @Override
    public WaiterResponse<DescribeWorkspaceResponse> waitUntilWorkspaceDeleted(DescribeWorkspaceRequest describeWorkspaceRequest) {
        return workspaceDeletedWaiter.run(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)));
    }

    @Override
    public WaiterResponse<DescribeWorkspaceResponse> waitUntilWorkspaceDeleted(DescribeWorkspaceRequest describeWorkspaceRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return workspaceDeletedWaiter.run(() -> client.describeWorkspace(applyWaitersUserAgent(describeWorkspaceRequest)),
                workspaceDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeAnomalyDetectorResponse>> anomalyDetectorActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAnomalyDetectorResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("anomalyDetector").field("status").field("statusCode").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("anomalyDetector").field("status").field("statusCode").value(), "CREATING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("anomalyDetector").field("status").field("statusCode").value(), "UPDATING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeAnomalyDetectorResponse>> anomalyDetectorDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeAnomalyDetectorResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("anomalyDetector").field("status").field("statusCode").value(), "DELETING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeScraperResponse>> scraperActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeScraperResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("scraper").field("status").field("statusCode").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("scraper").field("status").field("statusCode").value(), "CREATION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (scraper.status.statusCode=CREATION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeScraperResponse>> scraperDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeScraperResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    return Objects.equals(input.field("scraper").field("status").field("statusCode").value(), "DELETION_FAILED");
                },
                "A waiter acceptor with the matcher (path) was matched on parameter (scraper.status.statusCode=DELETION_FAILED) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeWorkspaceResponse>> workspaceActiveWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeWorkspaceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "ACTIVE");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "UPDATING");
        }));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "CREATING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeWorkspaceResponse>> workspaceDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeWorkspaceResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ResourceNotFoundException")));
        result.add(WaiterAcceptor.retryOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            return Objects.equals(input.field("workspace").field("status").field("statusCode").value(), "DELETING");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration anomalyDetectorActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration anomalyDetectorDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration scraperActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration scraperDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration workspaceActiveWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration workspaceDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(2)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AmpWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AmpRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AmpWaiter.Builder {
        private AmpClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private DefaultBuilder() {
        }

        @Override
        public AmpWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AmpWaiter.Builder client(AmpClient client) {
            this.client = client;
            return this;
        }

        public AmpWaiter build() {
            return new DefaultAmpWaiter(this);
        }
    }
}
