/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.services.acmpca.AcmPcaAsyncClient;
import software.amazon.awssdk.services.acmpca.model.AcmPcaRequest;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityAuditReportRequest;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityAuditReportResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCsrRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCsrResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateResponse;
import software.amazon.awssdk.services.acmpca.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultAcmPcaAsyncWaiter implements AcmPcaAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final AcmPcaAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<GetCertificateAuthorityCsrResponse> certificateAuthorityCSRCreatedWaiter;

    private final AsyncWaiter<GetCertificateResponse> certificateIssuedWaiter;

    private final AsyncWaiter<DescribeCertificateAuthorityAuditReportResponse> auditReportCreatedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultAcmPcaAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = AcmPcaAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.certificateAuthorityCSRCreatedWaiter = AsyncWaiter.builder(GetCertificateAuthorityCsrResponse.class)
                .acceptors(certificateAuthorityCSRCreatedWaiterAcceptors())
                .overrideConfiguration(certificateAuthorityCSRCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.certificateIssuedWaiter = AsyncWaiter.builder(GetCertificateResponse.class)
                .acceptors(certificateIssuedWaiterAcceptors())
                .overrideConfiguration(certificateIssuedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.auditReportCreatedWaiter = AsyncWaiter.builder(DescribeCertificateAuthorityAuditReportResponse.class)
                .acceptors(auditReportCreatedWaiterAcceptors())
                .overrideConfiguration(auditReportCreatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCertificateAuthorityAuditReportResponse>> waitUntilAuditReportCreated(
            DescribeCertificateAuthorityAuditReportRequest describeCertificateAuthorityAuditReportRequest) {
        return auditReportCreatedWaiter.runAsync(() -> client
                .describeCertificateAuthorityAuditReport(applyWaitersUserAgent(describeCertificateAuthorityAuditReportRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCertificateAuthorityAuditReportResponse>> waitUntilAuditReportCreated(
            DescribeCertificateAuthorityAuditReportRequest describeCertificateAuthorityAuditReportRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return auditReportCreatedWaiter.runAsync(() -> client
                .describeCertificateAuthorityAuditReport(applyWaitersUserAgent(describeCertificateAuthorityAuditReportRequest)),
                auditReportCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetCertificateAuthorityCsrResponse>> waitUntilCertificateAuthorityCSRCreated(
            GetCertificateAuthorityCsrRequest getCertificateAuthorityCsrRequest) {
        return certificateAuthorityCSRCreatedWaiter.runAsync(() -> client
                .getCertificateAuthorityCsr(applyWaitersUserAgent(getCertificateAuthorityCsrRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetCertificateAuthorityCsrResponse>> waitUntilCertificateAuthorityCSRCreated(
            GetCertificateAuthorityCsrRequest getCertificateAuthorityCsrRequest, WaiterOverrideConfiguration overrideConfig) {
        return certificateAuthorityCSRCreatedWaiter.runAsync(
                () -> client.getCertificateAuthorityCsr(applyWaitersUserAgent(getCertificateAuthorityCsrRequest)),
                certificateAuthorityCSRCreatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetCertificateResponse>> waitUntilCertificateIssued(
            GetCertificateRequest getCertificateRequest) {
        return certificateIssuedWaiter.runAsync(() -> client.getCertificate(applyWaitersUserAgent(getCertificateRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetCertificateResponse>> waitUntilCertificateIssued(
            GetCertificateRequest getCertificateRequest, WaiterOverrideConfiguration overrideConfig) {
        return certificateIssuedWaiter.runAsync(() -> client.getCertificate(applyWaitersUserAgent(getCertificateRequest)),
                certificateIssuedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super GetCertificateAuthorityCsrResponse>> certificateAuthorityCSRCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetCertificateAuthorityCsrResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "RequestInProgressException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetCertificateResponse>> certificateIssuedWaiterAcceptors() {
        List<WaiterAcceptor<? super GetCertificateResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "RequestInProgressException")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCertificateAuthorityAuditReportResponse>> auditReportCreatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCertificateAuthorityAuditReportResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("AuditReportStatus").value(), "SUCCESS");
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("AuditReportStatus").value(), "FAILED");
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration certificateAuthorityCSRCreatedWaiterConfig(
            WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration certificateIssuedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration auditReportCreatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(60);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(3)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static AcmPcaAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends AcmPcaRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements AcmPcaAsyncWaiter.Builder {
        private AcmPcaAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public AcmPcaAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public AcmPcaAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public AcmPcaAsyncWaiter.Builder client(AcmPcaAsyncClient client) {
            this.client = client;
            return this;
        }

        public AcmPcaAsyncWaiter build() {
            return new DefaultAcmPcaAsyncWaiter(this);
        }
    }
}
