/**
 * <h2>AWS CDK Assets</h2>
 * <!-- raw HTML omitted -->
 * <hr />
 * <p><img src="https://img.shields.io/badge/stability-Experimental-important.svg?style=for-the-badge" alt="Stability: Experimental" /></p>
 * <blockquote>
 * <p><strong>This is a <em>developer preview</em> (public beta) module. Releases might lack important features and might have
 * future breaking changes.</strong></p>
 * <p>This API is still under active development and subject to non-backward
 * compatible changes or removal in any future version. Use of the API is not recommended in production
 * environments. Experimental APIs are not subject to the Semantic Versioning model.</p>
 * </blockquote>
 * <hr />
 * <!-- raw HTML omitted -->
 * <p>Assets are local files or directories which are needed by a CDK app. A common
 * example is a directory which contains the handler code for a Lambda function,
 * but assets can represent any artifact that is needed for the app's operation.</p>
 * <p>When deploying a CDK app that includes constructs with assets, the CDK toolkit
 * will first upload all the assets to S3, and only then deploy the stacks. The S3
 * locations of the uploaded assets will be passed in as CloudFormation Parameters
 * to the relevant stacks.</p>
 * <p>The following JavaScript example defines an directory asset which is archived as
 * a .zip file and uploaded to S3 during deployment.</p>
 * <pre><code class="language-ts">const asset = new assets.Asset(this, 'SampleAsset', {
 *   path: path.join(__dirname, 'sample-asset-directory')
 * });
 * </code></pre>
 * <p>The following JavaScript example defines a file asset, which is uploaded as-is
 * to an S3 bucket during deployment.</p>
 * <pre><code class="language-ts">const asset = new assets.Asset(this, 'SampleAsset', {
 *   path: path.join(__dirname, 'file-asset.txt')
 * });
 * </code></pre>
 * <h2>Attributes</h2>
 * <p><code>Asset</code> constructs expose the following deploy-time attributes:</p>
 * <ul>
 * <li><code>s3BucketName</code> - the name of the assets S3 bucket.</li>
 * <li><code>s3ObjectKey</code> - the S3 object key of the asset file (whether it's a file or a zip archive)</li>
 * <li><code>s3Url</code> - the S3 URL of the asset (i.e. https://s3.us-east-1.amazonaws.com/mybucket/mykey.zip)</li>
 * </ul>
 * <p>In the following example, the various asset attributes are exported as stack outputs:</p>
 * <pre><code class="language-ts">    const asset = new assets.Asset(this, 'SampleAsset', {
 *       path: path.join(__dirname, 'sample-asset-directory')
 *     });
 * 
 *     new cdk.CfnOutput(this, 'S3BucketName', { value: asset.s3BucketName });
 *     new cdk.CfnOutput(this, 'S3ObjectKey', { value: asset.s3ObjectKey });
 *     new cdk.CfnOutput(this, 'S3URL', { value: asset.s3Url });
 * </code></pre>
 * <h2>Permissions</h2>
 * <p>IAM roles, users or groups which need to be able to read assets in runtime will should be
 * granted IAM permissions. To do that use the <code>asset.grantRead(principal)</code> method:</p>
 * <p>The following examples grants an IAM group read permissions on an asset:</p>
 * <pre><code class="language-ts">const group = new iam.Group(this, 'MyUserGroup');
 * asset.grantRead(group);
 * </code></pre>
 * <h2>How does it work?</h2>
 * <p>When an asset is defined in a construct, a construct metadata entry
 * <code>aws:cdk:asset</code> is emitted with instructions on where to find the asset and what
 * type of packaging to perform (<code>zip</code> or <code>file</code>). Furthermore, the synthesized
 * CloudFormation template will also include two CloudFormation parameters: one for
 * the asset's bucket and one for the asset S3 key. Those parameters are used to
 * reference the deploy-time values of the asset (using <code>{ Ref: &quot;Param&quot; }</code>).</p>
 * <p>Then, when the stack is deployed, the toolkit will package the asset (i.e. zip
 * the directory), calculate an MD5 hash of the contents and will render an S3 key
 * for this asset within the toolkit's asset store. If the file doesn't exist in
 * the asset store, it is uploaded during deployment.</p>
 * <blockquote>
 * <p>The toolkit's asset store is an S3 bucket created by the toolkit for each
 * environment the toolkit operates in (environment = account + region).</p>
 * </blockquote>
 * <p>Now, when the toolkit deploys the stack, it will set the relevant CloudFormation
 * Parameters to point to the actual bucket and key for each asset.</p>
 * <h2>CloudFormation Resource Metadata</h2>
 * <blockquote>
 * <p>NOTE: This section is relevant for authors of AWS Resource Constructs.</p>
 * </blockquote>
 * <p>In certain situations, it is desirable for tools to be able to know that a certain CloudFormation
 * resource is using a local asset. For example, SAM CLI can be used to invoke AWS Lambda functions
 * locally for debugging purposes.</p>
 * <p>To enable such use cases, external tools will consult a set of metadata entries on AWS CloudFormation
 * resources:</p>
 * <ul>
 * <li><code>aws:asset:path</code> points to the local path of the asset.</li>
 * <li><code>aws:asset:property</code> is the name of the resource property where the asset is used</li>
 * </ul>
 * <p>Using these two metadata entries, tools will be able to identify that assets are used
 * by a certain resource, and enable advanced local experiences.</p>
 * <p>To add these metadata entries to a resource, use the
 * <code>asset.addResourceMetadata(resource, property)</code> method.</p>
 * <p>See https://github.com/awslabs/aws-cdk/issues/1432 for more details</p>
 * 
 */
@software.amazon.jsii.Stability(software.amazon.jsii.Stability.Level.Experimental)
package software.amazon.awscdk.services.s3.assets;
