/**
 * <h1>Event Targets for Amazon EventBridge</h1>
 * <p>
 * This library contains integration classes to send Amazon EventBridge to any
 * number of supported AWS Services. Instances of these classes should be passed
 * to the <code>rule.addTarget()</code> method.
 * <p>
 * Currently supported are:
 * <p>
 * <ul>
 * <li><a href="#event-targets-for-amazon-eventbridge">Event Targets for Amazon EventBridge</a>
 * <p>
 * <ul>
 * <li><a href="#event-retry-policy-and-using-dead-letter-queues">Event retry policy and using dead-letter queues</a></li>
 * <li><a href="#invoke-a-lambda-function">Invoke a Lambda function</a></li>
 * <li><a href="#log-an-event-into-a-loggroup">Log an event into a LogGroup</a></li>
 * <li><a href="#start-a-codebuild-build">Start a CodeBuild build</a></li>
 * <li><a href="#start-a-codepipeline-pipeline">Start a CodePipeline pipeline</a></li>
 * <li><a href="#start-a-stepfunctions-state-machine">Start a StepFunctions state machine</a></li>
 * <li><a href="#queue-a-batch-job">Queue a Batch job</a></li>
 * <li><a href="#invoke-an-api-gateway-rest-api">Invoke an API Gateway REST API</a></li>
 * <li><a href="#invoke-an-api-destination">Invoke an API Destination</a></li>
 * <li><a href="#invoke-an-appsync-graphql-api">Invoke an AppSync GraphQL API</a></li>
 * <li><a href="#put-an-event-on-an-eventbridge-bus">Put an event on an EventBridge bus</a></li>
 * <li><a href="#run-an-ecs-task">Run an ECS Task</a>
 * <p>
 * <ul>
 * <li><a href="#tagging-tasks">Tagging Tasks</a></li>
 * <li><a href="#launch-type-for-ecs-task">Launch type for ECS Task</a></li>
 * <li><a href="#assign-public-ip-addresses-to-tasks">Assign public IP addresses to tasks</a></li>
 * <li><a href="#enable-amazon-ecs-exec-for-ecs-task">Enable Amazon ECS Exec for ECS Task</a></li>
 * </ul></li>
 * <li><a href="#schedule-a-redshift-query-serverless-or-cluster">Run a Redshift query</a></li>
 * </ul></li>
 * </ul>
 * <p>
 * See the README of the <code>aws-cdk-lib/aws-events</code> library for more information on
 * EventBridge.
 * <p>
 * <h2>Event retry policy and using dead-letter queues</h2>
 * <p>
 * The Codebuild, CodePipeline, Lambda, Kinesis Data Streams, StepFunctions, LogGroup, SQSQueue, SNSTopic and ECSTask targets support attaching a <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue and setting retry policies</a>. See the <a href="#invoke-a-lambda-function">lambda example</a>.
 * Use <a href="https://docs.aws.amazon.com/cdk/latest/guide/cfn_layer.html">escape hatches</a> for the other target types.
 * <p>
 * <h2>Invoke a Lambda function</h2>
 * <p>
 * Use the <code>LambdaFunction</code> target to invoke a lambda function.
 * <p>
 * The code snippet below creates an event rule with a Lambda function as a target
 * triggered for every events from <code>aws.ec2</code> source. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunc")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = handler.toString()"))
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "rule")
 *         .eventPattern(EventPattern.builder()
 *                 .source(List.of("aws.ec2"))
 *                 .build())
 *         .build();
 * 
 * Queue queue = new Queue(this, "Queue");
 * 
 * rule.addTarget(LambdaFunction.Builder.create(fn)
 *         .deadLetterQueue(queue) // Optional: add a dead letter queue
 *         .maxEventAge(Duration.hours(2)) // Optional: set the maxEventAge retry policy
 *         .retryAttempts(2)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Log an event into a LogGroup</h2>
 * <p>
 * Use the <code>LogGroup</code> target to log your events in a CloudWatch LogGroup.
 * <p>
 * For example, the following code snippet creates an event rule with a CloudWatch LogGroup as a target.
 * Every events sent from the <code>aws.ec2</code> source will be sent to the CloudWatch LogGroup.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * 
 * 
 * LogGroup logGroup = LogGroup.Builder.create(this, "MyLogGroup")
 *         .logGroupName("MyLogGroup")
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "rule")
 *         .eventPattern(EventPattern.builder()
 *                 .source(List.of("aws.ec2"))
 *                 .build())
 *         .build();
 * 
 * rule.addTarget(new CloudWatchLogGroup(logGroup));
 * </pre></blockquote>
 * <p>
 * A rule target input can also be specified to modify the event that is sent to the log group.
 * Unlike other event targets, CloudWatchLogs requires a specific input template format.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * LogGroup logGroup;
 * Rule rule;
 * 
 * 
 * rule.addTarget(CloudWatchLogGroup.Builder.create(logGroup)
 *         .logEvent(LogGroupTargetInput.fromObjectV2(LogGroupTargetInputOptions.builder()
 *                 .timestamp(EventField.fromPath("$.time"))
 *                 .message(EventField.fromPath("$.detail-type"))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you want to use static values to overwrite the <code>message</code> make sure that you provide a <code>string</code>
 * value.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * LogGroup logGroup;
 * Rule rule;
 * 
 * 
 * rule.addTarget(CloudWatchLogGroup.Builder.create(logGroup)
 *         .logEvent(LogGroupTargetInput.fromObjectV2(LogGroupTargetInputOptions.builder()
 *                 .message(JSON.stringify(Map.of(
 *                         "CustomField", "CustomValue")))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The cloudwatch log event target will create an AWS custom resource internally which will default
 * to set <code>installLatestAwsSdk</code> to <code>true</code>. This may be problematic for CN partition deployment. To
 * workaround this issue, set <code>installLatestAwsSdk</code> to <code>false</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.logs.*;
 * LogGroup logGroup;
 * Rule rule;
 * 
 * 
 * rule.addTarget(CloudWatchLogGroup.Builder.create(logGroup)
 *         .installLatestAwsSdk(false)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodeBuild build</h2>
 * <p>
 * Use the <code>CodeBuildProject</code> target to trigger a CodeBuild project.
 * <p>
 * The code snippet below creates a CodeCommit repository that triggers a CodeBuild project
 * on commit to the master branch. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codebuild.*;
 * import software.amazon.awscdk.services.codecommit.*;
 * 
 * 
 * Repository repo = Repository.Builder.create(this, "MyRepo")
 *         .repositoryName("aws-cdk-codebuild-events")
 *         .build();
 * 
 * Project project = Project.Builder.create(this, "MyProject")
 *         .source(Source.codeCommit(CodeCommitSourceProps.builder().repository(repo).build()))
 *         .build();
 * 
 * Queue deadLetterQueue = new Queue(this, "DeadLetterQueue");
 * 
 * // trigger a build when a commit is pushed to the repo
 * Rule onCommitRule = repo.onCommit("OnCommit", OnCommitOptions.builder()
 *         .target(CodeBuildProject.Builder.create(project)
 *                 .deadLetterQueue(deadLetterQueue)
 *                 .build())
 *         .branches(List.of("master"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodePipeline pipeline</h2>
 * <p>
 * Use the <code>CodePipeline</code> target to trigger a CodePipeline pipeline.
 * <p>
 * The code snippet below creates a CodePipeline pipeline that is triggered every hour
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codepipeline.*;
 * 
 * 
 * Pipeline pipeline = new Pipeline(this, "Pipeline");
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.expression("rate(1 hour)"))
 *         .build();
 * 
 * rule.addTarget(new CodePipeline(pipeline));
 * </pre></blockquote>
 * <p>
 * <h2>Start a StepFunctions state machine</h2>
 * <p>
 * Use the <code>SfnStateMachine</code> target to trigger a State Machine.
 * <p>
 * The code snippet below creates a Simple StateMachine that is triggered every minute with a
 * dummy object as input.
 * You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>
 * to the target.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.minutes(1)))
 *         .build();
 * 
 * Queue dlq = new Queue(this, "DeadLetterQueue");
 * 
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new ServicePrincipal("events.amazonaws.com"))
 *         .build();
 * StateMachine stateMachine = StateMachine.Builder.create(this, "SM")
 *         .definition(Wait.Builder.create(this, "Hello").time(WaitTime.duration(Duration.seconds(10))).build())
 *         .build();
 * 
 * rule.addTarget(SfnStateMachine.Builder.create(stateMachine)
 *         .input(RuleTargetInput.fromObject(Map.of("SomeParam", "SomeValue")))
 *         .deadLetterQueue(dlq)
 *         .role(role)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Queue a Batch job</h2>
 * <p>
 * Use the <code>BatchJob</code> target to queue a Batch job.
 * <p>
 * The code snippet below creates a Simple JobQueue that is triggered every hour with a
 * dummy object as input.
 * You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>
 * to the target.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ec2.*;
 * import software.amazon.awscdk.services.ecs.*;
 * import software.amazon.awscdk.services.batch.*;
 * import software.amazon.awscdk.services.ecs.ContainerImage;
 * 
 * Vpc vpc;
 * 
 * 
 * FargateComputeEnvironment computeEnvironment = FargateComputeEnvironment.Builder.create(this, "ComputeEnv")
 *         .vpc(vpc)
 *         .build();
 * 
 * JobQueue jobQueue = JobQueue.Builder.create(this, "JobQueue")
 *         .priority(1)
 *         .computeEnvironments(List.of(OrderedComputeEnvironment.builder()
 *                 .computeEnvironment(computeEnvironment)
 *                 .order(1)
 *                 .build()))
 *         .build();
 * 
 * EcsJobDefinition jobDefinition = EcsJobDefinition.Builder.create(this, "MyJob")
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "Container")
 *                 .image(ContainerImage.fromRegistry("test-repo"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .build())
 *         .build();
 * 
 * Queue queue = new Queue(this, "Queue");
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(BatchJob.Builder.create(jobQueue.getJobQueueArn(), jobQueue, jobDefinition.getJobDefinitionArn(), jobDefinition)
 *         .deadLetterQueue(queue)
 *         .event(RuleTargetInput.fromObject(Map.of("SomeParam", "SomeValue")))
 *         .retryAttempts(2)
 *         .maxEventAge(Duration.hours(2))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Invoke an API Gateway REST API</h2>
 * <p>
 * Use the <code>ApiGateway</code> target to trigger a REST API.
 * <p>
 * The code snippet below creates a Api Gateway REST API that is invoked every hour.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigateway.*;
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.minutes(1)))
 *         .build();
 * 
 * Function fn = Function.Builder.create(this, "MyFunc")
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .code(Code.fromInline("exports.handler = e =&gt; {}"))
 *         .build();
 * 
 * LambdaRestApi restApi = LambdaRestApi.Builder.create(this, "MyRestAPI").handler(fn).build();
 * 
 * Queue dlq = new Queue(this, "DeadLetterQueue");
 * 
 * rule.addTarget(
 * ApiGateway.Builder.create(restApi)
 *         .path("/*&#47;test")
 *         .method("GET")
 *         .stage("prod")
 *         .pathParameterValues(List.of("path-value"))
 *         .headerParameters(Map.of(
 *                 "Header1", "header1"))
 *         .queryStringParameters(Map.of(
 *                 "QueryParam1", "query-param-1"))
 *         .deadLetterQueue(dlq)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Invoke an API Gateway V2 HTTP API</h2>
 * <p>
 * Use the <code>ApiGatewayV2</code> target to trigger a HTTP API.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.apigatewayv2.*;
 * 
 * HttpApi httpApi;
 * Rule rule;
 * 
 * 
 * rule.addTarget(new ApiGatewayV2(httpApi));
 * </pre></blockquote>
 * <p>
 * <h2>Invoke an API Destination</h2>
 * <p>
 * Use the <code>targets.ApiDestination</code> target to trigger an external API. You need to
 * create an <code>events.Connection</code> and <code>events.ApiDestination</code> as well.
 * <p>
 * The code snippet below creates an external destination that is invoked every hour.
 * <p>
 * <blockquote><pre>
 * Connection connection = Connection.Builder.create(this, "Connection")
 *         .authorization(Authorization.apiKey("x-api-key", SecretValue.secretsManager("ApiSecretName")))
 *         .description("Connection with API Key x-api-key")
 *         .build();
 * 
 * ApiDestination destination = ApiDestination.Builder.create(this, "Destination")
 *         .connection(connection)
 *         .endpoint("https://example.com")
 *         .description("Calling example.com with API key x-api-key")
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.minutes(1)))
 *         .targets(List.of(new ApiDestination(destination)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also import an existing connection and destination
 * to create additional rules:
 * <p>
 * <blockquote><pre>
 * IConnection connection = Connection.fromEventBusArn(this, "Connection", "arn:aws:events:us-east-1:123456789012:event-bus/EventBusName", "arn:aws:secretsmanager:us-east-1:123456789012:secret:SecretName-f3gDy9");
 * 
 * String apiDestinationArn = "arn:aws:events:us-east-1:123456789012:api-destination/DestinationName";
 * ApiDestination destination = ApiDestination.fromApiDestinationAttributes(this, "Destination", ApiDestinationAttributes.builder().apiDestinationArn(apiDestinationArn).connection(connection).build());
 * 
 * Rule rule = Rule.Builder.create(this, "OtherRule")
 *         .schedule(Schedule.rate(Duration.minutes(10)))
 *         .targets(List.of(new ApiDestination(destination)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Invoke an AppSync GraphQL API</h2>
 * <p>
 * Use the <code>AppSync</code> target to trigger an AppSync GraphQL API. You need to
 * create an <code>AppSync.GraphqlApi</code> configured with <code>AWS_IAM</code> authorization mode.
 * <p>
 * The code snippet below creates an AppSync GraphQL API target that is invoked every hour, calling the <code>publish</code> mutation.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.appsync.*;
 * 
 * 
 * GraphqlApi api = GraphqlApi.Builder.create(this, "api")
 *         .name("api")
 *         .definition(Definition.fromFile("schema.graphql"))
 *         .authorizationConfig(AuthorizationConfig.builder()
 *                 .defaultAuthorization(AuthorizationMode.builder().authorizationType(AuthorizationType.IAM).build())
 *                 .build())
 *         .build();
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(AppSync.Builder.create(api)
 *         .graphQLOperation("mutation Publish($message: String!){ publish(message: $message) { message } }")
 *         .variables(RuleTargetInput.fromObject(Map.of(
 *                 "message", "hello world")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * You can pass an existing role with the proper permissions to be used for the target when the rule is triggered. The code snippet below uses an existing role and grants permissions to use the publish Mutation on the GraphQL API.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.appsync.*;
 * 
 * 
 * IGraphqlApi api = GraphqlApi.fromGraphqlApiAttributes(this, "ImportedAPI", GraphqlApiAttributes.builder()
 *         .graphqlApiId("&lt;api-id&gt;")
 *         .graphqlApiArn("&lt;api-arn&gt;")
 *         .graphQLEndpointArn("&lt;api-endpoint-arn&gt;")
 *         .visibility(Visibility.GLOBAL)
 *         .modes(List.of(AuthorizationType.IAM))
 *         .build());
 * 
 * Rule rule = Rule.Builder.create(this, "Rule").schedule(Schedule.rate(Duration.minutes(1))).build();
 * Role role = Role.Builder.create(this, "Role").assumedBy(new ServicePrincipal("events.amazonaws.com")).build();
 * 
 * // allow EventBridge to use the `publish` mutation
 * api.grantMutation(role, "publish");
 * 
 * rule.addTarget(AppSync.Builder.create(api)
 *         .graphQLOperation("mutation Publish($message: String!){ publish(message: $message) { message } }")
 *         .variables(RuleTargetInput.fromObject(Map.of(
 *                 "message", "hello world")))
 *         .eventRole(role)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Put an event on an EventBridge bus</h2>
 * <p>
 * Use the <code>EventBus</code> target to route event to a different EventBus.
 * <p>
 * The code snippet below creates the scheduled event rule that route events to an imported event bus.
 * <p>
 * <blockquote><pre>
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.expression("rate(1 minute)"))
 *         .build();
 * 
 * rule.addTarget(new EventBus(EventBus.fromEventBusArn(this, "External", "arn:aws:events:eu-west-1:999999999999:event-bus/test-bus")));
 * </pre></blockquote>
 * <p>
 * <h2>Run an ECS Task</h2>
 * <p>
 * Use the <code>EcsTask</code> target to run an ECS Task.
 * <p>
 * The code snippet below creates a scheduled event rule that will run the task described in <code>taskDefinition</code> every hour.
 * <p>
 * <h3>Tagging Tasks</h3>
 * <p>
 * By default, ECS tasks run from EventBridge targets will not have tags applied to
 * them. You can set the <code>propagateTags</code> field to propagate the tags set on the task
 * definition to the task initialized by the event trigger.
 * <p>
 * If you want to set tags independent of those applied to the TaskDefinition, you
 * can use the <code>tags</code> array. Both of these fields can be used together or separately
 * to set tags on the triggered task.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * ICluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(
 * EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .propagateTags(PropagatedTagSource.TASK_DEFINITION)
 *         .tags(List.of(Tag.builder()
 *                 .key("my-tag")
 *                 .value("my-tag-value")
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Launch type for ECS Task</h3>
 * <p>
 * By default, if <code>isEc2Compatible</code> for the <code>taskDefinition</code> is true, the EC2 type is used as
 * the launch type for the task, otherwise the FARGATE type.
 * If you want to override the default launch type, you can set the <code>launchType</code> property.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * ICluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .launchType(LaunchType.FARGATE)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Assign public IP addresses to tasks</h3>
 * <p>
 * You can set the <code>assignPublicIp</code> flag to assign public IP addresses to tasks.
 * If you want to detach the public IP address from the task, you have to set the flag <code>false</code>.
 * You can specify the flag <code>true</code> only when the launch type is set to FARGATE.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * import software.amazon.awscdk.services.ec2.*;
 * 
 * ICluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(
 * EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .assignPublicIp(true)
 *         .subnetSelection(SubnetSelection.builder().subnetType(SubnetType.PUBLIC).build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Enable Amazon ECS Exec for ECS Task</h3>
 * <p>
 * If you use Amazon ECS Exec, you can run commands in or get a shell to a container running on an Amazon EC2 instance or on AWS Fargate.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * ICluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .taskCount(1)
 *         .containerOverrides(List.of(ContainerOverride.builder()
 *                 .containerName("TheContainer")
 *                 .command(List.of("echo", EventField.fromPath("$.detail.event")))
 *                 .build()))
 *         .enableExecuteCommand(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Overriding Values in the Task Definition</h3>
 * <p>
 * You can override values in the task definition by setting the corresponding properties in the <code>EcsTaskProps</code>. All
 * values in the <a href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_TaskOverride.html"><code>TaskOverrides</code> API</a> are
 * supported.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * ICluster cluster;
 * TaskDefinition taskDefinition;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * rule.addTarget(EcsTask.Builder.create()
 *         .cluster(cluster)
 *         .taskDefinition(taskDefinition)
 *         .taskCount(1)
 * 
 *         // Overrides the cpu and memory values in the task definition
 *         .cpu("512")
 *         .memory("512")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Schedule a Redshift query (serverless or cluster)</h2>
 * <p>
 * Use the <code>RedshiftQuery</code> target to schedule an Amazon Redshift Query.
 * <p>
 * The code snippet below creates the scheduled event rule that route events to an Amazon Redshift Query
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.redshiftserverless.*;
 * 
 * CfnWorkgroup workgroup;
 * 
 * 
 * Rule rule = Rule.Builder.create(this, "Rule")
 *         .schedule(Schedule.rate(Duration.hours(1)))
 *         .build();
 * 
 * Queue dlq = new Queue(this, "DeadLetterQueue");
 * 
 * rule.addTarget(RedshiftQuery.Builder.create(workgroup.getAttrWorkgroupWorkgroupArn())
 *         .database("dev")
 *         .deadLetterQueue(dlq)
 *         .sql(List.of("SELECT * FROM foo", "SELECT * FROM baz"))
 *         .build());
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.events.targets;
