/**
 * <h1>AWS CodeBuild Construct Library</h1>
 * <p>
 * AWS CodeBuild is a fully managed continuous integration service that compiles
 * source code, runs tests, and produces software packages that are ready to
 * deploy. With CodeBuild, you don’t need to provision, manage, and scale your own
 * build servers. CodeBuild scales continuously and processes multiple builds
 * concurrently, so your builds are not left waiting in a queue. You can get
 * started quickly by using prepackaged build environments, or you can create
 * custom build environments that use your own build tools. With CodeBuild, you are
 * charged by the minute for the compute resources you use.
 * <p>
 * <h2>Source</h2>
 * <p>
 * Build projects are usually associated with a <em>source</em>, which is specified via
 * the <code>source</code> property which accepts a class that extends the <code>Source</code>
 * abstract base class.
 * The default is to have no source associated with the build project;
 * the <code>buildSpec</code> option is required in that case.
 * <p>
 * Here's a CodeBuild project with no source which simply prints <code>Hello, CodeBuild!</code>:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("echo \"Hello, CodeBuild!\""))))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3><code>CodeCommitSource</code></h3>
 * <p>
 * Use an AWS CodeCommit repository as the source of this build:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codecommit.*;
 * 
 * 
 * Repository repository = Repository.Builder.create(this, "MyRepo").repositoryName("foo").build();
 * Project.Builder.create(this, "MyFirstCodeCommitProject")
 *         .source(Source.codeCommit(CodeCommitSourceProps.builder().repository(repository).build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3><code>S3Source</code></h3>
 * <p>
 * Create a CodeBuild project with an S3 bucket as the source:
 * <p>
 * <blockquote><pre>
 * Bucket bucket = new Bucket(this, "MyBucket");
 * 
 * Project.Builder.create(this, "MyProject")
 *         .source(Source.s3(S3SourceProps.builder()
 *                 .bucket(bucket)
 *                 .path("path/to/file.zip")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The CodeBuild role will be granted to read just the given path from the given <code>bucket</code>.
 * <p>
 * <h3><code>GitHubSource</code> and <code>GitHubEnterpriseSource</code></h3>
 * <p>
 * These source types can be used to build code from a GitHub repository.
 * Example:
 * <p>
 * <blockquote><pre>
 * ISource gitHubSource = Source.gitHub(GitHubSourceProps.builder()
 *         .owner("awslabs")
 *         .repo("aws-cdk") // optional, default: undefined if unspecified will create organization webhook
 *         .webhook(true) // optional, default: true if `webhookFilters` were provided, false otherwise
 *         .webhookTriggersBatchBuild(true) // optional, default is false
 *         .webhookFilters(List.of(FilterGroup.inEventOf(EventAction.PUSH).andBranchIs("main").andCommitMessageIs("the commit message"), FilterGroup.inEventOf(EventAction.RELEASED).andBranchIs("main")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>GitHubSource</code> is also able to trigger all repos in GitHub Organizations
 * Example:
 * <p>
 * <blockquote><pre>
 * ISource gitHubSource = Source.gitHub(GitHubSourceProps.builder()
 *         .owner("aws")
 *         .webhookTriggersBatchBuild(true) // optional, default is false
 *         .webhookFilters(List.of(FilterGroup.inEventOf(EventAction.WORKFLOW_JOB_QUEUED).andRepositoryNameIs("aws-.*").andRepositoryNameIsNot("aws-cdk-lib")))
 *         .build());
 * </pre></blockquote>
 * <p>
 * To provide GitHub credentials, please either go to AWS CodeBuild Console to connect
 * or call <code>ImportSourceCredentials</code> to persist your personal access token.
 * Example:
 * <p>
 * <blockquote><pre>
 * aws codebuild import-source-credentials --server-type GITHUB --auth-type PERSONAL_ACCESS_TOKEN --token &lt;token_value&gt;
 * </pre></blockquote>
 * <p>
 * <h3><code>BitBucketSource</code></h3>
 * <p>
 * This source type can be used to build code from a BitBucket repository.
 * <p>
 * <blockquote><pre>
 * ISource bbSource = Source.bitBucket(BitBucketSourceProps.builder()
 *         .owner("owner")
 *         .repo("repo")
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>For all Git sources</h3>
 * <p>
 * For all Git sources, you can fetch submodules while cloning git repo.
 * <p>
 * <blockquote><pre>
 * ISource gitHubSource = Source.gitHub(GitHubSourceProps.builder()
 *         .owner("awslabs")
 *         .repo("aws-cdk")
 *         .fetchSubmodules(true)
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>BuildSpec</h2>
 * <p>
 * The build spec can be provided from a number of different sources
 * <p>
 * <h3>File path relative to the root of the source</h3>
 * <p>
 * You can specify a specific filename that exists within the project's source artifact to use as the buildspec.
 * <p>
 * <blockquote><pre>
 * Project project = Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromSourceFilename("my-buildspec.yml"))
 *         .source(Source.gitHub(GitHubSourceProps.builder()
 *                 .owner("awslabs")
 *                 .repo("aws-cdk")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will use <code>my-buildspec.yml</code> file within the <code>awslabs/aws-cdk</code> repository as the build spec.
 * <p>
 * <h3>File within the CDK project codebuild</h3>
 * <p>
 * You can also specify a file within your cdk project directory to use as the buildspec.
 * <p>
 * <blockquote><pre>
 * Project project = Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromAsset("my-buildspec.yml"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This file will be uploaded to S3 and referenced from the codebuild project.
 * <p>
 * <h3>Inline object</h3>
 * <p>
 * <blockquote><pre>
 * Project project = Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will result in the buildspec being rendered as JSON within the codebuild project, if you prefer it to be rendered as YAML, use <code>fromObjectToYaml</code>.
 * <p>
 * <blockquote><pre>
 * Project project = Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromObjectToYaml(Map.of(
 *                 "version", "0.2")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Artifacts</h2>
 * <p>
 * CodeBuild Projects can produce Artifacts and upload them to S3. For example:
 * <p>
 * <blockquote><pre>
 * Bucket bucket;
 * 
 * 
 * Project project = Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2")))
 *         .artifacts(Artifacts.s3(S3ArtifactsProps.builder()
 *                 .bucket(bucket)
 *                 .includeBuildId(false)
 *                 .packageZip(true)
 *                 .path("another/path")
 *                 .identifier("AddArtifact1")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Because we've not set the <code>name</code> property, this example will set the
 * <code>overrideArtifactName</code> parameter, and produce an artifact named as defined in
 * the Buildspec file, uploaded to an S3 bucket (<code>bucket</code>). The path will be
 * <code>another/path</code> and the artifact will be a zipfile.
 * <p>
 * <h2>CodePipeline</h2>
 * <p>
 * To add a CodeBuild Project as an Action to CodePipeline,
 * use the <code>PipelineProject</code> class instead of <code>Project</code>.
 * It's a simple class that doesn't allow you to specify <code>sources</code>,
 * <code>secondarySources</code>, <code>artifacts</code> or <code>secondaryArtifacts</code>,
 * as these are handled by setting input and output CodePipeline <code>Artifact</code> instances on the Action,
 * instead of setting them on the Project.
 * <p>
 * <blockquote><pre>
 * PipelineProject project = PipelineProject.Builder.create(this, "Project").build();
 * </pre></blockquote>
 * <p>
 * For more details, see the readme of the <code>&#64;aws-cdk/aws-codepipeline-actions</code> package.
 * <p>
 * <h2>Caching</h2>
 * <p>
 * You can save time when your project builds by using a cache. A cache can store reusable pieces of your build environment and use them across multiple builds. Your build project can use one of two types of caching: Amazon S3 or local. In general, S3 caching is a good option for small and intermediate build artifacts that are more expensive to build than to download. Local caching is a good option for large intermediate build artifacts because the cache is immediately available on the build host.
 * <p>
 * <h3>S3 Caching</h3>
 * <p>
 * With S3 caching, the cache is stored in an S3 bucket which is available
 * regardless from what CodeBuild instance gets selected to run your CodeBuild job
 * on. When using S3 caching, you must also add in a <code>cache</code> section to your
 * buildspec which indicates the files to be cached:
 * <p>
 * <blockquote><pre>
 * Bucket myCachingBucket;
 * 
 * 
 * Project.Builder.create(this, "Project")
 *         .source(Source.bitBucket(BitBucketSourceProps.builder()
 *                 .owner("awslabs")
 *                 .repo("aws-cdk")
 *                 .build()))
 * 
 *         .cache(Cache.bucket(myCachingBucket))
 * 
 *         // BuildSpec with a 'cache' section necessary for S3 caching. This can
 *         // also come from 'buildspec.yml' in your source.
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("..."))),
 *                 "cache", Map.of(
 *                         "paths", List.of("/root/cachedir/**&#47;*")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that two different CodeBuild Projects using the same S3 bucket will <em>not</em>
 * share their cache: each Project will get a unique file in the S3 bucket to store
 * the cache in.
 * <p>
 * <h3>Local Caching</h3>
 * <p>
 * With local caching, the cache is stored on the codebuild instance itself. This
 * is simple, cheap and fast, but CodeBuild cannot guarantee a reuse of instance
 * and hence cannot guarantee cache hits. For example, when a build starts and
 * caches files locally, if two subsequent builds start at the same time afterwards
 * only one of those builds would get the cache. Three different cache modes are
 * supported, which can be turned on individually.
 * <p>
 * <ul>
 * <li><code>LocalCacheMode.SOURCE</code> caches Git metadata for primary and secondary sources.</li>
 * <li><code>LocalCacheMode.DOCKER_LAYER</code> caches existing Docker layers.</li>
 * <li><code>LocalCacheMode.CUSTOM</code> caches directories you specify in the buildspec file.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .source(Source.gitHubEnterprise(GitHubEnterpriseSourceProps.builder()
 *                 .httpsCloneUrl("https://my-github-enterprise.com/owner/repo")
 *                 .build()))
 * 
 *         // Enable Docker AND custom caching
 *         .cache(Cache.local(LocalCacheMode.DOCKER_LAYER, LocalCacheMode.CUSTOM))
 * 
 *         // BuildSpec with a 'cache' section necessary for 'CUSTOM' caching. This can
 *         // also come from 'buildspec.yml' in your source.
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("..."))),
 *                 "cache", Map.of(
 *                         "paths", List.of("/root/cachedir/**&#47;*")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Environment</h2>
 * <p>
 * By default, projects use a small instance with an Ubuntu 18.04 image. You
 * can use the <code>environment</code> property to customize the build environment:
 * <p>
 * <ul>
 * <li><code>buildImage</code> defines the Docker image used. See <a href="#images">Images</a> below for
 * details on how to define build images.</li>
 * <li><code>certificate</code> defines the location of a PEM encoded certificate to import.</li>
 * <li><code>computeType</code> defines the instance type used for the build.</li>
 * <li><code>privileged</code> can be set to <code>true</code> to allow privileged access.</li>
 * <li><code>environmentVariables</code> can be set at this level (and also at the project
 * level).</li>
 * </ul>
 * <p>
 * Finally, you can also set the build environment <code>fleet</code> property to create
 * a reserved capacity project. See <a href="#fleet">Fleet</a> for more information.
 * <p>
 * <h2>Images</h2>
 * <p>
 * The CodeBuild library supports Linux, Windows, and Mac images via the
 * <code>LinuxBuildImage</code> (or <code>LinuxArmBuildImage</code>), <code>WindowsBuildImage</code>, and <code>MacBuildImage</code> classes, respectively.
 * With the introduction of Lambda compute support, the <code>LinuxLambdaBuildImage </code> (or <code>LinuxArmLambdaBuildImage</code>) class
 * is available for specifying Lambda-compatible images.
 * <p>
 * You can specify one of the predefined Windows/Linux images by using one
 * of the constants such as <code>WindowsBuildImage.WIN_SERVER_CORE_2019_BASE</code>,
 * <code>WindowsBuildImage.WINDOWS_BASE_2_0</code>, <code>LinuxBuildImage.STANDARD_2_0</code>,
 * <code>LinuxBuildImage.AMAZON_LINUX_2_5</code>, <code>MacBuildImage.BASE_14</code>, <code>LinuxArmBuildImage.AMAZON_LINUX_2_ARM</code>,
 * <code>LinuxLambdaBuildImage.AMAZON_LINUX_2_NODE_18</code> or <code>LinuxArmLambdaBuildImage.AMAZON_LINUX_2_NODE_18</code>.
 * <p>
 * Alternatively, you can specify a custom image using one of the static methods on
 * <code>LinuxBuildImage</code>:
 * <p>
 * <ul>
 * <li><code>LinuxBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])</code> to reference an image in any public or private Docker registry.</li>
 * <li><code>LinuxBuildImage.fromEcrRepository(repo[, tag])</code> to reference an image available in an
 * ECR repository.</li>
 * <li><code>LinuxBuildImage.fromAsset(parent, id, props)</code> to use an image created from a
 * local asset.</li>
 * <li><code>LinuxBuildImage.fromCodeBuildImageId(id)</code> to reference a pre-defined, CodeBuild-provided Docker image.</li>
 * </ul>
 * <p>
 * or one of the corresponding methods on <code>WindowsBuildImage</code>:
 * <p>
 * <ul>
 * <li><code>WindowsBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])</code></li>
 * <li><code>WindowsBuildImage.fromEcrRepository(repo[, tag, imageType])</code></li>
 * <li><code>WindowsBuildImage.fromAsset(parent, id, props, [, imageType])</code></li>
 * </ul>
 * <p>
 * or one of the corresponding methods on <code>MacBuildImage</code>:
 * <p>
 * <ul>
 * <li><code>MacBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }, imageType])</code></li>
 * <li><code>MacBuildImage.fromEcrRepository(repo[, tag, imageType])</code></li>
 * <li><code>MacBuildImage.fromAsset(parent, id, props, [, imageType])</code></li>
 * </ul>
 * <p>
 * or one of the corresponding methods on <code>LinuxArmBuildImage</code>:
 * <p>
 * <ul>
 * <li><code>LinuxArmBuildImage.fromDockerRegistry(image[, { secretsManagerCredentials }])</code></li>
 * <li><code>LinuxArmBuildImage.fromEcrRepository(repo[, tag])</code></li>
 * </ul>
 * <p>
 * <strong>Note:</strong> You cannot specify custom images on <code>LinuxLambdaBuildImage</code> or <code>LinuxArmLambdaBuildImage</code> images.
 * <p>
 * Note that the <code>WindowsBuildImage</code> version of the static methods accepts an optional parameter of type <code>WindowsImageType</code>,
 * which can be either <code>WindowsImageType.STANDARD</code>, the default, or <code>WindowsImageType.SERVER_2019</code>:
 * <p>
 * <blockquote><pre>
 * Repository ecrRepository;
 * 
 * 
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(WindowsBuildImage.fromEcrRepository(ecrRepository, "v1.0", WindowsImageType.SERVER_2019))
 *                 // optional certificate to include in the build image
 *                 .certificate(BuildEnvironmentCertificate.builder()
 *                         .bucket(Bucket.fromBucketName(this, "Bucket", "amzn-s3-demo-bucket"))
 *                         .objectKey("path/to/cert.pem")
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The following example shows how to define an image from a Docker asset:
 * <p>
 * <blockquote><pre>
 * .environment(BuildEnvironment.builder()
 *         .buildImage(LinuxBuildImage.fromAsset(this, "MyImage", DockerImageAssetProps.builder()
 *                 .directory(join(__dirname, "demo-image"))
 *                 .build()))
 *         .build())
 * .build();
 * </pre></blockquote>
 * <p>
 * The following example shows how to define an image from an ECR repository:
 * <p>
 * <blockquote><pre>
 * .environment(BuildEnvironment.builder()
 *         .buildImage(LinuxBuildImage.fromEcrRepository(ecrRepository, "v1.0"))
 *         .build())
 * .build();
 * </pre></blockquote>
 * <p>
 * The following example shows how to define an image from a private docker registry:
 * <p>
 * <blockquote><pre>
 * .environment(BuildEnvironment.builder()
 *         .buildImage(LinuxBuildImage.fromDockerRegistry("my-registry/my-repo", DockerImageOptions.builder()
 *                 .secretsManagerCredentials(secrets)
 *                 .build()))
 *         .build())
 * .build();
 * </pre></blockquote>
 * <p>
 * <h3>GPU images</h3>
 * <p>
 * The class <code>LinuxGpuBuildImage</code> contains constants for working with
 * <a href="https://aws.amazon.com/releasenotes/available-deep-learning-containers-images">AWS Deep Learning Container images</a>:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(LinuxGpuBuildImage.DLC_TENSORFLOW_2_1_0_INFERENCE)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * One complication is that the repositories for the DLC images are in
 * different accounts in different AWS regions.
 * In most cases, the CDK will handle providing the correct account for you;
 * in rare cases (for example, deploying to new regions)
 * where our information might be out of date,
 * you can always specify the account
 * (along with the repository name and tag)
 * explicitly using the <code>awsDeepLearningContainersImage</code> method:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(LinuxGpuBuildImage.awsDeepLearningContainersImage("tensorflow-inference", "2.1.0-gpu-py36-cu101-ubuntu18.04", "123456789012"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can reference an image available in an ECR repository using the <code>LinuxGpuBuildImage.fromEcrRepository(repo[, tag])</code> method.
 * <p>
 * <h3>Lambda images</h3>
 * <p>
 * The <code>LinuxLambdaBuildImage</code> (or <code>LinuxArmLambdaBuildImage</code>) class contains constants for working with
 * Lambda compute:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(LinuxLambdaBuildImage.AMAZON_LINUX_2_NODE_18)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/lambda.html">AWS Lambda compute in AWS CodeBuild</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h2>Fleet</h2>
 * <p>
 * By default, a CodeBuild project will request on-demand compute resources
 * to process your build requests. While being able to scale and handle high load,
 * on-demand resources can also be slow to provision.
 * <p>
 * Reserved capacity fleets are an alternative to on-demand.
 * Dedicated instances, maintained by CodeBuild,
 * will be ready to fulfill your build requests immediately.
 * Skipping the provisioning step in your project will reduce your build time,
 * at the cost of paying for these reserved instances, even when idling, until they are released.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/fleets.html">Working with reserved capacity in AWS CodeBuild</a> in the CodeBuild documentation.
 * <p>
 * <blockquote><pre>
 * Fleet fleet = Fleet.Builder.create(this, "Fleet")
 *         .computeType(FleetComputeType.MEDIUM)
 *         .environmentType(EnvironmentType.LINUX_CONTAINER)
 *         .baseCapacity(1)
 *         .build();
 * 
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .fleet(fleet)
 *                 .buildImage(LinuxBuildImage.STANDARD_7_0)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also import an existing fleet to share its resources
 * among several projects across multiple stacks:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .fleet(Fleet.fromFleetArn(this, "SharedFleet", "arn:aws:codebuild:us-east-1:123456789012:fleet/MyFleet:ed0d0823-e38a-4c10-90a1-1bf25f50fa76"))
 *                 .buildImage(LinuxBuildImage.STANDARD_7_0)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Attribute-based compute</h3>
 * <p>
 * You can use <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/fleets.html#fleets.attribute-compute">attribute-based compute</a> for your fleet by setting the <code>computeType</code> to <code>ATTRIBUTE_BASED</code>.
 * This allows you to specify the attributes in <code>computeConfiguration</code> such as vCPUs, memory, disk space, and the machineType.
 * After specifying some or all of the available attributes, CodeBuild will select the cheapest compute type from available instance types as that at least matches all given criteria.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Size;
 * 
 * 
 * Fleet fleet = Fleet.Builder.create(this, "MyFleet")
 *         .baseCapacity(1)
 *         .computeType(FleetComputeType.ATTRIBUTE_BASED)
 *         .environmentType(EnvironmentType.LINUX_CONTAINER)
 *         .computeConfiguration(ComputeConfiguration.builder()
 *                 .vCpu(2)
 *                 .memory(Size.gibibytes(4))
 *                 .disk(Size.gibibytes(10))
 *                 .machineType(MachineType.GENERAL)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Logs</h2>
 * <p>
 * CodeBuild lets you specify an S3 Bucket, CloudWatch Log Group or both to receive logs from your projects.
 * <p>
 * By default, logs will go to cloudwatch.
 * <p>
 * <h3>CloudWatch Logs Example</h3>
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .logging(LoggingOptions.builder()
 *                 .cloudWatch(CloudWatchLoggingOptions.builder()
 *                         .logGroup(new LogGroup(this, "MyLogGroup"))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>S3 Logs Example</h3>
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .logging(LoggingOptions.builder()
 *                 .s3(S3LoggingOptions.builder()
 *                         .bucket(new Bucket(this, "LogBucket"))
 *                         .build())
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Debugging builds interactively using SSM Session Manager</h2>
 * <p>
 * Integration with SSM Session Manager makes it possible to add breakpoints to your
 * build commands, pause the build there and log into the container to interactively
 * debug the environment.
 * <p>
 * To do so, you need to:
 * <p>
 * <ul>
 * <li>Create the build with <code>ssmSessionPermissions: true</code>.</li>
 * <li>Use a build image with SSM agent installed and configured (default CodeBuild images come with the image preinstalled).</li>
 * <li>Start the build with <a href="https://docs.aws.amazon.com/codebuild/latest/APIReference/API_StartBuild.html#CodeBuild-StartBuild-request-debugSessionEnabled">debugSessionEnabled</a> set to true.</li>
 * </ul>
 * <p>
 * If these conditions are met, execution of the command <code>codebuild-breakpoint</code>
 * will suspend your build and allow you to attach a Session Manager session from
 * the CodeBuild console.
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/session-manager.html">View a running build in Session
 * Manager</a>
 * in the CodeBuild documentation.
 * <p>
 * Example:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "Project")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(LinuxBuildImage.STANDARD_7_0)
 *                 .build())
 *         .ssmSessionPermissions(true)
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("codebuild-breakpoint", "./my-build.sh"))))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Credentials</h2>
 * <p>
 * CodeBuild allows you to store credentials used when communicating with various sources,
 * like GitHub:
 * <p>
 * <blockquote><pre>
 * GitHubSourceCredentials.Builder.create(this, "CodeBuildGitHubCreds")
 *         .accessToken(SecretValue.secretsManager("my-token"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * and BitBucket:
 * <p>
 * <blockquote><pre>
 * BitBucketSourceCredentials.Builder.create(this, "CodeBuildBitBucketCreds")
 *         .username(SecretValue.secretsManager("my-bitbucket-creds", SecretsManagerSecretOptions.builder().jsonField("username").build()))
 *         .password(SecretValue.secretsManager("my-bitbucket-creds", SecretsManagerSecretOptions.builder().jsonField("password").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: the credentials are global to a given account in a given region -
 * they are not defined per CodeBuild project.
 * CodeBuild only allows storing a single credential of a given type
 * (GitHub, GitHub Enterprise or BitBucket)
 * in a given account in a given region -
 * any attempt to save more than one will result in an error.
 * You can use the <a href="https://docs.aws.amazon.com/cli/latest/reference/codebuild/list-source-credentials.html"><code>list-source-credentials</code> AWS CLI operation</a>
 * to inspect what credentials are stored in your account.
 * <p>
 * <h2>Test reports</h2>
 * <p>
 * You can specify a test report in your buildspec:
 * <p>
 * <blockquote><pre>
 * Project project = Project.Builder.create(this, "Project")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 // ...
 *                 "reports", Map.of(
 *                         "myReport", Map.of(
 *                                 "files", "**&#47;*",
 *                                 "base-directory", "build/test-results")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will create a new test report group,
 * with the name <code>&lt;ProjectName&gt;-myReport</code>.
 * <p>
 * The project's role in the CDK will always be granted permissions to create and use report groups
 * with names starting with the project's name;
 * if you'd rather not have those permissions added,
 * you can opt out of it when creating the project:
 * <p>
 * <blockquote><pre>
 * Source source;
 * 
 * 
 * Project project = Project.Builder.create(this, "Project")
 *         .source(source)
 *         .grantReportGroupPermissions(false)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, you can specify an ARN of an existing resource group,
 * instead of a simple name, in your buildspec:
 * <p>
 * <blockquote><pre>
 * Source source;
 * 
 * 
 * // create a new ReportGroup
 * ReportGroup reportGroup = new ReportGroup(this, "ReportGroup");
 * 
 * Project project = Project.Builder.create(this, "Project")
 *         .source(source)
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 // ...
 *                 "reports", Map.of(
 *                         reportGroup.getReportGroupArn(), Map.of(
 *                                 "files", "**&#47;*",
 *                                 "base-directory", "build/test-results")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * For a code coverage report, you can specify a report group with the code coverage report group type.
 * <p>
 * <blockquote><pre>
 * Source source;
 * 
 * 
 * // create a new ReportGroup
 * ReportGroup reportGroup = ReportGroup.Builder.create(this, "ReportGroup")
 *         .type(ReportGroupType.CODE_COVERAGE)
 *         .build();
 * 
 * Project project = Project.Builder.create(this, "Project")
 *         .source(source)
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 // ...
 *                 "reports", Map.of(
 *                         reportGroup.getReportGroupArn(), Map.of(
 *                                 "files", "**&#47;*",
 *                                 "base-directory", "build/coverage-report.xml",
 *                                 "file-format", "JACOCOXML")))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you specify a report group, you need to grant the project's role permissions to write reports to that report group:
 * <p>
 * <blockquote><pre>
 * Project project;
 * ReportGroup reportGroup;
 * 
 * 
 * reportGroup.grantWrite(project);
 * </pre></blockquote>
 * <p>
 * The created policy will adjust to the report group type. If no type is specified when creating the report group the created policy will contain the action for the test report group type.
 * <p>
 * For more information on the test reports feature,
 * see the <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/test-reporting.html">AWS CodeBuild documentation</a>.
 * <p>
 * <h3>Report group deletion</h3>
 * <p>
 * When a report group is removed from a stack (or the stack is deleted), the report
 * group will be removed according to its removal policy (which by default will
 * simply orphan the report group and leave it in your AWS account). If the removal
 * policy is set to <code>RemovalPolicy.DESTROY</code>, the report group will be deleted as long
 * as it does not contain any reports.
 * <p>
 * To override this and force all reports to get deleted during report group deletion,
 * enable the <code>deleteReports</code> option as well as setting the removal policy to
 * <code>RemovalPolicy.DESTROY</code>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.*;
 * 
 * 
 * ReportGroup reportGroup = ReportGroup.Builder.create(this, "ReportGroup")
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .deleteReports(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Events</h2>
 * <p>
 * CodeBuild projects can be used either as a source for events or be triggered
 * by events via an event rule.
 * <p>
 * <h3>Using Project as an event target</h3>
 * <p>
 * The <code>aws-cdk-lib/aws-events-targets.CodeBuildProject</code> allows using an AWS CodeBuild
 * project as a AWS CloudWatch event rule target:
 * <p>
 * <blockquote><pre>
 * // start build when a commit is pushed
 * import software.amazon.awscdk.services.codecommit.*;
 * import software.amazon.awscdk.services.events.targets.*;
 * 
 * Repository codeCommitRepository;
 * Project project;
 * 
 * 
 * codeCommitRepository.onCommit("OnCommit", OnCommitOptions.builder()
 *         .target(new CodeBuildProject(project))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Using Project as an event source</h3>
 * <p>
 * To define Amazon CloudWatch event rules for build projects, use one of the <code>onXxx</code>
 * methods:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.targets.*;
 * Function fn;
 * Project project;
 * 
 * 
 * Rule rule = project.onStateChange("BuildStateChange", OnEventOptions.builder()
 *         .target(new LambdaFunction(fn))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>CodeStar Notifications</h2>
 * <p>
 * To define CodeStar Notification rules for Projects, use one of the <code>notifyOnXxx()</code> methods.
 * They are very similar to <code>onXxx()</code> methods for CloudWatch events:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.chatbot.*;
 * 
 * Project project;
 * 
 * 
 * SlackChannelConfiguration target = SlackChannelConfiguration.Builder.create(this, "MySlackChannel")
 *         .slackChannelConfigurationName("YOUR_CHANNEL_NAME")
 *         .slackWorkspaceId("YOUR_SLACK_WORKSPACE_ID")
 *         .slackChannelId("YOUR_SLACK_CHANNEL_ID")
 *         .build();
 * 
 * INotificationRule rule = project.notifyOnBuildSucceeded("NotifyOnBuildSucceeded", target);
 * </pre></blockquote>
 * <p>
 * <h2>Secondary sources and artifacts</h2>
 * <p>
 * CodeBuild Projects can get their sources from multiple places, and produce
 * multiple outputs. For example:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codecommit.*;
 * Repository repo;
 * Bucket bucket;
 * 
 * 
 * Project project = Project.Builder.create(this, "MyProject")
 *         .secondarySources(List.of(Source.codeCommit(CodeCommitSourceProps.builder()
 *                 .identifier("source2")
 *                 .repository(repo)
 *                 .build())))
 *         .secondaryArtifacts(List.of(Artifacts.s3(S3ArtifactsProps.builder()
 *                 .identifier("artifact2")
 *                 .bucket(bucket)
 *                 .path("some/path")
 *                 .name("file.zip")
 *                 .build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Note that the <code>identifier</code> property is required for both secondary sources and
 * artifacts.
 * <p>
 * The contents of the secondary source is available to the build under the
 * directory specified by the <code>CODEBUILD_SRC_DIR_&lt;identifier&gt;</code> environment variable
 * (so, <code>CODEBUILD_SRC_DIR_source2</code> in the above case).
 * <p>
 * The secondary artifacts have their own section in the buildspec, under the
 * regular <code>artifacts</code> one. Each secondary artifact has its own section, beginning
 * with their identifier.
 * <p>
 * So, a buildspec for the above Project could look something like this:
 * <p>
 * <blockquote><pre>
 * Project project = Project.Builder.create(this, "MyProject")
 *         // secondary sources and artifacts as above...
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("cd $CODEBUILD_SRC_DIR_source2", "touch output2.txt"))),
 *                 "artifacts", Map.of(
 *                         "secondary-artifacts", Map.of(
 *                                 "artifact2", Map.of(
 *                                         "base-directory", "$CODEBUILD_SRC_DIR_source2",
 *                                         "files", List.of("output2.txt")))))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Definition of VPC configuration in CodeBuild Project</h3>
 * <p>
 * Typically, resources in an VPC are not accessible by AWS CodeBuild. To enable
 * access, you must provide additional VPC-specific configuration information as
 * part of your CodeBuild project configuration. This includes the VPC ID, the
 * VPC subnet IDs, and the VPC security group IDs. VPC-enabled builds are then
 * able to access resources inside your VPC.
 * <p>
 * For further Information see https://docs.aws.amazon.com/codebuild/latest/userguide/vpc-support.html
 * <p>
 * <strong>Use Cases</strong>
 * VPC connectivity from AWS CodeBuild builds makes it possible to:
 * <p>
 * <ul>
 * <li>Run integration tests from your build against data in an Amazon RDS database that's isolated on a private subnet.</li>
 * <li>Query data in an Amazon ElastiCache cluster directly from tests.</li>
 * <li>Interact with internal web services hosted on Amazon EC2, Amazon ECS, or services that use internal Elastic Load Balancing.</li>
 * <li>Retrieve dependencies from self-hosted, internal artifact repositories, such as PyPI for Python, Maven for Java, and npm for Node.js.</li>
 * <li>Access objects in an Amazon S3 bucket configured to allow access through an Amazon VPC endpoint only.</li>
 * <li>Query external web services that require fixed IP addresses through the Elastic IP address of the NAT gateway or NAT instance associated with your subnet(s).</li>
 * </ul>
 * <p>
 * Your builds can access any resource that's hosted in your VPC.
 * <p>
 * <strong>Enable Amazon VPC Access in your CodeBuild Projects</strong>
 * <p>
 * Pass the VPC when defining your Project, then make sure to
 * give the CodeBuild's security group the right permissions
 * to access the resources that it needs by using the
 * <code>connections</code> object.
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * ApplicationLoadBalancer loadBalancer;
 * 
 * 
 * Vpc vpc = new Vpc(this, "MyVPC");
 * Project project = Project.Builder.create(this, "MyProject")
 *         .vpc(vpc)
 *         .buildSpec(BuildSpec.fromObject(Map.of()))
 *         .build();
 * 
 * project.connections.allowTo(loadBalancer, Port.tcp(443));
 * </pre></blockquote>
 * <p>
 * <h2>Project File System Location EFS</h2>
 * <p>
 * Add support for CodeBuild to build on AWS EFS file system mounts using
 * the new ProjectFileSystemLocation.
 * The <code>fileSystemLocations</code> property which accepts a list <code>ProjectFileSystemLocation</code>
 * as represented by the interface <code>IFileSystemLocations</code>.
 * The only supported file system type is <code>EFS</code>.
 * <p>
 * For example:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2")))
 *         .fileSystemLocations(List.of(FileSystemLocation.efs(EfsFileSystemLocationProps.builder()
 *                 .identifier("myidentifier2")
 *                 .location("myclodation.mydnsroot.com:/loc")
 *                 .mountPoint("/media")
 *                 .mountOptions("opts")
 *                 .build())))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Here's a CodeBuild project with a simple example that creates a project mounted on AWS EFS:
 * <p>
 * <a href="./test/integ.project-file-system-location.ts">Minimal Example</a>
 * <p>
 * <h2>Batch builds</h2>
 * <p>
 * To enable batch builds you should call <code>enableBatchBuilds()</code> on the project instance.
 * <p>
 * It returns an object containing the batch service role that was created,
 * or <code>undefined</code> if batch builds could not be enabled, for example if the project was imported.
 * <p>
 * <blockquote><pre>
 * Source source;
 * 
 * 
 * Project project = Project.Builder.create(this, "MyProject").source(source).build();
 * 
 * if (project.enableBatchBuilds()) {
 *     System.out.println("Batch builds were enabled");
 * }
 * </pre></blockquote>
 * <p>
 * <h2>Timeouts</h2>
 * <p>
 * There are two types of timeouts that can be set when creating your Project.
 * The <code>timeout</code> property can be used to set an upper limit on how long your Project is able to run without being marked as completed.
 * The default is 60 minutes.
 * An example of overriding the default follows.
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .timeout(Duration.minutes(90))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>queuedTimeout</code> property can be used to set an upper limit on how your Project remains queued to run.
 * There is no default value for this property.
 * As an example, to allow your Project to queue for up to thirty (30) minutes before the build fails,
 * use the following code.
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .queuedTimeout(Duration.minutes(30))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Limiting concurrency</h2>
 * <p>
 * By default if a new build is triggered it will be run even if there is a previous build already in progress.
 * It is possible to limit the maximum concurrent builds to value between 1 and the account specific maximum limit.
 * By default there is no explicit limit.
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .concurrentBuildLimit(1)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Visibility</h2>
 * <p>
 * When you can specify the visibility of the project builds. This setting controls whether the builds are publicly readable or remain private.
 * <p>
 * Visibility options:
 * <p>
 * <ul>
 * <li><code>PUBLIC_READ</code>: The project builds are visible to the public.</li>
 * <li><code>PRIVATE</code>: The project builds are not visible to the public.</li>
 * </ul>
 * <p>
 * Examples:
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .visibility(ProjectVisibility.PUBLIC_READ)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Auto retry limit</h2>
 * <p>
 * You can automatically retry your builds in AWS CodeBuild by setting <code>autoRetryLimit</code> property.
 * <p>
 * With auto-retry enabled, CodeBuild will automatically call RetryBuild using the project's service role after a failed build up to a specified limit.
 * <p>
 * For example, if the auto-retry limit is set to two, CodeBuild will call the RetryBuild API to automatically retry your build for up to two additional times.
 * <p>
 * <blockquote><pre>
 * Project.Builder.create(this, "MyProject")
 *         .autoRetryLimit(2)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.codebuild;
